/*
	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#include <qapplication.h>
#include <qlistview.h>
#include "git.h"
#include "mainimpl.h"
#include "domain.h"

using namespace QGit;

void StateInfo::S::clear() {

	sha = fn = dtSha = "";
	isM = allM = false;
	sel = true;
}

StateInfo::S& StateInfo::S::operator=(const StateInfo::S& newState) {

	if (&newState != this) {
		sha = newState.sha;
		fn = newState.fn;
		dtSha = newState.dtSha;
		sel = newState.sel;
		isM = newState.isM;
		allM = newState.allM;
	}
	return *this;
}

bool StateInfo::S::operator==(const StateInfo::S& newState) {

	if (&newState == this)
		return true;

	return (	sha == newState.sha			&&
			fn == newState.fn			&&
			dtSha == newState.dtSha		&&
			sel == newState.sel			&&
			isM == newState.isM			&&
			allM == newState.allM );
}

bool StateInfo::S::operator!=(const StateInfo::S& newState) {

	return !(StateInfo::S::operator==(newState));
}

void StateInfo::clear() {

	newS.clear();
	curS.clear();
}

StateInfo& StateInfo::operator=(const StateInfo& newState) {

	if (&newState != this)
		newS = newState.newS; // curS is mot modified to allow a rollback

	return *this;
}

bool StateInfo::operator==(const StateInfo& newState) {

	if (&newState == this)
		return true;

	return (newS == newState.newS); // compare is made on newS only
}

bool StateInfo::operator!=(const StateInfo& newState) {

	return !(StateInfo::operator==(newState));
}

// ************************* Domain ****************************

Domain::Domain(MainImpl* m, Git* g) : QObject(m), git(g) {

	d = this;
	st.clear();
	queuedSt.clear();
	busy = readyToDrag = dragging = dropping = false;
	popupType = 0;
}

MainImpl* Domain::m() const { return static_cast<MainImpl*>(parent()); }

bool Domain::requestPending() {

	return (!queuedSt.sha().isEmpty() && (st != queuedSt));
}

bool Domain::setReadyToDrag(bool b) {

	readyToDrag = (b && !busy && !dragging && !dropping);
	return readyToDrag;
}

bool Domain::setDragging(bool b) {

	bool dragFinished = (!b && dragging);

	dragging = (b && readyToDrag && !dropping);

	if (dragging)
		readyToDrag = false;

	if (dragFinished)
		flushQueue();

	return dragging;
}

bool Domain::flushQueue() {
// during dragging updateDomain() queues any
// update request so try to flush pending now

	if (!busy && requestPending()) {
		st = queuedSt;
		UPDATE_DOMAIN(d);
		return true;
	}
	return false;
}

void Domain::customEvent(QCustomEvent* e) {

	switch (e->type()) {
	case UPD_DOMAIN_EV:
		update();
		break;
	case MSG_EV:
		if (!busy && !requestPending())
			QApplication::postEvent(m(), new MessageEvent(((MessageEvent*)e)->data()));
		else // waiting for the end of updating
			statusBarRequest = ((MessageEvent*)e)->data();
		break;
	default:
		break;
	}
	QObject::customEvent(e);
}

void Domain::populateState() {

	const Rev* r = git->revLookup(st.sha());
	if (r)
		st.setIsMerge(r->parentsCount() > 1);
}

void Domain::update() {

	if (st.sha().isEmpty()) { // avoid spurious calls
		st.rollBack();
		return;
	}
	if (busy || dragging) {
		queuedSt = st;
		return;
	} else
		queuedSt.setSha("");

	busy = true;

	setReadyToDrag(false); // do not start any drag while updating

	populateState(); // complete any missing state information

	if (doUpdate())
		st.commit();
	else
		st.rollBack();

	busy = false;

	bool nextRequestPending = flushQueue();

	if (!nextRequestPending && !statusBarRequest.isEmpty()) {
		// update status bar when we are sure no more work is pending
		QApplication::postEvent(m(), new MessageEvent(statusBarRequest));
		statusBarRequest = "";
	}
	if (!nextRequestPending && popupType)
		sendPopupEvent();
}

void Domain::sendPopupEvent() {

	// call an async context popup, must be executed
	// after returning to event loop
	DeferredPopupEvent* e = new DeferredPopupEvent(popupData, popupType);
	QApplication::postEvent(m(), e);
	popupType = 0;
}

void Domain::on_contextMenu(const QString& data, int type) {

	popupType = type;
	popupData = data;

	if (busy)
		return; // we are in the middle of an update

	// if list view is already updated pop-up
	// context menu, otherwise it means update()
	// has still not been called, a pop-up request,
	// will be fired up at the end of next update()
	if ((type == POPUP_LIST_EV) && (data != st.sha()))
		return;

	sendPopupEvent();
}
