/*
	Description: file names persistant cache

	Author: Marco Costalba (C) 2005-2006

	Copyright: See COPYING file that comes with this distribution

*/
#include <qfile.h>
#include <qdir.h>
#include <qapplication.h>
#include <qdatastream.h>
#include "cache.h"

using namespace QGit;

bool Cache::update(const QString& workDir, const RevFileMap& rf,
                   const StrVect& dirs, const StrVect& files) {

	if (workDir.isEmpty() || rf.isEmpty())
		return false;

	QString gitDir(workDir + "/.git");
	QString path(gitDir + C_DAT_FILE);
	QString tmpPath(path + BAK_EXT);

	QDir dir;
	if (!dir.exists(gitDir)) {
		qDebug("Git directory not found, unable to save cache");
		return false;
	}
	QFile f(tmpPath);
	if (!f.open(IO_WriteOnly))
		return false;
	qDebug("Saving cache. Please wait...");

	// compress in memory before write to file
	QByteArray data;
	QDataStream stream(data, IO_WriteOnly);

	// Write a header with a "magic number" and a version
	stream << (Q_UINT32)C_MAGIC;
	stream << (Q_INT32)C_VERSION;

	stream << (Q_INT32)dirs.count();
	for (uint i = 0; i < dirs.count(); ++i)
		stream << dirs[i];

	stream << (Q_INT32)files.count();
	for (uint i = 0; i < files.count(); ++i)
		stream << files[i];

	loop(RevFileMap, it, rf) {

		if (it.key() == ZERO_SHA)
			continue;
		if (it.key() == CUSTOM_SHA)
			continue;
		if (it.key().startsWith("A")) // ALL_MERGE_FILES + rev sha
			continue;

		stream << it.key();
		stream << (*it).names;
		stream << (*it).dirs;
		stream << (*it).status;
		stream << (*it).mergeParent;
	}
	qDebug("Compressing data...");
	f.writeBlock(qCompress(data)); // no need to encode with QDataStream compressed data
	f.close();

	if (dir.exists(path)) { // rename C_DAT_FILE + BAK_EXT -> C_DAT_FILE
		if(!dir.remove(path)) {
			qDebug("access denied to " + path);
			dir.remove(tmpPath);
			return false;
		}
	}
	dir.rename(tmpPath, path);
 	qDebug("Done.");
	return true;
}

bool Cache::load(const QString& workDir, RevFileMap& rf,
			  StrVect& dirs, StrVect& files) {

	// remove any old cache file
	QDir dir;
	if (dir.exists(workDir + C_OLD_DAT_FILE))
		dir.remove(workDir + C_OLD_DAT_FILE);

	if (dir.exists(workDir + C_OLD_DAT_FILE_Z))
		dir.remove(workDir + C_OLD_DAT_FILE_Z);

	// check for cache file
	QString gitDir(workDir + "/.git");
	QString path(gitDir + C_DAT_FILE);
	QFile f(path);
	if (!f.exists())
		return true; // no cache file is not an error

	if (!f.open(IO_ReadOnly))
		return false;

	QDataStream* stream = new QDataStream(qUncompress(f.readAll()), IO_ReadOnly);

	Q_UINT32 magic;
	Q_INT32 version;
	Q_INT32 dirsNum, filesNum;
	*stream >> magic;
	*stream >> version;
	if (magic != C_MAGIC || version != C_VERSION) {
		f.close();
		delete stream;
		return false;
	}
	// read the data
	*stream >> dirsNum;
	dirs.resize(dirsNum);
	for (int i = 0; i < dirsNum; ++i)
		*stream >> dirs[i];

	*stream >> filesNum;
	files.resize(filesNum);
	for (int i = 0; i < filesNum; ++i)
		*stream >> files[i];

	QString sha;
	while (!stream->atEnd()) {
		*stream >> sha;
		RevFileMap::Iterator it = rf.insert(sha, RevFile());
		*stream >> (*it).names;
		*stream >> (*it).dirs;
		*stream >> (*it).status;
		*stream >> (*it).mergeParent;
	}
	f.close();
	delete stream;
	return true;
}
