/***************************************************************************
 $RCSfile: graphreport.cpp,v $
                             -------------------
    cvs         : $Id: graphreport.cpp,v 1.11 2006/02/08 15:03:39 cstim Exp $
    begin       : Mon Mar 01 2004
    copyright   : (C) 2004 by Martin Preuss
    email       : martin@libchipcard.de

 ***************************************************************************
 *          Please see toplevel file COPYING for license details           *
 ***************************************************************************/

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "graphreport.h"
#include "greportwidget.h"
#include "editgraphreport.h"
#include "transfinder.h"
#include "refpointer.h"
#include "kbanking.h"

#include <aqbanking/value.h>

#include <qpainter.h>
#include <qcolor.h>
#include <qbrush.h>
#include <qwidget.h>
#include <qmessagebox.h>
#include <qpixmap.h>
#include <qfont.h>
#include <qscrollview.h>



#define BORDER1 30
#define BORDER2 80
#define MARKLEN1 5
#define MARKLEN2 10
#define MARKLEN3 15
#define MARKLEN4 20
#define MINPTSPERUNIT 5

#define COLORWIDTH 24



GraphReport::Units GraphReport::unitsFromString(const char *s) {
  if (strcasecmp(s, "days")==0)
    return UnitsDays;
  else if (strcasecmp(s, "months")==0)
    return UnitsMonths;
  else if (strcasecmp(s, "years")==0)
    return UnitsMonths;
  else
    return UnitsUnknown;
}



const char *GraphReport::unitsToString(GraphReport::Units u) {
  switch(u) {
  case UnitsDays:   return "days";
  case UnitsMonths: return "months";
  case UnitsYears:  return "years";
  default:          return "unknown";
  }
}



GraphReport::GraphType GraphReport::graphTypeFromString(const char *s) {
  if (strcasecmp(s, "points")==0)
    return GraphTypePoints;
  else if (strcasecmp(s, "lines")==0)
    return GraphTypeLines;
  else if (strcasecmp(s, "bars")==0)
    return GraphTypeBars;
  return GraphTypeUnknown;
}



const char *GraphReport::graphTypeToString(GraphReport::GraphType t) {
  switch(t) {
  case GraphTypePoints: return "points";
  case GraphTypeLines:  return "lines";
  case GraphTypeBars:   return "bars";
  default:              return "unknown";
  }
}



GraphReport::ValueMode GraphReport::valueModeFromString(const char *s) {
  if (strcasecmp(s, "normal")==0)
    return ValueModeNormal;
  else if (strcasecmp(s, "absolute")==0)
    return ValueModeAbsolute;
  return ValueModeUnknown;
}



const char *GraphReport::valueModeToString(GraphReport::ValueMode m) {
  switch(m) {
  case ValueModeNormal:   return "normal";
  case ValueModeAbsolute: return "absolute";
  default:                return "unknown";
  }
}



GWEN_TYPE_UINT32 GraphReport::flagsFromDb(GWEN_DB_NODE *db, const char *name){
  GWEN_TYPE_UINT32 fl=0;
  int i;

  for (i=0; ; i++) {
    const char *s;

    s=GWEN_DB_GetCharValue(db, name, i, 0);
    if (!s)
      break;
    if (strcasecmp(s, "mark_x")==0)
      fl|=GRAPHREPORT_FLAGS_MARK_X;
    else if (strcasecmp(s, "mark_y")==0)
      fl|=GRAPHREPORT_FLAGS_MARK_Y;
    else if (strcasecmp(s, "cumulate")==0)
      fl|=GRAPHREPORT_FLAGS_CUMULATE;
    else if (strcasecmp(s, "raster_y")==0)
      fl|=GRAPHREPORT_FLAGS_RASTER_Y;
    else if (strcasecmp(s, "askForData")==0)
      fl|=GRAPHREPORT_FLAGS_ASK_DATA;
    else {
      DBG_ERROR(0, "Unknown flag \"%s\"", s);
    }
  }

  return fl;
}



void GraphReport::flagsToDb(GWEN_DB_NODE *db, const char *name,
			    GWEN_TYPE_UINT32 fl) {
  GWEN_DB_DeleteVar(db, name);
  if (fl & GRAPHREPORT_FLAGS_MARK_X)
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT, name, "mark_x");
  if (fl & GRAPHREPORT_FLAGS_MARK_Y)
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT, name, "mark_y");
  if (fl & GRAPHREPORT_FLAGS_CUMULATE)
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT, name, "cumulate");
  if (fl & GRAPHREPORT_FLAGS_RASTER_Y)
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT, name, "raster_y");
  if (fl & GRAPHREPORT_FLAGS_ASK_DATA)
    GWEN_DB_SetCharValue(db, GWEN_DB_FLAGS_DEFAULT, name, "askForData");
}





GraphReport::DsElement::DsElement(const QDate &date,
                                  const QString &text,
                                  const AB_VALUE *value)
:_date(date)
,_text(text)
,_value(0)
{
  if (value)
    _value=AB_Value_dup(value);
  else
    _value=AB_Value_new(0, 0);
}



GraphReport::DsElement::DsElement(const DsElement &e)
:_date(e._date)
,_text(e._text)
,_value(0)
{
  if (e._value)
    _value=AB_Value_dup(e._value);
  else
    _value=AB_Value_new(0, 0);
}



GraphReport::DsElement::~DsElement() {
  AB_Value_free(_value);
}



const QDate &GraphReport::DsElement::getDate() const {
  return _date;
}



const QString &GraphReport::DsElement::getText() const {
  return _text;
}



const AB_VALUE *GraphReport::DsElement::getValue() const {
  return _value;
}



void GraphReport::DsElement::addValue(const AB_VALUE *v) {
  AB_Value_AddValue(_value, v);
}



void GraphReport::DsElement::reset() {
  AB_Value_free(_value);
  _value=AB_Value_new(0, 0);
}




GraphReport::DsDay::DsDay(const QDate &date,
                          const QString &text)
:_me(DsElement(date, text))
{
}



GraphReport::DsDay::~DsDay() {
  reset();
}



const GraphReport::DsElement&
GraphReport::DsDay::getItsElement() const {
  return _me;
}



void GraphReport::DsDay::addElement(GraphReport::DsElement *e) {
  _me.addValue(e->getValue());
  _elements.push_back(e);
}



void GraphReport::DsDay::getMinMaxValues(double &minValue, double &maxValue){
  std::list<DsElement*>::iterator it;

  for (it=_elements.begin(); it!=_elements.end(); it++) {
    double d;

    d=AB_Value_GetValue((*it)->getValue());
    if (d<minValue)
      minValue=d;
    if (d>maxValue)
      maxValue=d;
  }
}



const std::list<GraphReport::DsElement*>&
GraphReport::DsDay::getElements() const {
  return _elements;
}



void GraphReport::DsDay::reset() {
  std::list<DsElement*>::iterator it;

  for (it=_elements.begin(); it!=_elements.end(); it++)
    delete (*it);
  _elements.clear();
}




GraphReport::DsMonth::DsMonth(const QDate &date,
                              const QString &text)
:_me(date, text)
{
}



GraphReport::DsMonth::~DsMonth() {
  reset();
}



const GraphReport::DsElement &GraphReport::DsMonth::getItsElement() const {
  return _me;
}



void GraphReport::DsMonth::addElement(GraphReport::DsElement *e) {
  DsDay *day;

  day=getDay(e->getDate());
  assert(day);
  _me.addValue(e->getValue());
  day->addElement(e);
}



GraphReport::DsDay *GraphReport::DsMonth::findDay(const QDate &date) {
  std::list<DsDay*>::iterator it;

  for (it=_days.begin(); it!=_days.end(); it++) {
    if ((*it)->getItsElement().getDate()==date)
      return *it;
  }

  return 0;
}



GraphReport::DsDay *GraphReport::DsMonth::getDay(const QDate &date) {
  DsDay *day;

  day=findDay(date);
  if (!day) {
    day=new DsDay(date);
    addDay(day);
  }
  return day;
}



void GraphReport::DsMonth::addDay(GraphReport::DsDay *d) {
  _days.push_back(d);
}



const std::list<GraphReport::DsDay*> &GraphReport::DsMonth::getDays() const {
  return _days;
}



void GraphReport::DsMonth::reset() {
  std::list<DsDay*>::iterator it;

  for (it=_days.begin(); it!=_days.end(); it++)
    delete *it;
  _days.clear();
}



void GraphReport::DsMonth::getMinMaxValues(double &minValue,
					   double &maxValue){
  std::list<DsDay*>::iterator it;

  for (it=_days.begin(); it!=_days.end(); it++)
    (*it)->getMinMaxValues(minValue, maxValue);
}



void GraphReport::DsMonth::getMinMaxDayValues(double &minValue,
					      double &maxValue){
  std::list<DsDay*>::iterator it;

  for (it=_days.begin(); it!=_days.end(); it++) {
    const DsElement &e=(*it)->getItsElement();
    double d;

    d=AB_Value_GetValue(e.getValue());
    if (d<minValue)
      minValue=d;
    if (d>maxValue)
      maxValue=d;
  }
}




GraphReport::DsYear::DsYear(const QDate &date,
                            const QString &text)
:_me(date, text)
{
}



GraphReport::DsYear::~DsYear() {
  reset();
}



const GraphReport::DsElement &GraphReport::DsYear::getItsElement() const {
  return _me;
}



void GraphReport::DsYear::addElement(GraphReport::DsElement *e) {
  DsMonth *m;

  m=getMonth(e->getDate());
  _me.addValue(e->getValue());
  m->addElement(e);
}



GraphReport::DsMonth *GraphReport::DsYear::findMonth(const QDate &date) {
  std::list<DsMonth*>::iterator it;
  int y;
  int m;

  y=date.year();
  m=date.month();
  for (it=_months.begin(); it!=_months.end(); it++) {
    if ((*it)->getItsElement().getDate().year()==y &&
        (*it)->getItsElement().getDate().month()==m)
      return *it;
  }

  return 0;
}



GraphReport::DsMonth *GraphReport::DsYear::getMonth(const QDate &date) {
  DsMonth *m;

  m=findMonth(date);
  if (!m) {
    m=new DsMonth(QDate(date.year(), date.month(), 1));
    addMonth(m);
  }

  return m;
}



void GraphReport::DsYear::addMonth(GraphReport::DsMonth *m) {
  _months.push_back(m);
}



const std::list<GraphReport::DsMonth*>&
GraphReport::DsYear::getMonths() const {
  return _months;
}



GraphReport::DsDay *GraphReport::DsYear::findDay(const QDate &date) {
  DsMonth *m;

  m=findMonth(date);
  if (m)
    return m->findDay(date);
  return 0;
}



GraphReport::DsDay *GraphReport::DsYear::getDay(const QDate &date) {
  DsMonth *m;

  m=getMonth(date);
  return m->getDay(date);
}



void GraphReport::DsYear::addDay(GraphReport::DsDay *d) {
  DsMonth *m;

  m=getMonth(d->getItsElement().getDate());
  m->addDay(d);
}



void
GraphReport::DsYear::getMinMaxValues(double &minValue, double &maxValue) {
  std::list<DsMonth*>::iterator it;

  for (it=_months.begin(); it!=_months.end(); it++)
    (*it)->getMinMaxValues(minValue, maxValue);
}



void
GraphReport::DsYear::getMinMaxDayValues(double &minValue, double &maxValue){
  std::list<DsMonth*>::iterator it;

  for (it=_months.begin(); it!=_months.end(); it++)
    (*it)->getMinMaxDayValues(minValue, maxValue);
}



void
GraphReport::DsYear::getMinMaxMonthValues(double &minValue, double &maxValue){
  std::list<DsMonth*>::iterator it;

  for (it=_months.begin(); it!=_months.end(); it++) {
    const DsElement &e=(*it)->getItsElement();
    double d;

    d=AB_Value_GetValue(e.getValue());
    if (d<minValue)
      minValue=d;
    if (d>maxValue)
      maxValue=d;
  }
}



void GraphReport::DsYear::reset() {
  std::list<DsMonth*>::iterator it;

  for (it=_months.begin(); it!=_months.end(); it++)
    delete *it;
  _months.clear();
}




GraphReport::DsSet::DsSet(const QString &text)
:_me(QDate::currentDate(), text)
{
}



GraphReport::DsSet::~DsSet() {
  reset();
}


const GraphReport::DsElement &GraphReport::DsSet::getItsElement() const {
  return _me;
}



void GraphReport::DsSet::addElement(GraphReport::DsElement *e) {
  DsYear *y;

  if (!_firstDate.isValid() ||
      (e->getDate()<_firstDate))
    _firstDate=e->getDate();

  if (!_lastDate.isValid() ||
      (e->getDate()>_lastDate))
    _lastDate=e->getDate();

  _me.addValue(e->getValue());

  y=getYear(e->getDate());
  y->addElement(e);
}



const QDate &GraphReport::DsSet::getFirstDate() const {
  return _firstDate;
}



const QDate &GraphReport::DsSet::getLastDate() const {
  return _lastDate;
}



void
GraphReport::DsSet::getMinMaxValues(double &minValue, double &maxValue){
  std::list<DsYear*>::iterator it;

  for (it=_years.begin(); it!=_years.end(); it++)
    (*it)->getMinMaxValues(minValue, maxValue);
}



void
GraphReport::DsSet::getMinMaxDayValues(double &minValue, double &maxValue){
  std::list<DsYear*>::iterator it;

  for (it=_years.begin(); it!=_years.end(); it++)
    (*it)->getMinMaxDayValues(minValue, maxValue);
}



void
GraphReport::DsSet::getMinMaxMonthValues(double &minValue, double &maxValue){
  std::list<DsYear*>::iterator it;

  for (it=_years.begin(); it!=_years.end(); it++)
    (*it)->getMinMaxMonthValues(minValue, maxValue);
}



void
GraphReport::DsSet::getMinMaxYearValues(double &minValue, double &maxValue){
  std::list<DsYear*>::iterator it;

  for (it=_years.begin(); it!=_years.end(); it++) {
    const DsElement &e=(*it)->getItsElement();
    double d;

    d=AB_Value_GetValue(e.getValue());
    if (d<minValue)
      minValue=d;
    if (d>maxValue)
      maxValue=d;
  }
}



GraphReport::DsYear *GraphReport::DsSet::findYear(const QDate &date) {
  std::list<DsYear*>::iterator it;
  int iy;

  iy=date.year();
  for (it=_years.begin(); it!=_years.end(); it++) {
    if ((*it)->getItsElement().getDate().year()==iy)
      return *it;
  }

  return 0;
}



GraphReport::DsYear *GraphReport::DsSet::getYear(const QDate &date) {
  DsYear *y;

  y=findYear(date);
  if (!y) {
    y=new DsYear(date);
    addYear(y);
  }

  return y;
}



void GraphReport::DsSet::addYear(GraphReport::DsYear *y) {
  _years.push_back(y);
}



const std::list<GraphReport::DsYear*> GraphReport::DsSet::getYears() const {
  return _years;
}



GraphReport::DsMonth *GraphReport::DsSet::findMonth(const QDate &date) {
  DsYear *y;

  y=findYear(date);
  if (y)
    return y->findMonth(date);
  return 0;
}



GraphReport::DsMonth *GraphReport::DsSet::getMonth(const QDate &date) {
  DsYear *y;

  y=getYear(date);
  return y->getMonth(date);
}



void GraphReport::DsSet::addMonth(GraphReport::DsMonth *m) {
  DsYear *y;

  y=getYear(m->getItsElement().getDate());
  y->addMonth(m);
}



GraphReport::DsDay *GraphReport::DsSet::findDay(const QDate &date) {
  DsYear *y;

  y=findYear(date);
  if (y)
    return y->findDay(date);
  return 0;
}



GraphReport::DsDay *GraphReport::DsSet::getDay(const QDate &date) {
  DsYear *y;

  y=getYear(date);
  return y->getDay(date);
}



void GraphReport::DsSet::addDay(GraphReport::DsDay *d) {
  DsYear *y;

  y=getYear(d->getItsElement().getDate());
  y->addDay(d);
}



void GraphReport::DsSet::reset() {
  std::list<DsYear*>::iterator it;

  for (it=_years.begin(); it!=_years.end(); it++)
    delete *it;
  _firstDate=QDate();
  _lastDate=QDate();
}

















GraphReport::GraphReport(KBanking *app)
  :Report(app, "GraphReport"){

}



GraphReport::~GraphReport() {
}



QString GraphReport::shortDescription() {
  return tr("Graphical Reports");
}



QString GraphReport::longDescription() {
  return "";
}



bool GraphReport::initProfile(GWEN_DB_NODE *dbProfile, QWidget *parent) {
  EditGraphReport w(app(), dbProfile, parent);

  w.setCaption(QWidget::tr("Create new profile"));
  if (w.exec()!=QDialog::Accepted)
    return false;

  return true;
}



bool GraphReport::editProfile(GWEN_DB_NODE *dbProfile, QWidget *parent) {
  EditGraphReport w(app(), dbProfile, parent);

  w.setCaption(QWidget::tr("Edit profile"));
  if (w.exec()!=QDialog::Accepted)
    return false;

  return true;
}



bool GraphReport::useProfile(GWEN_DB_NODE *dbProfile, QWidget *parent) {
  const char *s;

  _minValue=0;
  _maxValue=1;
  _firstDate=QDate();
  _lastDate=QDate();

  s=GWEN_DB_GetCharValue(dbProfile, "units", 0, "months");
  _units=unitsFromString(s);
  if (_units==UnitsUnknown) {
    QMessageBox::critical(parent,
                          tr("Settings Error"),
                          tr("Invalid value for units type in profile."),
			  tr("Dismiss"), QString::null);
    return false;
  }

  s=GWEN_DB_GetCharValue(dbProfile, "valueMode", 0, "normal");
  _valueMode=valueModeFromString(s);
  if (_valueMode==ValueModeUnknown) {
    QMessageBox::critical(parent,
                          tr("Settings Error"),
                          tr("Invalid value mode in profile."),
			  tr("Dismiss"), QString::null);
    return false;
  }

  s=GWEN_DB_GetCharValue(dbProfile, "graphType", 0, "lines");
  _graphType=graphTypeFromString(s);
  if (_graphType==GraphTypeUnknown) {
    QMessageBox::critical(parent,
                          tr("Settings Error"),
                          tr("Invalid graph type in profile."),
			  tr("Dismiss"), QString::null);
    return false;
  }

  _flags=flagsFromDb(dbProfile, "flags");

  _width=GWEN_DB_GetIntValue(dbProfile, "width", 0, 800);
  _height=GWEN_DB_GetIntValue(dbProfile, "height", 0, 600);
  if (_width<BORDER2*3) {
    QMessageBox::critical(parent,
			  tr("Settings Error"),
			  tr("Width in profile is too small."),
			  tr("Dismiss"), QString::null);
    return false;
  }
  if (_height<BORDER2*3) {
    QMessageBox::critical(parent,
			  tr("Settings Error"),
			  tr("Height in profile is too small."),
			  tr("Dismiss"), QString::null);
    return false;
  }

  /* get transactions */
  std::list<RefPointer<Transaction> > tl;
  if (_flags & GRAPHREPORT_FLAGS_ASK_DATA) {
    if (!TransactionFinder::selectTransactions(app(),
                                               &tl,
                                               tr("Select Transactions"),
                                               TRANSFINDER_FLAGS_USE_CATEGORY |
                                               TRANSFINDER_FLAGS_EXT_PAYEES,
                                               parent)) {
      return false;
    }
  }
  else {
    GWEN_DB_NODE *dbRule;

    dbRule=GWEN_DB_GetGroup(dbProfile, GWEN_PATH_FLAGS_NAMEMUSTEXIST, "rule");
    if (dbRule) {
      if (!app()->getMatchingTransactions(dbRule, tl)) {
        QMessageBox::critical(parent,
                              tr("Data Error"),
                              tr("No transactions match the "
                                 "predefined rule."),
                              tr("Dismiss"), QString::null);
        return false;
      }
    }
    else {
      QMessageBox::critical(parent,
                            tr("Settings Error"),
                            tr("No transaction matcher rule defined."),
                            tr("Dismiss"), QString::null);
      return false;
    }
  }

  if (tl.empty()) {
    QMessageBox::critical(parent,
                          tr("Data Error"),
                          tr("No transactions selected."),
                          tr("Dismiss"), QString::null);
    return false;
  }
  else {
    std::list<RefPointer<Transaction> >::iterator it;
    DsSet normalSet(tr("Data Set"));
    DsSet cumulatedSet(tr("Cumulated Data Set"));
    DsSet *ds;

    /* transform transaction list to data set */
    for (it=tl.begin(); it!=tl.end(); it++) {
      const GWEN_TIME *ti;

      ti=(*it).ref().getDate();
      if (!ti)
	ti=(*it).ref().getValutaDate();
      if (ti) {
	int y, m, d;
	QString qs;
	DsElement *e;
        AB_VALUE *v;

	if (GWEN_Time_GetBrokenDownDate(ti, &d, &m, &y)) {
	  DBG_ERROR(0, "Invalid date");
          abort();
	}
	if (!((*it).ref().getPurpose().empty()))
	  qs=QString::fromUtf8((*it).ref().getPurpose().front().c_str());
	QDate qd(y, m+1, d);
	v=AB_Value_dup((*it).ref().getValue());

	switch(_valueMode) {
	case ValueModeAbsolute:
	  if (AB_Value_IsNegative(v))
            AB_Value_Negate(v);
	  break;
	case ValueModeNormal:
	default:
	  break;
	}
	e=new DsElement(qd, qs, v);
        AB_Value_free(v);
	normalSet.addElement(e);
      }
    } // for
    tl.clear();

    if (_flags & GRAPHREPORT_FLAGS_CUMULATE) {
      if (_units==UnitsDays) {
	double cumVal=0;
	QDate d=normalSet.getFirstDate();
  
	while(d<=normalSet.getLastDate()) {
	  QDate dNext=d.addDays(1);
	  DsDay *dy;
	  DsElement *e;
	  AB_VALUE *v;
  
	  dy=normalSet.getDay(d);
	  e=new DsElement(d);

	  cumVal+=AB_Value_GetValue(dy->getItsElement().getValue());
	  v=AB_Value_new(cumVal, 0);
	  e->addValue(v);
	  AB_Value_free(v);
	  cumulatedSet.addElement(e);
	  d=dNext;
	}
	addDataSet(cumulatedSet);
	ds=&cumulatedSet;
      }
      else if (_units==UnitsMonths) {
	double cumVal=0;
	QDate d=normalSet.getFirstDate();
  
	while(d<=normalSet.getLastDate()) {
	  QDate dNext=d.addMonths(1);
	  DsMonth *dm;
	  DsElement *e;
	  AB_VALUE *v;

	  dm=normalSet.getMonth(d);
	  e=new DsElement(d, dm->getItsElement().getText());
  
	  cumVal+=AB_Value_GetValue(dm->getItsElement().getValue());
	  v=AB_Value_new(cumVal, 0);
	  e->addValue(v);
	  AB_Value_free(v);
	  cumulatedSet.addElement(e);
	  d=dNext;
	}
	addDataSet(cumulatedSet);
	ds=&cumulatedSet;
      }
      else {
        DBG_ERROR(0, "Cumulated flag not supported with these units");
	ds = 0;
      }
    }
    else {
      // add data set
      addDataSet(normalSet);
      ds=&normalSet;
    }

    // calculate some variables
    setup();

    // actually draw the graph
    QPainter p;
  
    RefPointer<QPixmap> pm;
    pm=new QPixmap(_width, _height
#if (QT_VERSION < 0x040000)
		   // qt4 doesn't have this argument here anymore
		   , COLORWIDTH
#endif
	);
    pm.ref().fill();
    if (!p.begin(pm.ptr())) {
      DBG_ERROR(0, "Could not begin to paint...");
      return false;
    }
  
    // draw data
    drawDataSet(&p, *ds);
    // draw coordinate system
    drawCoordSystem(&p);
    p.end();

    QScrollView *sw;

    sw=new QScrollView(parent, 0,
                       Qt::WType_TopLevel |
                       Qt::WStyle_NormalBorder |
                       Qt::WDestructiveClose);
    sw->resize(800, 600);
    GraphicReportWidget *w;
    w=new GraphicReportWidget(pm, parent);
    w->resize(pm.ref().width(), pm.ref().height());
    sw->addChild(w);
    sw->show();
  }

  return true;
}



void GraphReport::addDataSet(GraphReport::DsSet &ds) {
  if (!_firstDate.isValid() ||
      ds.getFirstDate()<_firstDate)
    _firstDate=ds.getFirstDate();
  if (!_lastDate.isValid() ||
      _lastDate<ds.getLastDate())
    _lastDate=ds.getLastDate();

  switch(_units) {
  case UnitsDays:
    ds.getMinMaxDayValues(_minValue, _maxValue);
    break;
  case UnitsYears:
    ds.getMinMaxYearValues(_minValue, _maxValue);
    break;
  case UnitsMonths:
  default:
    ds.getMinMaxMonthValues(_minValue, _maxValue);
    break;
  }

  DBG_ERROR(0, "Date range: %04d/%02d/%02d to %04d/%02d/%02d",
	    _firstDate.year(), _firstDate.month(), _firstDate.day(),
	    _lastDate.year(), _lastDate.month(), _lastDate.day());
  DBG_ERROR(0, "MinValue: %lf, MaxValue: %lf, Width: %d Height: %d",
            _minValue, _maxValue, _width, _height);
}



void GraphReport::setup() {
  double d;

  switch(_units) {
  case UnitsDays:
    _unitCount=_firstDate.daysTo(_lastDate)+1;
    break;
  case UnitsYears:
    _unitCount=(_lastDate.year()-_firstDate.year())+1;
    break;
  case UnitsMonths:
  default:
    _unitCount=((((_lastDate.year()-_firstDate.year())*12)+_lastDate.month())
                -_firstDate.month())+1;
    break;
  }

  _ptsPerUnit=((double)(_width-(2*BORDER2)))/((double)_unitCount);
  _nullX=BORDER2;

  // round up values
  if (1) {
    char numbuf[32];
    long d;
    int i;
    char j;
    int k;
    long minVal;
    long maxVal;
    std::string st;

    if (_minValue<0) {
      snprintf(numbuf, sizeof(numbuf), "%ld", ((long)(-_minValue)));
      j=numbuf[0]-'0'+1;
      k=strlen(numbuf);
      snprintf(numbuf, sizeof(numbuf), "%d", j);
      st=numbuf;
      for (i=1; i<k; i++)
        st+='0';
      sscanf(st.c_str(), "%ld", &d);
      minVal=-d;
    }
    else
      minVal=0;

    snprintf(numbuf, sizeof(numbuf), "%ld", ((long)(_maxValue)));
    j=numbuf[0]-'0'+1;
    k=strlen(numbuf);
    snprintf(numbuf, sizeof(numbuf), "%d", j);
    st=numbuf;
    for (i=1; i<k; i++)
      st+='0';
    sscanf(st.c_str(), "%ld", &d);
    maxVal=d;
    DBG_ERROR(0, "Values: _minValue=%lf (%ld), _maxValue=%lf (%ld)",
              _minValue, minVal, _maxValue, maxVal);

    if (labs(maxVal)>labs(minVal))
      d=labs(maxVal);
    else
      d=labs(minVal);

    snprintf(numbuf, sizeof(numbuf), "%ld", d);
    k=strlen(numbuf);
    if (k<2)
      d=1;
    else {
      j=numbuf[0]-'0';
      numbuf[0]='1';
      if (j>2) { /* will give more than 2 steps, so take this value */
	for (i=1; i<k; i++)
	  numbuf[i]='0';
      }
      else { /* otherwise go down to next step below */
	for (i=1; i<k-1; i++)
	  numbuf[i]='0';
	numbuf[k-1]=0;
      }
      sscanf(numbuf, "%ld", &d);
    }
    _minValue=(double)minVal;
    _maxValue=(double)maxVal;
    _yStep=(double)d;
    DBG_ERROR(0, "Step: %lf (%ld)", _yStep, d);
  }

  if (_minValue>0)
    d=_maxValue;
  else
    d=_maxValue-_minValue;

  _stretchY=d/(_height-(2*BORDER2));
  _nullY=((int)(_maxValue/_stretchY))+BORDER2;

  DBG_ERROR(0, "NullY: %d StretchY: %lf", _nullY, _stretchY);

  switch(_graphType) {
  case GraphTypePoints:
    _unitWidth=2;
    break;
  case GraphTypeBars:
    _unitWidth=(int)(0.75*_ptsPerUnit);
    break;
  case GraphTypeLines:
  default:
    _unitWidth=2;
    break;
  }

  if (_unitWidth<1 || _unitWidth>(int)_ptsPerUnit) {
    _unitWidth=1;
    _unitOffset=0;
  }
  else {
    _unitOffset=((int)_ptsPerUnit-_unitWidth)/2;
  }

  DBG_ERROR(0, "Units: %s, graphType: %s, valueMode: %s",
            unitsToString(_units),
            graphTypeToString(_graphType),
            valueModeToString(_valueMode));
  DBG_ERROR(0, "UnitCount: %d", _unitCount);

  DBG_ERROR(0, "_ptsPerUnit: %lf, _unitWidth: %d, _unitOffset: %d",
            _ptsPerUnit, _unitWidth, _unitOffset);

}



int GraphReport::dateToX(const QDate &d) {
  int u;

  switch(_units) {
  case UnitsDays:
    u=_firstDate.daysTo(d);
    break;
  case UnitsYears:
    u=(d.year()-_firstDate.year());
    break;
  case UnitsMonths:
  default:
    u=((((d.year()-_firstDate.year())*12)+d.month())
       -_firstDate.month());
    break;
  }

  return (int)(((double)u)*_ptsPerUnit)+_nullX;
}



int GraphReport::valueToY(double d) {
  return _nullY-((int)(d/_stretchY));
}



void GraphReport::drawCoordSystem(QPainter *qp) {
  QPen pen=QPen(QColor("black"), 2, Qt::SolidLine);
  QPen penDotted=QPen(QColor("gray"), 1, Qt::DotLine);
  QFontMetrics fm(qp->font());
  int th=fm.height();
  QString yTitle=tr("Value");
  int twY=fm.width(yTitle);
  int y;

  qp->setPen(pen);

  // draw Y
  qp->drawLine(BORDER2, BORDER2, BORDER2, _height-BORDER2);

  // draw X
  qp->drawLine(BORDER2-MARKLEN2, _nullY, _width-BORDER2, _nullY);

  qp->drawText(BORDER2-twY, BORDER2-th-2, yTitle);

  // handle Y line
  if (_flags & GRAPHREPORT_FLAGS_MARK_Y) {
    double val;
    int lastY;

    // go up
    val=_yStep;
    lastY=valueToY(0);
    while(val<=_maxValue) {
      y=valueToY(val);
      if (_flags & GRAPHREPORT_FLAGS_RASTER_Y) {
	qp->setPen(penDotted);
	qp->drawLine(_nullX, y, _width-BORDER2, y);
      }
      if ((lastY-y)>th) {
	qp->setPen(pen);
	qp->drawLine(_nullX-MARKLEN1, y, _nullX+MARKLEN1, y);
	QString t=QString::number(val, 'f', 2);
	qp->drawText(_nullX-MARKLEN2-fm.width(t), y, t);
      }
      val+=_yStep;
    }

    if (_minValue<0) {
      // go down also
      val=-_yStep;
      lastY=valueToY(0);
      while(val>=_minValue) {
	y=valueToY(val);
	if (_flags & GRAPHREPORT_FLAGS_RASTER_Y) {
	  qp->setPen(penDotted);
	  qp->drawLine(_nullX, y, _width-BORDER2, y);
	}
	if ((y-lastY)>th) {
	  qp->setPen(pen);
	  qp->drawLine(_nullX-MARKLEN1, y, _nullX+MARKLEN1, y);
	  QString t=QString::number(val, 'f', 2);
	  qp->drawText(_nullX-MARKLEN2-fm.width(t), y, t);
	}
	val-=_yStep;
      }
    }
  }

  // mark X line
  if (_flags & GRAPHREPORT_FLAGS_MARK_X) {
    if (_units==UnitsDays)
      drawCoordXMarksDays(qp);
    else if (_units==UnitsMonths)
      drawCoordXMarksMonths(qp);
    else {
      DBG_ERROR(0, "Not days");
    }
  }

}



void GraphReport::drawCoordXMarksDays(QPainter *qp) {
  QFontMetrics fm(qp->font());
  int th=fm.height();
  int twDay=fm.width("2006/01/01");
  int twWeek=fm.width("2006/53");
  int twMonth=fm.width("2006/01");
  int twYear=fm.width("2006");
  int y;
  int lastMonth;
  QDate d=_firstDate;
  int lastX;
  int lastDayX;
  int lastWeekX;
  int lastMonthX;
  int lastYearX;
  int labelWhat;
  QString qs;

  if (_ptsPerUnit>th)
    labelWhat=1; // days
  else if ((_ptsPerUnit*7)>twWeek)
    labelWhat=2; // weeks
  else if ((_ptsPerUnit*28)>twMonth)
    labelWhat=3; // months
  else if ((_ptsPerUnit*366)>twYear)
    labelWhat=4; // years
  else
    labelWhat=0; // nothing

  DBG_ERROR(0, "LabelWhat: %d (_ptsPrtUnit=%lf)", labelWhat, _ptsPerUnit);

  y=valueToY(0);
  lastX=lastDayX=lastWeekX=lastMonthX=lastYearX=dateToX(_firstDate);
  lastMonth=d.month();

  while(d<=_lastDate) {
    QDate dNext=d.addDays(1);
    int xBegin;
    int xEnd;

    xBegin=dateToX(d);
    xEnd=dateToX(dNext);
    if (d.dayOfWeek()==7 || (labelWhat==2 && dNext>_lastDate)) {
      if ((xEnd-lastWeekX)>MINPTSPERUNIT) {
	DBG_ERROR(0, "Week %d", d.weekNumber());
	qp->drawLine(xEnd, y+MARKLEN2, xEnd, y-MARKLEN2);
	if (labelWhat==2) {
          QString lbl=
            QString("%1/%2")
            .arg(d.weekNumber(), 2)
            .arg(d.year(), 4);
          int tw=fm.width(lbl);

          if ((xEnd-lastWeekX)>tw) {
            int r;

            r=(xEnd-lastWeekX-tw)/2;
            qp->drawText(lastWeekX+r,
                         y+MARKLEN3+th, lbl);
          }
	  else if ((xEnd-lastWeekX)>th) {
	    int r;
  
	    r=(xEnd-lastWeekX-th)/2;
	    qp->save();
	    qp->rotate(90);
	    qp->drawText(y+MARKLEN3,
			 -(xBegin+r),
			 QString("%1/%2")
			 .arg(d.weekNumber(), 2)
			 .arg(d.year(), 4));
	    qp->restore();
	  }
	  else {
	    DBG_ERROR(0, "Too small to show week (%d->%d)",
		      lastWeekX, xEnd);
	  }
	}
      }
      lastWeekX=xEnd;
    }
    if (dNext.month()!=d.month() || (labelWhat==3 && dNext>_lastDate)) {
      if ((xEnd-lastMonthX)>MINPTSPERUNIT) {
	qp->drawLine(xEnd, y+MARKLEN3, xEnd, y-MARKLEN3);

	if (labelWhat==3) {
	  if ((xEnd-lastMonthX)>twMonth) {
	    int r;

	    r=(xEnd-lastMonthX-twMonth)/2;
	    qp->drawText(lastMonthX+r,
			 y+MARKLEN3+th,
			 QString("%1/%2").arg(d.month(), 2).arg(d.year(), 4));
	  }
	  else if ((xEnd-lastMonthX)>th) {
	    int r;

	    r=(xEnd-lastMonthX-th)/2;
	    qp->save();
	    qp->rotate(90);
	    qp->drawText(y+MARKLEN3,
			 -(lastMonthX+r),
			 QString("%1/%2").arg(d.month(), 2).arg(d.year(), 4));
	    qp->restore();
	  }
	  else {
	    DBG_ERROR(0, "Too small to show date");
	  }
	}
      }
      lastMonthX=xEnd;
    }

    if (dNext.month()<d.month() || (labelWhat==4 && dNext>_lastDate)) {
      if ((xEnd-lastYearX)>MINPTSPERUNIT) {
	qp->drawLine(xEnd, y+MARKLEN4, xEnd, y-MARKLEN4);

	if (labelWhat==4) {
	  if ((xEnd-lastYearX)>twYear) {
	    int r;

	    r=(xEnd-lastYearX-twYear)/2;
	    qp->drawText(lastYearX+r,
			 y+MARKLEN3+th,
			 QString("%1").arg(d.year(), 4));
	  }
	  else if ((xEnd-lastYearX)>th) {
	    int r;

	    r=(xEnd-lastYearX-th)/2;
	    qp->save();
	    qp->rotate(90);
	    qp->drawText(y+MARKLEN3,
			 -(lastYearX+r),
			 QString("%1").arg(d.year(), 4));
	    qp->restore();
	  }
	  else {
	    DBG_ERROR(0, "Too small to show date");
	  }
	}
      }
      lastYearX=xEnd;
    }

    if ((xEnd-lastDayX)>MINPTSPERUNIT) {
      qp->drawLine(xEnd, y+MARKLEN1, xEnd, y-MARKLEN1);
      if (labelWhat==1) {
	if ((xEnd-xBegin)>twDay) {
          int r;
  
	  r=(xEnd-xBegin-twDay)/2;
	  qp->drawText(xBegin+r,
		       y+MARKLEN3+th,
		       d.toString("dd.MM.yyyy"));
        }
        else if ((xEnd-xBegin)>th) {
          int r;
          QString ds=d.toString("dd.MM.yyyy");

          DBG_ERROR(0, "Day: %s", ds.latin1());
	  r=(xEnd-xBegin-th)/2;
	  qp->save();
	  qp->rotate(90);

          qp->drawText(y+MARKLEN3,
		       -(xBegin+r),
                       ds);
	  qp->restore();
	}
	else {
	  DBG_ERROR(0, "Too small to show date");
	}
      }
    }
    lastDayX=xEnd;
    lastX=xEnd;

    lastMonth=d.month();

    d=dNext;
  }

  switch(labelWhat) {
  case 1: // days
    qs=tr("Days");
    break;
  case 2: // weeks
    qs=tr("Weeks");
    break;
  case 3: // months
    qs=tr("Months");
    break;
  case 4: // years
    qs=tr("Years");
    break;
  default:
    break;
  }

  if (!qs.isEmpty()) {
    qp->drawText(_width-BORDER2+2, _nullY+(th/2), qs);
  }
}



void GraphReport::drawCoordXMarksMonths(QPainter *qp) {
  QFontMetrics fm(qp->font());
  int th=fm.height();
  int twWeek=fm.width("2006/53");
  int twMonth=fm.width("2006/01");
  int twYear=fm.width("2006");
  int y;
  int lastMonth;
  QDate d=_firstDate;
  int lastX;
  int lastDayX;
  int lastWeekX;
  int lastMonthX;
  int labelWhat;
  QString qs;

  if (_ptsPerUnit>th)
    labelWhat=3; // months
  else if ((_ptsPerUnit*12)>twYear)
    labelWhat=4; // years
  else
    labelWhat=0; // nothing

  DBG_ERROR(0, "LabelWhat: %d (_ptsPrtUnit=%lf)", labelWhat, _ptsPerUnit);

  y=valueToY(0);
  lastX=lastDayX=lastWeekX=lastMonthX=dateToX(_firstDate);
  lastMonth=d.month();

  while(d<=_lastDate) {
    QDate dNext=d.addMonths(1);
    int xBegin;
    int xEnd;

    xBegin=dateToX(d);
    xEnd=dateToX(dNext);

    if (dNext.month()!=lastMonth || (labelWhat==3 && dNext>_lastDate)) {
      if ((xEnd-lastMonthX)>MINPTSPERUNIT) {
	qp->drawLine(xEnd, y+MARKLEN3, xEnd, y-MARKLEN3);

	if (labelWhat==3) {
	  if ((xEnd-lastMonthX)>twWeek) {
	    int r;

	    r=(xEnd-lastMonthX-twMonth)/2;
	    qp->drawText(lastMonthX+r,
			 y+MARKLEN3+th,
			 QString("%1/%2").arg(d.month(), 2).arg(d.year(), 4));
	  }
	  else if ((xEnd-lastWeekX)>th) {
	    int r;

	    r=(xEnd-lastMonthX-th)/2;
	    qp->save();
	    qp->rotate(90);
	    qp->drawText(y+MARKLEN3,
			 -(lastMonthX+r),
			 QString("%1/%2").arg(d.month(), 2).arg(d.year(), 4));
	    qp->restore();
	  }
	  else {
	    DBG_ERROR(0, "Too small to show date");
	  }
	}
      }
      lastMonthX=xEnd;
    }

    lastX=xEnd;

    lastMonth=d.month();

    d=dNext;
  }

  switch(labelWhat) {
  case 3: // months
    qs=tr("Months");
    break;
  case 4: // years
    qs=tr("Years");
    break;
  default:
    break;
  }

  if (!qs.isEmpty()) {
    qp->drawText(_width-BORDER2+2, _nullY+(th/2), qs);
  }

}




void GraphReport::drawDataSet(QPainter *qp, GraphReport::DsSet &ds) {
  QDate dt;
  int oldX=_nullX;
  int oldY=_nullY;

  switch(_graphType) {
  case GraphTypePoints:
  case GraphTypeBars:
    qp->setPen(QPen(QColor("black"), 1));
    qp->setBrush(QBrush(QColor("light blue")));
  case GraphTypeLines:
  default:
    qp->setPen(QPen(QColor("light blue"), 2));
    break;
  }

  dt=_firstDate;
  while(dt<=_lastDate) {
    const DsElement *e=0;
    DsDay *d;
    DsMonth *m;
    DsYear *yr;
    int x, y;
    double dv;

    x=dateToX(dt);

    switch(_units) {
    case UnitsDays:
      d=ds.findDay(dt);
      if (d)
	e=&(d->getItsElement());
      break;
    case UnitsYears:
      yr=ds.getYear(dt);
      if (yr)
	e=&(yr->getItsElement());
      break;
    case UnitsMonths:
    default:
      m=ds.getMonth(dt);
      if (m)
	e=&(m->getItsElement());
      break;
    }

    if (e)
      dv=AB_Value_GetValue(e->getValue());
    else
      dv=0;

    y=valueToY(dv);

    // now we have x and y
    switch(_graphType) {
    case GraphTypePoints:
      qp->drawPoint(x+_unitOffset, y);
      oldX=x+_unitOffset;
      oldY=y;
    case GraphTypeBars:
      qp->drawRect(x+_unitOffset, y, _unitWidth, _nullY-y);
      oldX=x+_unitOffset;
      oldY=y;
      break;
    case GraphTypeLines:
    default:
      qp->drawLine(oldX, oldY, x+_unitOffset, y);
      oldX=x+_unitOffset;
      oldY=y;
      break;
    }


    // advance to next
    switch(_units) {
    case UnitsDays:
      dt=dt.addDays(1);
      break;
    case UnitsYears:
      dt=dt.addYears(1);
      break;
    case UnitsMonths:
    default:
      dt=dt.addMonths(1);
      break;
    }
  } // while
}






