#
# Copyright (c) 2002, 2003, 2004 Art Haas
#
# This file is part of PythonCAD.
#
# PythonCAD is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# PythonCAD is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PythonCAD; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# classes for graphic objects
#

import warnings

from PythonCAD.Generic import style
from PythonCAD.Generic import color
from PythonCAD.Generic import linetype
from PythonCAD.Generic import baseobject
from PythonCAD.Generic import logger

class GraphicObject(baseobject.Subpart):
    """A class representing an object that is drawn.

This class is meant to be a base class for things like
line segments, circles, etc. The GraphicObject class
has one attribute that cannot be None:

style: The Style object

There are three other attributes that override values in
the style attribute:

color: A Color object
thickness: A positive float value
linetype: A Linetype object

A GraphicObject object has the following methods:

{get/set}Style(): Get/Set the Style of a GraphicObject.
{get/set}Color(): Get/Set the Color of a GraphicObject.
{get/set}Thickness(): Get/Set the thickness of a GraphicObject.
{get/set}Linetype(): Get/Set the Linetype of a GraphicObject.
    """
    messages = {
    'style_changed' : True,
    'color_changed' : True,
    'linetype_changed' : True,
    'thickness_changed' : True,
    }
    __defstyle = style.Style(u'Default Style',
                             linetype.Linetype(u'Solid', None),
                             color.Color(0xffffff),
                             1.0)

    def __init__(self, st=None, lt=None, col=None, t=None, **kw):
        """Initialize a GraphicObject.

GraphicObject([st, lt, col, t])

Optional arguments:
style: A style object that overrides the class default
linetype: A Linetype object that overrides the value in the Style
color: A Color object that overrides the value in the Style
thickness: A positive float that overrides the value in the Style
        """
        baseobject.Subpart.__init__(self, **kw)
        _st = st
        if _st is None:
            _st = GraphicObject.__defstyle
        else:
            if not isinstance(_st, style.Style):
                raise TypeError, "Invalid style: " + `_st`
        _col = _lt = _t = None
        if 'color' in kw:
            _col = kw['color']
        else:
            if 'col' in kw:
                warnings.warn("keyword 'col' is deprecated;  use 'color'",
                DeprecationWarning, stacklevel=2)
                _col = kw['col']
        if _col is not None:
            if not isinstance(_col, color.Color):
                _col = color.Color(_col)
            if _col == _st.getColor():
                _col = None
        _col = col
        if _col is not None:
            if not isinstance(_col, color.Color):
                _col = color.Color(col)
            if _col == _st.getColor():
                _col = None
        if 'linetype' in kw:
            _lt = kw['linetype']
        else:
            if 'lt' in kw:
                warnings.warn("keyword 'lt' is deprecated;  use 'linetype'",
                              DeprecationWarning, stacklevel=2)
                _lt = kw['lt']
        if _lt is not None:
            if not isinstance(_lt, linetype.Linetype):
                raise TypeError, "Invalid linetype: " + `_lt`
            if _lt == _st.getLinetype():
                _lt = None
        _lt = lt
        if _lt is not None:
            if not isinstance(_lt, linetype.Linetype):
                raise TypeError, "Invalid linetype: " + `_lt`
            if _lt == _st.getLinetype():
                _lt = None
        if 'thickness' in kw:
            _t = kw['thickness']
        else:
            if 'th' in kw:
                warnings.warn("keyword 'th' is deprecated;  use 'thickness'",
                               DeprecationWarning, stacklevel=2)
                _t = kw['th']
        if _t is not None:
            if not isinstance(_t, float):
                _t = float(_t)
            if _t < 0.0:
                raise ValueError, "Invalid thickness: %g" % _t
            if abs(_t - _st.getThickness()) < 1e-10:
                _t = None
        _t = t
        if _t is not None:
            if not isinstance(_t, float):
                _t = float(t)
            if _t < 0.0:
                raise ValueError, "Invalid thickness: %g" % _t
            if abs(_t - _st.getThickness()) < 1e-10:
                _t = None
        self.__style = _st
        self.__color = _col
        self.__linetype = _lt
        self.__thickness = _t

    def finish(self):
        self.__style = None
        self.__color = None
        self.__linetype = None
        self.__thickness = None
        baseobject.Subpart.finish(self)

    def getStyle(self):
        """Get the Style of the GraphicObject.

getStyle()

Returns the current Style of an object. The values in the
style may be overriden if any of the color, linetype, or
thickness attributes are not None.
        """
        _style = self.__style
        if _style is None:
            _style = GraphicObject.__defstyle
        return _style

    def setStyle(self, s):
        """Set the Style of the Object.

setStyle(s)

Setting the style of a GraphicObject unsets any overrides for
the object. Setting the Style to None restores the default
style.
        """
        if self.isLocked():
            raise RuntimeError, "Style change not allowed - objected locked."
        _s = s
        if _s is not None:
            if not isinstance(_s, style.Style):
                raise TypeError, "Invalid style: " + `_s`
        _cs = self.getStyle()
        if _s is None or _cs != _s:
            self.__style = _s
            self.__color = None
            self.__linetype = None
            self.__thickness = None
            self.sendMessage('style_changed', _cs)
            self.modified()

    style = property(getStyle, setStyle, None, "Object Style.")

    def getColor(self):
        """Get the Color of the Object.

getColor()
        """
        _color = self.__color
        if _color is None:
            if self.__style is not None:
                _color = self.__style.getColor()
            if _color is None:
                _color = GraphicObject.__defstyle.getColor()
        return _color

    def setColor(self, c=None):
        """Set the color of the object.

setColor([c])

Setting the color overrides the value in the Style. Setting
the color to None or invoking this method without arguments
restores the color value defined in the Style.
        """
        if self.isLocked():
            raise RuntimeError, "Color change not allowed - object locked."
        _c = c
        if _c is not None:
            if not isinstance(_c, color.Color):
                _c = color.Color(c)
        _oc = self.getColor()
        if _c is None or _oc != _c:
            self.__color = _c
            self.sendMessage('color_changed', _oc)
            self.modified()

    color = property(getColor, setColor, None, "Object color.")

    def getThickness(self):
        """Get the thickness of the GraphicObject.

getThickness()
        """
        _th = self.__thickness
        if _th is None:
            if self.__style is not None:
                _th = self.__style.getThickness()
            if _th is None:
                _th = GraphicObject.__defstyle.getThickness()
        return _th

    def setThickness(self, t=None):
        """Set the thickness of a object.

setThickness([t])

Setting the thickness overrides the value in the Style. Setting
the thickness to None, or invoking this method without an
argument, restores the thickness value defined in the Style.
        """
        if self.isLocked():
            raise RuntimeError, "Thickness change not allowed - object locked."
        _t = t
        if _t is not None:
            if not isinstance(_t, float):
                _t = float(t)
            if _t < 0.0:
                raise ValueError, "Invalid thickness: %g" % _t
        _ot = self.getThickness()
        if _t is None or abs(_ot - _t) > 1e-10:
            self.__thickness = _t
            self.sendMessage('thickness_changed', _ot)
            self.modified()

    thickness = property(getThickness, setThickness, None, "Object thickness.")

    def getLinetype(self):
        """Get the Linetype of the object.

getLinetype()
        """
        _lt = self.__linetype
        if _lt is None:
            if self.__style is not None:
                _lt = self.__style.getLinetype()
            if _lt is None:
                _lt = GraphicObject.__defstyle.getLinetype()
        return _lt

    def setLinetype(self, lt=None):
        """Set the Linetype of the GraphicObject.

setLinetype([lt])

Setting the Linetype overrides the value in the Sytle. Setting
the Linetype to None or invoking this method without arguments
restores the linetype value defined in the Style.
        """
        if self.isLocked():
            raise RuntimeError, "Linetype change not allowed - object locked."
        _lt = lt
        if _lt is not None:
            if not isinstance(_lt, linetype.Linetype):
                raise TypeError, "Invalid linetype: " + `_lt`
        _ol = self.getLinetype()
        if _lt is None or _ol != _lt:
            self.__linetype = _lt
            self.sendMessage('linetype_changed', _ol)
            self.modified()

    linetype = property(getLinetype, setLinetype, None, "Object Linetype.")

    def getGraphicValues(self, defstyle=None):
        if defstyle is not None and not isinstance(defstyle, style.Style):
            raise TypeError, "Invalid default style: " + `defstyle`
        _s = _c = _l = _t = None
        _ss = self.__style
        if defstyle is None or defstyle != _ss:
            _s = _ss.getStyleValues()
        _sc = self.__color
        if _sc is not None:
            _c = _sc.getColors()
        _sl = self.__linetype
        if _sl is not None:
            _l = (_sl.getName(), _sl.getList())
        _st = self.__thickness
        if _st is not None:
            _t = self.__thickness
        return _s, _l, _c, _t

    def getValues(self):
        """Return values comprising the GraphicObject.

getValues()

This method extends the Subpart::getValues() method.
        """
        _data = baseobject.Subpart.getValues(self)
        _data.setValue('style', self.__style.getStyleValues())
        if self.__color is not None:
            _data.setValue('color', self.__color.getColors())
        if self.__linetype is not None:
            _lt = self.__linetype
            _data.setValue('linetype', (_lt.getName(), _lt.getList()))
        if self.__thickness is not None:
            _data.setValue('thickness', self.__thickness)
        return _data
    
    def sendsMessage(self, m):
        if m in GraphicObject.messages:
            return True
        return baseobject.Subpart.sendsMessage(self, m)

#
# GraphicObject history class
#

class GraphicObjectLog(logger.Logger):
    def __init__(self, obj):
        if not isinstance(obj, GraphicObject):
            raise TypeError, "Invalid GraphicObject: " + `obj`
        logger.Logger.__init__(self)
        self.__obj = obj
        obj.connect('style_changed', self, GraphicObjectLog.styleChanged)
        obj.connect('linetype_changed', self, GraphicObjectLog.linetypeChanged)
        obj.connect('color_changed', self, GraphicObjectLog.colorChanged)
        obj.connect('thickness_changed', self, GraphicObjectLog.thicknessChanged)
        # obj.connect('attribute_changed', self, GraphicObjectLog.attrChanged)

    def detatch(self):
        self.__obj.disconnect(self)
        self.__obj = None

    def getObject(self):
        return self.__obj

    def styleChanged(self, tb, *args):
        _alen = len(args)
        if _alen < 1:
            raise ValueError, "Invalid argument count: %d" % _alen
        _style = args[0]
        if not isinstance(_style, style.Style):
            raise TypeError, "Invalid style: " + str(_style)
        self.saveUndoData('style_changed', _style.getStyleValues())

    def linetypeChanged(self, tb, *args):
        _alen = len(args)
        if _alen < 1:
            raise ValueError, "Invalid argument count: %d" % _alen
        _lt = args[0]
        if not isinstance(_lt, linetype.Linetype):
            raise TypeError, "Invalid linetype: " + str(_lt)
        self.saveUndoData('linetype_changed', (_lt.getName(), _lt.getList()))

    def colorChanged(self, tb, *args):
        _alen = len(args)
        if _alen < 1:
            raise ValueError, "Invalid argument count: %d" % _alen
        _color = args[0]
        if not isinstance(_color, color.Color):
            raise TypeError, "Invalid color: " + str(_color)
        self.saveUndoData('color_changed', _color.getColors())

    def thicknessChanged(self, tb, *args):
        _alen = len(args)
        if _alen < 1:
            raise ValueError, "Invalid argument count: %d" % _alen
        _t = args[0]
        if not isinstance(_t, float):
            _t = float(args[0])
        if _t < 0.0:
            raise ValueError, "Invalid thickness: %g" % _t
        self.saveUndoData('thickness_changed', _t)

    def attrChanged(self, obj, *args):
        _alen = len(args)
        if len(args) < 2:
            raise ValueError, "Invalid argument count: %d" % _alen
        _attr = args[0]
        if not isinstance(_attr, str):
            raise TypeError, "Unexpected attribute string: " + str(_attr)
        _v = args[1]
        if _attr == 'linetype':
            if not isinstance(_v, linetype.Linetype):
                raise TypeError, "Invalid linetype: " + `_v`
            _val = (_v.getName(), _v.getList())
        elif _attr == 'color':
            if not isinstance(_v, color.Color):
                raise TypeError, "Invalid color: " + `_v`
            _val = _v.getColors()
        elif _attr == 'style':
            if not isinstance(_v, style.Style):
                raise TypeError, "Invalid style: " + `_v`
            _l = _v.getLinetype.getList()
            _c = _v.getColor().getColors()
            _t = _v.getThickness()
            _val = (_l, _c, _t)
        elif _attr == 'thickness':
            _val = _v
            if not isinstance(_val, float):
                _val = float(_v)
        else:
            raise ValueError, "Unexpected attribute: %s" % _attr
        self.saveUndoData('attr_changed', _attr, _val)

    def execute(self, undo, *args):
        if undo is not True and undo is not False:
            raise ValueError, "Invalid undo value: " + str(undo)
        _alen = len(args)
        if len(args) == 0:
            raise ValueError, "No arguments to execute()"
        _obj = self.__obj
        _op = args[0]
        if _op == 'style_changed':
            if len(args) < 2:
                raise ValueError, "Invalid argument count: %d" % _alen
            _style = _obj.getStyle()
            if undo:
                self.saveRedoData(_op, _style.getStyleValues())
            else:
                self.saveUndoData(_op, _style.getStyleValues())
            self.ignore('style_changed')
            try:
                pass # fixme
            finally:
                self.receive('style_changed')
        elif _op == 'linetype_changed':
            if len(args) < 2:
                raise ValueError, "Invalid argument count: %d" % _alen
            _lt = _obj.getLinetype()
            _sv = (_lt.getName(), _lt.getList())
            if undo:
                self.saveRedoData(_op, _sv)
            else:
                self.saveUndoData(_op, _sv)
            self.ignore('linetype_changed')
            try:
                _name, _dlist = args[1]
                _lt = linetype.get_linetype_by_dashes(_dlist)
                if undo:
                    _obj.startUndo(True)
                    try:
                        _obj.setLinetype(_lt)
                    finally:
                        _obj.endUndo()
                else:
                    _obj.setLinetype(_lt)
            finally:
                self.receive('linetype_changed')
        elif _op == 'color_changed':
            if len(args) < 2:
                raise ValueError, "Invalid argument count: %d" % _alen
            _color = _obj.getColor()
            if undo:
                self.saveRedoData(_op, _color.getColors())
            else:
                self.saveUndoData(_op, _color.getColors())
            self.ignore('color_changed')
            try:
                _r, _g, _b = args[1]
                _col = color.get_color(_r, _g, _b)
                if undo:
                    _obj.startUndo(True)
                    try:
                        _obj.setColor(_col)
                    finally:
                        _obj.endUndo()
                else:
                    _obj.setColor(_col)
            finally:
                self.receive('color_changed')
        elif _op == 'thickness_changed':
            if len(args) < 2:
                raise ValueError, "Invalid argument count: %d" % _alen
            _t = _obj.getThickness()
            if undo:
                self.saveRedoData(_op, _t)
            else:
                self.saveUndoData(_op, _t)
            self.ignore('thickness_changed')
            try:
                _t = args[1]
                if undo:
                    _obj.startUndo(True)
                    try:
                        _obj.setThickness(_t)
                    finally:
                        _obj.endUndo()
                else:
                    _obj.setThickness(_t)
            finally:
                self.receive('thickness_changed')
        elif _op == 'attr_changed':
            raise ValueError, "Executing on 'attr_changed': " + str(args)
        else:
            raise ValueError, "Unexpected operation: %s" % _op
