#
# Copyright (c) 2002, 2003, 2004 Art Haas
#
# This file is part of PythonCAD.
#
# PythonCAD is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# PythonCAD is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with PythonCAD; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
#
#
# horizontal construction lines
#

from __future__ import generators

import conobject
import entity
import tolerance
import point
import quadtree
import logger

class HCLine(conobject.ConstructionObject):
    """A base class for horizontal construction lines.
    """
    messages = {
        'moved' : True,
        'keypoint_changed' : True
    }
    def __init__(self, p, **kw):
        """Instantiate an HCLine object.

HCLine(p)
        """
        _p = p
        if not isinstance(p, point.Point):
            _p = point.Point(p)
        conobject.ConstructionObject.__init__(self, **kw)
        self.__keypoint = _p
        _p.storeUser(self)
        _p.connect('moved', self, HCLine.movePoint)

    def __eq__(self, obj):
        """Compare one HCLine to another for equality.
        """
        if not isinstance(obj, HCLine):
            return False
        if obj is self:
            return True
        if abs(self.getLocation().y - obj.getLocation().y) < 1e-10:
            return True
        return False

    def __ne__(self, obj):
        """Compare one HCLine to another for inequality.
        """
        if not isinstance(obj, HCLine):
            return True
        if obj is self:
            return False
        if abs(self.getLocation().y - obj.getLocation().y) < 1e-10:
            return False
        return True

    def __str__(self):
        _x, _y = self.getLocation().getCoords()
        return "Horizontal Construction Line at y = %g" % self.__keypoint.y

    def getValues(self):
        return 'hcline', entity.Entity.getValues(self), self.__keypoint.getID()

    def getLocation(self):
        return self.__keypoint

    def setLocation(self, p):
        if self.isLocked():
            raise RuntimeError, "Setting keypoint not allowed - object locked."
        if not isinstance(p, point.Point):
            raise TypeError, "Invalid keypoint: " + `p`
        _kp = self.__keypoint
        if p is not _kp:
            _y = _kp.y
            _kp.disconnect(self)
            _kp.freeUser(self)
            self.__keypoint = p
            self.sendMessage('keypoint_changed', _kp)
            p.connect('moved', self, HCLine.movePoint)
            p.storeUser(self)
            if abs(_y - p.y) > 1e-10:
                self.sendMessage('moved', p.x, _y)
            self.sendMessage('modified')

    def mapCoords(self, x, y, tol=tolerance.TOL):
        """Return the nearest Point on the HCLine to a coordinate pair.

mapCoords(x, y[, tol])

The function has two required argument:

x: A Float value giving the "x" coordinate
y: A Float value giving the "y" coordinate

There is a single optional argument:

tol: A float value equal or greater than 0.0

This function is used to map a possibly near-by coordinate pair to
an actual Point on the HCLine. If the distance between the actual
Point and the coordinates used as an argument is less than the tolerance,
the actual Point is returned. Otherwise, this function returns None.
        """
        _x = x
        if not isinstance(_x, float):
            _x = float(x)
        _y = y
        if not isinstance(_y, float):
            _y = float(y)
        _t = tolerance.toltest(tol)
        _hy = self.__keypoint.y
        if abs(_hy - _y) < _t:
            return _x, _hy
        return None

    def inRegion(self, xmin, ymin, xmax, ymax, fully=False):
        """Return whether or not an HCLine passes through a region.

inRegion(xmin, ymin, xmax, ymax)

The first four arguments define the boundary. The method
will return True if the HCLine falls between ymin and ymax.
Otherwise the function will return False.
        """
        _xmin = xmin
        if not isinstance(_xmin, float):
            _xmin = float(xmin)
        _ymin = ymin
        if not isinstance(_ymin, float):
            _ymin = float(ymin)
        _xmax = xmax
        if not isinstance(_xmax, float):
            _xmax = float(xmax)
        if _xmax < _xmin:
            raise ValueError, "Illegal values: xmax < xmin"
        _ymax = ymax
        if not isinstance(_ymax, float):
            _ymax = float(ymax)
        if _ymax < _ymin:
            raise ValueError, "Illegal values: ymax < ymin"
        if fully is not True and fully is not False:
            raise ValueError, "Invalid flag: " + str(fully)
        if fully:
            return False
        _y = self.__keypoint.y
        return _ymin < _y < _ymax

    def move(self, dx, dy):
        """Move a HCLine

move(dx, dy)

The first argument gives the x-coordinate displacement,
and the second gives the y-coordinate displacement. Both
values should be floats.
        """
        if self.isLocked() or self.__keypoint.isLocked():
            raise RuntimeError, "Moving HCLine not allowed - object locked."
        _dx = dx
        if not isinstance(_dx, float):
            _dx = float(dx)
        _dy = dy
        if not isinstance(_dy, float):
            _dy = float(dy)
        if abs(_dx) > 1e-10 or abs(_dy) > 1e-10:
            _x, _y = self.__keypoint.getCoords()
            self.ignore('moved')
            self.__keypoint.move(_dx, _dy)
            self.receive('moved')
            self.sendMessage('moved', _x, _y)
            self.modified()

    def movePoint(self, p, *args):
        _plen = len(args)
        if _plen < 2:
            raise ValueError, "Invalid argument count: %d" % _plen
        _x = args[0]
        if not isinstance(_x, float):
            _x = float(args[0])
        _y = args[1]
        if not isinstance(_y, float):
            _y = float(args[1])
        if p is not self.__keypoint:
            raise ValueError, "Invalid point for HCLine::movePoint()" + `p`
        if abs(p.y - _y) > 1e-10:
            self.sendMessage('moved', _x, _y)
            self.modified()
            
    def clipToRegion(self, xmin, ymin, xmax, ymax):
        _xmin = xmin
        if not isinstance(_xmin, float):
            _xmin = float(xmin)
        _ymin = ymin
        if not isinstance(_ymin, float):
            _ymin = float(ymin)
        _xmax = xmax
        if not isinstance(_xmax, float):
            _xmax = float(xmax)
        if _xmax < _xmin:
            raise ValueError, "Illegal values: xmax < xmin"
        _ymax = ymax
        if not isinstance(_ymax, float):
            _ymax = float(ymax)
        if _ymax < _ymin:
            raise ValueError, "Illegal values: ymax < ymin"
        _y = self.__keypoint.y
        if _ymin < _y < _ymax:
            return _xmin, _y, _xmax, _y
        return None

    def sendsMessage(self, m):
        if m in HCLine.messages:
            return True
        return conobject.ConstructionObject.sendsMessage(self, m)

#
#
#

def intersect_region(hcl, xmin, ymin, xmax, ymax):
    if not isinstance(hcl, HCLine):
        raise TypeError, "Invalid HCLine: " + str(hcl)
    _xmin = xmin
    if not isinstance(_xmin, float):
        _xmin = float(xmin)
    _ymin = ymin
    if not isinstance(_ymin, float):
        _ymin = float(ymin)
    _xmax = xmax
    if not isinstance(_xmax, float):
        _xmax = float(xmax)
    if _xmax < _xmin:
        raise ValueError, "Illegal values: xmax < xmin"
    _ymax = ymax
    if not isinstance(_ymax, float):
        _ymax = float(ymax)
    if _ymax < _ymin:
        raise ValueError, "Illegal values: ymax < ymin"
    _x, _y = hcl.getLocation().getCoords()
    _x1 = _y1 = _x2 = _y2 = None
    if _ymin < _y < _ymax:
        _x1 = _xmin
        _y1 = _y
        _x2 = _xmax
        _y2 = _y
    return _x1, _y1, _x2, _y2

#
# Quadtree HCLine storage
#

class HCLineQuadtree(quadtree.Quadtree):
    def __init__(self):
        quadtree.Quadtree.__init__(self)

    def getNodes(self, *args):
        _alen = len(args)
        if _alen != 1:
            raise ValueError, "Expected 1 arguments, got %d" % _alen
        _y = args[0]
        if not isinstance(_y, float):
            _y = float(args[0])
        _nodes = [self.getTreeRoot()]
        while len(_nodes):
            _node = _nodes.pop()
            _bounds = _node.getBoundary()
            _ymin = _bounds[1]
            _ymax = _bounds[3]
            if _y < _ymin or _y > _ymax:
                continue
            if _node.hasSubnodes():
                _ymid = (_ymin + _ymax)/2.0
                _ne = _nw = _sw = _se = True
                if _y < _ymid: # hcline below
                    _nw = _ne = False
                else: # hcline above
                    _sw = _se = False
                if _ne:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.NENODE))
                if _nw:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.NWNODE))
                if _sw:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.SWNODE))
                if _se:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.SENODE))
            else:
                yield _node

    def addObject(self, obj):
        if not isinstance(obj, HCLine):
            raise TypeError, "Invalid HCLine object: " + `obj`
        if obj in self:
            return
        _x, _y = obj.getLocation().getCoords()
        _bounds = self.getTreeRoot().getBoundary()
        _xmin = _ymin = _xmax = _ymax = None
        _resize = False
        if _bounds is None: # first node in tree
            _resize = True
            _xmin = _x - 1.0
            _ymin = _y - 1.0
            _xmax = _x + 1.0
            _ymax = _y + 1.0
        else:
            _xmin, _ymin, _xmax, _ymax = _bounds
            if _x < _xmin:
                _xmin = _x - 1.0
                _resize = True
            if _x > _xmax:
                _xmax = _x + 1.0
                _resize = True
            if _y < _ymin:
                _ymin = _y - 1.0
                _resize = True
            if _y > _ymax:
                _ymax = _y + 1.0
                _resize = True
        if _resize:
            self.resize(_xmin, _ymin, _xmax, _ymax)
        for _node in self.getNodes(_y):
            _xmin, _ymin, _xmax, _ymax = _node.getBoundary()            
            if obj.inRegion(_xmin, _ymin, _xmax, _ymax):
                _node.addObject(obj)
        quadtree.Quadtree.addObject(self, obj)
        obj.connect('moved', self, HCLineQuadtree.moveHCLine)

    def old_addObject(self, obj):
        if not isinstance(obj, HCLine):
            raise TypeError, "Invalid HCLine object: " + `obj`
        if obj in self:
            return
        _x, _y = obj.getLocation().getCoords()
        _node = self.getTreeRoot()
        _bounds = _node.getBoundary()
        _xmin = _ymin = _xmax = _ymax = None
        _resize = False
        if _bounds is None: # first node in tree
            _resize = True
            _xmin = _x - 1.0
            _ymin = _y - 1.0
            _xmax = _x + 1.0
            _ymax = _y + 1.0
        else:
            _xmin, _ymin, _xmax, _ymax = _bounds
            if _x < _xmin:
                _xmin = _x - 1.0
                _resize = True
            if _x > _xmax:
                _xmax = _x + 1.0
                _resize = True
            if _y < _ymin:
                _ymin = _y - 1.0
                _resize = True
            if _y > _ymax:
                _ymax = _y + 1.0
                _resize = True
        if _resize:
            self.resize(_xmin, _ymin, _xmax, _ymax)
            _node = self.getTreeRoot()
        _nodes = []
        _nodes.append(_node)
        while len(_nodes):
            _node = _nodes.pop()
            _bounds = _node.getBoundary()
            _ymin = _bounds[1]
            _ymax = _bounds[3]
            if _node.hasSubnodes():
                _ymid = (_ymin + _ymax)/2.0
                _ne = _nw = _sw = _se = True
                if _y < _ymid: # hcline below
                    _nw = _ne = False
                else: # hcline above
                    _sw = _se = False
                if _ne:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.NENODE))
                if _nw:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.NWNODE))
                if _sw:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.SWNODE))
                if _se:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.SENODE))
            else:
                if (not (_y < _ymin) and (_y < _ymax)):
                    _node.addObject(obj)
        quadtree.Quadtree.addObject(self, obj)
        obj.connect('moved', self, HCLineQuadtree.moveHCLine)
        
    def delObject(self, obj):
        if obj not in self:
            return
        _pdict = {}
        _x, _y = obj.getLocation().getCoords()
        for _node in self.getNodes(_y):
            _node.delObject(obj)
            _parent = _node.getParent()
            if _parent is not None:                
                _pid = id(_parent)
                if _pid not in _pdict:
                    _pdict[_pid] = _parent
        quadtree.Quadtree.delObject(self, obj)
        obj.disconnect(self)
        for _parent in _pdict.values():
            self.purgeSubnodes(_parent)

    def old_delObject(self, obj):
        if obj not in self:
            return
        _x, _y = obj.getLocation().getCoords()
        _nodes = []
        _nodes.append(self.getTreeRoot())
        _pdict = {}
        while len(_nodes):
            _node = _nodes.pop()
            _bounds = _node.getBoundary()
            _ymin = _bounds[1]
            _ymax = _bounds[3]
            if _node.hasSubnodes():
                _ymid = (_ymin + _ymax)/2.0
                _ne = _nw = _sw = _se = True
                if _y < _ymid: # hcline below
                    _nw = _ne = False
                else: # hcline above
                    _sw = _se = False
                if _ne:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.NENODE))
                if _nw:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.NWNODE))
                if _sw:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.SWNODE))
                if _se:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.SENODE))
            else:
                if (not (_y < _ymin) and (_y < _ymax)):
                    _node.delObject(obj)
                    _parent = _node.getParent()
                    if _parent is not None:                
                        _pid = id(_parent)
                        if _pid not in _pdict:
                            _pdict[_pid] = _parent
        quadtree.Quadtree.delObject(self, obj)
        obj.disconnect(self)
        for _parent in _pdict.values():
            self.purgeSubnodes(_parent)

    def find(self, *args):
        _alen = len(args)
        if _alen < 1:
            raise ValueError, "Invalid argument count: %d" % _alen
        _y = args[0]
        if not isinstance(_y, float):
            _y = float(args[0])
        _t = tolerance.TOL
        if _alen > 1:
            _t = tolerance.toltest(args[1])
        if not len(self):
            return None
        _nodes = []
        _nodes.append(self.getTreeRoot())
        _hcl = _tsep = None
        _hdict = {}
        while len(_nodes):
            _node = _nodes.pop()
            _bounds = _node.getBoundary()
            _ymin = _bounds[1]
            _ymax = _bounds[3]
            if (_ymin - _t) < _y < (_ymax + _t):
                if _node.hasSubnodes():
                    for _subnode in _node.getSubnodes():
                        _bounds = _node.getBoundary()
                        _ymin = _bounds[1]
                        _ymax = _bounds[3]
                        if (_ymin - _t) < _y < (_ymax + _t):
                            _nodes.append(_subnode)
                else:
                    for _h in _node.getObjects():
                        _hid = id(_h)
                        if _hid not in _hdict:
                            _yh = _h.getLocation().y
                            _ysep = abs(_yh - _y)
                            if _ysep < _t:
                                if _hcl is None:
                                    _hcl = _h
                                    _tsep = _ysep
                                else:
                                    if _ysep < _tsep:
                                        _hcl = _h
                                        _tsep = _ysep
                            _hdict[_hid] = True
        return _hcl

    def moveHCLine(self, obj, *args):
        if obj not in self:
            raise ValueError, "HCLine not stored in Quadtree: " + `obj`
        _alen = len(args)
        if _alen < 2:
            raise ValueError, "Invalid argument count: %d" % _alen
        _x = args[0]
        if not isinstance(_x, float):
            _x = float(args[0])
        _y = args[1]
        if not isinstance(_y, float):
            _y = float(args[1])
        for _node in self.getNodes(_y):
            _node.delObject(obj) # hcline may not be in node
        quadtree.Quadtree.delObject(self, obj)
        obj.disconnect(self)
        self.addObject(obj)

    def old_moveHCLine(self, obj, *args):
        if obj not in self:
            raise ValueError, "HCLine not stored in Quadtree: " + `obj`
        _alen = len(args)
        if _alen < 2:
            raise ValueError, "Invalid argument count: %d" % _alen
        _x = args[0]
        if not isinstance(_x, float):
            _x = float(args[0])
        _y = args[1]
        if not isinstance(_y, float):
            _y = float(args[1])
        _nodes = []
        _nodes.append(self.getTreeRoot())
        while len(_nodes):
            _node = _nodes.pop()
            _bounds = _node.getBoundary()
            _ymin = _bounds[1]
            _ymax = _bounds[3]
            if _node.hasSubnodes():
                _ymid = (_ymin + _ymax)/2.0
                _ne = _nw = _sw = _se = True
                if _y < _ymid: # hcline below
                    _nw = _ne = False
                else: # hcline above
                    _sw = _se = False
                if _ne:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.NENODE))
                if _nw:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.NWNODE))
                if _sw:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.SWNODE))
                if _se:
                    _nodes.append(_node.getSubnode(quadtree.QTreeNode.SENODE))
            else:
                _node.delObject(obj) # hcline may not be in node
        quadtree.Quadtree.delObject(self, obj)
        obj.disconnect(self)
        self.addObject(obj)
    
    def getClosest(self, x, y, tol=tolerance.TOL):
        return self.find(y, tol)

    def getInRegion(self, xmin, ymin, xmax, ymax):
        _xmin = xmin
        if not isinstance(_xmin, float):
            _xmin = float(xmin)
        _ymin = ymin
        if not isinstance(_ymin, float):
            _ymin = float(ymin)
        _xmax = xmax
        if not isinstance(_xmax, float):
            _xmax = float(xmax)
        if _xmax < _xmin:
            raise ValueError, "Illegal values: xmax < xmin"
        _ymax = ymax
        if not isinstance(_ymax, float):
            _ymax = float(ymax)
        if _ymax < _ymin:
            raise ValueError, "Illegal values: ymax < ymin"
        _hcls = []        
        if not len(self):
            return _hcls
        _nodes = []
        _nodes.append(self.getTreeRoot())
        _hdict = {}
        while len(_nodes):
            _node = _nodes.pop()
            if _node.hasSubnodes():
                for _subnode in _node.getSubnodes():
                    _bounds = _subnode.getBoundary()
                    _bmin = _bounds[1]
                    _bmax = _bounds[3]
                    if ((_bmin > _ymax) or (_bmax < _ymin)):
                        continue
                    _nodes.append(_subnode)
            else:
                for _hcl in _node.getObjects():
                    _hid = id(_hcl)
                    if _hid not in _hdict:
                        if _hcl.inRegion(_xmin, _ymin, _xmax, _ymax):
                            _hcls.append(_hcl)
                        _hdict[_hid] = True
        return _hcls

#
# HCLine history class
#

class HCLineLog(logger.Logger):
    def __init__(self, h):
        if not isinstance(h, HCLine):
            raise TypeError, "Invalid HCLine: " + `h`
        logger.Logger.__init__(self)
        self.__h = h
        h.connect('keypoint_changed' ,self, HCLineLog.keypointChange)

    def detatch(self):
        self.__h.disconnect(self)
        self.__h = None

    def keypointChange(self, h, *args):
        _alen = len(args)
        if _alen < 1:
            raise ValueError, "Invalid argument count: %d" % _alen
        _old = args[0]
        if not isinstance(_old, point.Point):
            raise TypeError, "Invalid old endpoint: " + `_old`
        _oid = _old.getID()
        self.saveUndoData('keypoint_changed', _oid)

    def execute(self, undo, *args):
        if undo is not True and undo is not False:
            raise ValueError, "Invalid undo value: " + str(undo)
        _alen = len(args)
        if _alen == 0:
            raise ValueError, "No arguments to execute()"
        _h = self.__h
        _p = _h.getLocation()
        _op = args[0]
        if _op == 'keypoint_changed':
            if _alen < 2:
                raise ValueError, "Invalid argument count: %d" % _alen
            _oid = args[1]
            _cid = _p.getID()
            if undo:
                self.saveRedoData('keypoint_changed', _cid)
                self.ignore('keypoint_changed')
                try:
                    _parent = _h.getParent()
                    if _parent is None:
                        raise ValueError, "HCLine has no parent - cannot undo"
                    _op = _parent.getObject(_oid)
                    if _op is None or not isinstance(_op, point.Point):
                        raise ValueError, "Keypoint missing: id=%d" % _oid
                    _x1, _y1 = _p.getCoords()
                    _h.startUndo(True)
                    try:
                        _h.setLocation(_op)
                    finally:
                        _h.endUndo()
                    if abs(_op.y - _y1) > 1e-10:
                        _h.sendMessage('moved', _x1, _y1)
                finally:
                    self.receive('keypoint_changed')
            else:
                self.saveUndoData('keypoint_changed', _cid)
                _parent = _h.getParent()
                if _parent is None:
                    raise ValueError, "HCLine has no parent - cannot undo"
                _op = _parent.getObject(_oid)
                if _op is None or not isinstance(_op, point.Point):
                    raise ValueError, "Keypoint missing: id=%d" % _oid
                self.ignore('keypoint_changed')
                try:
                    _h.setLocation(_op)
                finally:
                    self.receive('keypoint_changed')
        else:
            raise ValueError, "Unexpected operation: %s" % _op
