// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.
#include "prim.h"
#include <boost/python/class.hpp>
#include <boost/python/return_internal_reference.hpp>
#include "rotate.h"

namespace visual {
	
Primitive::Primitive()
 : pos( &mtx, 0,0,0),
   axis( &mtx, 1,0,0),
   up( &mtx, 0,1,0),
   degenerate(true)
{
}

Primitive::Primitive( const Primitive& other)
	: DisplayObject( other), 
	pos( &mtx, other.pos.x, other.pos.y, other.pos.z),
	axis( &mtx, other.axis.x, other.axis.y, other.axis.z),
	up( &mtx, other.up.x, other.up.y, other.up.z),
	degenerate(other.degenerate),
	mwt( other.mwt),
	wlt( other.wlt)
{
	write_lock L(mtx);
}

void
Primitive::py_rotate( double angle, vector _axis, vector origin)
{
	tmatrix R;
	py_rotation( R, angle, _axis, origin);
	
	if (!axis.cross( up)) {
		up = vector( 1,0,0);
		if (!axis.cross( up))
			up = vector( 0,1,0);
	}

	pos = R * ( pos);
	axis = R.times_v( axis);
	up = R.times_v( up);
}

vector 
Primitive::getScale()
{
	return vector();
}

void
Primitive::set_pos( const vector& v)
{
	pos = v;
}

void
Primitive::set_x( const double& x)
{
	pos.set_x( x);
}

void
Primitive::set_y( const double& y)
{
	pos.set_y( y);
}

void
Primitive::set_z( const double& z)
{
	pos.set_z( z);
}

void
Primitive::set_axis( const vector& v)
{
	axis = v;
}

void
Primitive::set_up( const vector& v)
{
	up = v;
}

void
Primitive::set_color( const rgb& t)
{
	write_lock L( mtx);
	this->color = t;
}

void
Primitive::set_green( const float& green)
{
	write_lock L( mtx);
	this->color.g = green;
}

void
Primitive::set_blue( const float& blue)
{
	write_lock L(mtx);
	this->color.b = blue;
}

void
Primitive::set_red( const float& red)
{
	write_lock L(mtx);
	this->color.r = red;
}

rgb
Primitive::get_color()
{
	return color;
}

void 
Primitive::refreshCache() 
{
	degenerate = !( axis);

	if (!degenerate) {
		scale = getScale();

		vector W = axis.cross( up);
		if (!W) {
			W = axis.cross( vector(1,0,0));
			if (!W) {
				W = axis.cross( vector(0,1,0));
			}
		}
		W = W.norm();
		vector H = W.cross( axis ).norm();
		vector L = axis.norm();

		mwt.x_column( L * scale.x );
		mwt.y_column( H * scale.y );
		mwt.z_column( W * scale.z );
		mwt.w_column( pos );
		mwt.w_row();

		wlt.x_column( L.x, H.x, W.x );
		wlt.y_column( L.y, H.y, W.y );
		wlt.z_column( L.z, H.z, W.z );
		wlt.w_column();
		wlt.w_row();
	}
}

void
Primitive::set_pos_t( const boost::python::object& t)
{
	this->set_pos( vector(t));
}

void
Primitive::set_axis_t( const boost::python::object& t)
{
	this->set_axis( vector(t));
}

void
Primitive::set_up_t( const boost::python::object& t)
{
	this->set_up( vector(t));
}

void
primitive_init_type()
{
	using namespace boost::python;
	
	// No constructor is revealed to python for this base class.
	class_<Primitive, bases<DisplayObject>, boost::noncopyable>("primitive", no_init)
		.def( "_get_axis", &Primitive::get_axis, return_internal_reference<>())
		.def( "_set_axis", &Primitive::set_axis)
		.add_property( "color", &Primitive::get_color, &Primitive::set_color)
			.add_property( "red", &Primitive::get_red, &Primitive::set_red)
			.add_property( "green", &Primitive::get_green, &Primitive::set_green)
			.add_property( "blue", &Primitive::get_blue, &Primitive::set_blue)
		.def( "_set_pos", &Primitive::set_pos)
		.def( "_get_pos", &Primitive::get_pos, return_internal_reference<>())
			.add_property( "x", &Primitive::get_x, &Primitive::set_x)
			.add_property( "y", &Primitive::get_y, &Primitive::set_y)
			.add_property( "z", &Primitive::get_z, &Primitive::set_z)
		.def( "_set_up", &Primitive::set_up)
		.def( "_get_up", &Primitive::get_up, return_internal_reference<>())
		.def( "rotate", raw_function( py_rotate<Primitive>, 2))
#ifdef VISUAL_ALLOW_PYTHON_PRIMITIVES
		// The following functions are completely new for the purpose of implementing new primitives in python.
		.def( "glRender", &Primitive::glRender, "New primitives must override this function to preform rendering operations.")
		.def( "refreshCache", &Primitive::refreshCache
			, "New primitives may override this function to perform rendering operations.")
		// .def( "getScale", &Primitive::getScale, "New primitives may override this function to perform rendering operations.")
		.def( "rayIntersect", &Primitive::rayIntersect
			, "New Primitives must override this function to perform rendering operations.")
		.def( "read_lock", &Primitive::py_read_lock
			, "Obtain a thread-safe lock in the rendering loop.  Call unlock() when done to prevent a deadlock.")
		.def( "write_lock", &Primitive::py_write_lock
			, "Obtain a thread-safe lock in the Python application loop.  Use this function for *all* new setters.  Call unlock() when done to prevent a deadlock.")
		.def( "unlock", &Primitive::py_unlock
			, "Release the lock obtained by a call to write_lock() or read_lock().")
#endif // !VISUAL_ALLOW_PYTHON_PRIMITIVES
		;

}


} // !namespace visual
