// Copyright (c) 2000, 2001, 2002, 2003 by David Scherer and others.
// See the file license.txt for complete license terms.
// See the file authors.txt for a complete list of contributors.

#include "curve.h"
#include "slice.h"

#include <stdexcept>
#include <cassert>
#include <iostream>

#include <boost/python/class.hpp>
#include <boost/python/object.hpp>
#include <boost/python/object_operators.hpp>
#include <boost/python/return_internal_reference.hpp>
#include <boost/python/extract.hpp>

#include <boost/version.hpp>

#include GL_INCLUDE    // see platform.h

using boost::python::object;
using visual::python::slice;
using boost::python::make_tuple;

// Recall that the default constructor for object() is a reference to None.

namespace visual {

namespace {
	
// returns a pointer to the ith vector in the array.
double* index( array a, size_t i)
{
	// This is technically an unsafe cast since the alignment requirement
	// goes up for the cast.  It is made safe by padding actions within the Numeric
	// library itself, but the compiler doesn't know that, so I am just using a
	// raw cast vice a static_cast<>.
	return ((double*)data(a)) + i * 3;
}
	
} // !namespace visual::(unnamed)


curve::curve()
	: pos( 0), color( 0), antialias( false),
	degenerate(true), radius(0.0),
	preallocated_size(256),
	count(0)
{
	// Perform initial allocation of storage space for the buggar.
	std::vector<int> dims(2);
	dims[0] = 256;
	dims[1] = 3;
	pos = makeNum( dims);
	color = makeNum( dims);
	double* pos_i = index( pos, 0);
	double* color_i = index( color, 0);
	pos_i[0] = 0;
	pos_i[1] = 0;
	pos_i[2] = 0;
	color_i[0] = 1;
	color_i[1] = 1;
	color_i[2] = 1;

	
	const int curve_around = 4;

	for (int i=0; i<curve_around; i++) {
		curve_sc[i]  = std::cos(i * 2 * pi / curve_around);
		curve_sc[i+curve_around] = std::sin(i * 2 * pi / curve_around);
	}

	for (int i=0; i<128; i++) {
		curve_slice[i*2] = i*curve_around;
		curve_slice[i*2+1] = i*curve_around + 1;
		curve_slice[i*2 + 256] = i*curve_around + (curve_around - 1);
		curve_slice[i*2 + 257] = i*curve_around;
	}
}

curve::curve( const curve& other)
	: DisplayObject( other), pos( other.pos), color( other.color), 
	antialias( other.antialias), degenerate( other.degenerate), 
	radius( other.radius), preallocated_size( other.preallocated_size),
	count( other.count)
{
	const int curve_around = 4;
	for (int i=0; i<curve_around; i++) {
		curve_sc[i]  = std::cos(i * 2 * pi / curve_around);
		curve_sc[i+curve_around] = std::sin(i * 2 * pi / curve_around);
	}

	for (int i=0; i<128; i++) {
		curve_slice[i*2] = i*curve_around;
		curve_slice[i*2+1] = i*curve_around + 1;
		curve_slice[i*2 + 256] = i*curve_around + (curve_around - 1);
		curve_slice[i*2 + 257] = i*curve_around;
	}	
}

object
curve::get_pos()
{
	return pos[slice(0, (int)count)];
}

object
curve::get_color()
{
	return color[slice(0, (int)count)];
}

void
curve::set_length( size_t length)
{
	size_t npoints = count;
	if (npoints > length) // A shrink operation - never done by VPython.
		npoints = length;
	if (npoints == 0)
		// The first allocation.
		npoints = 1;
		
	if (length > preallocated_size) {
		std::vector<int> dims(2);
		dims[0] = 2 * length;
		dims[1] = 3;
		array n_pos = makeNum( dims);
		array n_color = makeNum( dims);
		std::memcpy( data( n_pos), data( pos), sizeof(double) * 3 * npoints);
		std::memcpy( data( n_color), data( color), sizeof(double) * 3 * npoints);
		pos = n_pos;
		color = n_color;
		preallocated_size = dims[0];
	}
	if (length > npoints) {
		// Copy the last good element to the new positions.
		const double* last_element = index( pos, npoints-1);
		double* element_i = index( pos, npoints);
		double* element_end = index( pos, length);
		while (element_i < element_end) {
			element_i[0] = last_element[0];
			element_i[1] = last_element[1];
			element_i[2] = last_element[2];
			element_i += 3;
		}
		
		last_element = index( color, npoints-1);
		element_i = index( color, npoints);
		element_end = index( color, length);
		while (element_i < element_end) {
			element_i[0] = last_element[0];
			element_i[1] = last_element[1];
			element_i[2] = last_element[2];
			element_i += 3;
		}
	}
	count = length;
}

// All forms of append set up and call this one.
void
curve::append_rgb( vector npos, double red, double blue, double green) 
{
	write_lock L(mtx);
	set_length( count+1);
	double* last_pos = index( pos, count-1);
	double* last_color = index( color, count-1);
	last_pos[0] = npos.x;
	last_pos[1] = npos.y;
	last_pos[2] = npos.z;
	if (red > 0)
		last_color[0] = red;
	if (blue > 0)
		last_color[1] = green;
	if (green > 0)
		last_color[2] = blue;
}

void
curve::append( vector npos)
{
	write_lock L(mtx);
	set_length( count+1);
	double* last_pos = index( pos, count-1);
	last_pos[0] = npos.x;
	last_pos[1] = npos.y;
	last_pos[2] = npos.z;	
}

void
curve::append( vector npos, rgb ncolor)
{
	write_lock L(mtx);
	set_length( count+1);
	double* last_pos = index( pos, count-1);
	double* last_color = index( color, count-1);
	last_pos[0] = npos.x;
	last_pos[1] = npos.y;
	last_pos[2] = npos.z;
	last_color[0] = ncolor.r;
	last_color[1] = ncolor.g;
	last_color[2] = ncolor.b;	
}

void
curve::set_pos( array n_pos)
{
	array_types t = type( n_pos);
	if (t != double_t) {
		n_pos = astype( n_pos, double_t);
	}
	std::vector<int> dims = shape( n_pos);
	if (dims.size() == 1 && !dims[0]) {
		write_lock lock(mtx);
		set_length(0);
		return;
	}
	if (dims.size() != 2) {
		throw std::invalid_argument( "pos must be an Nx3 array");
	}
	if (dims[1] == 2) {
		write_lock L(mtx);
		set_length( dims[0]);
		pos[make_tuple(slice(0, count), slice(0,2))] = n_pos;
		pos[make_tuple(slice(0, count), 2)] = 0.0;
		return;
	}
	else if (dims[1] == 3) {
		write_lock L(mtx);
		set_length( dims[0]);
		pos[make_tuple(slice(0, count), slice())] = n_pos;
		return;
	}
	else {
		throw std::invalid_argument( "pos must be an Nx3 array");
	}
}

void
curve::set_pos_l( const list& pos)
{
	this->set_pos( array(pos));
}

// Interpreted as an initial append operation, with no color specified.
void
curve::set_pos_v( const vector& npos)
{
	using namespace boost::python;
	tuple t_pos = make_tuple( npos.as_tuple());
	set_pos( array(t_pos));
}

void
curve::set_color( array n_color)
{
	array_types t = type(n_color);
	if (t != double_t) {
		n_color = astype( n_color, double_t);
	}
	std::vector<int> dims = shape( n_color);
	if (dims.size() == 1 && dims[0] == 3) {
		// A single color, broadcast across the entire (used) array.
		int npoints = (count) ? count : 1;
		write_lock L(mtx);
		color[slice(0,npoints)] = n_color;
		return;
	}
	if (dims.size() == 2 && dims[1] == 3) {
		if (dims[0] != count) {
			throw std::invalid_argument( "color must be the same length as pos.");
		}
		write_lock L(mtx);
		color[slice(0, count)] = n_color;
		return;
	}
	throw std::invalid_argument( "color must be an Nx3 array");
}

void
curve::set_color_l( const list& color)
{
	this->set_color( array(color));
}

void
curve::set_color_t( const tuple& color)
{
	this->set_color( array(color));
}

void
curve::set_color_v( const vector& n_color)
{
	set_color( array(make_tuple( n_color.as_tuple())));
}

void
curve::set_red( const array& red)
{
	write_lock L(mtx);
	set_length( shape(red).at(0));
	color[make_tuple(slice(0,count), 0)] = red;
}

void
curve::set_green( const array& green)
{
	write_lock L(mtx);
	set_length( shape(green).at(0));
	color[make_tuple(slice(0,count), 1)] = green;
}

void
curve::set_blue( const array& blue)
{
	write_lock L(mtx);
	set_length( shape(blue).at(0));
	color[make_tuple(slice(0,count), 2)] = blue;
}

void
curve::set_x( const array& x)
{
	write_lock L(mtx);
	set_length( shape(x).at(0));
	pos[make_tuple( slice(0, count), 0)] = x;
}

void
curve::set_y( const array& y)
{
	write_lock L(mtx);
	set_length( shape(y).at(0));
	pos[make_tuple( slice(0, count), 1)] = y;
}

void
curve::set_z( const array& z)
{
	write_lock L(mtx);
	set_length( shape(z).at(0));
	pos[make_tuple( slice(0, count), 2)] = z;
}

void
curve::set_red_l( const list& red)
{
	this->set_red( array(red));
}

void
curve::set_green_l( const list& green)
{
	this->set_green( array(green));
}

void
curve::set_blue_l( const list& blue)
{
	this->set_blue( array(blue));
}

void
curve::set_x_l( const list& x)
{
	set_x( array(x));
}

void
curve::set_y_l( const list& y)
{
	set_y( array(y));	
}

void
curve::set_z_l( const list& z)
{
	set_z( array(z));
}

void
curve::set_x_d( const double x)
{
	write_lock L(mtx);
	if (count == 0) {
		set_length(1);
	}
	pos[make_tuple(slice(0,count), 0)] = x;	
}

void
curve::set_y_d( const double y)
{
	write_lock L(mtx);
	if (count == 0) {
		set_length(1);
	}
	pos[make_tuple(slice(0,count), 1)] = y;	
}

void
curve::set_z_d( const double z)
{
	write_lock L(mtx);
	if (count == 0) {
		set_length(1);
	}
	pos[make_tuple(slice(0,count), 2)] = z;	
}

void
curve::set_red_d( const double red)
{
	write_lock L(mtx);
	if (count == 0) {
		set_length(1);
	}
	color[make_tuple(slice(0,count), 0)] = red;	
}

void
curve::set_green_d( const double green)
{
	write_lock L(mtx);
	if (count == 0) {
		set_length(1);
	}
	color[make_tuple(slice(0,count), 1)] = green;	
}

void
curve::set_blue_d( const double blue)
{
	write_lock L(mtx);
	if (count == 0) {
		set_length(1);
	}
	color[make_tuple(slice(0,count), 2)] = blue;	
}

void
curve::set_radius( const double& radius)
{
	write_lock L(mtx);
	this->radius = radius;
}

void
curve::set_antialias( bool aa)
{
	write_lock L(mtx);
	this->antialias = aa;
}

void
curve::refreshCache()
{
	if (count < 2 )
		degenerate = true;
	else degenerate = false;
}

void
curve::glRender( rView& view)
{
	if (degenerate)
		return;
	
	if (radius) {
		if (antialias)
			// Not the right way to go about AA, but oh, well.
			glEnable(GL_POLYGON_SMOOTH);

		thickline( view);

		if (antialias)
			glDisable(GL_POLYGON_SMOOTH);
	}
	else
		thinline( view);
}

void
curve::thickline( rView& view)
{
	// double* v, double* color, double radius, int count
	const int MAX_SEGMENTS = 1024;
	const int curve_around = 4;
	static vertex projected[MAX_SEGMENTS * curve_around];
	static float light[MAX_SEGMENTS * curve_around * 3];
	int step = (count + MAX_SEGMENTS-1) / MAX_SEGMENTS;

	vertex *pr = projected;
	float *lt = light;

	float *cost = curve_sc;
	float *sint = cost + curve_around;

	vector x, y, lastx, lasty;

	int npoints = 0;
	
	// pos and color iterators
	const double* v_i = (double*)( data(pos));
	const double* c_i = (double*)( data(color));

	for(size_t corner=0; corner < count/step; ++corner, v_i += step*3, c_i += step*3) {
		// The vector to which v_i currently points towards.
		vector current = vector( v_i[0], v_i[1], v_i[2]);
		view.ext_point( current);

		/* To construct the circle of vertices around a point on the curve,
		 * I need an orthonormal basis of the plane of the circle.  A and B,
		 * normalized vectors pointing toward the next and from the previous
		 * points on the curve, are vectors in the plane of the *curve* at
		 * this point.  The cross product and the difference of these vectors
		 * are orthogonal and lie in the appropriate plane.
		 */

		if (corner && (corner+1 < count/step)) {
			// Any point except the first and last.
			vector next( v_i[3], v_i[4], v_i[5]); // The next vector in pos
			vector prev( v_i[-3], v_i[-2], v_i[-1]); // The previous vector in pos
			vector A = next - current;
			vector B = current - prev;
			x = (A.norm()-B.norm()).norm();
			y = A.cross(B).norm();
			if (!x || !y) {
				x = lastx;
				y = lasty;
			}
		}
		else if (corner) {
			// The last point
			vector prev( v_i[-3], v_i[-2], v_i[-1]);
			vector A = current - prev;
			A = A.norm();
			x = A.cross( vector(0,0,1));
			if (!x)
				x = A.cross( vector(1,0,0));
			x = x.norm();
			y = A.cross(x);
		}
		else {
			// The first point
			vector next( v_i[3], v_i[4], v_i[5]);
			vector A = next - current;
			A = A.norm();
			x = A.cross( vector(0,0,1));
			if (!x)
				x = A.cross( vector(1,0,0));
			x = x.norm();
			y = A.cross(x);
		}

		// experimental: kink prevention
		if (x.dot(lastx) < 0)
			x = -x;
		if (y.dot(lasty) < 0)
			y = -y;
		lastx = x; lasty = y;

		float lx[2], ly[2];
		for(int li=0; li < view.lights.n_lights; li++) {
			lx[li] = view.lights.L[li].dot( x );
			ly[li] = view.lights.L[li].dot( y );
		}

		x = x * radius;
		y = y * radius;
		
		// Manipulate colors when we are in stereo mode.
		rgb rendered_color( static_cast<float>(c_i[0]), 
		                    static_cast<float>(c_i[1]),
		                    static_cast<float>(c_i[2]));
		if (view.anaglyph) {
			if (view.coloranaglyph) {
				rendered_color = rendered_color.unsaturate();
			}
			else {
				rendered_color.r = 
				rendered_color.g = 
				rendered_color.b = rendered_color.grayscale();
			}
		}
		
		npoints++;
		for (int a=0; a < curve_around; a++) {
			float c = cost[a];
			float s = sint[a];
			
			float illum = view.lights.ambient;
			for(int li=0; li < view.lights.n_lights; li++) {
				float dot = c*lx[li] + s*ly[li];
				if (dot > 0.0)
					illum += dot;
			}

			view.wct.project(current + x*c + y*s, *pr++);
			*(lt++) = illum * rendered_color.r;
			*(lt++) = illum * rendered_color.g;
			*(lt++) = illum * rendered_color.b;
		}
	}
	glEnableClientState(GL_VERTEX_ARRAY);
	glEnableClientState(GL_COLOR_ARRAY);
	glShadeModel(GL_SMOOTH);

	int *ind = curve_slice;
	for(int a=0; a<curve_around; a++) {
		int ai = a;
		if (a == curve_around-1) {
			ind += 256;
			ai = 0;
		}

		for(int i=0; i<npoints; i+=127) {
			glVertexPointer(4, GL_DOUBLE, sizeof(vertex), &projected[i*curve_around + ai].x);
			glColorPointer(3, GL_FLOAT, sizeof(float)*3, &light[(i*curve_around + ai)*3] );

			if (npoints-i < 128)
				glDrawElements(GL_TRIANGLE_STRIP, 2*(npoints-i), GL_UNSIGNED_INT, ind);
			else
				glDrawElements(GL_TRIANGLE_STRIP, 256, GL_UNSIGNED_INT, ind);
		}
	}
}

void
curve::thinline( rView& view )
{
	// The maximum number of line segments to draw.
	const int LINE_LENGTH = 1024;
	// Data storage for the color data.
	static float color_data[LINE_LENGTH][3];

	// The number of steps to take for each stride.
	// This evaluates to 1 when count < LINE_LENGTH, and something larger else.
	int step = (count + LINE_LENGTH - 1) / LINE_LENGTH;

	const double* p_i = (double*)( data(this->pos));
	const double* c_i = (double*)( data(this->color));

	for (size_t i = 0; i < count / step; ++i, p_i += step*3, c_i += step*3) {
		view.ext_point( p_i);
		
		if (view.anaglyph) {
			rgb rendered_color( c_i[0], c_i[1], c_i[2]);
			if (view.coloranaglyph) {
				rendered_color = rendered_color.unsaturate();
			}
			else {
				rendered_color.r = 
				rendered_color.g = 
				rendered_color.b = rendered_color.grayscale();
			}
			color_data[i][0] = rendered_color.r;
			color_data[i][1] = rendered_color.g;
			color_data[i][2] = rendered_color.b;
		}
	}
	
	view.wct.gl_load();

	glEnableClientState( GL_VERTEX_ARRAY);
	glEnableClientState( GL_COLOR_ARRAY);
	glVertexPointer( 3, GL_DOUBLE, 3*sizeof(double)*step, data(this->pos));
	if (view.anaglyph)
		glColorPointer( 3, GL_FLOAT, 3*sizeof(float), color_data);
	else
		glColorPointer( 3, GL_DOUBLE, 3*sizeof(double)*step, data(this->color));
	glShadeModel( GL_SMOOTH);

	glDrawArrays(GL_LINE_STRIP, 0, count/step);
	glLoadIdentity();
}

void
curve_init_type()
{
	using namespace boost::python;

	void (curve::*append_v_rgb)( vector, rgb) = &curve::append;
	void (curve::*append_v)( vector) = &curve::append;
	
	class_<curve, bases<DisplayObject>, boost::shared_ptr<curve> >( "curve")
		.def( init<const curve&>())
		.add_property( "radius", &curve::get_radius, &curve::set_radius)  // AKA thickness.
		.add_property( "antialias", &curve::get_antialias, &curve::set_antialias)
		.def( "_get_color", &curve::get_color)
		.def( "_set_color", &curve::set_color_l)
		.def( "_set_color", &curve::set_color_v)
		.def( "_set_color", &curve::set_color_t)
		.def( "_set_color", &curve::set_color)
		.def( "_set_red", &curve::set_red_l)
		.def( "_set_red", &curve::set_red_d)
		.def( "_set_red", &curve::set_red)
		.def( "_set_green", &curve::set_green_l)
		.def( "_set_green", &curve::set_green_d)
		.def( "_set_green", &curve::set_green)
		.def( "_set_blue", &curve::set_blue_l)
		.def( "_set_blue", &curve::set_blue_d)
		.def( "_set_blue", &curve::set_blue)
		.def( "_set_pos", &curve::set_pos_v)
		.def( "_get_pos", &curve::get_pos)
		.def( "_set_pos", &curve::set_pos_l)
		.def( "_set_pos", &curve::set_pos)
		.def( "_set_x", &curve::set_x_l)
		.def( "_set_x", &curve::set_x_d)
		.def( "_set_x", &curve::set_x)
		.def( "_set_y", &curve::set_y_l)
		.def( "_set_y", &curve::set_y_d)
		.def( "_set_y", &curve::set_y)
		.def( "_set_z", &curve::set_z_l)
		.def( "_set_z", &curve::set_z_d)
		.def( "_set_z", &curve::set_z)
#if BOOST_VERSION / 100 % 1000 >= 31
		.def( "append", &curve::append_rgb, 
			(args("pos"), args("r")=-1, args("g")=-1, args("b")=-1))
#else
# warning "curve.append( pos, r, g, b) is not available without Boost.Python 1.31 or higher"
#endif
		.def( "append", append_v_rgb, args( "pos", "color"))
		.def( "append", append_v, args("pos"))
		;
}


} // !namespace visual
