#!/usr/bin/env python

from __future__ import division # confidence high

CONTACT = "Michael Droettboom"
EMAIL = "mdroe@stsci.edu"

import cStringIO
from distutils.core import setup, Extension
from os.path import join
import os.path
import sys

######################################################################
# CONFIGURATION
# BUILD may be 'debug', 'profile', or 'release'
BUILD = 'release'
OPENMP = False

######################################################################
# Helper class
def write_if_different(filename, data):
    if os.path.exists(filename):
        fd = open(filename, 'r')
        original_data = fd.read()
        fd.close()
    else:
        original_data = None

    if original_data != data:
        fd = open(filename, 'w')
        fd.write(data)
        fd.close()

######################################################################
# NUMPY
try:
    import numpy
except ImportError:
    print "numpy must be installed to build pywcs."
    print "ABORTING."
    raise

major, minor, rest = numpy.__version__.split(".", 2)
if (major, minor) < (1, 3):
    print "numpy version 1.3 or later must be installed to build pywcs."
    print "ABORTING."
    raise ImportError

try:
    numpy_include = numpy.get_include()
except AttributeError:
    numpy_include = numpy.get_numpy_include()

######################################################################
# WCSLIB
WCSVERSION = "4.7"
WCSLIB = "wcslib" # Path to wcslib
WCSLIBC = join(WCSLIB, "C") # Path to wcslib source files
WCSFILES = [ # List of wcslib files to compile
    'flexed/wcsbth.c',
    'flexed/wcspih.c',
    'flexed/wcsulex.c',
    'flexed/wcsutrn.c',
    'cel.c',
    'lin.c',
    'log.c',
    'prj.c',
    'spc.c',
    'sph.c',
    'spx.c',
    'tab.c',
    'wcs.c',
    'wcsfix.c',
    'wcshdr.c',
    'wcsprintf.c',
    'wcsunits.c',
    'wcsutil.c']
WCSFILES = [join(WCSLIBC, x) for x in WCSFILES]

######################################################################
# WCSLIB CONFIGURATION

# The only configuration parameter needed at compile-time is how to
# specify a 64-bit signed integer.  Python's ctypes module can get us
# that information, but it is only available in Python 2.5 or later.
# If we can't be absolutely certain, we default to "long long int",
# which is correct on most platforms (x86, x86_64).  If we find
# platforms where this heuristic doesn't work, we may need to hardcode
# for them.
def determine_64_bit_int():
    try:
        try:
            import ctypes
        except ImportError:
            raise ValueError()

        if ctypes.sizeof(ctypes.c_longlong) == 8:
            return "long long int"
        elif ctypes.sizeof(ctypes.c_long) == 8:
            return "long int"
        elif ctypes.sizeof(ctypes.c_int) == 8:
            return "int"
        else:
            raise ValueError()

    except ValueError:
        return "long long int"

if os.path.exists("pywcs"):
    srcroot = 'pywcs'
else:
    srcroot = '.'
h_file = cStringIO.StringIO()
h_file.write("""
/* WCSLIB library version number. */
#define WCSLIB_VERSION %s

/* 64-bit integer data type. */
#define WCSLIB_INT64 %s
""" % (WCSVERSION, determine_64_bit_int()))
if sys.platform in ('win32', 'cygwin'):
    h_file.write("""
#define wcsset wcsset_
""")
write_if_different(join(srcroot, 'src', 'wcsconfig.h'), h_file.getvalue())

######################################################################
# GENERATE DOCSTRINGS IN C
sys.path.append(join('.', srcroot, "lib"))
docstrings = {}
execfile(join(srcroot, 'doc', 'docstrings.py'), docstrings)
keys = [key for key in docstrings.keys()
        if not key.startswith('__') and type(key) in (str, unicode)]
keys.sort()
for key in keys:
    docstrings[key] = docstrings[key].encode('utf8').lstrip() + '\0'

h_file = cStringIO.StringIO()
h_file.write("""/*
DO NOT EDIT!

This file is autogenerated by setup.py.  To edit its contents,
edit doc/docstrings.py
*/

#ifndef __DOCSTRINGS_H__
#define __DOCSTRINGS_H__

void fill_docstrings(void);

""")
for key in keys:
    val = docstrings[key]
    h_file.write('extern char doc_%s[%d];\n' % (key, len(val)))
h_file.write("\n#endif\n\n")

write_if_different(join(srcroot, 'src', 'docstrings.h'), h_file.getvalue())

c_file = cStringIO.StringIO()
c_file.write("""/*
DO NOT EDIT!

This file is autogenerated by setup.py.  To edit its contents,
edit doc/docstrings.py

The weirdness here with strncpy is because some C compilers, notably
MSVC, do not support string literals greater than 256 characters.
*/

#include <string.h>
#include "docstrings.h"

""")
for key in keys:
    val = docstrings[key]
    c_file.write('char doc_%s[%d];\n' % (key, len(val)))

c_file.write("\nvoid fill_docstrings(void)\n{\n")
for key in keys:
    val = docstrings[key]
    # For portability across various compilers, we need to fill the
    # docstrings in 256-character chunks
    for i in range(0, len(val), 256):
        chunk = val[i:i+256].encode("string_escape").replace('"', '\\"')
        c_file.write('   strncpy(doc_%s + %d, "%s", %d);\n' % (
                key, i, chunk, min(len(val) - i, 256)))
    c_file.write("\n")
c_file.write("\n}\n\n")

write_if_different(join(srcroot, 'src', 'docstrings.c'), c_file.getvalue())

######################################################################
# PYWCS-SPECIFIC AND WRAPPER SOURCE FILES
PYWCS_VERSION = '1.10'
VERSION = '%s-%s' % (PYWCS_VERSION, WCSVERSION)
PYWCS_SOURCES = [ # List of pywcs files to compile
    'distortion.c',
    'distortion_wrap.c',
    'docstrings.c',
    'pipeline.c',
    'pyutil.c',
    'pywcs.c',
    'pywcs_api.c',
    'sip.c',
    'sip_wrap.c',
    'str_list_proxy.c',
    'wcslib_wrap.c',
    'wcslib_tabprm_wrap.c',
    'wcslib_units_wrap.c',
    'wcslib_wtbarr_wrap.c']
PYWCS_SOURCES = [join('src', x) for x in PYWCS_SOURCES]

######################################################################
# DISTUTILS SETUP
libraries = []
define_macros = [('ECHO', None),
                 ('WCSTRIG_MACRO', None),
                 ('PYWCS_BUILD', None),
                 ('_GNU_SOURCE', None)]
undef_macros = []
extra_compile_args = []
if BUILD.lower() == 'debug':
    define_macros.append(('DEBUG', None))
    undef_macros.append('NDEBUG')
    if not sys.platform.startswith('sun') and \
       not sys.platform == 'win32':
        extra_compile_args.extend(["-fno-inline", "-O0", "-g"])
elif BUILD.lower() == 'profile':
    define_macros.append(('NDEBUG', None))
    undef_macros.append('DEBUG')
    if not sys.platform.startswith('sun'):
        extra_compile_args.extend(["-O3", "-g"])
elif BUILD.lower() == 'release':
    # Define ECHO as nothing to prevent spurious newlines from
    # printing within the libwcs parser
    define_macros.append(('NDEBUG', None))
    undef_macros.append('DEBUG')
else:
    raise ValueError("BUILD should be one of 'debug', 'profile', or 'release'")

if sys.platform == 'win32':
    define_macros.append(('YY_NO_UNISTD_H', None))
    define_macros.append(('_CRT_SECURE_NO_WARNINGS', None))

if sys.platform.startswith('linux'):
    define_macros.append(('HAVE_SINCOS', None))

if not sys.platform.startswith('sun') and \
   not sys.platform == 'win32':
    if OPENMP:
        extra_compile_args.append('-fopenmp')
        libraries.append('gomp')
    else:
        extra_compile_args.extend(['-Wno-unknown-pragmas'])

PYWCS_EXTENSIONS = [
    Extension('pywcs._pywcs',
              WCSFILES + PYWCS_SOURCES,
              include_dirs =
              [numpy_include,
               join(srcroot, WCSLIBC),
               WCSLIBC,
               join(srcroot, "src")
               ],
              define_macros=define_macros,
              undef_macros=undef_macros,
              extra_compile_args=extra_compile_args,
              libraries=libraries
              )
    ]

pkg = ["pywcs", "pywcs.tests"]

setupargs = {
    'version' :     VERSION,
    'description':  "Python wrappers to WCSLIB",
    'author' :      CONTACT,
    'author_email': EMAIL,
    'url' :         "http://projects.scipy.org/astropy/astrolib/wiki/WikiStart",
    'platforms' :   ["unix","windows"],
    'ext_modules' : PYWCS_EXTENSIONS,
    'data_files' : [
        ( 'pywcs/include', ['src/*.h']),
        ( 'pywcs/include/wcslib', [ WCSLIBC + '/*.h'] ),
        ( 'pywcs/tests/maps', ['lib/tests/maps/*.fits']),
        ( 'pywcs/tests/spectra', ['lib/tests/spectra/*.fits'])
        ],
    'package_dir' : {pkg[0]: 'lib', pkg[1]: 'lib/tests'},
}

