"""
  Filename:  glpk.py (define a class for calling GLPK from Python)

-- This code is part of the Python-GLPK interface.
--
-- Copyright (C) 2005, Joao Pedro Pedroso
-- Faculdade de Ciencias, Universidade do Porto
-- Porto, Portugal. All rights reserved. E-mail: <jpp@fc.up.pt>.
--
-- GLPK is free software; you can redistribute it and/or modify it
-- under the terms of the GNU General Public License as published by
-- the Free Software Foundation; either version 2, or (at your option)
-- any later version.
--
-- GLPK is distributed in the hope that it will be useful, but WITHOUT
-- ANY WARRANTY; without even the implied warranty of MERCHANTABILITY
-- or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public
-- License for more details.
--
-- You should have received a copy of the GNU General Public License
-- along with GLPK; see the file COPYING. If not, write to the Free
-- Software Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA
-- 02110-1301, USA.
"""

from glpkpi import *
from array import *

class lpx:
    def __init__(self, mod_file = None, dat_file = None):
        if mod_file == None:
            self.lp = lpx_create_prob();
        elif mod_file.find(".mps") != -1:
            self.lp = lpx_read_mps(mod_file);
        elif mod_file.find(".mod") != -1:
            if dat_file == None:
                self.lp = lpx_read_model(mod_file, None, None);
            else:
                self.lp = lpx_read_model(mod_file, dat_file, None);
        elif mod_file.find(".lp") != -1:
            self.lp = lpx_read_cpxlp(mod_file);

        lpx_set_int_parm(self.lp, LPX_K_MSGLEV, 0);  # silent /verbose modes

        self.names = {}		# map variable names to indices (for integer vars)
        self.indices = {}	# map variable indices to names (for integer vars)
        self.cvars = []		# continuous variables
        self.ivars = []		# integer variables
        self.nvars = lpx_get_num_cols(self.lp)
        self.lb = []		# lower bounds
        self.ub = []		# upper bounds
        for i in range(1,self.nvars+1):
            name = lpx_get_col_name(self.lp, i)
            if lpx_get_class(self.lp) == LPX_LP or lpx_get_col_kind(self.lp, i) == LPX_CV:
                self.cvars.append(i)
            elif lpx_get_col_kind(self.lp, i) == LPX_IV:
                self.names[i] = name
                self.indices[name] = i
                self.ivars.append(i)
                self.lb.append(lpx_get_col_lb(self.lp, i))
                self.ub.append(lpx_get_col_ub(self.lp, i))
            else:
                print "unkown col kind"
                raise Exception
        

    def solve(self):
        if lpx_get_class(self.lp) == LPX_LP:
            lpx_simplex(self.lp)
        else:	# problem is MIP
            lpx_set_int_parm(self.lp, LPX_K_DUAL, 1);    # use dual simplex if sol is dual feasible
            lpx_simplex(self.lp)
            lpx_integer(self.lp)

        #if lpx_get_status(self.lp) != LPX_OPT:
        #    raise Exception

        self.instanciate_solution()


    def instanciate_solution(self):
        """update self.sol and self.x"""
        sol = {}
        x = array('d', [0. for i in range(self.nvars+1)])
        name = lpx_get_obj_name(self.lp)
        if lpx_get_class(self.lp) == LPX_LP:
            value = lpx_get_obj_val(self.lp)
            x[0] = value
            sol[name] = value
            for i in range(1,lpx_get_num_cols(self.lp)+1):
                name = lpx_get_col_name(self.lp, i)
                value = lpx_get_col_prim(self.lp, i)
                sol[name] = value
                x[i] = value
        else:	# problem is MIP
            value = lpx_mip_obj_val(self.lp)
            x[0] = value
            sol[name] = value
            for i in range(1,lpx_get_num_cols(self.lp)+1):
                name = lpx_get_col_name(self.lp, i)
                if lpx_get_col_kind(self.lp, i) == LPX_CV:
                    value = lpx_mip_col_val(self.lp, i)
                elif lpx_get_col_kind(self.lp, i) == LPX_IV:
                    # to avoid finite precision problems:
                    value = int(round(lpx_mip_col_val(self.lp, i),0))
                else:
                    print "unkown col kind"
                    raise Exception
                sol[name] = value
                x[i] = value

        self.sol = sol
        self.x = x


if __name__ == "__main__":
    print "starting..."
    example = lpx("examples/example.mod")
    example.solve()
    print "solution:", example.sol
    print "solution is also here:", example.x
    
