import cStringIO

from Ft.Lib.Uri import OsPathToUri
from Ft.Xml.InputSource import InputSourceFactory, DefaultFactory
from Ft.Xml.Xslt import XsltException, Error
from Ft.Xml.Xslt.Processor import Processor

from Xml.Xslt import test_harness


sheet_str_1 = """<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
version="1.0">

  <xsl:include href='Xml/Xslt/Core/addr_book1.xsl'/>

  <xsl:template match='PHONENUM' priority='10'>
    <xsl:element name='p'>
      <xsl:attribute name='align'>center</xsl:attribute>
      <xsl:value-of select='@DESC'/><xsl:text>: </xsl:text>
      <xsl:apply-templates/>
    </xsl:element>
  </xsl:template>

</xsl:stylesheet>
"""

expected_1 = """<html>
  <head>
    <meta http-equiv='Content-Type' content='text/html; charset=iso-8859-1'>
    <title>Address Book</title>
  </head>
  <body>
    <h1>Tabulate Just Names and Phone Numbers</h1>
    <table>
      <tr>
        <td align='center'><b>Pieter Aaron</b></td>
        <td>
          <p align='center'>Work: 404-555-1234</p>
          <p align='center'>Fax: 404-555-4321</p>
          <p align='center'>Pager: 404-555-5555</p>
        </td>
      </tr>
      <tr>
        <td align='center'><b>Emeka Ndubuisi</b></td>
        <td>
          <p align='center'>Work: 767-555-7676</p>
          <p align='center'>Fax: 767-555-7642</p>
          <p align='center'>Pager: 800-SKY-PAGEx767676</p>
        </td>
      </tr>
      <tr>
        <td align='center'><b>Vasia Zhugenev</b></td>
        <td>
          <p align='center'>Work: 000-987-6543</p>
          <p align='center'>Cell: 000-000-0000</p>
        </td>
      </tr>
    </table>
  </body>
</html>"""


STY_CIRC = """<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

  <xsl:include href="self-include.xsl"/>

</xsl:stylesheet>"""


STY_A = """<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

  <!-- this is stylesheet A -->
  <xsl:output method="text"/>
  <xsl:template match="/">hello world</xsl:template>

</xsl:stylesheet>"""

STY_B = """<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

  <!-- this is stylesheet B -->
  <xsl:include href="A.xsl"/>

</xsl:stylesheet>"""

STY_C = """<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

  <!-- this is stylesheet C -->
  <xsl:include href="A.xsl"/>

</xsl:stylesheet>"""

STY_D = """<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

  <!-- this is stylesheet D -->
  <xsl:include href="B.xsl"/>
  <xsl:include href="C.xsl"/>

</xsl:stylesheet>"""

STY_IDENTITY = """<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet version="1.0"
  xmlns:xsl="http://www.w3.org/1999/XSL/Transform">

  <xsl:output method="xml" indent="no"/>

  <xsl:template match="@*|node()">
    <xsl:copy>
      <xsl:apply-templates select="@*|node()"/>
    </xsl:copy>
  </xsl:template>

</xsl:stylesheet>"""

STY_XINC = """<?xml version="1.0" encoding="utf-8"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="1.0">

  <xsl:output method="text" />

  <xsl:template match="/">
    <xsl:variable name="content"><xi:include xmlns:xi="http://www.w3.org/2001/XInclude" parse="text" href="content.ent"/></xsl:variable>
    <xsl:choose>
      <xsl:when test="string($content)">The XInclude was processed. Here is the text:&#10;<xsl:value-of select="$content"/></xsl:when>
      <xsl:otherwise>The XInclude was not processed.</xsl:otherwise>
    </xsl:choose>
  </xsl:template>

</xsl:stylesheet>"""

XINC_CONTENT = """hello world"""

XINC_EXPECTED_1 = """The XInclude was processed. Here is the text:
hello world"""

XINC_EXPECTED_2 = """The XInclude was not processed."""

DUMMY_XML = """<?xml version="1.0" encoding="utf-8"?><dummy/>"""

from Ft.Lib.Uri import FtUriResolver
class TestResolver(FtUriResolver):

    def resolve(self, uri, baseUri=None):
        if uri.endswith('self-include.xsl'):
            return cStringIO.StringIO(STY_CIRC)
        elif uri.endswith('identity.xsl'):
            return cStringIO.StringIO(STY_IDENTITY)
        elif uri.endswith('xinc.xsl'):
            return cStringIO.StringIO(STY_XINC)
        elif uri.endswith('A.xsl'):
            return cStringIO.StringIO(STY_A)
        elif uri.endswith('B.xsl'):
            return cStringIO.StringIO(STY_B)
        elif uri.endswith('C.xsl'):
            return cStringIO.StringIO(STY_C)
        elif uri.endswith('D.xsl'):
            return cStringIO.StringIO(STY_D)
        elif uri.endswith('content.ent'):
            return cStringIO.StringIO(XINC_CONTENT)
        else:
            raise ValueError("can't resolve %s" % uri)


def Test(tester):

    source = test_harness.FileInfo(uri="Xml/Xslt/Core/addr_book1.xml")
    sty = test_harness.FileInfo(string=sheet_str_1)
    test_harness.XsltTest(tester, source, [sty], expected_1,
                          title='xsl:include')

    styfactory = InputSourceFactory(resolver=TestResolver())

    source = test_harness.FileInfo(string=DUMMY_XML)
    sty_uri = OsPathToUri('self-include.xsl', attemptAbsolute=True)
    sty = test_harness.FileInfo(string=STY_CIRC, baseUri=sty_uri)
    test_harness.XsltTest(tester, source, [sty], None,
                          title='circular xsl:include (direct)',
                          exceptionCode=Error.CIRCULAR_INCLUDE,
                          stylesheetInputFactory=styfactory
                          )

    source = test_harness.FileInfo(string=DUMMY_XML)
    sty_uri = OsPathToUri('D.xsl', attemptAbsolute=True)
    sty = test_harness.FileInfo(string=STY_D, baseUri=sty_uri)
    test_harness.XsltTest(tester, source, [sty], None,
                          title='circular xsl:include (indirect)',
                          exceptionCode=Error.CIRCULAR_INCLUDE,
                          stylesheetInputFactory=styfactory
                          )

    source = test_harness.FileInfo(string=DUMMY_XML)
    sty_uri = OsPathToUri('identity.xsl', attemptAbsolute=True)
    sty = test_harness.FileInfo(string=STY_IDENTITY, baseUri=sty_uri)
    sty_isrc = sty.toInputSource(styfactory)
    processor = Processor()
    try:
        processor.appendStylesheet(sty_isrc)
        processor.appendStylesheet(sty_isrc)
    except XsltException, exception:
        tester.compare(Error.CIRCULAR_INCLUDE, exception.errorCode, stackLevel=2)
    else:
        tester.error("XsltException(%d) not raised" % exceptionCode,
                     stackLevel=2)

    tester.startTest('process XInclude in stylesheet')
    source = test_harness.FileInfo(string=DUMMY_XML)
    src_isrc = source.toInputSource(DefaultFactory)
    sty_uri = OsPathToUri('xinclude.xsl', attemptAbsolute=True)
    sty = test_harness.FileInfo(string=STY_XINC, baseUri=sty_uri)
    sty_isrc = sty.toInputSource(styfactory, processIncludes=True)
    processor = Processor()
    processor.appendStylesheet(sty_isrc)
    result = processor.run(src_isrc)
    tester.compare(XINC_EXPECTED_1, result)
    tester.testDone()

    tester.startTest('ignore XInclude in stylesheet')
    source = test_harness.FileInfo(string=DUMMY_XML)
    src_isrc = source.toInputSource(DefaultFactory)
    sty_uri = OsPathToUri('xinclude.xsl', attemptAbsolute=True)
    sty = test_harness.FileInfo(string=STY_XINC, baseUri=sty_uri)
    sty_isrc = sty.toInputSource(styfactory, processIncludes=False)
    processor = Processor()
    processor.appendStylesheet(sty_isrc)
    result = processor.run(src_isrc)
    tester.compare(XINC_EXPECTED_2, result)
    tester.testDone()

    return
