
import time, string, sha

from Ft.Xml.Lib import TreeCompare

from Ft.Server import FTSERVER_NAMESPACE

from Ft.Server.Common import ResourceTypes, CreationParams, Schema
from Ft.Xml.XLink import XLINK_NAMESPACE

import test_helper
from Ft import Rdf

def test_container(tester):

    tester.startTest("Container Creation w/ forced type")

    x = 1
    
    repo = test_helper.GetRepo(tester)

    test = repo.fetchResource('test')

    src="""<ftss:Container xmlns:ftss="%s"><ftss:Children/><foo/></ftss:Container>""" % (FTSERVER_NAMESPACE)

    cont = test.createDocument('cont',src,forcedType = ResourceTypes.ResourceType.CONTAINER)

    tester.compare('/test/cont',cont.getAbsolutePath())

    tester.compare(ResourceTypes.ResourceType.CONTAINER,cont.getResourceType())
    tester.compare(1,cont.isResourceType(ResourceTypes.ResourceType.CONTAINER))
    tester.compare(1,cont.isResourceType(ResourceTypes.ResourceType.XML_DOCUMENT))
    tester.compare(1,cont.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(1,string.find(cont.getContent(),'<foo') != -1)
    repo.txCommit()
    tester.testDone()

    tester.startTest("Container Creation w/o forced type")

    repo = test_helper.GetRepo(tester)

    test = repo.fetchResource('test')

    src="""<ftss:Container xmlns:ftss="%s"><ftss:Children/><foo/></ftss:Container>""" % (FTSERVER_NAMESPACE)


    cont = test.createDocument('cont2',src)
    tester.compare('/test/cont2',cont.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.CONTAINER,cont.getResourceType())
    tester.compare(1,cont.isResourceType(ResourceTypes.ResourceType.CONTAINER))
    tester.compare(1,cont.isResourceType(ResourceTypes.ResourceType.XML_DOCUMENT))
    tester.compare(1,cont.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(1,string.find(cont.getContent(),'<foo') != -1)
    repo.txCommit()
    tester.testDone()



def test_xml_document(tester):

    tester.startTest("XML Document Creation w/ forcedType")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<dummy/>"""
    doc = test.createDocument('doc1',src,forcedType = ResourceTypes.ResourceType.XML_DOCUMENT)
    tester.compare('/test/doc1',doc.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.XML_DOCUMENT,doc.getResourceType())
    tester.compare(1,doc.isResourceType(ResourceTypes.ResourceType.XML_DOCUMENT))
    tester.compare(1,doc.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,doc.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    repo.txCommit()
    tester.testDone()

    tester.startTest("XML Document Creation w/o forcedType")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<dummy/>"""
    doc = test.createDocument('doc2',src)
    tester.compare('/test/doc2',doc.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.XML_DOCUMENT,doc.getResourceType())
    tester.compare(1,doc.isResourceType(ResourceTypes.ResourceType.XML_DOCUMENT))
    tester.compare(1,doc.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,doc.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    repo.txCommit()
    tester.testDone()



def test_xslt_document(tester):

    tester.startTest("XSLT Document Creation w/ forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<?xml version="1.0"?>
<xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform" version="1.0">
  <xsl:template match="/">
  Sheet 2
  </xsl:template>
</xsl:stylesheet>"""
    sheet = test.createDocument('sheet1',src,forcedType = ResourceTypes.ResourceType.XSLT_DOCUMENT)
    tester.compare('/test/sheet1',sheet.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.XSLT_DOCUMENT,sheet.getResourceType())
    tester.compare(1,sheet.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    tester.compare(1,sheet.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,sheet.isResourceType(ResourceTypes.ResourceType.SCHEMATRON_DOCUMENT))
    repo.txCommit()
    tester.testDone()

    tester.startTest("XSLT Document Creation w/o forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    sheet = test.createDocument('sheet2',src)
    tester.compare('/test/sheet2',sheet.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.XSLT_DOCUMENT,sheet.getResourceType())
    tester.compare(1,sheet.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    tester.compare(1,sheet.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,sheet.isResourceType(ResourceTypes.ResourceType.SCHEMATRON_DOCUMENT))
    repo.txCommit()
    tester.testDone()


def test_schematron_document(tester):

    tester.startTest("Schematron Document Creation w forcedType")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<schema xmlns='http://www.ascc.net/xml/schematron'>
  <pattern name='Root'>
    <rule context="/docelem">
      <assert test='@v'>
        VALIDATION ERROR: The root element must have a v attribute
      </assert>      
    </rule>
  </pattern>
</schema>"""
    schema = test.createDocument('schema1',src,forcedType = ResourceTypes.ResourceType.SCHEMATRON_DOCUMENT)
    tester.compare('/test/schema1',schema.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.SCHEMATRON_DOCUMENT,schema.getResourceType())
    tester.compare(1,schema.isResourceType(ResourceTypes.ResourceType.SCHEMATRON_DOCUMENT))
    tester.compare(1,schema.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,schema.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    repo.txCommit()
    tester.testDone()

    tester.startTest("Schematron Document Creation w/o forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    schema = test.createDocument('schema2',src)
    tester.compare('/test/schema2',schema.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.SCHEMATRON_DOCUMENT,schema.getResourceType())
    tester.compare(1,schema.isResourceType(ResourceTypes.ResourceType.SCHEMATRON_DOCUMENT))
    tester.compare(1,schema.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,schema.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    repo.txCommit()
    tester.testDone()



def test_xpath_document_definition(tester):

    tester.startTest("XPath Doc Def Document Creation w/ forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<?xml version='1.0' encoding='UTF-8'?>
<ftss:DocDef xmlns:ftss='http://xmlns.4suite.org/reserved'>
  <ftss:BaseNames/>
  <ftss:NsMappings>
    <ftss:NsMapping>
      <ftss:Prefix>ft</ftss:Prefix>
      <ftss:Uri>foo.com</ftss:Uri>
    </ftss:NsMapping>
  </ftss:NsMappings>
  <ftss:RdfMappings>
    <ftss:RdfMapping>
      <ftss:Subject>$uri</ftss:Subject>
      <ftss:Predicate>"NAME"</ftss:Predicate>
      <ftss:Object>//ft:name</ftss:Object>
    </ftss:RdfMapping>
  </ftss:RdfMappings>
  <ftss:CreationParams FullTextIndex='0' EnforceSchema='0'>
    <ftss:Validator type='http://schemas.4suite.org/4ss#documentdefinition.no_validation'/>
  </ftss:CreationParams>
</ftss:DocDef>"""
    dd = test.createDocument('dd1',src,forcedType = ResourceTypes.ResourceType.XPATH_DOCUMENT_DEFINITION)
    tester.compare('/test/dd1',dd.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.XPATH_DOCUMENT_DEFINITION,dd.getResourceType())
    tester.compare(1,dd.isResourceType(ResourceTypes.ResourceType.XPATH_DOCUMENT_DEFINITION))
    tester.compare(1,dd.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,dd.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    maps = [("$uri",'"NAME"',"//ft:name",Rdf.OBJECT_TYPE_UNKNOWN)]
    nss = {'ft':'foo.com'}
    cp = CreationParams.CreationParams()
    test_helper.CompareNsMap(tester,nss,dd.getNsMap())
    test_helper.CompareRdfMaps(tester,maps,dd.getRdfMaps())
    test_helper.CompareCreationParams(tester,cp,dd.getCreationParams())
    repo.txCommit()
    tester.testDone()


    tester.startTest("XPath Doc Def Document Creation w/o forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    dd = test.createDocument('dd2',src)
    tester.compare('/test/dd2',dd.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.XPATH_DOCUMENT_DEFINITION,dd.getResourceType())
    tester.compare(1,dd.isResourceType(ResourceTypes.ResourceType.XPATH_DOCUMENT_DEFINITION))
    tester.compare(1,dd.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,dd.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    maps = [("$uri",'"NAME"',"//ft:name",Rdf.OBJECT_TYPE_UNKNOWN)]
    nss = {'ft':'foo.com'}
    cp = CreationParams.CreationParams()
    test_helper.CompareNsMap(tester,nss,dd.getNsMap())
    test_helper.CompareRdfMaps(tester,maps,dd.getRdfMaps())
    test_helper.CompareCreationParams(tester,cp,dd.getCreationParams())
    repo.txCommit()
    tester.testDone()



def test_xslt_document_definition(tester):

    tester.startTest("XSLT Doc Def Document Creation w/ forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<?xml version='1.0' encoding='UTF-8'?>
<xsl:stylesheet xmlns:xsl='http://www.w3.org/1999/XSL/Transform' xmlns:ftss='http://xmlns.4suite.org/reserved' xmlns:rdf='http://www.w3.org/1999/02/22-rdf-syntax-ns#' version='1.0'>
  <xsl:param name='ftss:path'/>
  <xsl:template match='/'>
    <rdf:RDF>
      <rdf:Description about='{$ftss:path}'>
        <Test>Test Value</Test>
      </rdf:Description>
    </rdf:RDF>
  </xsl:template>
  <ftss:BaseNames/>
  <ftss:CreationParams EnforceSchema='0' FullTextIndex='1'>
    <ftss:Validator type='http://schemas.4suite.org/4ss#documentdefinition.no_validation'/>
  </ftss:CreationParams>
</xsl:stylesheet>"""
    dd = test.createDocument('dd3',src,forcedType = ResourceTypes.ResourceType.XSLT_DOCUMENT_DEFINITION)
    tester.compare('/test/dd3',dd.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.XSLT_DOCUMENT_DEFINITION,dd.getResourceType())
    tester.compare(1,dd.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT_DEFINITION))
    tester.compare(1,dd.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,dd.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    cp = CreationParams.CreationParams(fullTextIndex = 1)
    test_helper.CompareCreationParams(tester,cp,dd.getCreationParams())
    repo.txCommit()
    tester.testDone()

    tester.startTest("XSLT Doc Def Document Creation w/o forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    dd = test.createDocument('dd4',src)
    tester.compare('/test/dd4',dd.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.XSLT_DOCUMENT_DEFINITION,dd.getResourceType())
    tester.compare(1,dd.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT_DEFINITION))
    tester.compare(1,dd.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,dd.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    cp = CreationParams.CreationParams(fullTextIndex = 1)
    test_helper.CompareCreationParams(tester,cp,dd.getCreationParams())
    repo.txCommit()
    tester.testDone()


def test_rdf_document(tester):

    tester.startTest("RDF Document Creation w/ forcedType")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<rdf:RDF
    xmlns:rdf='http://www.w3.org/1999/02/22-rdf-syntax-ns#'
    xmlns:dc='http://purl.org/metadata/dublin_core#'>
    <rdf:Description about='http://www.dlib.org'>
      <dc:Title>D-Lib Program - Research in Digital Libraries</dc:Title>
      <dc:Date>1995-01-07</dc:Date>
      <dc:Type>World Wide Web Home Page</dc:Type>
    </rdf:Description>
  </rdf:RDF>
"""
    rdoc = test.createDocument('rdoc1',src,forcedType = ResourceTypes.ResourceType.RDF_DOCUMENT)
    tester.compare('/test/rdoc1',rdoc.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.RDF_DOCUMENT,rdoc.getResourceType())
    tester.compare(1,rdoc.isResourceType(ResourceTypes.ResourceType.RDF_DOCUMENT))
    tester.compare(1,rdoc.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,rdoc.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    repo.txCommit()
    tester.testDone()

    tester.startTest("RDF Document Creation w/o forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    rdoc = test.createDocument('rdoc2',src)
    tester.compare('/test/rdoc2',rdoc.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.RDF_DOCUMENT,rdoc.getResourceType())
    tester.compare(1,rdoc.isResourceType(ResourceTypes.ResourceType.RDF_DOCUMENT))
    tester.compare(1,rdoc.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,rdoc.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    repo.txCommit()
    tester.testDone()



def test_command(tester):

    tester.startTest("Command Document Creation w forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<ftss:Command xmlns:ftss="%s" xmlns:dc="%s" xmlns:xlink="%s" name='foo'>
  <dc:Description>This is a test command</dc:Description>
  <ftss:Example>foo --o=1</ftss:Example>
  <ftss:VerboseDescription>This is a very verbose description</ftss:VerboseDescription>
  <ftss:Options>
    <ftss:Option short-name='s' long-name='sample'>
      <dc:Description>A option</dc:Description>
      <ftss:SubOptions>
        <dc:Description>A sub option</dc:Description>
        <ftss:Option short-name='p' long-name='people'/>
      </ftss:SubOptions>
    </ftss:Option>
    <ftss:TypedOption short-name='s' long-name='sample'>
      <dc:Description>A typed option</dc:Description>
      <ftss:Allowed>1</ftss:Allowed>
    </ftss:TypedOption>
    <ftss:ExclusiveOption>
      <dc:Description>A exclusive option</dc:Description>
      <ftss:Option short-name='q' long-name='quite'/>
    </ftss:ExclusiveOption>
  </ftss:Options>
  <ftss:Arguments>
    <ftss:Argument name='foo' requirements='1'>
      <dc:Description>An argument</dc:Description>
    </ftss:Argument>
  </ftss:Arguments>
  <ftss:SubCommands>
    <ftss:CommandReference xlink:type="simple" xlink:href="bar" xlink:actuate="onLoad" xlink:show="embed"/>
  </ftss:SubCommands>
</ftss:Command>
""" % (FTSERVER_NAMESPACE,Schema.DC, XLINK_NAMESPACE)
    com = test.createDocument('com1',str(src),forcedType = ResourceTypes.ResourceType.COMMAND)
    tester.compare('/test/com1',com.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.COMMAND,com.getResourceType())
    tester.compare(1,com.isResourceType(ResourceTypes.ResourceType.COMMAND))
    tester.compare(1,com.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,com.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    repo.txCommit()
    tester.testDone()

    tester.startTest("Command Document Creation w/o forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    com = test.createDocument('com2',str(src))
    tester.compare('/test/com2',com.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.COMMAND,com.getResourceType())
    tester.compare(1,com.isResourceType(ResourceTypes.ResourceType.COMMAND))
    tester.compare(1,com.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,com.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    repo.txCommit()
    tester.testDone()


def test_alias(tester):

    tester.startTest("Alias Document Creation w forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    test.createDocument('adoc','<foo/>')
    src="""<ftss:Alias xmlns:ftss="%s" reference='/test/adoc'/>"""% (FTSERVER_NAMESPACE)
    a = test.createDocument('a1',src,forcedType = ResourceTypes.ResourceType.ALIAS)
    tester.compare('/test/a1',a.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.ALIAS,a.getResourceType())
    tester.compare(1,a.isResourceType(ResourceTypes.ResourceType.ALIAS))
    tester.compare(1,a.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,a.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    tester.compare(test.fetchResource('adoc'),a.getReference())
    repo.txCommit()
    tester.testDone()

    tester.startTest("Alias Document Creation w/o forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    a = test.createDocument('a2',src)
    tester.compare('/test/a2',a.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.ALIAS,a.getResourceType())
    tester.compare(1,a.isResourceType(ResourceTypes.ResourceType.ALIAS))
    tester.compare(1,a.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,a.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    tester.compare(test.fetchResource('adoc'),a.getReference())
    repo.txCommit()
    tester.testDone()



def test_user(tester):
    tester.startTest("User Document Creation w forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<ftss:User xmlns:ftss="%s" userName='u1'>
  <ftss:PasswdHash>%s</ftss:PasswdHash>
<ftss:Data>
  <ftss:Entry key='foo'>bar</ftss:Entry>
</ftss:Data>
</ftss:User>""" % (FTSERVER_NAMESPACE,sha.new('foo').hexdigest())
    u = test.createDocument('u1',src,forcedType = ResourceTypes.ResourceType.USER)
    tester.compare('/test/u1',u.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.USER,u.getResourceType())
    tester.compare(1,u.isResourceType(ResourceTypes.ResourceType.USER))
    tester.compare(1,u.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,u.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    tester.compare(sha.new('foo').hexdigest(),u.getPassword())
    tester.compare('bar',u['foo'])
    tester.compareIn(repo.getAllUserNames(),'u1')
    repo.txCommit()
    tester.testDone()

    tester.startTest("User Document Creation w/o forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<ftss:User xmlns:ftss="%s" userName='u2'>
  <ftss:PasswdHash>%s</ftss:PasswdHash>
<ftss:Data>
  <ftss:Entry key='foo'>bar</ftss:Entry>
</ftss:Data>
</ftss:User>""" % (FTSERVER_NAMESPACE,sha.new('foo').hexdigest())
    u = test.createDocument('u2',src)
    tester.compare('/test/u2',u.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.USER,u.getResourceType())
    tester.compare(1,u.isResourceType(ResourceTypes.ResourceType.USER))
    tester.compare(1,u.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,u.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    tester.compare(sha.new('foo').hexdigest(),u.getPassword())
    tester.compare('bar',u['foo'])
    tester.compareIn(repo.getAllUserNames(),'u2')
    repo.txCommit()
    tester.testDone()


def test_group(tester):

    tester.startTest("Group Document Creation w forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<ftss:Group xmlns:ftss="%s" xmlns:xlink="%s" name='g1'>
  <ftss:Members>
    <ftss:MemberReference xlink:type="simple" xlink:href="/test/u1;metadata" xlink:actuate="onLoad" xlink:show="embed"/>
  </ftss:Members>
</ftss:Group>""" % (FTSERVER_NAMESPACE,XLINK_NAMESPACE,)
    g = test.createDocument('g1',str(src),forcedType = ResourceTypes.ResourceType.GROUP)
    tester.compare('/test/g1',g.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.GROUP,g.getResourceType())
    tester.compare(1,g.isResourceType(ResourceTypes.ResourceType.GROUP))
    tester.compare(1,g.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,g.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    tester.compareIn(repo.getAllGroupNames(),'g1')
    tester.compareIn(repo.getAllGroupPaths(),'/test/g1')
    tester.compareIn(g.getMembers(),repo.fetchResource('/test/u1'))
    repo.txCommit()
    tester.testDone()

    tester.startTest("Group Document Creation w/o forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<ftss:Group xmlns:ftss="%s" xmlns:xlink="%s" name='g2'>
  <ftss:Members>
    <ftss:MemberReference xlink:type="simple" xlink:href="/test/u1;metadata" xlink:actuate="onLoad" xlink:show="embed"/>
  </ftss:Members>
</ftss:Group>""" % (FTSERVER_NAMESPACE,XLINK_NAMESPACE,)
    g = test.createDocument('g2',str(src))
    tester.compare('/test/g2',g.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.GROUP,g.getResourceType())
    tester.compare(1,g.isResourceType(ResourceTypes.ResourceType.GROUP))
    tester.compare(1,g.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,g.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    tester.compareIn(repo.getAllGroupNames(),'g1')
    tester.compareIn(repo.getAllGroupPaths(),'/test/g1')
    tester.compareIn(g.getMembers(),repo.fetchResource('/test/u1'))
    repo.txCommit()
    tester.testDone()


def test_server(tester):

    tester.startTest("Server Document Creation w forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    src="""<ftss:Server xmlns:ftss="%s" xmlns:dc="%s" name='foo'>
  <dc:Description>This is a test server</dc:Description>
  <ftss:Port>8080</ftss:Port>
  <ftss:Handler>Ft.Server.Server.FtRpc.FtRpcHandler</ftss:Handler>
  <ftss:Status running='1'/>
</ftss:Server>
""" % (FTSERVER_NAMESPACE,Schema.DC)
    ser = test.createDocument('ser1',src,forcedType = ResourceTypes.ResourceType.SERVER)
    tester.compare('/test/ser1',ser.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.SERVER,ser.getResourceType())
    tester.compare(1,ser.isResourceType(ResourceTypes.ResourceType.SERVER))
    tester.compare(1,ser.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,ser.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    repo.txCommit()
    tester.testDone()

    tester.startTest("Server Document Creation w/o forced Type")
    repo = test_helper.GetRepo(tester)
    test = repo.fetchResource('test')
    ser = test.createDocument('ser2',src)
    tester.compare('/test/ser2',ser.getAbsolutePath())
    tester.compare(ResourceTypes.ResourceType.SERVER,ser.getResourceType())
    tester.compare(1,ser.isResourceType(ResourceTypes.ResourceType.SERVER))
    tester.compare(1,ser.isResourceType(ResourceTypes.ResourceType.RAW_FILE))
    tester.compare(0,ser.isResourceType(ResourceTypes.ResourceType.XSLT_DOCUMENT))
    repo.txCommit()
    tester.testDone()





def cleanup(tester):

    tester.startTest("Clean Up")
    repo = test_helper.GetRepo(tester)
    if repo.hasResource('/test'):
        t = repo.fetchResource('/test')
        t.delete()
    test = repo.createContainer('/test')

    repo.txCommit()
    tester.testDone()


def Test(tester):

    cleanup(tester)
    test_container(tester)
    test_xml_document(tester)
    test_xslt_document(tester)
    test_schematron_document(tester)
    test_xpath_document_definition(tester)
    test_xslt_document_definition(tester)
    test_rdf_document(tester)
    test_command(tester)
    test_server(tester)
    test_alias(tester)
    test_user(tester)
    test_group(tester)



