import urllib, os, types, string
from Ft.Lib import Uri

TEST_DOT_PY_BANGPATH = "#!/usr/bin/env python"

# Test cases for BaseJoin() ==================================================
# (base, relative, expected)
basejoin_test_cases = [
    ('','file:docs/xml/4XLink.api','file:docs/xml/4XLink.api'),
    ('file:docs/xml/','file:docs/xml/4XLink.api','file:docs/xml/4XLink.api'),
    ('file:/usr/local/lib/xslt/bar.xslt','http://4Suite.org/foo.xslt','http://4Suite.org/foo.xslt'),
    ]

# Test cases for Absolutize() ================================================
#CURRENT_DOC_URI = 'http://spam.com/bacon/eggs.xml'
BASE_URI = ('http://a/b/c/d;p?q',
            'http://a/b/c/d;p?q=1/2',
            'http://a/b/c/d;p=1/2?q',
            'fred:///s//a/b/c',
            'http:///s//a/b/c',
            )
# (ref, base, expected)
absolutize_test_cases = [
    # http://lists.w3.org/Archives/Public/uri/2004Feb/0114.html
    ('../c',  'foo:a/b', 'foo:c'),
    ('foo:.', 'foo:a',   'foo:'),
    ('/foo/../../../bar', 'zz:abc', 'zz:/bar'),
    ('/foo/../bar',       'zz:abc', 'zz:/bar'),
    ('foo/../../../bar',  'zz:abc', 'zz:bar'),
    ('foo/../bar',        'zz:abc', 'zz:bar'),
    ('zz:.',              'zz:abc', 'zz:'),
    ('/.'      , BASE_URI[0], 'http://a/'),
    ('/.foo'   , BASE_URI[0], 'http://a/.foo'),
    ('.foo'    , BASE_URI[0], 'http://a/b/c/.foo'),

    # http://gbiv.com/protocols/uri/test/rel_examples1.html
    # examples from RFC 2396
    ('g:h'     , BASE_URI[0], 'g:h'),
    ('g'       , BASE_URI[0], 'http://a/b/c/g'),
    ('./g'     , BASE_URI[0], 'http://a/b/c/g'),
    ('g/'      , BASE_URI[0], 'http://a/b/c/g/'),
    ('/g'      , BASE_URI[0], 'http://a/g'),
    ('//g'     , BASE_URI[0], 'http://g'),
    # changed with RFC 2396bis
    #('?y'      , BASE_URI[0], 'http://a/b/c/d;p?y'),
    ('?y'      , BASE_URI[0], 'http://a/b/c/d;p?y'),
    ('g?y'     , BASE_URI[0], 'http://a/b/c/g?y'),
    # changed with RFC 2396bis
    #('#s'      , BASE_URI[0], CURRENT_DOC_URI + '#s'),
    ('#s'      , BASE_URI[0], 'http://a/b/c/d;p?q#s'),
    ('g#s'     , BASE_URI[0], 'http://a/b/c/g#s'),
    ('g?y#s'   , BASE_URI[0], 'http://a/b/c/g?y#s'),
    (';x'      , BASE_URI[0], 'http://a/b/c/;x'),
    ('g;x'     , BASE_URI[0], 'http://a/b/c/g;x'),
    ('g;x?y#s' , BASE_URI[0], 'http://a/b/c/g;x?y#s'),
    # changed with RFC 2396bis
    #(''        , BASE_URI[0], CURRENT_DOC_URI),
    (''        , BASE_URI[0], 'http://a/b/c/d;p?q'),
    ('.'       , BASE_URI[0], 'http://a/b/c/'),
    ('./'      , BASE_URI[0], 'http://a/b/c/'),
    ('..'      , BASE_URI[0], 'http://a/b/'),
    ('../'     , BASE_URI[0], 'http://a/b/'),
    ('../g'    , BASE_URI[0], 'http://a/b/g'),
    ('../..'   , BASE_URI[0], 'http://a/'),
    ('../../'  , BASE_URI[0], 'http://a/'),
    ('../../g' , BASE_URI[0], 'http://a/g'),
    ('../../../g', BASE_URI[0], ('http://a/../g', 'http://a/g')),
    ('../../../../g', BASE_URI[0], ('http://a/../../g', 'http://a/g')),
    # changed with RFC 2396bis
    #('/./g', BASE_URI[0], 'http://a/./g'),
    ('/./g', BASE_URI[0], 'http://a/g'),
    # changed with RFC 2396bis
    #('/../g', BASE_URI[0], 'http://a/../g'),
    ('/../g', BASE_URI[0], 'http://a/g'),
    ('g.', BASE_URI[0], 'http://a/b/c/g.'),
    ('.g', BASE_URI[0], 'http://a/b/c/.g'),
    ('g..', BASE_URI[0], 'http://a/b/c/g..'),
    ('..g', BASE_URI[0], 'http://a/b/c/..g'),
    ('./../g', BASE_URI[0], 'http://a/b/g'),
    ('./g/.', BASE_URI[0], 'http://a/b/c/g/'),
    ('g/./h', BASE_URI[0], 'http://a/b/c/g/h'),
    ('g/../h', BASE_URI[0], 'http://a/b/c/h'),
    ('g;x=1/./y', BASE_URI[0], 'http://a/b/c/g;x=1/y'),
    ('g;x=1/../y', BASE_URI[0], 'http://a/b/c/y'),
    ('g?y/./x', BASE_URI[0], 'http://a/b/c/g?y/./x'),
    ('g?y/../x', BASE_URI[0], 'http://a/b/c/g?y/../x'),
    ('g#s/./x', BASE_URI[0], 'http://a/b/c/g#s/./x'),
    ('g#s/../x', BASE_URI[0], 'http://a/b/c/g#s/../x'),
    ('http:g', BASE_URI[0], ('http:g', 'http://a/b/c/g')),
    ('http:', BASE_URI[0], ('http:', BASE_URI[0])),
    # not sure where this one originated
    ('/a/b/c/./../../g', BASE_URI[0], 'http://a/a/g'),

    # http://gbiv.com/protocols/uri/test/rel_examples2.html
    # slashes in base URI's query args
    ('g'       , BASE_URI[1], 'http://a/b/c/g'),
    ('./g'     , BASE_URI[1], 'http://a/b/c/g'),
    ('g/'      , BASE_URI[1], 'http://a/b/c/g/'),
    ('/g'      , BASE_URI[1], 'http://a/g'),
    ('//g'     , BASE_URI[1], 'http://g'),
    # changed in RFC 2396bis
    #('?y'      , BASE_URI[1], 'http://a/b/c/?y'),
    ('?y'      , BASE_URI[1], 'http://a/b/c/d;p?y'),
    ('g?y'     , BASE_URI[1], 'http://a/b/c/g?y'),
    ('g?y/./x' , BASE_URI[1], 'http://a/b/c/g?y/./x'),
    ('g?y/../x', BASE_URI[1], 'http://a/b/c/g?y/../x'),
    ('g#s'     , BASE_URI[1], 'http://a/b/c/g#s'),
    ('g#s/./x' , BASE_URI[1], 'http://a/b/c/g#s/./x'),
    ('g#s/../x', BASE_URI[1], 'http://a/b/c/g#s/../x'),
    ('./'      , BASE_URI[1], 'http://a/b/c/'),
    ('../'     , BASE_URI[1], 'http://a/b/'),
    ('../g'    , BASE_URI[1], 'http://a/b/g'),
    ('../../'  , BASE_URI[1], 'http://a/'),
    ('../../g' , BASE_URI[1], 'http://a/g'),

    # http://gbiv.com/protocols/uri/test/rel_examples3.html
    # slashes in path params
    # all of these changed in RFC 2396bis
    ('g'       , BASE_URI[2], 'http://a/b/c/d;p=1/g'),
    ('./g'     , BASE_URI[2], 'http://a/b/c/d;p=1/g'),
    ('g/'      , BASE_URI[2], 'http://a/b/c/d;p=1/g/'),
    ('g?y'     , BASE_URI[2], 'http://a/b/c/d;p=1/g?y'),
    (';x'      , BASE_URI[2], 'http://a/b/c/d;p=1/;x'),
    ('g;x'     , BASE_URI[2], 'http://a/b/c/d;p=1/g;x'),
    ('g;x=1/./y', BASE_URI[2], 'http://a/b/c/d;p=1/g;x=1/y'),
    ('g;x=1/../y', BASE_URI[2], 'http://a/b/c/d;p=1/y'),
    ('./'      , BASE_URI[2], 'http://a/b/c/d;p=1/'),
    ('../'     , BASE_URI[2], 'http://a/b/c/'),
    ('../g'    , BASE_URI[2], 'http://a/b/c/g'),
    ('../../'  , BASE_URI[2], 'http://a/b/'),
    ('../../g' , BASE_URI[2], 'http://a/b/g'),

    # http://gbiv.com/protocols/uri/test/rel_examples4.html
    # double and triple slash, unknown scheme
    ('g:h'     , BASE_URI[3], 'g:h'),
    ('g'       , BASE_URI[3], 'fred:///s//a/b/g'),
    ('./g'     , BASE_URI[3], 'fred:///s//a/b/g'),
    ('g/'      , BASE_URI[3], 'fred:///s//a/b/g/'),
    ('/g'      , BASE_URI[3], 'fred:///g'),  # may change to fred:///s//a/g
    ('//g'     , BASE_URI[3], 'fred://g'),   # may change to fred:///s//g
    ('//g/x'   , BASE_URI[3], 'fred://g/x'), # may change to fred:///s//g/x
    ('///g'    , BASE_URI[3], 'fred:///g'),
    ('./'      , BASE_URI[3], 'fred:///s//a/b/'),
    ('../'     , BASE_URI[3], 'fred:///s//a/'),
    ('../g'    , BASE_URI[3], 'fred:///s//a/g'),
    ('../../'  , BASE_URI[3], 'fred:///s//'),    # may change to fred:///s//a/../
    ('../../g' , BASE_URI[3], 'fred:///s//g'),   # may change to fred:///s//a/../g
    ('../../../g', BASE_URI[3], 'fred:///s/g'),  # may change to fred:///s//a/../../g
    ('../../../../g', BASE_URI[3], 'fred:///g'), # may change to fred:///s//a/../../../g

    # http://gbiv.com/protocols/uri/test/rel_examples5.html
    # double and triple slash, well-known scheme
    ('g:h'     , BASE_URI[4], 'g:h'),
    ('g'       , BASE_URI[4], 'http:///s//a/b/g'),
    ('./g'     , BASE_URI[4], 'http:///s//a/b/g'),
    ('g/'      , BASE_URI[4], 'http:///s//a/b/g/'),
    ('/g'      , BASE_URI[4], 'http:///g'),  # may change to http:///s//a/g
    ('//g'     , BASE_URI[4], 'http://g'),   # may change to http:///s//g
    ('//g/x'   , BASE_URI[4], 'http://g/x'), # may change to http:///s//g/x
    ('///g'    , BASE_URI[4], 'http:///g'),
    ('./'      , BASE_URI[4], 'http:///s//a/b/'),
    ('../'     , BASE_URI[4], 'http:///s//a/'),
    ('../g'    , BASE_URI[4], 'http:///s//a/g'),
    ('../../'  , BASE_URI[4], 'http:///s//'),    # may change to http:///s//a/../
    ('../../g' , BASE_URI[4], 'http:///s//g'),   # may change to http:///s//a/../g
    ('../../../g', BASE_URI[4], 'http:///s/g'),  # may change to http:///s//a/../../g
    ('../../../../g', BASE_URI[4], 'http:///g'), # may change to http:///s//a/../../../g

    # from Dan Connelly's tests in http://www.w3.org/2000/10/swap/uripath.py
    ("bar:abc", "foo:xyz", "bar:abc"),
    ('../abc', 'http://example/x/y/z', 'http://example/x/abc'),
    ('http://example/x/abc', 'http://example2/x/y/z', 'http://example/x/abc'),
    ('../r', 'http://ex/x/y/z', 'http://ex/x/r'),
    ('q/r', 'http://ex/x/y', 'http://ex/x/q/r'),
    ('q/r#s', 'http://ex/x/y', 'http://ex/x/q/r#s'),
    ('q/r#s/t', 'http://ex/x/y', 'http://ex/x/q/r#s/t'),
    ('ftp://ex/x/q/r', 'http://ex/x/y', 'ftp://ex/x/q/r'),
    ('', 'http://ex/x/y', 'http://ex/x/y'),
    ('', 'http://ex/x/y/', 'http://ex/x/y/'),
    ('', 'http://ex/x/y/pdq', 'http://ex/x/y/pdq'),
    ('z/', 'http://ex/x/y/', 'http://ex/x/y/z/'),
    ('#Animal', 'file:/swap/test/animal.rdf', 'file:/swap/test/animal.rdf#Animal'),
    ('../abc', 'file:/e/x/y/z', 'file:/e/x/abc'),
    ('/example/x/abc', 'file:/example2/x/y/z', 'file:/example/x/abc'),
    ('../r', 'file:/ex/x/y/z', 'file:/ex/x/r'),
    ('/r', 'file:/ex/x/y/z', 'file:/r'),
    ('q/r', 'file:/ex/x/y', 'file:/ex/x/q/r'),
    ('q/r#s', 'file:/ex/x/y', 'file:/ex/x/q/r#s'),
    ('q/r#', 'file:/ex/x/y', 'file:/ex/x/q/r#'),
    ('q/r#s/t', 'file:/ex/x/y', 'file:/ex/x/q/r#s/t'),
    ('ftp://ex/x/q/r', 'file:/ex/x/y', 'ftp://ex/x/q/r'),
    ('', 'file:/ex/x/y', 'file:/ex/x/y'),
    ('', 'file:/ex/x/y/', 'file:/ex/x/y/'),
    ('', 'file:/ex/x/y/pdq', 'file:/ex/x/y/pdq'),
    ('z/', 'file:/ex/x/y/', 'file:/ex/x/y/z/'),
    ('file://meetings.example.com/cal#m1', 'file:/devel/WWW/2000/10/swap/test/reluri-1.n3', 'file://meetings.example.com/cal#m1'),
    ('file://meetings.example.com/cal#m1', 'file:/home/connolly/w3ccvs/WWW/2000/10/swap/test/reluri-1.n3', 'file://meetings.example.com/cal#m1'),
    ('./#blort', 'file:/some/dir/foo', 'file:/some/dir/#blort'),
    ('./#', 'file:/some/dir/foo', 'file:/some/dir/#'),
    # Ryan Lee
    ("./", "http://example/x/abc.efg", "http://example/x/"),

    #
    # Graham Klyne's tests
    # http://www.ninebynine.org/Software/HaskellUtils/Network/UriTest.xls
    # 01-31 are from Connelly's cases
    #
    # 32-49
    ('./q:r', 'http://ex/x/y', 'http://ex/x/q:r'),
    ('./p=q:r', 'http://ex/x/y', 'http://ex/x/p=q:r'),
    ('?pp/rr', 'http://ex/x/y?pp/qq', 'http://ex/x/y?pp/rr'),
    ('y/z', 'http://ex/x/y?pp/qq', 'http://ex/x/y/z'),
    ('local/qual@domain.org#frag', 'mailto:local', 'mailto:local/qual@domain.org#frag'),
    ('more/qual2@domain2.org#frag', 'mailto:local/qual1@domain1.org', 'mailto:local/more/qual2@domain2.org#frag'),
    ('y?q', 'http://ex/x/y?q', 'http://ex/x/y?q'),
    ('/x/y?q', 'http://ex?p', 'http://ex/x/y?q'),
    ('c/d',  'foo:a/b', 'foo:a/c/d'),
    ('/c/d', 'foo:a/b', 'foo:/c/d'),
    ('', 'foo:a/b?c#d', 'foo:a/b?c'),
    ('b/c', 'foo:a', 'foo:b/c'),
    ('../b/c', 'foo:/a/y/z', 'foo:/a/b/c'),
    ('./b/c', 'foo:a', 'foo:b/c'),
    ('/./b/c', 'foo:a', 'foo:/b/c'),
    ('../../d', 'foo://a//b/c', 'foo://a/d'),
    ('.', 'foo:a', 'foo:'),
    ('..', 'foo:a', 'foo:'),
    #
    # 50-57 (cf. TimBL comments --
    #  http://lists.w3.org/Archives/Public/uri/2003Feb/0028.html,
    #  http://lists.w3.org/Archives/Public/uri/2003Jan/0008.html)
    ('abc', 'http://example/x/y%2Fz', 'http://example/x/abc'),
    ('../../x%2Fabc', 'http://example/a/x/y/z', 'http://example/a/x%2Fabc'),
    ('../x%2Fabc', 'http://example/a/x/y%2Fz', 'http://example/a/x%2Fabc'),
    ('abc', 'http://example/x%2Fy/z', 'http://example/x%2Fy/abc'),
    ('q%3Ar', 'http://ex/x/y', 'http://ex/x/q%3Ar'),
    ('/x%2Fabc', 'http://example/x/y%2Fz', 'http://example/x%2Fabc'),
    ('/x%2Fabc', 'http://example/x/y/z', 'http://example/x%2Fabc'),
    ('/x%2Fabc', 'http://example/x/y%2Fz', 'http://example/x%2Fabc'),
    #
    # 70-77
    ('local2@domain2', 'mailto:local1@domain1?query1', 'mailto:local2@domain2'),
    ('local2@domain2?query2', 'mailto:local1@domain1', 'mailto:local2@domain2?query2'),
    ('local2@domain2?query2', 'mailto:local1@domain1?query1', 'mailto:local2@domain2?query2'),
    ('?query2', 'mailto:local@domain?query1', 'mailto:local@domain?query2'),
    ('local@domain?query2', 'mailto:?query1', 'mailto:local@domain?query2'),
    ('?query2', 'mailto:local@domain?query1', 'mailto:local@domain?query2'),
    ('http://example/a/b?c/../d', 'foo:bar', 'http://example/a/b?c/../d'),
    ('http://example/a/b#c/../d', 'foo:bar', 'http://example/a/b#c/../d'),
    #
    # 82-88
    ('http:this', 'http://example.org/base/uri', 'http:this'),
    ('http:this', 'http:base', 'http:this'),
    ('.//g', 'f:/a', 'f://g'),
    ('b/c//d/e', 'f://example.org/base/a', 'f://example.org/base/b/c//d/e'),
    ('m2@example.ord/c2@example.org', 'mid:m@example.ord/c@example.org', 'mid:m@example.ord/m2@example.ord/c2@example.org'),
    ('mini1.xml', 'file:///C:/DEV/Haskell/lib/HXmlToolbox-3.01/examples/', 'file:///C:/DEV/Haskell/lib/HXmlToolbox-3.01/examples/mini1.xml'),
    ('../b/c', 'foo:a/y/z', 'foo:a/b/c'),

]

# Test cases for URI reference syntax ========================================
good_URI_references = [
    'file:///foo/bar',
    'mailto:user@host?subject=blah',
    'dav:', # empty opaque part / rel-path allowed by RFC 2396bis
    'about:', # empty opaque part / rel-path allowed by RFC 2396bis
    #
    # the following test cases are from a Perl script by David A. Wheeler
    # at http://www.dwheeler.com/secure-programs/url.pl
    'http://www.yahoo.com',
    'http://www.yahoo.com/',
    'http://1.2.3.4/',
    'http://www.yahoo.com/stuff',
    'http://www.yahoo.com/stuff/',
    'http://www.yahoo.com/hello%20world/',
    'http://www.yahoo.com?name=obi',
    'http://www.yahoo.com?name=obi+wan&status=jedi',
    'http://www.yahoo.com?onery',
    'http://www.yahoo.com#bottom',
    'http://www.yahoo.com/yelp.html#bottom',
    'https://www.yahoo.com/',
    'ftp://www.yahoo.com/',
    'ftp://www.yahoo.com/hello',
    'demo.txt',
    'demo/hello.txt',
    'demo/hello.txt?query=hello#fragment',
    '/cgi-bin/query?query=hello#fragment',
    '/demo.txt',
    '/hello/demo.txt',
    'hello/demo.txt',
    '/',
    '',
    '#',
    '#here',
    # Wheeler's script says these are invalid, but they aren't
    'http://www.yahoo.com?name=%00%01',
    'http://www.yaho%6f.com',
    'http://www.yahoo.com/hello%00world/',
    'http://www.yahoo.com/hello+world/',
    'http://www.yahoo.com?name=obi&',
    'http://www.yahoo.com?name=obi&type=',
    'http://www.yahoo.com/yelp.html#',
    '//',
    #
    # the following test cases are from a Haskell program by Graham Klyne
    # at http://www.ninebynine.org/Software/HaskellUtils/Network/URITest.hs
    'http://example.org/aaa/bbb#ccc',
    'mailto:local@domain.org',
    'mailto:local@domain.org#frag',
    'HTTP://EXAMPLE.ORG/AAA/BBB#CCC',
    '//example.org/aaa/bbb#ccc',
    '/aaa/bbb#ccc',
    'bbb#ccc',
    '#ccc',
    '#',
    #'/', # repeat of test above
    "A'C",
    #-- escapes
    'http://example.org/aaa%2fbbb#ccc',
    'http://example.org/aaa%2Fbbb#ccc',
    '%2F',
    'aaa%2Fbbb',
    #-- ports
    'http://example.org:80/aaa/bbb#ccc',
    'http://example.org:/aaa/bbb#ccc',
    'http://example.org./aaa/bbb#ccc',
    'http://example.123./aaa/bbb#ccc',
    #-- bare authority
    'http://example.org',
    #-- IPv6 literals (from RFC2732):
    'http://[FEDC:BA98:7654:3210:FEDC:BA98:7654:3210]:80/index.html',
    'http://[1080:0:0:0:8:800:200C:417A]/index.html',
    'http://[3ffe:2a00:100:7031::1]',
    'http://[1080::8:800:200C:417A]/foo',
    'http://[::192.9.5.5]/ipng',
    'http://[::FFFF:129.144.52.38]:80/index.html',
    'http://[2010:836B:4179::836B:4179]',
    '//[2010:836B:4179::836B:4179]',
    #-- Random other things that crop up
    'http://example/Andr&#567;',
    'file:///C:/DEV/Haskell/lib/HXmlToolbox-3.01/examples/',
    ]

bad_URI_references = [
    'beepbeep\x07\x07',
    '\n',
    '::', # not OK, per Roy Fielding on the W3C uri list on 2004-04-01
    #
    # the following test cases are from a Perl script by David A. Wheeler
    # at http://www.dwheeler.com/secure-programs/url.pl
    'http://www yahoo.com',
    'http://www.yahoo.com/hello world/',
    'http://www.yahoo.com/yelp.html#"',
    #
    # the following test cases are from a Haskell program by Graham Klyne
    # at http://www.ninebynine.org/Software/HaskellUtils/Network/URITest.hs
    '[2010:836B:4179::836B:4179]',
    ' ',
    '%',
    'A%Z',
    '%ZZ',
    '%AZ',
    'A C',
    r"A\'C",
    'A`C',
    'A<C',
    'A>C',
    'A^C',
    r'A\\C',
    'A{C',
    'A|C',
    'A}C',
    'A[C',
    'A]C',
    'A[**]C',
    'http://[xyz]/',
    'http://]/',
    'http://example.org/[2010:836B:4179::836B:4179]',
    'http://example.org/abc#[2010:836B:4179::836B:4179]',
    'http://example.org/xxx/[qwerty]#a[b]',
    #
    # from a post to the W3C uri list on 2004-02-17
    'http://w3c.org:80path1/path2',
    ]

# Test cases for UriToOsPath =================================================
#
# Each tuple is (URI, expected for Windows, expected for POSIX).
# None means a UriException is expected.
fileUris = [
    ('file:x/y/z',         r'x\y\z',       'x/y/z'),
    ('file:/x/y/z',        r'\x\y\z',      '/x/y/z'),
    ('file:/x/y/z/',       '\\x\\y\\z\\',  '/x/y/z/'),
    ('file:///x/y/z',      r'\x\y\z',      '/x/y/z'),
    ('file:///x/y/z/',     '\\x\\y\\z\\',  '/x/y/z/'),
    ('file:///x/y/z?q1=1&q2=2', r'\x\y\z', '/x/y/z'),
    ('file:///x/y/z#frag', r'\x\y\z',      '/x/y/z'),
    ('file:///c:/x/y/z',   r'C:\x\y\z',    '/c:/x/y/z'),
    ('file:///c|/x/y/z',   r'C:\x\y\z',    '/c|/x/y/z'),
    ('file:///c:/x:/y/z',  r'C:\x:\y\z',   '/c:/x:/y/z'),
    ('file://c:/x/y/z',    r'C:\x\y\z',    None),
    ('file://host/share/x/y/z', r'\\host\share\x\y\z', None),
    ('file:////host/share/x/y/z', r'\\host\share\x\y\z', '//host/share/x/y/z'),
    ('file://host/c:/x/y/z', r'\\host\c:\x\y\z', None),
    ('file://localhost/x/y/z', r'\x\y\z',  '/x/y/z'),
    ('file://localhost/c:/x/y/z', r'C:\x\y\z', '/c:/x/y/z'),
    ('file:///C:%5Cx%5Cy%5Cz', r'C:\x\y\z', r'/C:\x\y\z'),
    ('file:///C:%2Fx%2Fy%2Fz', r'C:\x\y\z', r'/C:\/x\/y\/z'),
    ('file:///water%3D%E6%B0%B4', '\\water=\xe6\xb0\xb4', '/water=\xe6\xb0\xb4'),
    (u'file:///water%3D%E6%B0%B4', u'\\water=\u6c34', u'/water=\u6c34'),
    ]

# Test cases for OsPathToUri =================================================
#
# Each tuple is (OS path, expected for Windows, expected for POSIX).
# None means a UriException is expected.

if Uri.WINDOWS_SLASH_COMPAT:
    filePaths = [
        ('x/y/z',     'file:x/y/z',              'file:x/y/z'),
        ('/x/y/z',    'file:///x/y/z',           'file:///x/y/z'),
        ('//x/y/z',   'file://x/y/z',            'file:////x/y/z'),
        ('///x/y/z',  'file:///x/y/z',           'file://///x/y/z'),
    ]
else:
    filePaths = [
        ('x/y/z',     'file:x%2Fy%2Fz',          'file:x/y/z'),
        ('/x/y/z',    'file:%2Fx%2Fy%2Fz',       'file:///x/y/z'),
        ('//x/y/z',   'file:%2F%2Fx%2Fy%2Fz',    'file:////x/y/z'),
        ('///x/y/z',  'file:%2F%2F%2Fx%2Fy%2Fz', 'file://///x/y/z'),
    ]

filePaths += [
    ('C:\\',      'file:///C:/',             'file:C%3A%5C'),
    (r'C:\a\b\c', 'file:///C:/a/b/c',        'file:C%3A%5Ca%5Cb%5Cc'),
    (r'\a\b\c',   'file:///a/b/c',           'file:%5Ca%5Cb%5Cc'),
    (r'a\b\c',    'file:a/b/c',              'file:a%5Cb%5Cc'),
    (r'C:\a\b\c\d\..\..\C', 'file:///C:/a/b/C',
     'file:C%3A%5Ca%5Cb%5Cc%5Cd%5C..%5C..%5CC'),
    (r'C:\Documents and Settings\Your Name\Local Settings\Temp',
     'file:///C:/Documents%20and%20Settings/Your%20Name/Local%20Settings/Temp',
     'file:C%3A%5CDocuments%20and%20Settings%5CYour%20Name%5CLocal%20Settings%5CTemp'),
    ('.',         'file:.',                  'file:.'),
    ('..\Y\Z',    'file:../Y/Z',             'file:..%5CY%5CZ'),
    ('a!;b?c#d^e()', 'file:a%21%3Bb%3Fc%23d%5Ee%28%29',
     'file:a%21%3Bb%3Fc%23d%5Ee%28%29'),
    (u'98.6\xb0F', 'file:98.6%C2%B0F',       'file:98.6%C2%B0F'),
    (u'water=\u6c34', 'file:water%3D%E6%B0%B4', 'file:water%3D%E6%B0%B4'),
    (r'\\h\s\x\y\z', 'file://h/s/x/y/z',     'file:%5C%5Ch%5Cs%5Cx%5Cy%5Cz'),
    (r'\\h\$c$\x\y\z', 'file://h/%24c%24/x/y/z',
     'file:%5C%5Ch%5C%24c%24%5Cx%5Cy%5Cz'),
    (r'\\localhost\share\x\y\z', 'file://localhost/share/x/y/z',
     'file:%5C%5Clocalhost%5Cshare%5Cx%5Cy%5Cz'),
    ]


# Test cases for NormalizeCase ===============================================
#
# Each tuple is (URI, expected w/o normalizing host, expected with norm. host)
caseNormalizationTests = [
    ('HTTP://www.EXAMPLE.com/', 'http://www.EXAMPLE.com/', 'http://www.example.com/'),
    ('example://A/b/c/%7bfoo%7d', 'example://A/b/c/%7Bfoo%7D', 'example://a/b/c/%7Bfoo%7D'),
    ]


# Test cases for NormalizePercentEncoding ====================================
#
# Each tuple is (URI, expected)
pctEncNormalizationTests = [
    ('http://host/%7Euser/x/y/z', 'http://host/~user/x/y/z'),
    ('http://host/%7euser/x/y/z', 'http://host/~user/x/y/z'),
    ('example://A/b/c/%7bfoo%7d', 'example://A/b/c/%7bfoo%7d'),
    ]


# Test cases for NormalizePathSegments =======================================
#
# Each tuple is (path, expected)
pathSegmentNormalizationTests = [
    ('/a/b/../../c', '/c'),
    ('a/b/../../c', 'a/b/../../c'),
    ('/a/b/././c', '/a/b/c'),
    ('a/b/././c', 'a/b/././c'),
    ('/a/b/../c/././d', '/a/c/d'),
    ('a/b/../c/././d', 'a/b/../c/././d'),
    ]


# Test cases for MakeUrllibSafe ==============================================
#
# Each tuple is (URI, expected)
makeUrllibSafeTests = [
    # Martin Duerst's IDN tests in http://www.w3.org/2004/04/uri-rel-test.html
    ('http://www.w%33.org', 'http://www.w3.org'), # 101
    ('http://r%C3%A4ksm%C3%B6rg%C3%A5s.josefsson.org', 'http://xn--rksmrgs-5wao1o.josefsson.org'), # 111
    ('http://%E7%B4%8D%E8%B1%86.w3.mag.keio.ac.jp', 'http://xn--99zt52a.w3.mag.keio.ac.jp'), # 112
    ('http://www.%E3%81%BB%E3%82%93%E3%81%A8%E3%81%86%E3%81%AB%E3%81%AA%E3%81'
     '%8C%E3%81%84%E3%82%8F%E3%81%91%E3%81%AE%E3%82%8F%E3%81%8B%E3%82%89%E3%81'
     '%AA%E3%81%84%E3%81%A9%E3%82%81%E3%81%84%E3%82%93%E3%82%81%E3%81%84%E3%81'
     '%AE%E3%82%89%E3%81%B9%E3%82%8B%E3%81%BE%E3%81%A0%E3%81%AA%E3%81%8C%E3%81'
     '%8F%E3%81%97%E3%81%AA%E3%81%84%E3%81%A8%E3%81%9F%E3%82%8A%E3%81%AA%E3%81'
     '%84.w3.mag.keio.ac.jp/',
     'http://www.xn--n8jaaaaai5bhf7as8fsfk3jnknefdde3fg11amb5gzdb4wi9bya3kc6lr'
     'a.w3.mag.keio.ac.jp/'), # 121
    ('http://%E3%81%BB%E3%82%93%E3%81%A8%E3%81%86%E3%81%AB%E3%81%AA%E3%81%8C'
     '%E3%81%84%E3%82%8F%E3%81%91%E3%81%AE%E3%82%8F%E3%81%8B%E3%82%89%E3%81%AA'
     '%E3%81%84%E3%81%A9%E3%82%81%E3%81%84%E3%82%93%E3%82%81%E3%81%84%E3%81%AE'
     '%E3%82%89%E3%81%B9%E3%82%8B%E3%81%BE%E3%81%A0%E3%81%AA%E3%81%8C%E3%81%8F'
     '%E3%81%97%E3%81%AA%E3%81%84%E3%81%A8%E3%81%9F%E3%82%8A%E3%81%AA%E3%81%84'
     '.%E3%81%BB%E3%82%93%E3%81%A8%E3%81%86%E3%81%AB%E3%81%AA%E3%81%8C%E3%81'
     '%84%E3%82%8F%E3%81%91%E3%81%AE%E3%82%8F%E3%81%8B%E3%82%89%E3%81%AA%E3%81'
     '%84%E3%81%A9%E3%82%81%E3%81%84%E3%82%93%E3%82%81%E3%81%84%E3%81%AE%E3%82'
     '%89%E3%81%B9%E3%82%8B%E3%81%BE%E3%81%A0%E3%81%AA%E3%81%8C%E3%81%8F%E3%81'
     '%97%E3%81%AA%E3%81%84%E3%81%A8%E3%81%9F%E3%82%8A%E3%81%AA%E3%81%84.%E3'
     '%81%BB%E3%82%93%E3%81%A8%E3%81%86%E3%81%AB%E3%81%AA%E3%81%8C%E3%81%84%E3'
     '%82%8F%E3%81%91%E3%81%AE%E3%82%8F%E3%81%8B%E3%82%89%E3%81%AA%E3%81%84%E3'
     '%81%A9%E3%82%81%E3%81%84%E3%82%93%E3%82%81%E3%81%84%E3%81%AE%E3%82%89%E3'
     '%81%B9%E3%82%8B%E3%81%BE%E3%81%A0%E3%81%AA%E3%81%8C%E3%81%8F%E3%81%97%E3'
     '%81%AA%E3%81%84%E3%81%A8%E3%81%9F%E3%82%8A%E3%81%AA%E3%81%84.w3.mag.keio'
     '.ac.jp/',
     'http://xn--n8jaaaaai5bhf7as8fsfk3jnknefdde3fg11amb5gzdb4wi9bya3kc6lra.xn'
     '--n8jaaaaai5bhf7as8fsfk3jnknefdde3fg11amb5gzdb4wi9bya3kc6lra.xn--n8jaaaa'
     'ai5bhf7as8fsfk3jnknefdde3fg11amb5gzdb4wi9bya3kc6lra.w3.mag.keio.ac.jp/'), #122
    # Unicode versions of above
    (u'http://www.w%33.org', u'http://www.w3.org'), # 101
    (u'http://r%C3%A4ksm%C3%B6rg%C3%A5s.josefsson.org', u'http://xn--rksmrgs-5wao1o.josefsson.org'), # 111
    (u'http://%E7%B4%8D%E8%B1%86.w3.mag.keio.ac.jp', u'http://xn--99zt52a.w3.mag.keio.ac.jp'), # 112
    (u'http://www.%E3%81%BB%E3%82%93%E3%81%A8%E3%81%86%E3%81%AB%E3%81%AA%E3%81'
     u'%8C%E3%81%84%E3%82%8F%E3%81%91%E3%81%AE%E3%82%8F%E3%81%8B%E3%82%89%E3%81'
     u'%AA%E3%81%84%E3%81%A9%E3%82%81%E3%81%84%E3%82%93%E3%82%81%E3%81%84%E3%81'
     u'%AE%E3%82%89%E3%81%B9%E3%82%8B%E3%81%BE%E3%81%A0%E3%81%AA%E3%81%8C%E3%81'
     u'%8F%E3%81%97%E3%81%AA%E3%81%84%E3%81%A8%E3%81%9F%E3%82%8A%E3%81%AA%E3%81'
     u'%84.w3.mag.keio.ac.jp/',
     u'http://www.xn--n8jaaaaai5bhf7as8fsfk3jnknefdde3fg11amb5gzdb4wi9bya3kc6lr'
     u'a.w3.mag.keio.ac.jp/'), # 121
    (u'http://%E3%81%BB%E3%82%93%E3%81%A8%E3%81%86%E3%81%AB%E3%81%AA%E3%81%8C'
     u'%E3%81%84%E3%82%8F%E3%81%91%E3%81%AE%E3%82%8F%E3%81%8B%E3%82%89%E3%81%AA'
     u'%E3%81%84%E3%81%A9%E3%82%81%E3%81%84%E3%82%93%E3%82%81%E3%81%84%E3%81%AE'
     u'%E3%82%89%E3%81%B9%E3%82%8B%E3%81%BE%E3%81%A0%E3%81%AA%E3%81%8C%E3%81%8F'
     u'%E3%81%97%E3%81%AA%E3%81%84%E3%81%A8%E3%81%9F%E3%82%8A%E3%81%AA%E3%81%84'
     u'.%E3%81%BB%E3%82%93%E3%81%A8%E3%81%86%E3%81%AB%E3%81%AA%E3%81%8C%E3%81'
     u'%84%E3%82%8F%E3%81%91%E3%81%AE%E3%82%8F%E3%81%8B%E3%82%89%E3%81%AA%E3%81'
     u'%84%E3%81%A9%E3%82%81%E3%81%84%E3%82%93%E3%82%81%E3%81%84%E3%81%AE%E3%82'
     u'%89%E3%81%B9%E3%82%8B%E3%81%BE%E3%81%A0%E3%81%AA%E3%81%8C%E3%81%8F%E3%81'
     u'%97%E3%81%AA%E3%81%84%E3%81%A8%E3%81%9F%E3%82%8A%E3%81%AA%E3%81%84.%E3'
     u'%81%BB%E3%82%93%E3%81%A8%E3%81%86%E3%81%AB%E3%81%AA%E3%81%8C%E3%81%84%E3'
     u'%82%8F%E3%81%91%E3%81%AE%E3%82%8F%E3%81%8B%E3%82%89%E3%81%AA%E3%81%84%E3'
     u'%81%A9%E3%82%81%E3%81%84%E3%82%93%E3%82%81%E3%81%84%E3%81%AE%E3%82%89%E3'
     u'%81%B9%E3%82%8B%E3%81%BE%E3%81%A0%E3%81%AA%E3%81%8C%E3%81%8F%E3%81%97%E3'
     u'%81%AA%E3%81%84%E3%81%A8%E3%81%9F%E3%82%8A%E3%81%AA%E3%81%84.w3.mag.keio'
     u'.ac.jp/',
     u'http://xn--n8jaaaaai5bhf7as8fsfk3jnknefdde3fg11amb5gzdb4wi9bya3kc6lra.xn'
     u'--n8jaaaaai5bhf7as8fsfk3jnknefdde3fg11amb5gzdb4wi9bya3kc6lra.xn--n8jaaaa'
     u'ai5bhf7as8fsfk3jnknefdde3fg11amb5gzdb4wi9bya3kc6lra.w3.mag.keio.ac.jp/'), #122
    ]

winMakeUrllibSafeTests = [
    ('file:///C:/path/to/file', 'file:///C|/path/to/file'),
    ('http://foo/bar:baz/', 'http://foo/bar:baz/'),
    ]


# Test cases for PublicIdToUrn and UrnToPublicId =============================
#
publicIdTests = [
    # examples from RFC 3151
    ("ISO/IEC 10179:1996//DTD DSSSL Architecture//EN",
     "urn:publicid:ISO%2FIEC+10179%3A1996:DTD+DSSSL+Architecture:EN"),
    ("ISO 8879:1986//ENTITIES Added Latin 1//EN",
     "urn:publicid:ISO+8879%3A1986:ENTITIES+Added+Latin+1:EN"),
    ("-//OASIS//DTD DocBook XML V4.1.2//EN",
     "urn:publicid:-:OASIS:DTD+DocBook+XML+V4.1.2:EN"),
    ("+//IDN example.org//DTD XML Bookmarks 1.0//EN//XML",
     "urn:publicid:%2B:IDN+example.org:DTD+XML+Bookmarks+1.0:EN:XML"),
    ("-//ArborText::prod//DTD Help Document::19970708//EN",
     "urn:publicid:-:ArborText;prod:DTD+Help+Document;19970708:EN"),
    ("foo",
     "urn:publicid:foo"),
    ("3+3=6",
     #"urn:publicid:3%2B3=6" # RFC 2396
     "urn:publicid:3%2B3%3D6"), # RFC 2396bis
    ("-//Acme, Inc.//DTD Book Version 1.0",
     #"urn:publicid:-:Acme,+Inc.:DTD+Book+Version+1.0" # RFC 2396
     "urn:publicid:-:Acme%2C+Inc.:DTD+Book+Version+1.0"), # RFC 2396bis
]


def Test(tester):

    tester.startGroup("PublicIdToUrn")
    for publicid, urn in publicIdTests:
        tester.startTest(publicid)
        tester.compare(urn, Uri.PublicIdToUrn(publicid))
        tester.testDone()
    tester.groupDone()

    tester.startGroup("UrnToPublicId")
    for publicid, urn in publicIdTests:
        tester.startTest(urn)
        tester.compare(publicid, Uri.UrnToPublicId(urn))
        tester.testDone()
    tester.groupDone()

    tester.startGroup("URI reference syntax")
    for testuri in good_URI_references:
        tester.startTest("Good URI ref: %s" % repr(testuri))
        tester.compare(1, Uri.MatchesUriRefSyntax(testuri), "Mistakenly tests as invalid")
        tester.testDone()
    for testuri in bad_URI_references:
        tester.startTest("Bad URI ref: %s" % repr(testuri))
        tester.compare(0, Uri.MatchesUriRefSyntax(testuri), "Mistakenly tests as valid")
        tester.testDone()
    tester.groupDone()

    tester.startGroup('Absolutize')
    for uriRef, baseUri, expectedUri in absolutize_test_cases:
        tester.startTest('base=%r ref=%r' % (baseUri, uriRef))
        res = Uri.Absolutize(uriRef, baseUri)
        # in a couple cases, there's more than one correct result
        if type(expectedUri) == types.TupleType:
            tester.compare(1, res in expectedUri, 'Invalid result')
        else:
            tester.compare(expectedUri, res, 'Invalid result')
        tester.testDone()
    tester.groupDone()

    tester.startGroup('BaseJoin')
    for base, relative, expectedUri in basejoin_test_cases:
        tester.startTest('base=%r rel=%r' % (base,relative))
        res = Uri.BaseJoin(base, relative)
        tester.compare(expectedUri, res, 'Invalid result')
        tester.testDone()
    tester.groupDone()

    tester.startGroup('UriToOsPath')
    for osname in ('posix', 'nt'):
        tester.startGroup(osname)
        for subgroupname in ('absolute', 'relative'):
            tester.startGroup(subgroupname)
            for uri, nt_path, posix_path in fileUris:
                if subgroupname == 'relative':
                    if uri[:5] == 'file:':
                        uri = uri[5:]
                    else:
                        break
                if type(uri) == types.UnicodeType:
                    testname = repr(uri)
                else:
                    testname = uri
                tester.startTest(testname)
                if osname == 'nt':
                    path = nt_path
                elif osname == 'posix':
                    path = posix_path
                else:
                    break
                if path is None:
                     tester.testException(Uri.UriToOsPath, (uri,),
                                          Uri.UriException, kwargs={'osname': osname})
                else:
                    tester.compare(path, Uri.UriToOsPath(uri, osname=osname))
                tester.testDone()
            tester.groupDone()
        tester.groupDone()
    tester.groupDone()

    tester.startGroup('OsPathToUri')
    for osname in ('posix', 'nt'):
        tester.startGroup(osname)
        for path, nt_uri, posix_uri in filePaths:
            if type(path) == types.UnicodeType:
                testname = repr(path)
            else:
                testname = path
            tester.startTest(testname)
            if osname == 'nt':
                uri = nt_uri
            elif osname == 'posix':
                uri = posix_uri
            else:
                break
            if uri is None:
                 tester.testException(Uri.OsPathToUri, (path,),
                                      Uri.UriException, kwargs={'osname': osname})
            else:
                tester.compare(uri, Uri.OsPathToUri(path, osname=osname))
            tester.testDone()
        tester.groupDone()
    tester.groupDone()

    tester.startGroup('NormalizeCase')
    for uri, expected0, expected1 in caseNormalizationTests:
        testname = uri
        uri = Uri.SplitUriRef(uri)
        tester.startTest(testname)
        tester.compare(expected0, Uri.UnsplitUriRef(Uri.NormalizeCase(uri)))
        tester.testDone()
        tester.startTest(testname + ' (host too)')
        tester.compare(expected1, Uri.UnsplitUriRef(Uri.NormalizeCase(uri, doHost=1)))
        tester.testDone()
    tester.groupDone()

    tester.startGroup('NormalizePercentEncoding')
    for uri, expected in pctEncNormalizationTests:
        testname = uri
        tester.startTest(testname)
        tester.compare(expected, Uri.NormalizePercentEncoding(uri))
        tester.testDone()
    tester.groupDone()

    tester.startGroup('NormalizePathSegments')
    for path, expected in pathSegmentNormalizationTests:
        testname = path
        tester.startTest(testname)
        tester.compare(expected, Uri.NormalizePathSegments(path))
        tester.testDone()
    tester.groupDone()

    tester.startGroup('NormalizePathSegmentsInUri')
    for path, expectedpath in pathSegmentNormalizationTests:
        # for non-hierarchical scheme, no change expected in every case
        uri = 'urn:bogus:%s?a=1&b=2#frag' % path
        expected = 'urn:bogus:%s?a=1&b=2#frag' % path
        testname = uri
        tester.startTest(testname)
        tester.compare(expected, Uri.NormalizePathSegmentsInUri(uri))
        tester.testDone()
    for path, expectedpath in pathSegmentNormalizationTests:
        if path[:1] == '/':
            # hierarchical scheme
            uri = 'file://root:changeme@host%s?a=1&b=2#frag' % path
            expected = 'file://root:changeme@host%s?a=1&b=2#frag' % expectedpath
            testname = uri
            tester.startTest(testname)
            tester.compare(expected, Uri.NormalizePathSegmentsInUri(uri))
            tester.testDone()
    tester.groupDone()

    tester.startGroup('UriDict')
    tester.startTest('file:/// and file://localhost/ equivalence')
    uris = Uri.UriDict()
    uris['file:///path/to/resource'] = 1
    uris['file://localhost/path/to/resource'] = 2
    tester.compare(2, uris['file:///path/to/resource'], 'RFC 1738 localhost support failed')
    tester.testDone()
    for uri, expected, junk in caseNormalizationTests:
        tester.startTest('%s and %s equivalence' % (uri, expected))
        uris[uri] = 1
        uris[expected] = 2
        tester.compare(2, uris[uri])
        tester.testDone()
    for uri, expected in pctEncNormalizationTests:
        tester.startTest('%s and %s equivalence' % (uri, expected))
        uris[uri] = 1
        uris[expected] = 2
        tester.compare(2, uris[uri])
        tester.testDone()
    tester.groupDone()

    tester.startGroup("MakeUrllibSafe")
    tests = makeUrllibSafeTests
    if os.name == 'nt':
        tests += winMakeUrllibSafeTests
    for uri, expected in makeUrllibSafeTests:
        if type(uri) == types.UnicodeType:
            test_title = repr(uri)
        else:
            test_title = uri
        tester.startTest(test_title)
        res = Uri.MakeUrllibSafe(uri)
        tester.compare(expected, res)
        tester.testDone()
    tester.groupDone()

    tester.startGroup("Basic Uri Resolver")
    data = [('http://foo.com/root/', 'path', 'http://foo.com/root/path'),
            ('http://foo.com/root',  'path', 'http://foo.com/path'),
            ]
    for base,uri,exp in data:
        tester.startTest("normalize: %s %s" % (base, uri))
        res = Uri.BASIC_RESOLVER.normalize(uri, base)
        tester.compare(exp, res)
        tester.testDone()

    base = 'foo:foo.com'
    uri = 'path'
    tester.startTest("normalize: %s %s" % (base, uri))
    tester.testException(Uri.BASIC_RESOLVER.normalize, (uri, base), Uri.UriException)
    tester.testDone()

    tester.startTest('resolve')
    base = os.getcwd()
    if base[-1] != os.sep:
        base += os.sep
    stream = Uri.BASIC_RESOLVER.resolve('test.py', Uri.OsPathToUri(base))
    tester.compare(TEST_DOT_PY_BANGPATH, string.rstrip(stream.readline()))
    stream.close()
    tester.testDone()

    tester.startTest('generate')
    uuid = Uri.BASIC_RESOLVER.generate()
    tester.compare('urn:uuid:', uuid[:9])

    tester.testDone()

    tester.groupDone()

    tester.startGroup("SchemeRegistryResolver")

    def evalSchemeHandler(uri, base=None):
        if base: uri = base+uri
        uri = uri[5:]
        return str(eval(uri))

    def shiftSchemeHandler(uri, base=None):
        if base: uri = base+uri
        uri = uri[6:]
        return ''.join([ chr(ord(c)+1) for c in uri])

    resolver = Uri.SchemeRegistryResolver({'eval': evalSchemeHandler,
                                           'shift': shiftSchemeHandler,
                                           })

    scheme_cases = [(None, 'eval:150-50', '100'),
            (None, 'shift:abcde', 'bcdef'),
            ('eval:150-', '50', '100'),
            ('shift:ab', 'cde', 'bcdef'),
        ]

    for base, relative, expected in scheme_cases:
        tester.startTest("URI: base=%s uri=%s" % (base, relative))

        res = resolver.resolve(relative, base)
        tester.compare(expected, res)

        tester.testDone()

    resolver.handlers[None] = shiftSchemeHandler
    del resolver.handlers['shift']

    for base, relative, expected in scheme_cases:
        tester.startTest("URI: base=%s uri=%s" % (base, relative))

        res = resolver.resolve(relative, base)
        tester.compare(expected, res)

        tester.testDone()

    tester.groupDone()

    return


if __name__ == '__main__':
    from Ft.Lib.TestSuite import Tester
    tester = Tester.Tester()
    Test(tester)

