#include "Python.h"

#define IS_XMLSPACE(c) (((c) == 0x09) || \
                        ((c) == 0x0A) || \
                        ((c) == 0x0D) || \
                        ((c) == 0x20))

static PyObject *XmlStrip(PyUnicodeObject *str, int left, int right)
{
  Py_UNICODE *p = PyUnicode_AS_UNICODE(str);
  int start = 0;
  int end = PyUnicode_GET_SIZE(str);

  if (left)
    while (start < end && IS_XMLSPACE(p[start]))
      start++;

  if (right)
    while (end > start && IS_XMLSPACE(p[end-1]))
      end--;
  
  if (start == 0 && end == PyUnicode_GET_SIZE(str)) {
    /* couldn't strip anything off, return original string */
    Py_INCREF(str);
    return (PyObject *) str;
  }
  return PySequence_GetSlice((PyObject *)str, start, end);
}

static char lstrip_doc[] =
"XmlStrLStrip(S) -> unicode\n\
\n\
Return a copy of the string S with leading whitespace removed.";

static PyObject *PyXmlStrLStrip(PyObject *self, PyObject *args)
{
  PyObject *_argc0, *result;  /*Unicode objects*/

  if (!PyArg_ParseTuple(args,"O:XmlStrLStrip",&_argc0))
    return NULL;

  _argc0 = PyUnicode_FromObject(_argc0);
  if (!_argc0) {
    if (PyErr_ExceptionMatches(PyExc_TypeError))
      PyErr_Format(PyExc_TypeError, 
                   "argument must be unicode or string, %.80s found.",
                   _argc0->ob_type->tp_name);
    return NULL;
  }
  result = XmlStrip((PyUnicodeObject *)_argc0, 1, 0);

  /*PyUnicode_FromObject always adds one REF so do a DECREF on it*/
  Py_DECREF(_argc0);

  /*Has a REF of one now so we don't need to INCREF it*/
  return result;
}


static char rstrip_doc[] =
"XmlStrRStrip(S) -> unicode\n\
\n\
Return a copy of the string S with trailing whitespace removed.";

static PyObject *PyXmlStrRStrip(PyObject * self, PyObject *args)
{
  PyObject *_argc0, *result;  /*Unicode objects*/

  if (!PyArg_ParseTuple(args,"O:XmlStrRStrip",&_argc0))
    return NULL;

  _argc0 = PyUnicode_FromObject(_argc0);
  if (!_argc0) {
    if (PyErr_ExceptionMatches(PyExc_TypeError))
      PyErr_Format(PyExc_TypeError, 
                   "argument must be unicode or string, %.80s found.",
                   _argc0->ob_type->tp_name);
    return NULL;
  }
  result = XmlStrip((PyUnicodeObject *)_argc0, 0, 1);

  /*PyUnicode_FromObject always adds one REF so do a DECREF on it*/
  Py_DECREF(_argc0);

  /*Has a REF of one now so we don't need to INCREF it*/
  return result;
}


static char strip_doc[] =
"XmlStrStrip(S) -> unicode\n\
\n\
Return a copy of the string S with leading and trailing whitespace removed.";

static PyObject *PyXmlStrStrip(PyObject * self, PyObject *args)
{
  PyObject *_argc0, *result;  /*Unicode objects*/

  if (!PyArg_ParseTuple(args,"O:XmlStrStrip",&_argc0))
    return NULL;

  _argc0 = PyUnicode_FromObject(_argc0);
  if (!_argc0) {
    if (PyErr_ExceptionMatches(PyExc_TypeError))
      PyErr_Format(PyExc_TypeError, 
                   "argument must be unicode or string, %.80s found.",
                   _argc0->ob_type->tp_name);
    return NULL;
  }

  result = XmlStrip((PyUnicodeObject *)_argc0, 1, 1);

  /*PyUnicode_FromObject always adds one REF so do a DECREF on it*/
  Py_DECREF(_argc0);

  /*Has a REF of one now so we don't need to INCREF it*/
  return result;
}

static char isspace_doc[] =
"IsXmlSpace(S) -> bool\n\
\n\
Return True if there are only whitespace characters in S, False otherwise.";

static PyObject *PyIsXmlSpace(PyObject *self, PyObject *args)
{
  PyObject *str;
  Py_UNICODE *p, *e;

  if (!PyArg_ParseTuple(args,"O:IsXmlSpace", &str))
    return NULL;

  str = PyUnicode_FromObject(str);
  if (!str) {
    if (PyErr_ExceptionMatches(PyExc_TypeError))
      PyErr_Format(PyExc_TypeError, 
                   "argument must be unicode or string, %.80s found.",
                   str->ob_type->tp_name);
    return NULL;
  }

  p = PyUnicode_AS_UNICODE(str);
  e = p + PyUnicode_GET_SIZE(str);
  while (p < e) {
    if (!IS_XMLSPACE(*p)) {
      Py_INCREF(Py_False);
      return Py_False;
    }
    p++;
  }

  Py_INCREF(Py_True);
  return Py_True;
}

static PyMethodDef module_methods[] = {
  { "XmlStrLStrip", PyXmlStrLStrip, METH_VARARGS, lstrip_doc },
  { "XmlStrRStrip", PyXmlStrRStrip, METH_VARARGS, rstrip_doc },
  { "XmlStrStrip", PyXmlStrStrip, METH_VARARGS, strip_doc },
  { "IsXmlSpace", PyIsXmlSpace, METH_VARARGS, isspace_doc },
  { NULL, NULL }
};

DL_EXPORT(void) initXmlStripc(void) {
  PyObject *module;

  module = Py_InitModule("XmlStripc", module_methods);
  if (!module) 
    return;
 
  return;
}
