#include "domlette.h"

/*Internal Interfaces*/

#define Text_VerifyState(ob)                     \
  if (!PyText_Check(ob) ||                       \
      ((PyTextObject *)(ob))->nodeValue == NULL) \
     return DOMException_InvalidStateErr("Text in inconsistent state");

static PyObject *my_node_type, *my_node_name;

/*External Interfaces*/

static struct PyMethodDef Text_methods[] = {
  NODE_METHODS,
  CHARACTERDATA_METHODS,
  {NULL, NULL}      /* sentinel */
};


PyTextObject *Text_CloneNode(PyObject *node, int deep,
			     PyNodeObject *newOwnerDocument)
{

  long tempId;
  PyObject *nodeValue;

  if (!PyDocument_Check(newOwnerDocument)) {
    PyErr_SetString(PyExc_TypeError,"Argument must be a document");
    return NULL;
  }

  nodeValue = PyObject_GetAttrString(node, "nodeValue");
  nodeValue = DOMString_FromObjectInplace(nodeValue);
  if (nodeValue == NULL) return NULL;

  return Document_CreateTextNode((PyDocumentObject *)newOwnerDocument,
				 nodeValue, &tempId);
}

/*
  Type Interfaces
*/

static PyObject *text_getattr(PyTextObject *self, char *name)
{
  PyObject *rt = NULL;

  Text_VerifyState(self);

  if (!strcmp(name,"data")) {
    rt = (PyObject *)self->nodeValue;
  }
  else if (!strcmp(name,"nodeValue")) {
    rt = (PyObject *)self->nodeValue;
  }
  else if (!strcmp(name,"nodeType")) {
    rt = my_node_type;
  }
  else if (!strcmp(name,"nodeName")) {
    rt = my_node_name;
  }
  if (rt) {
    Py_INCREF(rt);
    return rt;
  }
  return node_getattr((PyNodeObject*)self,name, Text_methods);
}


static void text_dealloc(PyTextObject *node)
{
  PyObject_GC_UnTrack((PyObject *) node);

  Py_XDECREF(node->nodeValue);
  node->nodeValue = NULL;
  Node_Del(node);
}


static int text_clear(PyTextObject *self)
{
  /*Called when we need to break cycles*/
  return node_clear((PyNodeObject *)self);
}

static int text_traverse(PyTextObject *self, visitproc visit, void *arg)
{
  return node_traverse((PyNodeObject *)self,visit,arg);
}

static PyObject *text_repr(PyTextObject *text)
{
  char buf[256];
  sprintf(buf, "<cText at %p>", text);
  return PyString_FromString(buf);
}

int text_test_ref_counts(PyObject *tester,PyTextObject *node,long *childCtr,PyObject *internDict,int base) {
  /*Ref count is base + 1.  Out nodeValue should have a ref count of 1 but is not interned*/
  PyObject_CallMethod(tester,"startTest","s","Node");
  if (!PyObject_CallMethod(tester,"compare","ll",base+1,node->ob_refcnt)) return 0;
  PyObject_CallMethod(tester,"testDone","");
  if (!TestRefCount(tester,(PyObject *)node->nodeValue,1, "nodeValue")) return 0;
  return 1;
}

PyTypeObject PyDomletteText_Type = {
    PyObject_HEAD_INIT(0)
    0,
    "cDomlette.Text",
    GC_TP_BASICSIZE(PyTextObject),
    0,
    (destructor)text_dealloc,   /*tp_dealloc*/
    (printfunc)0,  /*tp_print*/
    (getattrfunc)text_getattr,  /*tp_getattr*/
    (setattrfunc)characterdata_setattr,  /*tp_setattr*/
    0,                          /*tp_compare*/
    (reprfunc)text_repr,          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,              /*tp_as_sequence*/
    0,              /*tp_as_mapping*/
    0,                             /*tp_hash*/
    0,          /*tp_call*/
    0,          /*tp_str*/
    0,                      /*tp_getattro*/
    0,          /*tp_setattro*/
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_HAVE_GC,
    0,                          /* tp_doc */
    (traverseproc)text_traverse,  /* tp_traverse */
    (inquiry)text_clear,          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
};

int DomletteText_Init(void)
{
  PyDomletteText_Type.ob_type = &PyType_Type;

  my_node_type = PyInt_FromLong(TEXT_NODE);
  if (my_node_type == NULL) return 0;

  my_node_name = PyUnicode_DecodeASCII("#text", 5, NULL);
  if (my_node_name == NULL) return 0;

  return 1;
}

void DomletteText_Fini(void)
{
  Py_DECREF(my_node_type);
  Py_DECREF(my_node_name);
}
