#include "Python.h"
#include "state_machine.h"
#include "debug.h"

#ifdef DEBUG_STATE_TABLE
static char *state_names[MAX_STATE] = {
  "ERROR_STATE",
  "START_STATE",
  "PARSE_STREAM_STATE",
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  "ENTITY_REF_CALLBACK",
  "START_ELEMENT_CALLBACK",
  "END_ELEMENT_CALLBACK",
  "CHARACTER_DATA_CALLBACK",
  "COMMENT_CALLBACK",
  "PI_CALLBACK",
  "START_NS_SCOPE_CALLBACK",
  "END_NS_SCOPE_CALLBACK",
};

static char *event_names[MAX_STATE] = {
  "ERROR_EVENT",
  "PARSE_RESUME_EVENT",
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  0,
  "ENTITY_REF_EVENT",
  "START_ELEMENT_EVENT",
  "END_ELEMENT_EVENT",
  "CHARACTER_DATA_EVENT",
  "COMMENT_EVENT",
  "PI_EVENT",
  "START_NS_SCOPE_EVENT",
  "END_NS_SCOPE_EVENT",
};

#endif

#ifdef DEBUG_STATE_TABLE
static void errorHandler(void *userState, void *params)
{
  fprintf(stderr,"Pass through Error Handler\n");
}
#else
#define errorHandler NULL
#endif


int StateTable_Initialize(void *userState)
{
  State *state = (State *)userState;
  int i;

  /*Initalize all to None so that we know what to free when we are done*/
  for (i=0;i<MAX_STATE;i++) {
    state->table[i] = NULL;
#ifdef DEBUG_STATE_TABLE
    if (state_names[i] == NULL) {
      char buf[10];
      sprintf(buf, "%d", i);
      state_names[i] = strdup(buf);
    }
    if (event_names[i] == NULL) {
      char buf[10];
      sprintf(buf, "%d", i);
      event_names[i] = strdup(buf);
    }      
#endif
  }

  /* Add the error state */
  if (StateTable_AddStateWithHandler(userState, ERROR_STATE, errorHandler) == 0)
    return 0;

  /* Add the initial state */
  if (StateTable_AddState(userState, START_STATE) == 0) {
    StateTable_Finalize(userState);
    return 0;
  }

  return 1;
}


void StateTable_Finalize(void *userState)
{
  State *state = (State *)userState;
  StateTableEntry *ste;
  int i;

  /*Initalize all to None so that we know what to free when we are done*/
  for (i=0;i<MAX_STATE;i++) {
    if ((ste = state->table[i]) != NULL) {
      if (ste->transitions) 
	free(ste->transitions);

      if (ste->destruct != NULL)
        ste->destruct(ste->params);

      free(ste);
      state->table[i] = NULL;
    }
  }
}


int StateTable_AddStateWithHandlerParams(void *userState, short id,
                                         void (*handler)(void *, void *), 
                                         void *params,
                                         void (*destruct)(void *))
{
  State *state = (State *)userState;
  StateTableEntry *ste;
  
#ifdef DEBUG_STATE_TABLE
  fprintf(stderr, "Add state %s\n", state_names[id]);
#endif
  if ((ste = (StateTableEntry *)malloc(sizeof(StateTableEntry))) == NULL) {
    PyErr_NoMemory();
    return 0;
  }

  ste->transitions = NULL;
  ste->handler = handler;
  ste->params = params;
  ste->destruct = destruct;

  state->table[id] = ste;

  /*Always add a transition from this state to the error state*/
  if (!StateTable_AddTransition(userState, id, ERROR_EVENT, ERROR_STATE)) {
    free(ste);
    state->table[id] = NULL;
    return 0;
  }

  return 1;
}


int StateTable_AddTransition(void *userState, short from, short event, 
                             short to)
{
  State *state = (State *)userState;
  StateTableEntry *ste = state->table[from];
  short *ptransitions;
  size_t len=0;

#ifdef DEBUG_STATE_TABLE
  fprintf(stderr, "add transition from %s to %s on event %s\n", 
          state_names[from], state_names[to], event_names[event]);
#endif
  if (ste == NULL) {
    PyErr_Format(PyExc_SystemError, "Unable to set transition from state %d", 
                 from);
    return 0;
  }
  
  ptransitions = ste->transitions;
  if (ptransitions == NULL) {
    if ((ptransitions = (short *)malloc(3 * sizeof(short))) == NULL) {
      PyErr_NoMemory();
      return 0;
    }
    ste->transitions = ptransitions;
  } else {
    while (*ptransitions++) len++;
    ptransitions = (short *)realloc(ste->transitions, 
                                    (len + 3) * sizeof(short));
    if (ptransitions == NULL) {
      PyErr_NoMemory();
      return 0;
    }
    /* save the new array */
    ste->transitions = ptransitions;

    /* move to the new entry */
    ptransitions += len;
  }

  /* add the event to state mapping */
  *ptransitions++ = event;
  *ptransitions++ = to;

  /* terminate the list */
  *ptransitions = 0;

  return 1;
}


void StateTable_Transit(void *userState, short event)
{
  State *state = (State *)userState;
  StateTableEntry *entry = state->table[state->current];
  short new_state_id = ERROR_STATE;

  /* find the state this event transitions to */
  if (entry) {
    short *ptransitions = entry->transitions;
    while (*ptransitions) {
      if (*ptransitions == event)
        new_state_id = *(ptransitions+1);
      ptransitions += 2;
    }
  }

#ifdef DEBUG_STATE_TABLE
  fprintf(stderr, "transition from %s to %s on event %s\n", 
          state_names[state->current], state_names[new_state_id], 
          event_names[event]);
#endif
  
  entry = state->table[new_state_id];
  if (entry) {
    state->current = new_state_id;
    if (entry->handler) {
      entry->handler(userState, entry->params);
    }
  } else {
    /* switch to error state if there is nowhere else to go */
    state->current = ERROR_STATE;
  }
}


void StateTable_SetState(void *userState, short new)
{
  State *state = (State *)userState;

  state->current = new;
}

void _StateTable_SignalError(void *userState, char *filename, int lineno)
{
  if (!PyErr_Occurred()) {
    PyErr_Format(PyExc_SystemError, "%s:%d: Error signaled without exception",
                 filename, lineno);
  }
#if defined(DEBUG_PARSER) || defined(DEBUG_STATE_TABLE)
  fprintf(stderr, "Error signaled in %s:%d\n", filename, lineno);
#endif

  StateTable_Transit(userState, ERROR_EVENT);
}
