#include "expat_module.h"
#include "parse_event_handler.h"
#include "xptr.h"

/** forwards ***********************************************************/

static void processingInstructionStateHandler(void *userState, void *params);
static void characterDataStateHandler(void *userState, void *params);
static void startNsScopeStateHandler(void *userState, void *params);
static void startElementStateHandler(void *userState, void *params);
static void endElementStateHandler(void *userState, void *params);
static void commentStateHandler(void *userState, void *params);

static void completeText(ParserState *state);
static void xinclude(ParserState *state);

#ifdef DEBUG_PARSER
static void dumpParserState(ParserState *state);
#endif

static PyObject *MakeQName(PyObject *prefix, PyObject *localName);

/** handlers ***********************************************************/

static void startNsScopeStateHandler(void *userState, void *params)
{
  ParserState *state = (ParserState *)userState;

#ifdef DEBUG_PARSER
  fprintf(stderr, "NSDecl started\n");
  fprintf(stderr, "URI:\n");
  _PyObject_Dump(state->expat_uri);
  fprintf(stderr, "Prefix:\n");
  _PyObject_Dump(state->expat_prefix);
#endif

  PyDict_SetItem(state->new_nss, state->expat_prefix, state->expat_uri);
  return;
}

static void startElementStateHandler(void *userState, void *params)
{
  ParserState *state = (ParserState *)userState;
  PyElementObject *elem = NULL;
  UniversalName *un;
  int i;
  int *new_preserve_state;
  static PyObject *processIncludes, *xi_include;
  static PyObject *asterisk, *stripElements;
  static PyObject *xmlns;
  PyObject *key, *value;
  PyObject *strip_list;
  PyObject *xml_base;

#ifdef DEBUG_PARSER
  fprintf(stderr, "Start element\n");
  fprintf(stderr, "name:\n");
  _PyObject_Dump(state->expat_name);
#endif

  /* for getattr (needs to be PyString) */
  if (processIncludes == NULL) {
    processIncludes = PyString_InternFromString("processIncludes");
    if (processIncludes == NULL) {
      StateTable_SignalError(state);
      return;
    }
  }

  /* for getattr (needs to be PyString) */
  if (stripElements == NULL) {
    stripElements = PyString_InternFromString("stripElements");
    if (stripElements == NULL) {
      StateTable_SignalError(state);
      return;
    }
  }

  if (xi_include == NULL) {
    xi_include = PyUnicode_DecodeASCII("include", 7, NULL);
    if (xi_include == NULL) {
      StateTable_SignalError(state);
      return;
    }
  }

  if (asterisk == NULL) {
    asterisk = PyUnicode_DecodeASCII("*", 1, NULL);
    if (asterisk == NULL) {
      StateTable_SignalError(state);
      return;
    }
  }

  if (xmlns == NULL) {
    xmlns = PyUnicode_DecodeASCII("xmlns", 5, NULL);
    if (xmlns == NULL) {
      StateTable_SignalError(state);
      return;
    }
  }

  /* Call Complete Text to make sure there are no additional text nodes that need to be created*/
  completeText(state);

  if (state->xinclude_content_depth) {
    /* a child of an XInclude element */
    state->xinclude_content_depth++;
    PyDict_Clear(state->new_nss);
    return;
  }

  /* Call build universal name to get the current NS. */
  un = buildUniversalName(state->expat_name);
  if (un == NULL) {
    StateTable_SignalError(state);
    return;
  }

  /** XInclude processing **********************************************/

  value = PyObject_GetAttr(state->input_source, processIncludes);
  if (PyObject_IsTrue(value) &&
      !PyObject_Compare(un->uri, g_xincludeNamespace) &&
      !PyObject_Compare(un->local, xi_include)) {
    Py_DECREF(value);
    destroyUniversalName(un);
    xinclude(state);
    return;
  }
  Py_DECREF(value);

  /* Now create the element */
  elem = Document_CreateElementNS(state->ownerDoc, un->uri, un->qname, 
                                  un->prefix, un->local, state->docIx);
  destroyUniversalName(un);

  if (elem == NULL) {
    StateTable_SignalError(state);
    return;
  }

  /* Check for preserve state */
  if ((new_preserve_state = (int *)malloc(sizeof(int))) == NULL) {
    Py_DECREF(elem);
    PyErr_NoMemory();
    StateTable_SignalError(state);
    return;
  }

  *new_preserve_state = *((int *)_stack_peek(state->preserve_state));
  xml_base = (PyObject *)_stack_peek(state->xml_base_stack);

  /** XSLT strip-elements **********************************************/

  /* stripElements is a list of tuples */
  strip_list = PyObject_GetAttr(state->input_source, stripElements);
  for (i=0; i < PyList_GET_SIZE(strip_list); i++) {
    PyObject *strip_item = PyList_GET_ITEM(strip_list, i);
    PyObject *strip_uri = PyTuple_GET_ITEM(strip_item, 0);
    PyObject *strip_local = PyTuple_GET_ITEM(strip_item, 1);
    PyObject *strip_flag = PyTuple_GET_ITEM(strip_item, 2);

    /* Each strip element specifies a NS and a local name,
       The localname can be a name or *
       If the localname is * then the NS could be None.
       ns:local is a complete match
       ns:* matches any element in the given namespace
       * matches any element.

       NOTE:  There are precedence rules to stripping as (according to XSLT)
       the matches should be treated as node tests.  The precedence is based
       on level of specificity.  This code assumes the list of white space
       rules has already been sorted by precedence.
    */

    if (/* if localName == '*' and no namespace or the namespace matches, or */
        (!PyObject_Compare(strip_local, asterisk) &&
         (strip_uri == Py_None ||
          !PyObject_Compare(strip_uri, elem->namespaceURI))) ||
        /* the localName and namespaceURI match */
        (!PyObject_Compare(strip_local, elem->localName) &&
         !PyObject_Compare(strip_uri, elem->namespaceURI))) {
      *new_preserve_state = !PyInt_AS_LONG(strip_flag);
      break;
    }
  }
  Py_DECREF(strip_list);

  /** namespaces *******************************************************/

  /* new_nss is a dictionary where key is the prefix and value is the uri */
  i = 0;
  while (PyDict_Next(state->new_nss, &i, &key, &value)) {
    PyAttrObject *nsattr;
    PyObject *prefix, *localName, *qualifiedName;

    if (key != Py_None) {
      prefix = xmlns;
      localName = key;
    } else {
      prefix = key;
      localName = xmlns;
    }

    qualifiedName = MakeQName(prefix, localName);
    if (qualifiedName == NULL) {
      Py_DECREF(elem);
      StateTable_SignalError(state);
      return;
    }

    nsattr = Element_SetAttributeNS(elem, g_xmlnsNamespace, qualifiedName, 
                                    prefix, localName, value);
    Py_DECREF(qualifiedName);
    if (nsattr == NULL) {
      Py_DECREF(elem);
      StateTable_SignalError(state);
      return;
    }
    Py_DECREF(nsattr);
  }
  /* make sure children don't set these namespaces */
  PyDict_Clear(state->new_nss);

  /** attributes *******************************************************/

  /* expat_atts is a dictionary where key is the expanded name and value
     is the attribute value.
  */
  i = 0;
  while (PyDict_Next(state->expat_atts, &i, &key, &value)) {
    PyAttrObject *attr;
    un = buildUniversalName(key);
    if (un == NULL) {
      Py_DECREF(elem);
      StateTable_SignalError(state);
      return;
    }

    attr = Element_SetAttributeNS(elem, un->uri, un->qname, un->prefix,
                                  un->local, value);
    destroyUniversalName(un);
    if (attr == NULL) {
      Py_DECREF(elem);
      StateTable_SignalError(state);
      return;
    }

    /* check for xml:space and xml:base */
    if (!PyObject_Compare(attr->namespaceURI, g_xmlNamespace)) {
      PyObject *base = PyUnicode_DecodeASCII("base", 4, NULL);
      PyObject *space = PyUnicode_DecodeASCII("space", 5, NULL);

      if (!PyObject_Compare(attr->localName, base)) {
        /* use a borrowed value to simplify cleanup */
        xml_base = value;
      } else if (!PyObject_Compare(attr->localName, space)) {
        PyObject *space_preserve = PyUnicode_DecodeASCII("preserve", 8, NULL);
        PyObject *space_default = PyUnicode_DecodeASCII("default", 7, NULL);
        if (!PyObject_Compare(value, space_preserve))
          *new_preserve_state = 1;
        else if (!PyObject_Compare(value, space_default))
          *new_preserve_state = 0;;
        Py_DECREF(space_preserve);
        Py_DECREF(space_default);
      }
      Py_DECREF(base);
      Py_DECREF(space);
    }

    /* All done with the attribute, let the element own it */
    Py_DECREF(attr);
  }

  /* Set XML base */
  Py_DECREF(elem->xmlBase);
  Py_INCREF(xml_base);
  elem->xmlBase = xml_base;

  /* save states on their respective stacks */
  _stack_push(state->preserve_state, new_preserve_state);
  _stack_push(state->xml_base_stack, xml_base);
  _stack_push(state->node_stack, elem);
  _stack_push(state->elem_depth_event_stack, state->curr_elem_depth_event);
  return;
}


static void endElementStateHandler(void *userState, void *params)
{
  ParserState *state = (ParserState *)userState;
  int *top_ps;
  int *event;
  void *ptr;
  PyNodeObject *new_elem;
  PyNodeObject *top_node;

#ifdef DEBUG_PARSER
  fprintf(stderr, "End element\n");
  fprintf(stderr, "name:\n");
  _PyObject_Dump(state->expat_name);
#endif

  if (state->xinclude_content_depth){
    state->xinclude_content_depth--;
    return;
  }

  /*Complete Any outstanding text nodes*/
  completeText(state);

  /*Pop the last element off of the stack*/
  _stack_pop(state->node_stack, (void *)&new_elem);

  /*Pop off the preserve state*/
  if (!_stack_pop(state->preserve_state, &ptr)){
    top_ps = (int *)ptr;
    free(top_ps);
  }

  /*Pop off the xml base state*/
  if (!_stack_pop(state->xml_base_stack, &ptr)){
    /*Py_DECREF((PyObject *)ptr);*/
  }

  /*Append us to the tree*/
  top_node = (PyNodeObject *)_stack_peek(state->node_stack);
  Node_AppendChild(top_node, new_elem);
  /*Append child adds a REF*/
  Py_DECREF(new_elem);

  /*Make any special state transitions associated with the end of this element */
  if (!_stack_pop(state->elem_depth_event_stack, &ptr)){
    event = (int *)ptr;
    if (*event) StateTable_Transit(state, (short)*event);
    free(event);
  }
}


static void characterDataStateHandler(void *userState, void *params)
{
  ParserState *state = (ParserState *)userState;
  PyObject *temp;

#ifdef DEBUG_PARSER
  fprintf(stderr, "New character data: ");
  fprintf(stderr,"Original: \n");
  _PyObject_Dump(state->curr_text);
  fprintf(stderr,"Added: \n");
  _PyObject_Dump(state->expat_charbuf);
#endif

  if (state->curr_text) {
    /* existing text */
    temp = PySequence_Concat(state->curr_text, (PyObject *)state->expat_charbuf);
    Py_DECREF(state->curr_text);
    state->curr_text = temp;
  } else {
    /* first text */
    state->curr_text = (PyObject *)state->expat_charbuf;
    Py_INCREF(state->curr_text);
  }

#ifdef DEBUG_PARSER
  fprintf(stderr, "New: ");
  _PyObject_Dump(state->curr_text);
#endif
}


static void processingInstructionStateHandler(void *userState, void *params)
{
  ParserState *state = (ParserState *)userState;
  PyNodeObject *new_pi;
  PyNodeObject *top_node;

  completeText(state);

  new_pi =(PyNodeObject *)Document_CreateProcessingInstruction(state->ownerDoc,
							       state->expat_target,
							       state->expat_data,
							       state->docIx);
  top_node = (PyNodeObject *)_stack_peek(state->node_stack);
  Node_AppendChild(top_node, new_pi);

  /*Node Append Child added a ref*/
  Py_DECREF(new_pi);

}


static void commentStateHandler(void *userState, void *params)
{
  ParserState *state = (ParserState *)userState;
  PyNodeObject *new_comment;
  PyNodeObject *top_node;

  completeText(state);

  new_comment = (PyNodeObject *)Document_CreateComment(state->ownerDoc,
                                                       state->expat_data,
                                                       state->docIx);
  top_node = (PyNodeObject *)_stack_peek(state->node_stack);
  Node_AppendChild(top_node, new_comment);
  /*Append Child adds a ref*/
  Py_DECREF(new_comment);
}


/* local support routines */

#define IS_XMLSPACE(c) (((c) == 0x09) || \
			((c) == 0x0A) || \
			((c) == 0x0D) || \
			((c) == 0x20))

static void completeText(ParserState *state)
{
  PyNodeObject *new_text;
  PyNodeObject *top_node;
  int preserve_state;
  int i;

  if (state->curr_text) {
    preserve_state = *((int *)_stack_peek(state->preserve_state));

    /* Always preserve the text node if it contains non XML space chars */
    for (i = 0; i < PyUnicode_GET_SIZE(state->curr_text); i++) {
      Py_UNICODE ch = PyUnicode_AS_UNICODE(state->curr_text)[i];
      if (!IS_XMLSPACE(ch)) {
	preserve_state = 1;
	break;
      }
    }

    if (preserve_state) {
      /* New reference */
      new_text = (PyNodeObject *)Document_CreateTextNode(state->ownerDoc,
							 state->curr_text,
							 state->docIx);

      top_node = (PyNodeObject *)_stack_peek(state->node_stack);

      /* Adds reference */
      Node_AppendChild(top_node, new_text);

      Py_DECREF(new_text);
    }
    Py_DECREF(state->curr_text);
    state->curr_text = NULL;
  }
}


static void xinclude(ParserState *state)
{
  int i, xi_parse_text=0, xi_start_state=PARSE_STREAM_STATE;
  PyObject *new_input_source;

  PyObject *attr_name, *attr_value, *href = NULL;
  PyObject *save_name, *save_atts;
  PyObject *modules, *cdom_mod, *xptr_spec;

  PyObject *href_str = PyUnicode_DecodeASCII("href", 4, NULL);
  PyObject *encoding_str = PyUnicode_DecodeASCII("encoding", 8, NULL);
  PyObject *parse_str = PyUnicode_DecodeASCII("parse", 5, NULL);
  PyObject *text_str = PyUnicode_DecodeASCII("text", 4, NULL);
  PyObject *xml_str = PyUnicode_DecodeASCII("xml", 3, NULL);

  /* encodings MUST be ascii per the XML spec so OK to use a PyString */
  PyObject *encoding = PyString_FromString("UTF-8");

  href = NULL;
  i = 0;
  while (PyDict_Next(state->expat_atts, &i, &attr_name, &attr_value)) {
    /* because these attributes have no namespace, it is ok to compare the
       name from expat.
    */
    if (PyObject_Compare(attr_name, href_str) == 0) {
      href = attr_value;
      Py_INCREF(href);
    } else if (PyObject_Compare(attr_name, encoding_str) == 0) {
      /* delete the previous encoding */
      Py_DECREF(encoding);
      /* encodings MUST be ascii per the XML spec */
      encoding = PyUnicode_AsASCIIString(attr_value);
    } else if (PyObject_Compare(attr_name, parse_str) == 0) {
      if (PyObject_Compare(attr_value, text_str) == 0)
        xi_parse_text = 1;
      else if (PyObject_Compare(attr_value, xml_str) != 0) {
        Py_XDECREF(href);
        Py_DECREF(encoding);
        Py_DECREF(href_str);
        Py_DECREF(encoding_str);
        Py_DECREF(parse_str);
        Py_DECREF(text_str);
        Py_DECREF(xml_str);
        XIncludeException_InvalidParseAttr(attr_value);
        StateTable_SignalError(state);
        return;
      }
    }
  }
  Py_DECREF(href_str);
  Py_DECREF(encoding_str);
  Py_DECREF(parse_str);
  Py_DECREF(text_str);
  Py_DECREF(xml_str);

  if (href == NULL) {
    /* Early exit if href attr is missing*/
    Py_DECREF(encoding);
    XIncludeException_MissingHref();
    StateTable_SignalError(state);
    return;
  }

  /* Invoke resolve on the input_source to get the next input source*/
  new_input_source = PyObject_CallMethod(state->input_source, "resolve",
                                         "OOsi", href, Py_None, "XINCLUDE", 0);
  if (new_input_source == NULL) {
    /* FIXME: Propagate the current error object */
    Py_DECREF(encoding);
    StateTable_SignalError(state);
    return;
  }

  /* Get any XPointer fragment */
  modules = PyImport_GetModuleDict(); /* borrowed */
  cdom_mod = PyDict_GetItemString(modules, "Ft.Xml.cDomlette"); /* borrowed */
  if (!cdom_mod) {
    cdom_mod = PyImport_ImportModule("Ft.Xml.cDomlette");
    if (cdom_mod == NULL) {
      StateTable_SignalError(state);
      return;
    }

    /* Make it a borrowed reference, a copy exists in sys.modules */
    Py_DECREF(cdom_mod);
  }

  xptr_spec = PyObject_CallMethod(cdom_mod, "ProcessFragment", "O", href);
  /* We shall use xptr_spec as the key to holding all the refs of the state specs */
  if (xptr_spec == NULL) {
    StateTable_SignalError(state);
    return;
  }
  else if (xptr_spec != Py_None) {
    xi_start_state = handle_xpointer(state, xptr_spec);
    if (xi_start_state == 0) {
      StateTable_SignalError(state);
      return;
    }
  }
  Py_DECREF(xptr_spec);
  Py_DECREF(href);

#ifdef DEBUG_PARSER
  fprintf(stderr, "Begin XInclude parsing.\n");
#endif

  if (xi_parse_text) {
    char buf[BUFSIZ];
    size_t len;
    PyObject *stream, *data;

    /* Get the stream from the new input source */
    stream = PyObject_GetAttrString(new_input_source, "stream");

    StateTable_Transit(state, PARSE_RESUME_EVENT);

    do {
      len = readFromObject(stream, buf, BUFSIZ);
      if (PyErr_Occurred()) {
        Py_DECREF(new_input_source);
        Py_DECREF(encoding);
        Py_DECREF(stream);
        StateTable_SignalError(state);
        return;
      }

      /* A bit convoluted, but we must consider matters such as ws stripping,
         so better to handle this using the normal parse handling flow*/
      data = PyUnicode_Decode(buf, len, PyString_AS_STRING(encoding), NULL);
      if (data == NULL) {
        Py_DECREF(new_input_source);
        Py_DECREF(encoding);
        Py_DECREF(stream);
        StateTable_SignalError(state);
        return;
      }

      state->expat_charbuf = data;
      StateTable_Transit(state, CHARACTER_DATA_EVENT);
      StateTable_Transit(state, PARSE_RESUME_EVENT);
      Py_DECREF(data);
    } while (len >= BUFSIZ);

    Py_DECREF(stream);
  } else {
    XML_Parser parser;
    PyObject *current_input_source;
    int status;

    parser = XML_ParserCreateNS(NULL, EXPAT_NSSEP);
    if (parser == NULL) {
      PyErr_NoMemory();
      Py_DECREF(new_input_source);
      Py_DECREF(encoding);
      StateTable_SignalError(state);
      return;
    }

    /*Save a copy of the current input source*/
    current_input_source = state->input_source;
    state->input_source = new_input_source;

    initParser(&parser, state);

    /*Reset the state machine, save name and attrs*/
    save_name = state->expat_name;
    save_atts = state->expat_atts;
    StateTable_SetState(state, (short)xi_start_state);

    status = doParse(parser, state, new_input_source);

    /* Restore previous state */
    StateTable_SetState(state, START_ELEMENT_CALLBACK);
    state->input_source = current_input_source;
    state->expat_name = save_name;
    state->expat_atts = save_atts;

    XML_ParserFree(parser);

    if (status == 0) {
#ifdef DEBUG_PARSER
      fprintf(stderr, "XInclude parsing failed.\n");
#endif
      Py_DECREF(new_input_source);
      Py_DECREF(encoding);
      StateTable_SignalError(state);
      return;
    }

#ifdef DEBUG_PARSER
    fprintf(stderr, "End XInclude parsing.\n");
#endif

    /*Set the state back*/
  }

  Py_DECREF(new_input_source);
  Py_DECREF(encoding);
  state->xinclude_content_depth++;
  return;
}


/* debugging aids */
#ifdef DEBUG_PARSER
static void dumpParserState(ParserState *state)
{
  fprintf(stderr, "Current Text: ");
  _PyObject_Dump(state->curr_text);
  fprintf(stderr, "\n");
}
#endif

/* State table set-up */


int initializeStateTableForParseEvents(ParserState *state)
{
  /* Add state and transitions for start-tags */
  if (StateTable_AddStateWithHandler(state, START_ELEMENT_CALLBACK,
                                     startElementStateHandler) == 0)
    return 0;
  if (StateTable_AddTransition(state, PARSE_STREAM_STATE, START_ELEMENT_EVENT,
                               START_ELEMENT_CALLBACK) == 0)
    return 0;
  if (StateTable_AddTransition(state, START_ELEMENT_CALLBACK,
                               PARSE_RESUME_EVENT, PARSE_STREAM_STATE) == 0)
    return 0;

  /* Add state and transitions for end-tags */
  if (StateTable_AddStateWithHandler(state, END_ELEMENT_CALLBACK,
                                     endElementStateHandler) == 0)
    return 0;
  if (StateTable_AddTransition(state, PARSE_STREAM_STATE, END_ELEMENT_EVENT,
                               END_ELEMENT_CALLBACK) == 0)
    return 0;
  if (StateTable_AddTransition(state, END_ELEMENT_CALLBACK,
                               PARSE_RESUME_EVENT, PARSE_STREAM_STATE) == 0)
    return 0;

  /* Add state and transitions for start namespace scope */
  if (StateTable_AddStateWithHandler(state, START_NS_SCOPE_CALLBACK,
                                     startNsScopeStateHandler) == 0)
    return 0;
  if (StateTable_AddTransition(state, PARSE_STREAM_STATE, START_NS_SCOPE_EVENT,
                               START_NS_SCOPE_CALLBACK) == 0)
    return 0;
  if (StateTable_AddTransition(state, START_NS_SCOPE_CALLBACK,
                               PARSE_RESUME_EVENT, PARSE_STREAM_STATE) == 0)
    return 0;

  /* Add state and transitions for character data */
  if (StateTable_AddStateWithHandler(state, CHARACTER_DATA_CALLBACK,
                                     characterDataStateHandler) == 0)
    return 0;
  if (StateTable_AddTransition(state, PARSE_STREAM_STATE, CHARACTER_DATA_EVENT,
                               CHARACTER_DATA_CALLBACK) == 0)
    return 0;
  if (StateTable_AddTransition(state, CHARACTER_DATA_CALLBACK,
                               PARSE_RESUME_EVENT, PARSE_STREAM_STATE) == 0)
    return 0;

  /* Add state and transitions for processing instructions */
  if (StateTable_AddStateWithHandler(state, PI_CALLBACK,
                                     processingInstructionStateHandler) == 0)
    return 0;
  if (StateTable_AddTransition(state, PARSE_STREAM_STATE, PI_EVENT,
                               PI_CALLBACK) == 0)
    return 0;
  if (StateTable_AddTransition(state, PI_CALLBACK, PARSE_RESUME_EVENT,
                               PARSE_STREAM_STATE) == 0)
    return 0;

  /* Add state and transitions for comments */
  if (StateTable_AddStateWithHandler(state, COMMENT_CALLBACK,
                                     commentStateHandler) == 0)
    return 0;
  if (StateTable_AddTransition(state, PARSE_STREAM_STATE, COMMENT_EVENT,
                               COMMENT_CALLBACK) == 0)
    return 0;
  if (StateTable_AddTransition(state, COMMENT_CALLBACK, PARSE_RESUME_EVENT,
                               PARSE_STREAM_STATE) == 0)
    return 0;

  return 1;
}

static PyObject *MakeQName(PyObject *prefix, PyObject *localName)
{
  PyObject *nodeName;

  if (PyObject_IsTrue(prefix)) {
    PyObject *u, *v;
    u = PyUnicode_DecodeASCII(":", 1, NULL);
    if (u == NULL) return NULL;

    v = PySequence_Concat(prefix, u);
    Py_DECREF(u);
    if (v == NULL) return NULL;

    nodeName = PySequence_Concat(v, localName);
    Py_DECREF(v);
    if (nodeName == NULL) return NULL;
  } else {
    Py_INCREF(localName);
    nodeName = localName;
  }
  return nodeName;
}

/*
  Create a Universal Name structure from the expat name.
  If the name from the document had a prefix, then expat name is:
    namespace-uri + sep + localName + sep + prefix,
  otherwise if there is a default namespace:
    namespace-uri + sep + localName
  lastly, it could just be:
    localName
 */

UniversalName *buildUniversalName(PyObject *expanded_name)
{
  Py_UNICODE nssep = EXPAT_NSSEP;
  int len = PyUnicode_GET_SIZE(expanded_name);
  const Py_UNICODE *p = PyUnicode_AS_UNICODE(expanded_name);
  int i, j;
  UniversalName *un;
  /*ListElmt *node = NULL;*/

  un = PyMem_New(UniversalName, 1);
  if (un == NULL) {
    PyErr_NoMemory();
    return NULL;
  }

  /* scan for beginning of localName */
  for (i = 0; i < len && p[i] != nssep; i++);

  if (i == len) {
    /* just a localName, set namespace and prefix to None. */
    un->uri = Py_None;
    un->local = expanded_name;
    un->prefix = Py_None;
    un->qname = expanded_name;
    Py_INCREF(un->uri);
    Py_INCREF(un->local);
    Py_INCREF(un->prefix);
    Py_INCREF(un->qname);
    return un;
  }

  /* found a namespace uri */
  if ((un->uri = PyUnicode_FromUnicode(p, i)) == NULL) {
    PyMem_Del(un);
    return NULL;
  }
  i++;

  for (j = i; j < len && p[j] != nssep; j++);

  if ((un->local = PyUnicode_FromUnicode(p + i, j - i)) == NULL) {
    Py_DECREF(un->uri);
    PyMem_Del(un);
    return NULL;
  }
  j++;

  if (j < len) {
    /* a prefix is given as well */
    if ((un->prefix = PyUnicode_FromUnicode(p + j, len - j)) == NULL) {
      Py_DECREF(un->local);
      Py_DECREF(un->uri);
      PyMem_Del(un);
      return NULL;
    }
    if ((un->qname = MakeQName(un->prefix, un->local)) == NULL) {
      Py_DECREF(un->prefix);
      Py_DECREF(un->local);
      Py_DECREF(un->uri);
      PyMem_Del(un);
      return NULL;
    }
  } else {
    un->prefix = Py_None;
    Py_INCREF(Py_None);
    un->qname = un->local;
    Py_INCREF(un->qname);
  }

#ifdef DEBUG_PARSER
  fputs("-- Universal Name ------------------------------------------\n", 
        stderr);
  PRINT_OBJECT(" namespace-uri: ", un->uri);
  PRINT_OBJECT("    local-name: ", un->local);
  PRINT_OBJECT("        prefix: ", un->prefix);
  PRINT_OBJECT("qualified-name: ", un->qname);
#endif
  return un;
}

void destroyUniversalName(UniversalName *un)
{
#ifdef DEBUG_PARSER
  fputs("destroyUniversalName\n", stderr);
  _PyObject_Dump(un->local);
  _PyObject_Dump(un->uri);
  _PyObject_Dump(un->prefix);
  _PyObject_Dump(un->qname);
  fputs("\n", stderr);
#endif
  Py_DECREF(un->qname);
  Py_DECREF(un->prefix);
  Py_DECREF(un->local);
  Py_DECREF(un->uri);
  PyMem_Del(un);
}

