#include "domlette.h"

/** Domlette only implementation **************************************/

static PyObject *get_all_ns_domlette(PyNodeObject *node, PyObject *nss)
{
  if (PyElement_Check(node)) {
    PyObject *key, *attr;
    PyObject *namespaceURI, *prefix;
    int i;

    /* add the declaration if prefix is not already defined */
    if (PyElement_NAMESPACE_URI(node) != Py_None &&
        PyDict_GetItem(nss, PyElement_PREFIX(node)) == NULL) {
      if (PyDict_SetItem(nss, PyElement_PREFIX(node),
                         PyElement_NAMESPACE_URI(node)) == -1) {
        return NULL;
      }
    }
    
    /* now process this element's attributes */
    i = 0;
    while (PyDict_Next(PyElement_ATTRIBUTES(node), &i, &key, &attr)) {
      /* get the prefix/namespaceURI pair to add */
      switch (PyObject_RichCompareBool(PyAttr_NAMESPACE_URI(attr),
                                       g_xmlnsNamespace, Py_EQ)) {
      case 0:
        /* normal attribute */
        namespaceURI = PyAttr_NAMESPACE_URI(attr);
        prefix = PyAttr_PREFIX(attr);
        break;
      case 1: 
        /* namespace attribute */
        namespaceURI = PyAttr_NODE_VALUE(attr);
        prefix = PyAttr_PREFIX(attr);
        if (prefix != Py_None) {
          /* xmlns:foo = 'namespaceURI'; prefix='xmlns' */
          prefix = PyAttr_LOCAL_NAME(attr);
        }
        if (PyUnicode_GET_SIZE(namespaceURI) == 0) {
          /* empty string; remove prefix binding */
          /* NOTE: in XML Namespaces 1.1 it would be possible to do this
             for all prefixes, for now just the default namespace */
          if (prefix == Py_None) {
            namespaceURI = Py_None;
          }
        }
        break;
      default:
        return NULL;
      }

      /* add the declaration if prefix is not already defined */
      if (namespaceURI != Py_None && PyDict_GetItem(nss, prefix) == NULL) {
        if (PyDict_SetItem(nss, prefix, namespaceURI) == -1) {
          return NULL;
        }
      }
    }
  }    

  if (PyNode_PARENT_NODE(node) != Py_None) {
    nss = get_all_ns_domlette((PyNodeObject *)PyNode_PARENT_NODE(node), nss);
  }

  return nss;
}


static PyObject *seek_nss_domlette(PyNodeObject *node, PyObject *nss)
{
  int i;
  PyObject *children;
  PyObject *child;
  PyObject *key, *attr;

  if (PyElement_Check(node)) {
    /* don't replace existing ns decls */
    if (PyDict_GetItem(nss, PyElement_PREFIX(node)) == NULL) {
      if (PyDict_SetItem(nss, PyElement_PREFIX(node), 
                         PyElement_NAMESPACE_URI(node)) == -1) {
        return NULL;
      }
    }
    
    /* now process this element's attributes */
    i = 0;
    while (PyDict_Next(PyElement_ATTRIBUTES(node), &i, &key, &attr)) {
      PyObject *namespaceURI, *prefix;

      /* get the prefix/namespaceURI pair to add */
      switch (PyObject_RichCompareBool(PyAttr_NAMESPACE_URI(attr),
                                       g_xmlnsNamespace, Py_EQ)) {
      case 0:
        /* normal attribute */
        namespaceURI = PyAttr_NAMESPACE_URI(attr);
        prefix = PyAttr_PREFIX(attr);
        break;
      case 1: 
        /* namespace attribute */
        namespaceURI = PyAttr_NODE_VALUE(attr);
        prefix = PyAttr_PREFIX(attr);
        if (prefix != Py_None) {
          /* xmlns:foo = 'namespaceURI'; prefix='xmlns' */
          prefix = PyAttr_LOCAL_NAME(attr);
        }
        if (PyUnicode_GET_SIZE(namespaceURI) == 0) {
          /* empty string; remove prefix binding */
          /* NOTE: in XML Namespaces 1.1 it would be possible to do this
             for all prefixes, for now just the default namespace */
          if (prefix == Py_None) {
            namespaceURI = Py_None;
          }
        }
        break;
      default:
        return NULL;
      }

      /* add the declaration if prefix is not already defined */
      if (PyDict_GetItem(nss, prefix) == NULL) {
        if (PyDict_SetItem(nss, prefix, namespaceURI) == -1) {
          return NULL;
        }
      }
    }

    children = PyElement_CHILDNODES(node);
  }
  else if (PyDocument_Check(node)) {
    children = PyDocument_CHILDNODES(node);
  }
  else {
    /* non-container node */
    return nss;
  }

  for (i = 0; i < PyList_GET_SIZE(children); i++) {
    child = PyList_GET_ITEM(children, i);
    if (PyElement_Check(child)) {
      if (seek_nss_domlette((PyNodeObject *)child, nss) == NULL) {
        return NULL;
      }
    }
  }

  return nss;
}

/** DOM API implementation ********************************************/

static PyObject *get_all_ns_dom(PyObject *node, PyObject *nss)
{
  PyObject *obj;
  long nodeType;
  PyObject *tuple;

  obj = PyObject_GetAttrString(node, "nodeType");
  if (obj == NULL) {
    return NULL;
  }
  nodeType = PyInt_AsLong(obj);
  Py_DECREF(obj);
  if (PyErr_Occurred()) {
    return NULL;
  }

  if (nodeType == ELEMENT_NODE) {
    PyObject *namespaceURI, *prefix;
    PyObject *sequence;
    int i;

    /* add the element's namespace declaration */
    namespaceURI = PyObject_GetAttrString(node, "namespaceURI");
    namespaceURI = DOMString_FromObjectInplace(namespaceURI);
    prefix = PyObject_GetAttrString(node, "prefix");
    prefix = DOMString_FromObjectInplace(prefix);
    if (namespaceURI == NULL || prefix == NULL) {
      Py_XDECREF(namespaceURI);
      Py_XDECREF(prefix);
      return NULL;
    }
    
    /* add the declaration if prefix is not already defined */
    if (namespaceURI != Py_None && PyDict_GetItem(nss, prefix) == NULL) {
      if (PyDict_SetItem(nss, prefix, namespaceURI) == -1) {
        Py_DECREF(namespaceURI);
        Py_DECREF(prefix);
        return NULL;
      }
    }
    Py_DECREF(namespaceURI);
    Py_DECREF(prefix);
    
    /* now process this element's attributes */
    obj = PyObject_GetAttrString(node, "attributes");
    if (obj == NULL) {
      return NULL;
    }

    sequence = PyObject_CallMethod(obj, "values", NULL);
    Py_DECREF(obj);
    if (sequence == NULL) {
      return NULL;
    }

    tuple = PySequence_Tuple(sequence);
    Py_DECREF(sequence);
    if (tuple == NULL) {
      return NULL;
    }

    for (i = 0; i < PyTuple_GET_SIZE(tuple); i++) {
      PyObject *attr = PyTuple_GET_ITEM(tuple, i);
      if (attr == NULL) {
        Py_DECREF(tuple);
        return NULL;
      }

      namespaceURI = PyObject_GetAttrString(attr, "namespaceURI");
      namespaceURI = DOMString_FromObjectInplace(namespaceURI);
      prefix = PyObject_GetAttrString(attr, "prefix");
      prefix = DOMString_FromObjectInplace(prefix);
      if (namespaceURI == NULL || prefix == NULL) {
        Py_XDECREF(namespaceURI);
        Py_XDECREF(prefix);
        Py_DECREF(tuple);
        return NULL;
      }

      /* get the prefix/namespaceURI pair to add */
      switch (PyObject_RichCompareBool(namespaceURI, g_xmlnsNamespace,
                                       Py_EQ)) {
      case 0:
        /* normal attribute */
        break;
      case 1: 
        /* namespace attribute */
        Py_DECREF(namespaceURI);
        namespaceURI = PyObject_GetAttrString(attr, "value");
        namespaceURI = DOMString_FromObjectInplace(namespaceURI);
        if (namespaceURI == NULL) {
          Py_DECREF(prefix);
          Py_DECREF(tuple);
          return NULL;
        }
        if (prefix != Py_None) {
          /* xmlns:foo = 'namespaceURI'; prefix='xmlns' */
          Py_DECREF(prefix);
          prefix = PyObject_GetAttrString(attr, "localName");
          prefix = DOMString_FromObjectInplace(prefix);
          if (prefix == NULL) {
            Py_DECREF(namespaceURI);
            Py_DECREF(tuple);
            return NULL;
          }
        }
        if (PyUnicode_GET_SIZE(namespaceURI) == 0) {
          /* empty string; remove prefix binding */
          /* NOTE: in XML Namespaces 1.1 it would be possible to do this
             for all prefixes, for now just the default namespace */
          if (prefix == Py_None) {
            Py_DECREF(namespaceURI);
            Py_INCREF(Py_None);
            namespaceURI = Py_None;
          }
        }
        break;
      default:
        Py_DECREF(namespaceURI);
        Py_DECREF(prefix);
        Py_DECREF(tuple);
        return NULL;
      }

      /* add the declaration if prefix is not already defined */
      if (namespaceURI != Py_None && PyDict_GetItem(nss, prefix) == NULL) {
        if (PyDict_SetItem(nss, prefix, namespaceURI) == -1) {
          Py_DECREF(namespaceURI);
          Py_DECREF(prefix);
          Py_DECREF(tuple);
          return NULL;
        }
      }
      
      Py_DECREF(namespaceURI);
      Py_DECREF(prefix);
    }
    
    Py_DECREF(tuple);
  }
  
  if (nodeType == ATTRIBUTE_NODE) 
    obj = PyObject_GetAttrString(node, "ownerElement");
  else
    obj = PyObject_GetAttrString(node, "parentNode");
  if (obj == NULL) {
    return NULL;
  }
  else if (obj != Py_None) {
    nss = get_all_ns_dom(obj, nss);
  }
  Py_DECREF(obj);

  return nss;
}


static PyObject *seek_nss_dom(PyObject *node, PyObject *nss)
{
  PyObject *obj, *tuple;
  long nodeType;
  int i;

  obj = PyObject_GetAttrString(node, "nodeType");
  if (obj == NULL) {
    return NULL;
  }
  nodeType = PyInt_AsLong(obj);
  Py_DECREF(obj);
  if (PyErr_Occurred()) {
    return NULL;
  }

  if (nodeType == ELEMENT_NODE) {
    /* don't add ns decl for default namespace as it is initially defined */
    PyObject *namespaceURI, *prefix;
    PyObject *sequence;

    namespaceURI = PyObject_GetAttrString(node, "namespaceURI");
    namespaceURI = DOMString_FromObjectInplace(namespaceURI);
    prefix = PyObject_GetAttrString(node, "prefix");
    prefix = DOMString_FromObjectInplace(prefix);
    if (namespaceURI == NULL || prefix == NULL) {
      Py_XDECREF(namespaceURI);
      Py_XDECREF(prefix);
      return NULL;
    }

    /* don't replace existing ns decls */
    if (PyDict_GetItem(nss, prefix) == NULL) {
      if (PyDict_SetItem(nss, prefix, namespaceURI) == -1) {
        Py_DECREF(namespaceURI);
        Py_DECREF(prefix);
        return NULL;
      }
    }

    Py_DECREF(namespaceURI);
    Py_DECREF(prefix);
    
    /* now process this element's attributes */
    obj = PyObject_GetAttrString(node, "attributes");
    if (obj == NULL) {
      return NULL;
    }

    sequence = PyObject_CallMethod(obj, "values", NULL);
    Py_DECREF(obj);
    if (sequence == NULL) {
      return NULL;
    }

    tuple = PySequence_Tuple(sequence);
    Py_DECREF(sequence);
    if (tuple == NULL) {
      return NULL;
    }

    for (i = 0; i < PyTuple_GET_SIZE(tuple); i++) {
      PyObject *attr = PyTuple_GET_ITEM(tuple, i);
      if (attr == NULL) {
        Py_DECREF(tuple);
        return NULL;
      }

      namespaceURI = PyObject_GetAttrString(attr, "namespaceURI");
      namespaceURI = DOMString_FromObjectInplace(namespaceURI);
      prefix = PyObject_GetAttrString(attr, "prefix");
      prefix = DOMString_FromObjectInplace(prefix);
      if (namespaceURI == NULL || prefix == NULL) {
        Py_XDECREF(namespaceURI);
        Py_XDECREF(prefix);
        Py_DECREF(tuple);
        return NULL;
      }

      /* get the prefix/namespaceURI pair to add */
      switch (PyObject_RichCompareBool(namespaceURI, g_xmlnsNamespace, 
                                       Py_EQ)) {
      case 0:
        /* normal attribute */
        break;
      case 1: 
        /* namespace attribute */
        Py_DECREF(namespaceURI);
        namespaceURI = PyObject_GetAttrString(attr, "value");
        namespaceURI = DOMString_FromObjectInplace(namespaceURI);
        if (namespaceURI == NULL) {
          Py_DECREF(prefix);
          Py_DECREF(tuple);
          return NULL;
        }
        if (prefix != Py_None) {
          /* xmlns:foo = 'namespaceURI'; prefix='xmlns' */
          Py_DECREF(prefix);
          prefix = PyObject_GetAttrString(attr, "localName");
          prefix = DOMString_FromObjectInplace(prefix);
          if (prefix == NULL) {
            Py_DECREF(namespaceURI);
            Py_DECREF(tuple);
            return NULL;
          }
        }
        if (PyUnicode_GET_SIZE(namespaceURI) == 0) {
          /* empty string; remove prefix binding */
          /* NOTE: in XML Namespaces 1.1 it would be possible to do this
             for all prefixes, for now just the default namespace */
          if (prefix == Py_None) {
            Py_DECREF(namespaceURI);
            Py_INCREF(Py_None);
            namespaceURI = Py_None;
          }
        }
        break;
      default:
        Py_DECREF(namespaceURI);
        Py_DECREF(prefix);
        Py_DECREF(tuple);
        return NULL;
      }

      /* add the declaration if prefix is not already defined */
      if (PyDict_GetItem(nss, prefix) == NULL) {
        if (PyDict_SetItem(nss, prefix, namespaceURI) == -1) {
          Py_DECREF(namespaceURI);
          Py_DECREF(prefix);
          Py_DECREF(tuple);
          return NULL;
        }
      }
      
      Py_DECREF(namespaceURI);
      Py_DECREF(prefix);
    }
    Py_DECREF(tuple);
  }

  /* process the children of this node */
  obj = PyObject_GetAttrString(node, "childNodes");
  if (obj == NULL) {
    return NULL;
  }

  tuple = PySequence_Tuple(obj);
  Py_DECREF(obj);
  if (tuple == NULL) {
    return NULL;
  }

  for (i = 0; i < PyTuple_GET_SIZE(tuple); i++) {
    if (seek_nss_dom(PyTuple_GET_ITEM(tuple, i), nss) == NULL) {
      Py_DECREF(tuple);
      return NULL;
    }
  }
  Py_DECREF(tuple);
  
  return nss;
}

/** Public Methods ****************************************************/

char GetAllNs_doc[] = \
"GetAllNs(node) -> dict\n\
\n\
Get all of the namespaces defined in scope of this node.";

PyObject *Domlette_GetAllNs(PyObject *self, PyObject *args)
{
  PyObject *node;
  PyObject *nss;
  PyObject *prefix;
  PyObject *result;

  if (!PyArg_ParseTuple(args, "O:GetAllNs", &node))
    return NULL;

  nss = PyDict_New();
  if (nss == NULL) return NULL;

  /* add the XML namespace */
  prefix = PyUnicode_DecodeASCII("xml", 3, NULL);
  if (prefix == NULL) {
    Py_DECREF(nss);
    return NULL;
  }
  if (PyDict_SetItem(nss, prefix, g_xmlNamespace) == -1) {
    Py_DECREF(nss);
    Py_DECREF(prefix);
    return NULL;
  }
  Py_DECREF(prefix);
  
  if (PyNode_Check(node)) 
    result = get_all_ns_domlette((PyNodeObject *) node, nss);
  else
    result = get_all_ns_dom(node, nss);

  if (result == NULL) {
    Py_DECREF(nss);
  }

  return result;
}

char SeekNss_doc[] = \
"SeekNss(node) -> dict\n\
\n\
Traverses the tree to seek an approximate set of defined namespaces.";

PyObject *Domlette_SeekNss(PyObject *self, PyObject *args)
{
  PyObject *node;
  PyObject *nss;
  PyObject *result;

  if (!PyArg_ParseTuple(args, "O:SeekNss", &node))
    return NULL;

  nss = PyDict_New();
  if (nss == NULL) return NULL;

  if (PyNode_Check(node)) 
    result = seek_nss_domlette((PyNodeObject *)node, nss);
  else
    result = seek_nss_dom(node, nss);

  if (result == NULL) {
    Py_DECREF(nss);
  }
  else {
    /* don't expose the implied default namespace */
    if (PyDict_GetItem(nss, Py_None) == Py_None) {
      if (PyDict_DelItem(nss, Py_None) == -1) {
        Py_DECREF(nss);
        return NULL;
      }
    }
  }

  return result;
}
