#include "domlette.h"

#define Comment_VerifyState(ob)                          \
  if (!PyComment_Check(ob) ||                            \
      ((PyCommentObject *)(ob))->nodeValue == NULL)      \
     return DOMException_InvalidStateErr("Comment in inconsistent state");
      
static PyObject *my_node_type, *my_node_name;

/** C API **************************************************************/

PyCommentObject *Comment_CloneNode(PyObject *node, int deep,
				   PyNodeObject *newOwnerDocument)
{
  long tempId;
  PyObject *nodeValue;
  PyCommentObject *newNode;

  if (!PyDocument_Check(newOwnerDocument)) {
    PyErr_SetString(PyExc_TypeError,"newOwnerDocument must be a cDocument");
    return NULL;
  }

  nodeValue = PyObject_GetAttrString(node, "nodeValue");
  nodeValue = DOMString_FromObjectInplace(nodeValue);
  if (nodeValue == NULL) return NULL;

  newNode = Document_CreateComment((PyDocumentObject *)newOwnerDocument,
				   nodeValue, &tempId);
  Py_DECREF(nodeValue);

  return newNode;
}

/** Python API *********************************************************/

static struct PyMethodDef PyComment_methods[] = {
  NODE_METHODS,
  CHARACTERDATA_METHODS,
  {NULL, NULL}      /* sentinel */
};

/** Python Slot Methods ************************************************/

static PyObject *comment_getattr(PyCommentObject *self, char *name)
{
  PyObject *rt = NULL;

  Comment_VerifyState(self);

  if (!strcmp(name, "data")) {
    rt = (PyObject *)self->nodeValue;
  }
  else if (!strcmp(name, "nodeValue")) {
    rt = (PyObject *)self->nodeValue;
  }
  else if (!strcmp(name, "nodeType")) {
    rt = my_node_type;
  }
  else if (!strcmp(name, "nodeName")) {
    rt = my_node_name;
  }
  if (rt) {
    Py_INCREF(rt);
    return rt;
  }

  return node_getattr((PyNodeObject*)self, name, PyComment_methods);
}


static void comment_dealloc(PyCommentObject *node)
{
  PyObject_GC_UnTrack((PyObject *) node);

  Py_XDECREF(node->nodeValue);
  node->nodeValue = NULL;

  Node_Del(node);
}

static char *comment_format_str = "<cComment 0x%p>";

static PyObject *comment_repr(PyCommentObject *comment)
{
  char buf[256];
  sprintf(buf, comment_format_str, comment);
  return PyString_FromString(buf);
}

int comment_test_ref_counts(PyObject *tester,PyCommentObject *node,long *childCtr,PyObject *internDict,int base) {
  /*Ref count is base + 1.  Out nodeValue should have a ref count of 1 but is not interned*/
  PyObject_CallMethod(tester,"startTest","s","Node");
  if (!PyObject_CallMethod(tester,"compare","ll",base+1,node->ob_refcnt)) return 0;
  PyObject_CallMethod(tester,"testDone","");
  if (!TestRefCount(tester,(PyObject *)node->nodeValue,1, "nodeValue")) return 0;
  return 1;
}

PyTypeObject PyDomletteComment_Type = {
    PyObject_HEAD_INIT(0)
    0,
    "cDomlette.Comment",
    GC_TP_BASICSIZE(PyCommentObject),
    0,
    (destructor)comment_dealloc,    /*tp_dealloc*/
    (printfunc)0,   /*tp_print*/
    (getattrfunc)comment_getattr,   /*tp_getattr*/
    (setattrfunc)characterdata_setattr,  /*tp_setattr*/
    0,                          /*tp_compare*/
    (reprfunc)comment_repr,          /*tp_repr*/
    0,                          /*tp_as_number*/
    0,              /*tp_as_sequence*/
    0,              /*tp_as_mapping*/
    0,                             /*tp_hash*/
    0,          /*tp_call*/
    0,          /*tp_str*/
    0,                      /*tp_getattro*/
    0,          /*tp_setattro*/
    0,                          /* tp_as_buffer */
    Py_TPFLAGS_DEFAULT | Py_TPFLAGS_HAVE_GC,
    0,                          /* tp_doc */
    (traverseproc)node_traverse,  /* tp_traverse */
    (inquiry)node_clear,          /* tp_clear */
    0,                          /* tp_richcompare */
    0,                          /* tp_weaklistoffset */
};

int DomletteComment_Init(void)
{
  PyDomletteComment_Type.ob_type = &PyType_Type;

  my_node_type = PyInt_FromLong(COMMENT_NODE);
  if (my_node_type == NULL) return 0;

  my_node_name = PyUnicode_DecodeASCII("#comment", 8, NULL);
  if (my_node_name == NULL) return 0;

  return 1;
}

void DomletteComment_Fini(void)
{
  Py_DECREF(my_node_type);
  Py_DECREF(my_node_name);
}
