########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Xml/FtMiniDom/NonvalReader.py,v 1.14 2004/06/09 01:41:09 mbrown Exp $
"""
Minidom non-validating reader; requires pyexpat and expat w/DTD support.

If pyexpat is not installed at all, then importing this module should
be harmless. If pyexpat is installed, but the underlying expat does not
support DTDs, then the import will fail with a SystemExit.

Copyright 2004 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

from Ft.Xml import READ_EXTERNAL_DTD, HAS_PYEXPAT, VerifyPyexpat
from Ft.Xml import ReaderException
import Handler

class PyexpatNonvalReader(Handler.Handler):

    ### Reader Interface ###

    def fromSrc(self, source, parseParamEntities=READ_EXTERNAL_DTD):
        self.initHandler(source, parseParamEntities)
        self.prepareParser()

        self.parse(source)

        self.reset()
        return self.getRootNode()

    def prepareParser(self):
        parser = self.parser = expat.ParserCreate(self.inputSource.encoding)
        if self.parseParamEntities:
            parser.SetParamEntityParsing(expat.XML_PARAM_ENTITY_PARSING_ALWAYS)
        parser.StartElementHandler = self.startElement
        parser.EndElementHandler = self.endElement
        parser.CharacterDataHandler = self.characters
        parser.ProcessingInstructionHandler = self.processingInstruction
        parser.CommentHandler = self.comment
        parser.ExternalEntityRefHandler = self.externalEntityRef
        parser.StartDoctypeDeclHandler = self.startDoctypeDecl
        parser.EndDoctypeDeclHandler = self.endDoctypeDecl
        parser.UnparsedEntityDeclHandler = self.unparsedEntityDecl
        return

    def parse(self, source):
        self.parser.SetBase(source.uri)
        try:
            self.parser.ParseFile(source.stream)
        except expat.error:
            raise ReaderException(ReaderException.XML_PARSE_ERROR,
                                  self.parser.GetBase(),
                                  self.parser.ErrorLineNumber,
                                  self.parser.ErrorColumnNumber,
                                  expat.ErrorString(self.parser.ErrorCode))

        return

    def error(self, message):
        raise ReaderException(ReaderException.XML_PARSE_ERROR,
                              self.parser.GetBase(),
                              self.parser.ErrorLineNumber,
                              self.parser.ErrorColumnNumber,
                              message)

    def externalEntityRef(self, context, base, sysid, pubid):
        source = self.inputSource.resolve(sysid, pubid, 'EXTERNAL ENTITY')

        orig_parser = self.parser
        orig_source = self.inputSource

        self.inputSource = source
        self.parser = self.parser.ExternalEntityParserCreate(context)
        self.parser.SetBase(source.uri)
        self.parser.ParseFile(source.stream)
        source.stream.close()

        self.inputSource = orig_source
        self.parser = orig_parser
        return 1

    def reset(self):
        self.parser = None
        return


if HAS_PYEXPAT:
    VerifyPyexpat(dtdSupport=1, featureName="FtMiniDom's non-validating reader")
    from xml.parsers import expat
    NonvalReader = PyexpatNonvalReader
else:
    # Create a dummy reader class
    class NonvalReader:
        def __init__(self, *varargs, **varkw):
            VerifyPyexpat(dtdSupport=1, featureName="FtMiniDom's non-validating reader")
        # stub for completeness
        def fromSrc(self, src):
            raise NotImplementedError("Did the FtMiniDom non-validating reader import correctly?")


def nonvalParse(src, readExtDtd=READ_EXTERNAL_DTD):
    r = NonvalReader()
    return r.fromSrc(src, readExtDtd)

