########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Xml/FtMiniDom/Handler.py,v 1.13 2003/11/27 05:39:55 mbrown Exp $
"""
Handler to construct an FtMiniDom tree from SAX events

Copyright 2003 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

from types import TupleType
from xml.sax import xmlreader

from Ft.Xml import XML_NAMESPACE, EMPTY_NAMESPACE, XMLNS_NAMESPACE
from Ft.Xml import SplitQName
from Ft.Xml import ReaderException
from Ft.Xml.XInclude import XINCLUDE_NAMESPACE, XIncludeException

from Strip import XmlStrStrip
from DomTree import implementation

class Handler:

    def initHandler(self, inputSource, parseParamEntities):
        self.inputSource = inputSource
        self.parseParamEntities = parseParamEntities

        # Init our root node
        document = implementation.createRootNode(inputSource.uri)
        self._rootNode = self._ownerDoc = document

        # Setup the stack which keeps track of the nesting of DOM nodes.
        self._nodeStack = [self._rootNode]
        self._namespaces = [{'xml': XML_NAMESPACE, 'xmlns':XMLNS_NAMESPACE}]
        self._currText = ''

        self._preserveStateStack = [1]  #initialize to preserve
        self._stripElements = inputSource.stripElements

        # Setup our include depth
        self._includeDepth = 0

        self._xmlBaseStack = [inputSource.uri]
        self._rootNode.xmlBase = inputSource.uri

        # For XIncludes
        self._visited_hrefs = []
        self._ignore_top_level_whitespace = [1]

        # For disabling comments and processing instructions while
        # processing DTD
        self._inDTD = 0
        return

    def getRootNode(self):
        self._completeTextNode()
        return self._rootNode


    ### Reader Interface (stubs for completeness) ###

    def prepareParser(self):
        raise NotImplementedError("Subclass must override")

    def parse(self, source):
        raise NotImplementedError("Subclass must override")

    def error(self, message):
        raise NotImplementedError("Subclass must override")

    ### ContentHandler Interface ###

    def startElement(self, name, attribs):
        """Signals the start of an element in non-namespace mode.

        The name parameter contains the raw XML 1.0 name of the
        element type as a string and the attrs parameter holds an
        instance of the Attributes class containing the attributes of
        the element."""

        self._completeTextNode()

        self._ignore_top_level_whitespace.append(0)

        # Create a copy of our parents namespaces
        self._namespaces.append(self._namespaces[-1].copy())

        #Read our attributes
        for curr_attrib_key in attribs.keys():
            if type(curr_attrib_key) is TupleType:
                local = curr_attrib_key[1]
                prefix = None
                if curr_attrib_key[0]:
                    raise RuntimeError("Namespaces in validating docs not supported")
            else:
                (prefix, local) = SplitQName(curr_attrib_key)
            if not local:
                raise ReaderException(ReaderException.INVALID_XMLNS, curr_attrib_key)
            if prefix == 'xmlns':
                self._namespaces[-1][local] = attribs[curr_attrib_key] or EMPTY_NAMESPACE
            elif prefix is None and local == 'xmlns':
                if not attribs[curr_attrib_key]:
                    if self._namespaces[-1].has_key(None):
                        del self._namespaces[-1][None]
                else:
                    self._namespaces[-1][None] = attribs[curr_attrib_key]

        if type(name) is TupleType:
            local = name[1]
            prefix = None
            if name[0]:
                raise RuntimeError("Namespaces in validating docs not supported")
        else:
            (prefix, local) = SplitQName(name)

        # For consistency with cDomlette
        if prefix and not self._namespaces[-1].has_key(prefix):
            self.error('unbound prefix')

        if self._namespaces[-1].has_key(prefix):
            namespace = self._namespaces[-1][prefix]
        else:
            namespace = EMPTY_NAMESPACE

        attrs = {}
        qnames = {}
        for curr_attrib_key in attribs.keys():
            if type(curr_attrib_key) is TupleType:
                a_local = curr_attrib_key[1]
                a_prefix = None
                if curr_attrib_key[0]:
                    raise RuntimeError("Namespaces in validating docs not supported")
            else:
                (a_prefix, a_local) = SplitQName(curr_attrib_key)
            if a_prefix:
                if self._namespaces[-1].has_key(a_prefix):
                    ns = self._namespaces[-1][a_prefix]
                else:
                    # For consistency with cDomlette
                    self.error('unbound prefix')
            else:
                ns = EMPTY_NAMESPACE

            # For consistency with cDomlette
            if attrs.has_key((ns, a_local)):
                self.error('duplicate attribute')

            attrs[(ns, a_local)] = attribs[curr_attrib_key]
            qnames[(ns, a_local)] = curr_attrib_key


        nsattribs = xmlreader.AttributesNSImpl(attrs, qnames)

        #Adjust local variables
        (name, qname) = ((namespace or None, local), name)

        #See if we need to handle XInclude

        if self.inputSource.processIncludes and name == (XINCLUDE_NAMESPACE, 'include'):

            if self._includeDepth:
                self._includeDepth = self._includeDepth + 1
            else:
                #Looks like it is a GO!!
                href = nsattribs.get((EMPTY_NAMESPACE, 'href'))
                if not href:
                    raise XIncludeException(XIncludeException.MISSING_HREF)
                source = self.inputSource.resolve(href, '', 'XINCLUDE')
                if source.uri in self._visited_hrefs:
                    raise XIncludeException(
                        XIncludeException.CIRCULAR_INCLUDE_ERROR, href
                        )

                parse = nsattribs.get((EMPTY_NAMESPACE, 'parse'))
                if not parse or parse == 'xml':

                    self._visited_hrefs.append(source.uri)
                    self._ignore_top_level_whitespace.append(1)

                    orig_parser = self.parser
                    orig_source = self.inputSource

                    self.prepareParser()
                    self.inputSource = source

                    self.parse(source)

                    self.parser = orig_parser
                    self.inputSource = orig_source

                    del self._visited_hrefs[-1]
                    del self._ignore_top_level_whitespace[-1]
                else:
                    self._currText = self._currText + source.stream.read()
                source.stream.close()
                self._includeDepth = 1
        else:  #end of the check for XInclude processing
            #Create the new Element
            new_element = self._ownerDoc.createElementNS(namespace, qname)

            nextBase = self._xmlBaseStack[-1]

            for attr_qname in nsattribs.getQNames():

                attr_ns = nsattribs.getNameByQName(attr_qname)[0]
                (attr_prefix, attr_local) = SplitQName(attr_qname)

                if attr_prefix is None and attr_local == 'xmlns':
                    attr_ns = XMLNS_NAMESPACE
                    attr_key = (attr_ns, None)
                else:
                    attr_key = (attr_ns, attr_local)

                new_element.setAttributeNS(attr_ns, attr_qname,
                                           nsattribs.getValueByQName(attr_qname))

                #Look for a change in xml:base
                if (XML_NAMESPACE,'base') == attr_key:
                    nextBase = nsattribs.getValueByQName(attr_qname)


            new_element.xmlBase = new_element.baseURI = nextBase
            self._xmlBaseStack.append(nextBase)

            new_pstate = self._preserveStateStack[-1]
            for (uri, local, strip) in self._stripElements:
                if (uri, local) in [(new_element.namespaceURI,
                                     new_element.localName), (EMPTY_NAMESPACE, '*'),
                                    (new_element.namespaceURI, '*')
                                    ]:
                    new_pstate = not strip
                    break
            self._preserveStateStack.append(new_pstate)

            self._nodeStack.append(new_element)
        return

    def endElement(self, name):
        """Signals the end of an element in non-namespace mode.

        The name parameter contains the name of the element type, just
        as with the startElement event."""

        del self._namespaces[-1]

        if self._includeDepth:
            self._includeDepth = self._includeDepth - 1
            del self._ignore_top_level_whitespace[-1]
            return

        self._completeTextNode()
        del self._ignore_top_level_whitespace[-1]

        new_element = self._nodeStack[-1]

        del self._preserveStateStack[-1]

        del self._xmlBaseStack[-1]

        del self._nodeStack[-1]
        self._nodeStack[-1].appendChild(new_element)
        return

    def characters(self, data):
        """Receive notification of character data.

        The Parser will call this method to report each chunk of
        character data. SAX parsers may return all contiguous
        character data in a single chunk, or they may split it into
        several chunks; however, all of the characters in any single
        event must come from the same external entity so that the
        Locator provides useful information."""
        if self._includeDepth: return
        self._currText = self._currText + data
        return

    def processingInstruction(self, target, data):
        """Receive notification of a processing instruction.

        The Parser will invoke this method once for each processing
        instruction found: note that processing instructions may occur
        before or after the main document element.

        A SAX parser should never report an XML declaration (XML 1.0,
        section 2.8) or a text declaration (XML 1.0, section 4.3.1)
        using this method."""
        if self._inDTD or self._includeDepth: return
        self._completeTextNode()
        pi = self._ownerDoc.createProcessingInstruction(target, data)
        pi.xmlBase = pi.baseURI = self._xmlBaseStack[-1]
        self._nodeStack[-1].appendChild(pi)
        return

    ### LexicalHandler Interface ###

    def comment(self, data):
        """Reports a comment anywhere in the document (including the
        DTD and outside the document element).

        content is a string that holds the contents of the comment."""
        if self._inDTD or self._includeDepth: return
        self._completeTextNode()
        comment = self._ownerDoc.createComment(data)
        comment.xmlBase = comment.baseURI = self._xmlBaseStack[-1]
        self._nodeStack[-1].appendChild(comment)
        return

    def startDoctypeDecl(self, name, systemId, publicId, hasInternalSubset):
        """Report the start of the DTD declarations, if the document
        has an associated DTD.

        A startEntity event will be reported before declaration events
        from the external DTD subset are reported, and this can be
        used to infer from which subset DTD declarations derive.

        name is the name of the document element type, publicId the
        public identifier of the DTD (or None if none were supplied)
        and systemId the system identfier of the external subset (or
        None if none were supplied)."""
        self._ownerDoc.publicId = publicId
        self._ownerDoc.systemId = systemId
        self._inDTD = 1
        return

    def endDoctypeDecl(self):
        "Signals the end of DTD declarations."
        self._inDTD = 0
        return

    ### DTDHandler Interface ###

    def unparsedEntityDecl(self, name, base, systemId, publicId, notationName):
        "Handle an unparsed entity declaration event."
        resolver = self.inputSource.getUriResolver()
        uri = resolver.normalize(systemId, base)
        self._ownerDoc.unparsedEntities[name] = uri
        return

    ### Internal Functions ###

    def _completeTextNode(self):
        if not self._currText:
            return

        data = self._currText
        self._currText = u''

        # Ignore any top-level whitespace
        whitespace_only = not XmlStrStrip(data)
        if whitespace_only and self._ignore_top_level_whitespace[-1]:
            return

        if self._preserveStateStack[-1] or not whitespace_only:
            # Create a new text node
            text = self._ownerDoc.createTextNode(data)
            text.xmlBase = text.baseURI = self._xmlBaseStack[-1]
            top_node = self._nodeStack[-1]
            top_node.appendChild(text)
        return

