########################################################################
#
# File Name:            SoapHandler.py
#
# Documentation:        http://docs.4suite.org/4SS/SoapApi.py.html
#
"""
Exposes the SCore interface over SOAP
WWW: http://4suite.org         e-mail: support@4suite.org

Copyright (c) 2000-2001 Fourthought, Inc. USA.   All Rights Reserved.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""

import string, base64, sys, cStringIO, traceback
from Ft.Xml import XPath
from Ft.Xml import EMPTY_NAMESPACE

from Ft.Server.Server.Drivers import FtssInputSource

from Ft.Server.Server.Http import BaseRequestHandler, Status

from Ft.Server.Common import ResourceTypes
from xml.dom import Node

from Ft.Server import FTSERVER_NAMESPACE
FTSS_SOAP_NS = '%s#services' % FTSERVER_NAMESPACE

XMLSCHEMA_INSTANCE_NS = 'http://www.w3.org/1999/XMLSchema-instance'
SOAP_NS = "http://schemas.xmlsoap.org/soap/envelope/"
SOAP_ENCODING_STYLE = "http://schemas.xmlsoap.org/soap/encoding/"



class SoapHandler(BaseRequestHandler.BaseRequestHandler):


    #This dictionary is to be used by overridden implementations to look up the correct actions.
    #The default action is to map the URI to a SOAP Server instance.
    #Then call the correct action on the instance.  Since 90% of all of the parameters are strings,
    #All will be type cast to strings by the "default" handler.
    #The biggest exception is self will be the actual resource that was requested.
    NS_TO_HANDLER_MAPPING = {
        }  

    def do_POST(self):
        """
        Handle a POST request that is a SOAP message
        """

        #Look for the SOAP Action
        for name in self.headers_in.keys():
            if name.lower() == 'soapaction':
                break
        else:
            raise Exception("Request did not contain the soapaction header")

        #Read in the request document
        doc = self.reader.fromString(self.body)

        #Look four our handler function
        #this will raise an exception if not handled
        handlerfunc,reqbody = self._checkHeader(doc)

        #Get a connection to the repository
        repo = self.getRepository(sendError = 1)
        if repo is None:
            #An error was sent, just return
            return

        commit = 0
        try:
            #Fall into the main handling
            try:
                respbody = self._prepResponse(reqbody)
                commit = handlerfunc(repo, reqbody, respbody,self.server.errorLog)
                print commit
            except:
                commit = 0
                #Reset the response body
                respbody = self._generateFault(reqbody)
                
            return self._finalizeResponse(reqbody, respbody)
        finally:
            if commit:
                repo.txCommit()
            else:
                repo.txRollback()


    def _checkHeader(self, doc):
        #Read in the SOAP request envelope
        #Make sure it is a valid request for us to handle
       
        soap_envelope = doc.documentElement
        if (soap_envelope.namespaceURI, soap_envelope.localName) != (SOAP_NS, "Envelope"):
            #FIXME: Use HTTP SoapAction header for screening insead.
            #If it's present but the above condition is false, we send a client fault
            #FIXME: This should return a SOAP client fault
            raise Exception('Invalid SOAP message')

        soap_body = filter(lambda x: (x.namespaceURI, x.localName) == (SOAP_NS, "Body"), soap_envelope.childNodes)[0]
        request_el = filter(lambda x: x.nodeType == Node.ELEMENT_NODE, soap_body.childNodes)[0]
        request_ns = request_el.namespaceURI
        request_name = request_el.localName

        if not self.body:
            raise Exception('POST ERROR: missing body')

        handlerfunc = None
        #Determine our handler
        if self.NS_TO_HANDLER_MAPPING.has_key(request_ns):
            handlerfunc1 = self.NS_TO_HANDLER_MAPPING[request_ns]
            if handlerfunc1.has_key(request_name):
                handlerfunc = handlerfunc1[request_name]
            else:
                raise Exception("No handler for SOAP request: " + request_name)
        elif request_ns == FTSS_SOAP_NS:
            handlerfunc = GenericResourceMethod
        else:
            #Otherwise no clue!!!
            raise Exception("No handler for SOAP namespace: " + request_ns)

        return handlerfunc, request_el

    def _prepResponse(self, doc):
        #Create the response document
        dt = Domlette.implementation.createDocumentType("SOAP-ENV:Envelope", '', '')
        respdoc = Domlette.implementation.createDocument(SOAP_NS, "SOAP-ENV:Envelope", dt)
        respdocel = respdoc.documentElement
        respdocel.setAttributeNS(SOAP_NS, "SOAP-ENV:encodingStyle", SOAP_ENCODING_STYLE)
        respbody = respdoc.createElementNS(SOAP_NS, "SOAP-ENV:Body")
        respdocel.appendChild(respbody)
        return respbody

    def _finalizeResponse(self, reqbody, respbody):
        #Serialize the results
        self.status = Status.HTTP_OK
        Domlette.PrettyPrint(respbody.ownerDocument, stream=self.wfile)
        return


    def _generateFault(self,reqbody):

        #Create a new respbody
        respbody = self._prepResponse(reqbody)

        eType, ex = sys.exc_info()[:2]

        doc = respbody.ownerDocument

        if 0:
            #FIXME Capture specific exceptions
            pass
        else:
            #Unknown Fault
            faultCode = "Server"
            faultString = str(ex)
            faultFactor = None
            st = cStringIO.StringIO()
            traceback.print_exc(file=st)
            detailStr = base64.encodestring(st.getvalue())
            detail = doc.createElementNS('','detail')
            tb = doc.createElementNS(FTSS_SOAP_NS,'ftsoap:traceback')
            tb.appendChild(doc.createTextNode(detailStr))
            detail.appendChild(tb)

        fault = doc.createElementNS(SOAP_NS,'SOAP-ENV:Fault')
        fc = doc.createElementNS('','faultCode')
        fc.appendChild(doc.createTextNode(str(faultCode)))
        fault.appendChild(fc)

        fs = doc.createElementNS('','faultString')
        fs.appendChild(doc.createTextNode(str(faultString)))
        fault.appendChild(fs)

        if faultFactor:
            ff = doc.createElementNS('','faultFactor')
            ff.appendChild(doc.createTextNode(str(faultFactor)))
            fault.appendChild(ff)

        if detail:
            fault.appendChild(detail)

        respbody.appendChild(fault)
        return respbody


def GenericResourceMethod(repo,reqel,respbody,logFile):

    #If, we have fallen to this then we want to perform a generic request on the
    #On a resource.  Use the path attribute to get the resource
    path = reqel.getAttributeNS(EMPTY_NAMESPACE,'src-path')
    if not path:
        raise "Missing required 'src-path' attribute"

    res = repo.fetchResource(path)

    #Now, mapp the resource type to the local implementation
    l = g_resourceTypeMapping[res.resourceType](res)
    return l.execute(reqel,respbody,logFile)

    

class RawFileSoapImp:
    def __init__(self,res):
        self.resource = res

    def execute(self,reqel,respbody,logFile):
        """
        See if we define the requested action, ifso call that, otherwise, just call it on the resource
        """
        mname = reqel.localName[0].lower() + reqel.localName[1:]
        print mname
        if hasattr(self,mname):
            #Call it special
            return getattr(self,mname)(res,reqel,respbody)

        #If not, just call them all assuming that ever attribute mapps to a string
        args = {}
        for (ns,name),value in reqel.attributes.items():
            if ns == EMPTY_NAMESPACE:
                if name == 'src-path': continue
                while '-' in name:
                    index = name.find('-')
                    name = name[:index] + name[index+1].upper() + name[index+2:]
                args[str(name)] = value.value


        #See if there is a src document
        if len(reqel.childNodes) and reqel.childNodes[0].nodeType == Node.TEXT_NODE:
            #Assume it is the src argument
            args['src'] = unicode(base64.decodestring(reqel.childNodes[0].data))
        
        meth = getattr(self.resource, mname)
        msg = "%s(" % mname
        for name, value in args.items():
            msg += "%s='%s'," % (name, value)
        msg = msg[:-1] + ')'
        logFile.info("Execute SOAP command: %s on %s" % (msg, self.resource.getAbsolutePath()))
        res = meth(**args)
        if hasattr(res,'resourceType'):
            self._makeResourceResponse(res,respbody)
        return 1


    def _makeResourceResponse(self, res,respbody):
        doc = respbody.ownerDocument
        cont = res.getContent()
        imt = res.getImt()
        result = doc.createElementNS(FTSS_SOAP_NS,'ftsoap:%s' % self.resourceNameMapping[res.resourceType])
        result.setAttributeNS('', 'path', res.getAbsolutePath())
        result.setAttributeNS('', 'imt', res.getImt())
        result.appendChild(doc.createTextNode(base64.encodestring(cont)))
        respbody.appendChild(result)

    resourceNameMapping = {ResourceTypes.ResourceType.RAW_FILE:'RawFile',
                           ResourceTypes.ResourceType.XML_DOCUMENT:'XmlDocument',
                           ResourceTypes.ResourceType.CONTAINER:'Container',
                           }

g_resourceTypeMapping = {ResourceTypes.ResourceType.RAW_FILE:RawFileSoapImp,
                         ResourceTypes.ResourceType.XML_DOCUMENT:RawFileSoapImp,
                         ResourceTypes.ResourceType.CONTAINER:RawFileSoapImp,
                         }




def FetchResource(repo, reqel, respbody):
    doc = respbody.ownerDocument
    context = XPath.Context.Context(reqel)
    #context = XPath.Context.Context(reqel, processorNss={'ft': XMLSERVER_NS})
    uri = XPath.Evaluate('string(@uri)', context=context)
    try:
        res = repo.fetchResource(uri)
    except:
        GenerateFault(respbody)
        return 0
    _MakeResourceResponse(res,respbody)
    return 0


def CreateResource(repo, reqel, respbody):
    doc = respbody.ownerDocument
    context = XPath.Context.Context(reqel, processorNss={'ft': FTSS_SOAP_NS})
    cont = XPath.Evaluate('ft:Container', context=context)
    ddName = XPath.Evaluate('string(@doc-def)', context=context)

    if cont:
        context = XPath.Context.Context(cont[0], processorNss={'ft': FTSS_SOAP_NS})
        uri = XPath.Evaluate('string(@uri)', context=context)
        cp = XPath.Evaluate('number(@create-parents)', context=context)
        try:
            res = repo.createContainer(uri, createParents=cp)
        except:
            GenerateFault(respbody)
            return 0
    elif ddName:
        uri = XPath.Evaluate('string(@uri)', context=context)
        source = base64.decodestring(XPath.Evaluate('string(node())',
                                                    context=context))
        res = repo.createDocument(ddName, uri, source)
    else:
        try:
            raise "Unknown Resource Type"
        except:
            GenerateFault(respbody)
            return 0
    _MakeResourceResponse(res,respbody)
    return 1


def DeleteResource(repo, reqel, respbody):
    doc = respbody.ownerDocument
    context = XPath.Context.Context(reqel)
    #context = XPath.Context.Context(reqel, processorNss={'ft': XMLSERVER_NS})
    uri = XPath.Evaluate('string(@uri)', context=context)
    try:
        repo.deleteResource(uri)
    except:
        GenerateFault(respbody)
        return 0

    return 1



