########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Server/Client/Commands/Create/UriReference.py,v 1.6 2004/09/23 20:53:28 mbrown Exp $
"""
Implementation of '4ss create urireference' command
(functions defined here are used by the Ft.Lib.CommandLine framework)

Copyright 2004 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

__doc__ = """This command creates a new URI Reference resource in a \
4Suite repository. A URI Reference resource is like a Raw File but \
its content is always obtained on demand by dereferencing the stored \
URI. The first dereference operation occurs when this command is run, \
so the resource identified by the URI must be retrievable at that \
time. Resource metadata (size, etc.) from the last dereference \
operation is stored in the repository. Media type information is not \
obtained automatically; if a media type is not given on the command \
line, it will default to text/plain. If necessary, you will be \
prompted first for credentials and access info to connect to the \
repository."""

import sys, os

from Ft.Lib import Uri
from Ft.Server.Client.Commands import CommandUtil


def Run(options,args):

    src = args['sourceUri']
    base = Uri.OsPathToUri(os.getcwd(), attemptAbsolute=True)
    if base[-1] != '/':
        base += '/'

    newsrc = Uri.Absolutize(src, base)
    newsrc = Uri.PercentEncode(newsrc, encodeReserved=False)
    if src != newsrc:
        sys.stderr.write("URI '%s' normalized to %s\n" % (src, newsrc))
        while True:
            sys.stderr.write('Continue? (y/n): ')
            sys.stderr.flush()
            answer = sys.stdin.readline()
            if not answer:
                return
            elif answer[:1] in 'nN':
                return
            elif answer[:1] in 'yY':
                break

    if Uri.GetScheme(newsrc) == 'file':
        localpath = Uri.UriToOsPath(newsrc)
        if not os.path.exists(localpath):
            sys.stderr.write("A 'file' URI must point to a file that exists.\n")
            sys.stderr.write("URI %s -> nonexistent local path %s\n" % (newsrc, localpath))
            sys.stderr.flush()
            return

    path = args['name']
    imt = options.get('imt','text/plain')

    repo = CommandUtil.GetRepository(options,'4ss.create.urireference')
    if repo is not None:
        commit = False
        try:
            doc = repo.createUriReferenceFile(path, imt, newsrc)
            commit = True
        finally:
            if commit:
                repo.txCommit()
            else:
                repo.txRollback()
    return


def Register():

    from Ft.Lib.CommandLine import Options, Command, Arguments

    cmd = Command.Command('urireference',
                          'Create a new URI Reference in a repository',
                          '--imt=text/html /testref.html http://4suite.org/',
                          __doc__,
                          function = Run,
                          arguments = [Arguments.RequiredArgument('name',
                                                                  "The name (repo path) of the new UriReference resource",
                                                                  str),
                                       Arguments.RequiredArgument('sourceUri',
                                                                  "The URI to reference",
                                                                  str),
                                       ],
                          options = Options.Options([Options.Option(None,
                                                                    'imt=',
                                                                    'The referenced resource\'s Internet Media Type. Default: text/plain.'),
                                                     ]),

                          fileName = __file__
                          )
    return cmd
