########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Server/Client/Commands/Create/Document.py,v 1.20 2004/09/23 20:53:28 mbrown Exp $
"""
Implementation of '4ss create document' command
(functions defined here are used by the Ft.Lib.CommandLine framework)

Copyright 2004 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

__doc__ = """This command creates a new XML Document resource in a \
4Suite repository. An XML Document resource is used to store XML-based \
data in the repository. If you are storing non-XML data, see the \
'4ss create rawfile' command. If necessary, you will be prompted first \
for credentials and access info to connect to the repository."""

import sys

from Ft.Lib import Uri
from Ft.Lib.CommandLine.CommandLineUtil import SourceArgToInputSource
from Ft.Server.Client import FtServerClientException
from Ft.Server.Client.Commands import CommandUtil
from Ft.Server.Common import ResourceTypes
from Ft.Xml.InputSource import DefaultFactory

forcedTypeMapping = {'xml': ResourceTypes.ResourceType.XML_DOCUMENT,
                     'xslt': ResourceTypes.ResourceType.XSLT_DOCUMENT,
                     'schematron': ResourceTypes.ResourceType.SCHEMATRON_DOCUMENT,
                     'rdf': ResourceTypes.ResourceType.RDF_DOCUMENT,
                     'container': ResourceTypes.ResourceType.CONTAINER,
                     'user': ResourceTypes.ResourceType.USER,
                     'xpathdocdef': ResourceTypes.ResourceType.XPATH_DOCUMENT_DEFINITION,
                     'xsltdocdef': ResourceTypes.ResourceType.XSLT_DOCUMENT_DEFINITION,
                     'group': ResourceTypes.ResourceType.GROUP,
                     'alias': ResourceTypes.ResourceType.ALIAS,
                     }


def Run(options, args):
    sourceUri = args['source-uri']
    try:
        source_isrc = SourceArgToInputSource(sourceUri, DefaultFactory)
    except ValueError, e:
        sys.stderr.write(str(e)+'\n')
        sys.stderr.flush()
        return

    content = source_isrc.read()
    source_isrc.close()

    path = args['path']
    aliases = args.get('alias',[])
    imt = options.get('imt', 'text/xml')
    docDef = options.get('docdef', None)
    forcedType = forcedTypeMapping.get(options.get('type'))

    repo = CommandUtil.GetRepository(options,'4ss.create.document')
    if repo is not None:
        commit = False
        try:
            doc = repo.createDocument(path, content, imt=imt, docDef=docDef,
                                      forcedType=forcedType)
            for alias in aliases:
                doc.addAlias(alias)

            commit = True
        finally:
            try:
                if commit:
                    repo.txCommit()
                else:
                    repo.txRollback()
            except:
                pass

    return


def Register():
    from Ft.Lib.CommandLine import Options, Command, Arguments

    cmd = Command.Command('document',
                          'Create a new XML Document in a repository',
                          '/repo/path/to/dummy.xml dummy.xml',
                          __doc__,
                          function = Run,
                          arguments = [Arguments.RequiredArgument('path',
                                                                  "The repo path of the document to create",
                                                                  str),
                                       Arguments.RequiredArgument('source-uri',
                                                                  "The URI to read the document's content from, or '-' for stdin",
                                                                  str),
                                       Arguments.ZeroOrMoreArgument('alias',
                                                                    "Additional aliases to assign to the new document",
                                                                    str),
                                       ],
                          options = Options.Options([Options.Option('i',
                                                                    'imt=IMT',
                                                                    'The document\'s Internet Media Type. Default: text/xml'),
                                                     Options.Option('D',
                                                                    'docdef=docdef',
                                                                    'The path of a Document Definition (already in the repo) to use, '
                                                                    ' if RDF statements are to be derived from the document'),
                                                     Options.TypedOption('t',
                                                                         'type=',
                                                                         'The 4SS document type. If none specified, then the' \
                                                                         ' type is inferred from the document\'s content',
                                                                         [('xml',"generic XML Document (this is the default)"),
                                                                          ('xslt', "XSLT Document"),
                                                                          ('schematron', "Schematron Document"),
                                                                          ('rdf',"RDF Document"),
                                                                          ('container',"Container"),
                                                                          ('user',"User"),
                                                                          ('xpathdocdef',"XPath Document Definition"),
                                                                          ('xsltdocdef',"XSLT Document Definition"),
                                                                          ('group',"Group"),
                                                                          ('alias',"Alias")
                                                                          ]),
                                                     ]),

                          fileName = __file__,
                          )
    return cmd

