########################################################################
#
# File Name:            Memory.py
#
# Documentation:        http://docs.4suite.org/4Rdf/Drivers/Memory.py.html
#
"""
Non-persistent RDF model back-end
WWW: http://4suite.org/4RDF         e-mail: support@4suite.org

Copyright (c) 1999,2000 Fourthought Inc, USA.   All Rights Reserved.
See  http://4suite.org/COPYRIGHT  for license and copyright information
"""


import re

#from Ft.Rdf.Drivers import DataBaseExceptions
from Ft.Rdf.Statement import Statement
from Ft.Rdf import Model, RdfException
#from Ft.Lib import Set
from Ft.Rdf import OBJECT_TYPE_RESOURCE, OBJECT_TYPE_UNKNOWN
from Ft.Rdf.Drivers import PROPERTIES


def CreateDb(dbName, modelName='default'):
    return DbAdapter(dbName, modelName)

#Identical function for Memory driver
GetDb = CreateDb

def DestroyDb(dbName, modelName='default'):
    pass

def ExistsDb(dbName, modelName='default'):
    return 0

def ForceUnicode(*args):
    #Especially needed because of what seems to be a weid bug in Python 2.2
    #than can cause failed compares between identical strings & unicode objs
    #in certain obscure cases we run into in this driver (nested functions, etc.)
    return tuple([ a and unicode(a) or a for a in args ])

class DbAdapter:
    def __init__(self, name, modelName='default'):
        self._acl = {}
        self._statements = {modelName: []}
        self._bound = {}
        self._modelName = modelName
        self.props = {PROPERTIES.OBJECT_TYPE_SUPPORTED: 1}
        return

    ### Transactional Interface ###

    def begin(self):
        return

    def commit(self):
        return

    def rollback(self):
        return

    ### Operations ###

    def add(self, statements):
        # stored statements -> statement tuple
        self._statements[self._modelName].extend(statements)
        return

    def remove(self, statements):
        for s in statements:
            self.removePattern(s[0], s[1], s[2], s[3], s[4], {})
        return

    def removePattern(self, subject, predicate, object, statementUri,
                      scope, flags):
        (subject, predicate, object, statementUri, scope) = ForceUnicode(
            subject, predicate, object, statementUri, scope
            )
        # we use not not because '' and None are equivalent to us
        command = g_removes[(not not subject,
                             not not predicate,
                             not not object,
                             not not statementUri,
                             not not scope,
                             )]
        operators = (g_operators[flags.get('subjectFlags')](subject or ''),
                     g_operators[flags.get('predicateFlags')](predicate or ''),
                     g_operators[flags.get('objectFlags')](object or ''),
                     g_operators[flags.get('statementUriFlags')](statementUri or ''),
                     g_operators[flags.get('scopeFlags')](scope or ''),
                     )
        self._statements[self._modelName] = filter(lambda s, f=command, o=operators:
                                             not f(o, s),
                                             self._statements[self._modelName])
        return

    ### Queries

    def properties(self, scope):
        stmts = self.complete(None, None, None, None, scope, {})
        pdict = {}
        for s in stmts: pdict[s[1]] = None
        return pdict.keys()

    def resources(self, scope):
        stmts = self.complete(None, None, None, None, scope, {})
        rdict = {}
        for s in stmts:
            rdict[s[0]] = None
            rdict[s[1]] = None
        return rdict.keys()

    def complete(self, subject, predicate, object, statementUri, scope,
                 flags):
        (subject, predicate, object, statementUri, scope) = ForceUnicode(
            subject, predicate, object, statementUri, scope
            )
        command = g_completes[(not not subject,
                               not not predicate,
                               not not object,
                               not not statementUri,
                               not not scope,
                               )]
        operators = (g_operators[flags.get('subjectFlags')](subject or ''),
                     g_operators[flags.get('predicateFlags')](predicate or ''),
                     g_operators[flags.get('objectFlags')](object or ''),
                     g_operators[flags.get('statementUriFlags')](statementUri or ''),
                     g_operators[flags.get('scopeFlags')](scope or ''),
                     )

        return filter(lambda s, f=command, o=operators:
                      f(o, s),
                      self._statements[self._modelName])

    def size(self, scope):
        if scope:
            return reduce(lambda r, s, u=scope:
                          r + (s[4] == u),
                          self._statements[self._modelName], 0)
        else:
            return len(self._statements[self._modelName])

    def contains(self, subject, predicate, object, statementUri, scope,
                 flags):
        (subject, predicate, object, statementUri, scope) = ForceUnicode(
            subject, predicate, object, statementUri, scope
            )
        command = g_contains[(not not subject,
                              not not predicate,
                              not not object,
                              not not statementUri,
                              not not scope,
                              )]
        operators = (g_operators[flags.get('subjectFlags')](subject or ''),
                     g_operators[flags.get('predicateFlags')](predicate or ''),
                     g_operators[flags.get('objectFlags')](object or ''),
                     g_operators[flags.get('statementUriFlags')](statementUri or ''),
                     g_operators[flags.get('scopeFlags')](scope or ''),
                     )

        size = reduce(lambda r, s, f=command, o=operators:
                      r + (f(o, s) and 1 or 0),
                      self._statements[self._modelName], 0)
        return size > 0

    def bind(self, object, name, scope):
        if not self._bound.has_key(scope):
            self._bound[scope] = {}

        self._bound[scope][name] = object

    def unbind(self, name, scope):
        if not self._bound.has_key(scope):
            return

        info = self._bound[scope].get(name)
        if info:
            del self._bound[scope][name]

    def lookup(self, name, scope):
        if not self._bound.has_key(scope):
            return None

        return self._bound[scope].get(name)

    def keys(self, scope):
        if not scope:
            result = []
            for bindings in self._bound.values():
                result.extend(bindings.keys())
            return result
        else:
            if self._bound.has_key(scope):
                result = self._bound[scope].keys()
            else:
                result = []
        return result

    def has_key(self, name, scope):
        if not scope:
            result = reduce(lambda a, b, n=name:
                            a + b.has_key(n),
                            self._bound.values(), 0)
        else:
            if self._bound.has_key(scope):
                result = self._bound[scope].has_key(name)
            else:
                result = 0

        return result


    ## Utilities for performance, primarily in Versa ##
    def subjectsFromPredAndObjs(self, predicate, objects, scope):
        """
        Get a list of subjects with the given predicate and objects
        """
        resDict = {}
        if predicate is not None:
            for object in objects:
                for s in self.complete(None, predicate, object, None,
                                       scope, {}):
                    resDict[s[0]] = 1
        else:
            #FIXME: for purposes of Versa, we should not be using null as a wildcard, it seems
            for s in self._statements[self._modelName]:
                resDict[s[0]] = 1
        return resDict.keys()

    def subjectsFromPredsAndObj(self, predicates, object, scope):
        """
        Get a list of subjects with the given predicates and object
        """
        resDict = {}
        if object is not None:
            for p in predicates:
                for s in self.complete(None, p, object, None, scope, {}):
                    resDict[s[0]] = 1
        else:
            #FIXME: for purposes of Versa, we should not be using null as a wildcard, it seems
            for s in self._statements[self._modelName]:
                resDict[s[0]] = 1
        return resDict.keys()

    def objectsFromSubAndPreds(self, subject, predicates, scope):
        """
        Get a list of objects with the given predicates and subject
        """
        resDict = {}
        if subject is not None:
            for predicate in predicates:
                for s in self.complete(subject, predicate, None, None,
                                       scope, {}):
                    resDict[(s[2], s[5])] = 1
        else:
            #FIXME: for purposes of Versa, we should not be using null as a wildcard, it seems
            for s in self._statements[self._modelName]:
                resDict[(s[2], s[5])] = 1
        return resDict.keys()

    def isResource(self, res):
        #return [ s for s in self._statements[self._modelName] if res == s[0] or res == s[1] ]
        r = 0
        for s in self._statements[self._modelName]:
            if res == s[0]:
                r = 1
                break
        return not not r


def _regexCompile(cmd):
    try:
        return re.compile(cmd).match
    except re.error, e:
        raise RdfException(RdfException.INVALID_REGEX_STATEMENT, cmd, str(e))

def _regexICCompile(cmd):
    try:
        return re.compile(cmd,re.IGNORECASE).match
    except re.error, e:
        raise RdfException(RdfException.INVALID_REGEX_STATEMENT, cmd, str(e))


g_operators = {
    None : lambda s: lambda a, b=s: a == b,
    Model.NORMAL : lambda s: lambda a, b=s: a == b,
    Model.IGNORE_CASE : lambda s: lambda a, b=s.lower(): a.lower() == b,
    Model.REGEX : _regexCompile,
    Model.IGNORE_CASE + Model.REGEX : _regexICCompile,
    }

g_completes = {}
g_removes = {}
g_contains = {}

for bits in range(32):
    key = (bits & 16 > 0,
           bits & 8 > 0,
           bits & 4 > 0,
           bits & 2 > 0,
           bits & 1)

    # where f = comparision function for each item of a statement tuple
    #       s = the statement tuple

    parts = []
    if bits & 16:
        parts.append('f[0](s[0])')
    if bits & 8:
        parts.append('f[1](s[1])')
    if bits & 4:
        parts.append('f[2](s[2])')
    if bits & 2:
        parts.append('f[3](s[3])')
    if bits & 1:
        parts.append('f[4](s[4])')

    if parts:
        body = ' and '.join(parts)
    else:
        body = '1'

    g_completes[key] = eval('lambda f, s: %s' % body)
    g_removes[key] = eval('lambda f, s: %s' % body)

    if parts:
        body = ' and '.join(parts)
    else:
        body = '1'
    g_contains[key] = eval('lambda f, s: %s' % body)
