########################################################################
# $Header: /var/local/cvsroot/4Suite/Ft/Lib/ProgressIndicator.py,v 1.6 2004/08/05 22:13:46 mbrown Exp $
"""
Progress indicator

Copyright 2004 Fourthought, Inc. (USA).
Detailed license and copyright information: http://4suite.org/COPYRIGHT
Project home, documentation, distributions: http://4suite.org/
"""

import sys, time, os

from Ft.Lib import Terminfo

class ProgressIndicator:
    """
    A progress indicator intended for terminal output (relies on ^H).

    Indicator style, given as constructor argument, can be
    0: percentage; 1: bar; or 2: both. Default is 0.

    If using styles 1 or 2, an optional width argument
    for the bar portion can also be given (default 60).

    Example usage:
    # First emit whatever prefaces the indicator, if desired
    print " status:",
    sys.__stdout__.flush()
    # Create a new indicator
    p = ProgressIndicator(2)
    p.newIndicator()
    # With each iteration through a task, or as often as you want,
    # call updateProgress(), passing 2 numbers: amount completed,
    # and total amount to do.
    limit = 300000
    for i in range(limit):
        p.updateProgress(i, limit)
    print
    """

    if os.name == 'nt' or os.name == 'dos':
        # dark/light shades... thanks Oleg Broytmann.
        # OK for all MS-DOS codepages except 864 (Arabic) & 874 (Thai)
        # and python doesn't seem to mind writing the non-ASCII chars.
        _hashchar = '\xb2'
        _blankchar = '\xb0'
    else:
        _hashchar = '*'
        _blankchar = ' '
    _current = 0
    _total = 0
    def __init__(self, prefix, stream=sys.stdout):
        if type(prefix) == type(u''):
            # avoid possible UnicodeError during output
            self.prefix = prefix.encode('ascii','replace')
        else:
            self.prefix = prefix
        self.stream = stream
        self.columns = Terminfo.GetColumns(stream)
        return

    def newIndicator(self, total):
        """
        Start a new indicator at 00%.
        Optional style and width arguments are same as constructor.
        """

        #Prefix |bar|  %100
        self._barwidth = self.columns - 3 - len(self.prefix) - 4

        self._current = 0
        self._total = total
        self._showProgress()
        return

    def _erase(self):
        self.stream.write("\r")
        #print "\x08" * (self._barwidth + 6) + " " * (self._barwidth + 5) + "\x08" * (self._barwidth+6),


    def message(self,message):
        # left-justify message within column width, padding or trimming as
        # required.
        message = '%-*.*s' % (self.columns, self.columns, message)

        self._erase()
        #self._erasePrefix()
        self.stream.write(message + "\n");
        self._showProgress()

    def updateProgress(self, cur):
        """
        Update an existing indicator to reflect given progress.
        Arguments are amount completed so far, and total to do.
        For example, if 4 out of 30 have been completed, call
        updateProgress(4,30).
        """
        self._current = cur
        self._erase()
        #self._erasePrefix()
        self._showProgress()
        return

    def _showProgress(self):

        hashwidth = int(float(self._current+1)/self._total * (self._barwidth - 2))
        pct = int((self._current+1)*100/self._total)
        self.stream.write("%s |%s%s %s%%" % (
            self.prefix,
            self._hashchar * hashwidth,
            self._blankchar * (self._barwidth - hashwidth - 2) + "|",
            " " * (pct < 100) + "%02d" % pct))
        self.stream.flush()
        return

class AutoProgressIndicator(ProgressIndicator):
    def __init__(self, prefix, total, step=1, stream=sys.stdout):
        ProgressIndicator.__init__(self, prefix, stream)
        self.newIndicator(total)
        self._step = 1
        return

    def advance(self):
        self.updateProgress(self._current + self._step)
        return
