"""
Tests for generalized linear models. Majority of code either directly borrowed
or closely adapted from statsmodels package. Model results verfiied using glm
function in R and GLM function in statsmodels.
"""

__author__ = 'Taylor Oshan tayoshan@gmail.com'

import numpy as np
import pysal.lib
import unittest
import math
from ..glm import GLM
from ..family import Gaussian, Poisson, Binomial, QuasiPoisson


class TestGaussian(unittest.TestCase):
    """
    Tests for Poisson GLM
    """

    def setUp(self):
        db = pysal.lib.io.open(pysal.lib.examples.get_path('columbus.dbf'),'r')
        y = np.array(db.by_col("HOVAL"))
        self.y = np.reshape(y, (49,1))
        X = []
        X.append(db.by_col("INC"))
        X.append(db.by_col("CRIME"))
        self.X = np.array(X).T

    def testIWLS(self):
        model = GLM(self.y, self.X, family=Gaussian())
        results = model.fit()
        self.assertAlmostEqual(results.n, 49)
        self.assertAlmostEqual(results.df_model, 2)
        self.assertAlmostEqual(results.df_resid, 46)
        self.assertAlmostEqual(results.aic, 408.73548964604873)
        self.assertAlmostEqual(results.bic, 10467.991340493107)
        self.assertAlmostEqual(results.deviance, 10647.015074206196)
        self.assertAlmostEqual(results.llf, -201.36774482302437)
        self.assertAlmostEqual(results.null_deviance, 16367.794631703124)
        self.assertAlmostEqual(results.scale, 231.45684943926514)
        np.testing.assert_allclose(results.params, [ 46.42818268,   0.62898397,
            -0.48488854])
        np.testing.assert_allclose(results.bse, [ 13.19175703,   0.53591045,
            0.18267291])
        np.testing.assert_allclose(results.cov_params(),
                [[  1.74022453e+02,  -6.52060364e+00,  -2.15109867e+00],
                [ -6.52060364e+00,   2.87200008e-01,   6.80956787e-02],
                [ -2.15109867e+00,   6.80956787e-02,   3.33693910e-02]])
        np.testing.assert_allclose(results.tvalues, [ 3.51948437,  1.17367365,
            -2.65440864])
        np.testing.assert_allclose(results.pvalues, [ 0.00043239,  0.24052577,
            0.00794475], atol=1.0e-8)
        np.testing.assert_allclose(results.conf_int(),
                [[ 20.57281401,  72.28355135],
                [ -0.42138121,   1.67934915],
                [ -0.84292086,  -0.12685622]])
        np.testing.assert_allclose(results.normalized_cov_params,
                [[  7.51857004e-01,  -2.81720055e-02,  -9.29373521e-03],
                [ -2.81720055e-02,   1.24083607e-03,   2.94204638e-04],
                [ -9.29373521e-03,   2.94204638e-04,   1.44171110e-04]])
        np.testing.assert_allclose(results.mu,
                [ 51.08752105,  50.66601521,  41.61367567,  33.53969014,
                28.90638232,  43.87074227,  51.64910882,  34.92671563,
                42.69267622,  38.49449134,  20.92815471,  25.25228436,
                29.78223486,  25.02403635,  29.07959539,  24.63352275,
                34.71372149,  33.40443052,  27.29864225,  65.86219802,
                33.69854751,  37.44976435,  50.01304928,  36.81219959,
                22.02674837,  31.64775955,  27.63563294,  23.7697291 ,
                22.43119725,  21.76987089,  48.51169321,  49.05891819,
                32.31656426,  44.20550354,  35.49244888,  51.27811308,
                36.55047181,  27.37048914,  48.78812922,  57.31744163,
                51.22914162,  54.70515578,  37.06622277,  44.5075759 ,
                41.24328983,  49.93821824,  44.85644299,  40.93838609,  47.32045464])
        self.assertAlmostEqual(results.pearson_chi2, 10647.015074206196)
        np.testing.assert_allclose(results.resid_response,
                [ 29.37948195,  -6.09901421, -15.26367567,  -0.33968914,
                -5.68138232, -15.12074227,  23.35089118,   2.19828437,
                9.90732178,  57.90551066,  -1.22815371,  -5.35228436,
                11.91776614,  17.87596565, -11.07959539,  -5.83352375,
                7.03627851,  26.59556948,   3.30135775,  15.40479998,
                -13.72354751,  -6.99976335,  -2.28004728,  16.38780141,
                -4.12674837, -11.34776055,   6.46436506,  -0.9197291 ,
                10.06880275,   0.73012911, -16.71169421,  -8.75891919,
                -8.71656426, -15.75550254,  -8.49244888, -14.97811408,
                6.74952719,  -4.67048814,  -9.18813122,   4.63255937,
                -9.12914362, -10.37215578, -11.36622177, -11.0075759 ,
                -13.51028983,  26.16177976,  -2.35644299, -14.13838709, -11.52045564])
        np.testing.assert_allclose(results.resid_working,
                [ 29.37948195,  -6.09901421, -15.26367567,  -0.33968914,
                -5.68138232, -15.12074227,  23.35089118,   2.19828437,
                9.90732178,  57.90551066,  -1.22815371,  -5.35228436,
                11.91776614,  17.87596565, -11.07959539,  -5.83352375,
                7.03627851,  26.59556948,   3.30135775,  15.40479998,
                -13.72354751,  -6.99976335,  -2.28004728,  16.38780141,
                -4.12674837, -11.34776055,   6.46436506,  -0.9197291 ,
                10.06880275,   0.73012911, -16.71169421,  -8.75891919,
                -8.71656426, -15.75550254,  -8.49244888, -14.97811408,
                6.74952719,  -4.67048814,  -9.18813122,   4.63255937,
                -9.12914362, -10.37215578, -11.36622177, -11.0075759 ,
                -13.51028983,  26.16177976,  -2.35644299, -14.13838709, -11.52045564])
        np.testing.assert_allclose(results.resid_pearson,
                [ 29.37948195,  -6.09901421, -15.26367567,  -0.33968914,
                -5.68138232, -15.12074227,  23.35089118,   2.19828437,
                9.90732178,  57.90551066,  -1.22815371,  -5.35228436,
                11.91776614,  17.87596565, -11.07959539,  -5.83352375,
                7.03627851,  26.59556948,   3.30135775,  15.40479998,
                -13.72354751,  -6.99976335,  -2.28004728,  16.38780141,
                -4.12674837, -11.34776055,   6.46436506,  -0.9197291 ,
                10.06880275,   0.73012911, -16.71169421,  -8.75891919,
                -8.71656426, -15.75550254,  -8.49244888, -14.97811408,
                6.74952719,  -4.67048814,  -9.18813122,   4.63255937,
                -9.12914362, -10.37215578, -11.36622177, -11.0075759 ,
                -13.51028983,  26.16177976,  -2.35644299, -14.13838709, -11.52045564])
        np.testing.assert_allclose(results.resid_anscombe,
                [ 29.37948195,  -6.09901421, -15.26367567,  -0.33968914,
                -5.68138232, -15.12074227,  23.35089118,   2.19828437,
                9.90732178,  57.90551066,  -1.22815371,  -5.35228436,
                11.91776614,  17.87596565, -11.07959539,  -5.83352375,
                7.03627851,  26.59556948,   3.30135775,  15.40479998,
                -13.72354751,  -6.99976335,  -2.28004728,  16.38780141,
                -4.12674837, -11.34776055,   6.46436506,  -0.9197291 ,
                10.06880275,   0.73012911, -16.71169421,  -8.75891919,
                -8.71656426, -15.75550254,  -8.49244888, -14.97811408,
                6.74952719,  -4.67048814,  -9.18813122,   4.63255937,
                -9.12914362, -10.37215578, -11.36622177, -11.0075759 ,
                -13.51028983,  26.16177976,  -2.35644299, -14.13838709, -11.52045564])
        np.testing.assert_allclose(results.resid_deviance,
                [ 29.37948195,  -6.09901421, -15.26367567,  -0.33968914,
                -5.68138232, -15.12074227,  23.35089118,   2.19828437,
                9.90732178,  57.90551066,  -1.22815371,  -5.35228436,
                11.91776614,  17.87596565, -11.07959539,  -5.83352375,
                7.03627851,  26.59556948,   3.30135775,  15.40479998,
                -13.72354751,  -6.99976335,  -2.28004728,  16.38780141,
                -4.12674837, -11.34776055,   6.46436506,  -0.9197291 ,
                10.06880275,   0.73012911, -16.71169421,  -8.75891919,
                -8.71656426, -15.75550254,  -8.49244888, -14.97811408,
                6.74952719,  -4.67048814,  -9.18813122,   4.63255937,
                -9.12914362, -10.37215578, -11.36622177, -11.0075759 ,
                -13.51028983,  26.16177976,  -2.35644299, -14.13838709, -11.52045564])
        np.testing.assert_allclose(results.null,
                [ 38.43622447,  38.43622447,  38.43622447,  38.43622447,
                38.43622447,  38.43622447,  38.43622447,  38.43622447,
                38.43622447,  38.43622447,  38.43622447,  38.43622447,
                38.43622447,  38.43622447,  38.43622447,  38.43622447,
                38.43622447,  38.43622447,  38.43622447,  38.43622447,
                38.43622447,  38.43622447,  38.43622447,  38.43622447,
                38.43622447,  38.43622447,  38.43622447,  38.43622447,
                38.43622447,  38.43622447,  38.43622447,  38.43622447,
                38.43622447,  38.43622447,  38.43622447,  38.43622447,
                38.43622447,  38.43622447,  38.43622447,  38.43622447,
                38.43622447,  38.43622447,  38.43622447,  38.43622447,
                38.43622447,  38.43622447,  38.43622447,  38.43622447,  38.43622447])
        self.assertAlmostEqual(results.D2, .349514377851)
        self.assertAlmostEqual(results.adj_D2, 0.32123239427957673)
        self.assertAlmostEqual(results.tr_S, 3.0)

class TestPoisson(unittest.TestCase):

    def setUp(self):
        db = pysal.lib.io.open(pysal.lib.examples.get_path('columbus.dbf'),'r')
        y = np.array(db.by_col("HOVAL"))
        y = np.reshape(y, (49,1))
        self.y = np.round(y).astype(int)
        X = []
        X.append(db.by_col("INC"))
        X.append(db.by_col("CRIME"))
        self.X = np.array(X).T

    def testIWLS(self):
        model = GLM(self.y, self.X, family=Poisson())
        results = model.fit()
        self.assertAlmostEqual(results.n, 49)
        self.assertAlmostEqual(results.df_model, 2)
        self.assertAlmostEqual(results.df_resid, 46)
        self.assertAlmostEqual(results.aic, 500.85184179938756)
        self.assertAlmostEqual(results.bic, 51.436404535087661)
        self.assertAlmostEqual(results.deviance, 230.46013824817649)
        self.assertAlmostEqual(results.llf, -247.42592089969378)
        self.assertAlmostEqual(results.null_deviance, 376.97293610347361)
        self.assertAlmostEqual(results.scale, 1.0)
        np.testing.assert_allclose(results.params, [ 3.92159085,  0.01183491,
            -0.01371397], atol=1.0e-8)
        np.testing.assert_allclose(results.bse, [ 0.13049161,  0.00511599,
            0.00193769], atol=1.0e-8)
        np.testing.assert_allclose(results.cov_params(),
                [[  1.70280610e-02,  -6.18628383e-04,  -2.21386966e-04],
                [ -6.18628383e-04,   2.61733917e-05,   6.77496445e-06],
                [ -2.21386966e-04,   6.77496445e-06,   3.75463502e-06]])
        np.testing.assert_allclose(results.tvalues, [ 30.0524361 ,   2.31331634,
            -7.07748998])
        np.testing.assert_allclose(results.pvalues, [  2.02901657e-198,
            2.07052532e-002,   1.46788805e-012])
        np.testing.assert_allclose(results.conf_int(),
                [[  3.66583199e+00,   4.17734972e+00],
                [  1.80774841e-03,   2.18620753e-02],
                [ -1.75117666e-02,  -9.91616901e-03]])
        np.testing.assert_allclose(results.normalized_cov_params,
                [[  1.70280610e-02,  -6.18628383e-04,  -2.21386966e-04],
                [ -6.18628383e-04,   2.61733917e-05,   6.77496445e-06],
                [ -2.21386966e-04,   6.77496445e-06,   3.75463502e-06]])
        np.testing.assert_allclose(results.mu,
                [ 51.26831574,  50.15022766,  40.06142973,  34.13799739,
                28.76119226,  42.6836241 ,  55.64593703,  34.08277997,
                40.90389582,  37.19727958,  23.47459217,  26.12384057,
                29.78303507,  25.96888223,  29.14073823,  26.04369592,
                34.18996367,  32.28924005,  27.42284396,  72.69207879,
                33.05316347,  36.52276972,  49.2551479 ,  35.33439632,
                24.07252457,  31.67153709,  27.81699478,  25.38021219,
                24.31759259,  23.13586161,  48.40724678,  48.57969818,
                31.92596006,  43.3679231 ,  34.32925819,  51.78908089,
                34.49778584,  27.56236198,  48.34273194,  57.50829097,
                50.66038226,  54.68701352,  35.77103116,  43.21886784,
                40.07615759,  49.98658004,  43.13352883,  40.28520774,  46.28910294])
        self.assertAlmostEqual(results.pearson_chi2, 264.62262932090221)
        np.testing.assert_allclose(results.resid_response,
                [ 28.73168426,  -5.15022766, -14.06142973,  -1.13799739,
                -5.76119226, -13.6836241 ,  19.35406297,   2.91722003,
                12.09610418,  58.80272042,  -3.47459217,  -6.12384057,
                12.21696493,  17.03111777, -11.14073823,  -7.04369592,
                7.81003633,  27.71075995,   3.57715604,   8.30792121,
                -13.05316347,  -6.52276972,  -1.2551479 ,  17.66560368,
                -6.07252457, -11.67153709,   6.18300522,  -2.38021219,
                7.68240741,  -1.13586161, -16.40724678,  -8.57969818,
                -7.92596006, -15.3679231 ,  -7.32925819, -15.78908089,
                8.50221416,  -4.56236198,  -8.34273194,   4.49170903,
                -8.66038226, -10.68701352,  -9.77103116,  -9.21886784,
                -12.07615759,  26.01341996,  -1.13352883, -13.28520774, -10.28910294])
        np.testing.assert_allclose(results.resid_working,
                [ 1473.02506034,  -258.28508941,  -563.32097891,   -38.84895192,
                -165.69875817,  -584.06666725,  1076.97496919,    99.42696848,
                494.77778514,  2187.30123163,   -81.56463405,  -159.97823479,
                363.858295  ,   442.27909165,  -324.64933645,  -183.44387481,
                267.02485844,   894.75938   ,    98.09579187,   603.9200634 ,
                -431.44834594,  -238.2296165 ,   -61.82249568,   624.20344168,
                -146.18099686,  -369.65551968,   171.99262399,   -60.41029031,
                186.81765356,   -26.27913713,  -794.22964417,  -416.79914795,
                -253.04388425,  -666.47490701,  -251.6079969 ,  -817.70198717,
                293.30756327,  -125.74947222,  -403.31045369,   258.31051005,
                -438.73827602,  -584.440853  ,  -349.51985996,  -398.42903071,
                -483.96599444,  1300.32189904,   -48.89309853,  -535.19735391,
                -476.27334527])
        np.testing.assert_allclose(results.resid_pearson,
                [ 4.01269878, -0.72726045, -2.221602  , -0.19477008, -1.07425881,
                -2.09445239,  2.59451042,  0.49969118,  1.89131202,  9.64143836,
                -0.71714142, -1.19813392,  2.23861212,  3.34207756, -2.0637814 ,
                -1.3802231 ,  1.33568403,  4.87662684,  0.68309584,  0.97442591,
                -2.27043598, -1.07931992, -0.17884182,  2.97186889, -1.23768025,
                -2.07392709,  1.1723155 , -0.47246327,  1.55789092, -0.23614708,
                -2.35819937, -1.23096188, -1.40274877, -2.33362391, -1.25091503,
                -2.19400568,  1.44755952, -0.8690235 , -1.19989348,  0.59230634,
                -1.21675413, -1.44515442, -1.63370888, -1.40229988, -1.90759306,
                3.67934693, -0.17259375, -2.09312684, -1.51230062])
        np.testing.assert_allclose(results.resid_anscombe,
                [ 3.70889134, -0.74031295, -2.37729865, -0.19586855, -1.11374751,
                -2.22611959,  2.46352013,  0.49282126,  1.80857757,  8.06444452,
                -0.73610811, -1.25061371,  2.10820431,  3.05467547, -2.22437611,
                -1.45136173,  1.28939698,  4.35942058,  0.66904552,  0.95674923,
                -2.45438937, -1.11429881, -0.17961012,  2.76715848, -1.29658591,
                -2.22816691,  1.13269136, -0.48017382,  1.48562248, -0.23812278,
                -2.51664399, -1.2703721 , -1.4683091 , -2.49907536, -1.30026484,
                -2.32398309,  1.39380683, -0.89495368, -1.23735395,  0.58485202,
                -1.25435224, -1.4968484 , -1.71888038, -1.45756652, -2.01906267,
                3.41729922, -0.17335867, -2.22921828, -1.57470549])
        np.testing.assert_allclose(results.resid_deviance,
                [ 3.70529668, -0.74027329, -2.37536322, -0.19586751, -1.11349765,
                -2.22466106,  2.46246446,  0.4928057 ,  1.80799655,  8.02696525,
                -0.73602255, -1.25021555,  2.10699958,  3.05084608, -2.22214376,
                -1.45072221,  1.28913747,  4.35106213,  0.6689982 ,  0.95669662,
                -2.45171913, -1.11410444, -0.17960956,  2.76494217, -1.29609865,
                -2.22612429,  1.13247453, -0.48015254,  1.48508549, -0.23812   ,
                -2.51476072, -1.27015583, -1.46777697, -2.49699318, -1.29992892,
                -2.32263069,  1.39348459, -0.89482132, -1.23715363,  0.58483655,
                -1.25415329, -1.49653039, -1.7181055 , -1.45719072, -2.01791949,
                3.41437156, -0.1733581 , -2.22765605, -1.57426046])
        np.testing.assert_allclose(results.null,
                [ 38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,  38.42857143])
        self.assertAlmostEqual(results.D2, .388656011675)
        self.assertAlmostEqual(results.adj_D2, 0.36207583826952761)#.375648692774)

    def testQuasi(self):
        model = GLM(self.y, self.X, family=QuasiPoisson())
        results = model.fit()
        self.assertAlmostEqual(results.n, 49)
        self.assertAlmostEqual(results.df_model, 2)
        self.assertAlmostEqual(results.df_resid, 46)
        self.assertTrue(math.isnan(results.aic))
        self.assertAlmostEqual(results.bic, 51.436404535087661)
        self.assertAlmostEqual(results.deviance, 230.46013824817649)
        self.assertTrue(math.isnan(results.llf))
        self.assertAlmostEqual(results.null_deviance, 376.97293610347361)
        self.assertAlmostEqual(results.scale, 5.7526658548022223)
        np.testing.assert_allclose(results.params, [ 3.92159085,  0.01183491,
            -0.01371397], atol=1.0e-8)
        np.testing.assert_allclose(results.bse, [ 0.31298042,  0.01227057,
            0.00464749], atol=1.0e-8)
        np.testing.assert_allclose(results.cov_params(),
                [[  9.79567451e-02,  -3.55876238e-03,  -1.27356524e-03],
                [ -3.55876238e-03,   1.50566777e-04,   3.89741067e-05],
                [ -1.27356524e-03,   3.89741067e-05,   2.15991606e-05]])
        np.testing.assert_allclose(results.tvalues, [ 12.52982796,   0.96449604,
            -2.95083339])
        np.testing.assert_allclose(results.pvalues, [  5.12737770e-36,
            3.34797291e-01,   3.16917819e-03])
        np.testing.assert_allclose(results.conf_int(),
                [[ 3.3081605 ,  4.53502121],
                [-0.01221495,  0.03588478],
                [-0.02282288, -0.00460506]], atol=1.0e-8)
        np.testing.assert_allclose(results.normalized_cov_params,
                [[  1.70280610e-02,  -6.18628383e-04,  -2.21386966e-04],
                [ -6.18628383e-04,   2.61733917e-05,   6.77496445e-06],
                [ -2.21386966e-04,   6.77496445e-06,   3.75463502e-06]])
        np.testing.assert_allclose(results.mu,
                [ 51.26831574,  50.15022766,  40.06142973,  34.13799739,
                28.76119226,  42.6836241 ,  55.64593703,  34.08277997,
                40.90389582,  37.19727958,  23.47459217,  26.12384057,
                29.78303507,  25.96888223,  29.14073823,  26.04369592,
                34.18996367,  32.28924005,  27.42284396,  72.69207879,
                33.05316347,  36.52276972,  49.2551479 ,  35.33439632,
                24.07252457,  31.67153709,  27.81699478,  25.38021219,
                24.31759259,  23.13586161,  48.40724678,  48.57969818,
                31.92596006,  43.3679231 ,  34.32925819,  51.78908089,
                34.49778584,  27.56236198,  48.34273194,  57.50829097,
                50.66038226,  54.68701352,  35.77103116,  43.21886784,
                40.07615759,  49.98658004,  43.13352883,  40.28520774,  46.28910294])
        self.assertAlmostEqual(results.pearson_chi2, 264.62262932090221)
        np.testing.assert_allclose(results.resid_response,
                [ 28.73168426,  -5.15022766, -14.06142973,  -1.13799739,
                -5.76119226, -13.6836241 ,  19.35406297,   2.91722003,
                12.09610418,  58.80272042,  -3.47459217,  -6.12384057,
                12.21696493,  17.03111777, -11.14073823,  -7.04369592,
                7.81003633,  27.71075995,   3.57715604,   8.30792121,
                -13.05316347,  -6.52276972,  -1.2551479 ,  17.66560368,
                -6.07252457, -11.67153709,   6.18300522,  -2.38021219,
                7.68240741,  -1.13586161, -16.40724678,  -8.57969818,
                -7.92596006, -15.3679231 ,  -7.32925819, -15.78908089,
                8.50221416,  -4.56236198,  -8.34273194,   4.49170903,
                -8.66038226, -10.68701352,  -9.77103116,  -9.21886784,
                -12.07615759,  26.01341996,  -1.13352883, -13.28520774, -10.28910294])
        np.testing.assert_allclose(results.resid_working,
                [ 1473.02506034,  -258.28508941,  -563.32097891,   -38.84895192,
                -165.69875817,  -584.06666725,  1076.97496919,    99.42696848,
                494.77778514,  2187.30123163,   -81.56463405,  -159.97823479,
                363.858295  ,   442.27909165,  -324.64933645,  -183.44387481,
                267.02485844,   894.75938   ,    98.09579187,   603.9200634 ,
                -431.44834594,  -238.2296165 ,   -61.82249568,   624.20344168,
                -146.18099686,  -369.65551968,   171.99262399,   -60.41029031,
                186.81765356,   -26.27913713,  -794.22964417,  -416.79914795,
                -253.04388425,  -666.47490701,  -251.6079969 ,  -817.70198717,
                293.30756327,  -125.74947222,  -403.31045369,   258.31051005,
                -438.73827602,  -584.440853  ,  -349.51985996,  -398.42903071,
                -483.96599444,  1300.32189904,   -48.89309853,  -535.19735391,
                -476.27334527])
        np.testing.assert_allclose(results.resid_pearson,
                [ 4.01269878, -0.72726045, -2.221602  , -0.19477008, -1.07425881,
                -2.09445239,  2.59451042,  0.49969118,  1.89131202,  9.64143836,
                -0.71714142, -1.19813392,  2.23861212,  3.34207756, -2.0637814 ,
                -1.3802231 ,  1.33568403,  4.87662684,  0.68309584,  0.97442591,
                -2.27043598, -1.07931992, -0.17884182,  2.97186889, -1.23768025,
                -2.07392709,  1.1723155 , -0.47246327,  1.55789092, -0.23614708,
                -2.35819937, -1.23096188, -1.40274877, -2.33362391, -1.25091503,
                -2.19400568,  1.44755952, -0.8690235 , -1.19989348,  0.59230634,
                -1.21675413, -1.44515442, -1.63370888, -1.40229988, -1.90759306,
                3.67934693, -0.17259375, -2.09312684, -1.51230062])
        np.testing.assert_allclose(results.resid_anscombe,
                [ 3.70889134, -0.74031295, -2.37729865, -0.19586855, -1.11374751,
                -2.22611959,  2.46352013,  0.49282126,  1.80857757,  8.06444452,
                -0.73610811, -1.25061371,  2.10820431,  3.05467547, -2.22437611,
                -1.45136173,  1.28939698,  4.35942058,  0.66904552,  0.95674923,
                -2.45438937, -1.11429881, -0.17961012,  2.76715848, -1.29658591,
                -2.22816691,  1.13269136, -0.48017382,  1.48562248, -0.23812278,
                -2.51664399, -1.2703721 , -1.4683091 , -2.49907536, -1.30026484,
                -2.32398309,  1.39380683, -0.89495368, -1.23735395,  0.58485202,
                -1.25435224, -1.4968484 , -1.71888038, -1.45756652, -2.01906267,
                3.41729922, -0.17335867, -2.22921828, -1.57470549])
        np.testing.assert_allclose(results.resid_deviance,
                [ 3.70529668, -0.74027329, -2.37536322, -0.19586751, -1.11349765,
                -2.22466106,  2.46246446,  0.4928057 ,  1.80799655,  8.02696525,
                -0.73602255, -1.25021555,  2.10699958,  3.05084608, -2.22214376,
                -1.45072221,  1.28913747,  4.35106213,  0.6689982 ,  0.95669662,
                -2.45171913, -1.11410444, -0.17960956,  2.76494217, -1.29609865,
                -2.22612429,  1.13247453, -0.48015254,  1.48508549, -0.23812   ,
                -2.51476072, -1.27015583, -1.46777697, -2.49699318, -1.29992892,
                -2.32263069,  1.39348459, -0.89482132, -1.23715363,  0.58483655,
                -1.25415329, -1.49653039, -1.7181055 , -1.45719072, -2.01791949,
                3.41437156, -0.1733581 , -2.22765605, -1.57426046])
        np.testing.assert_allclose(results.null,
                [ 38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,
                38.42857143,  38.42857143,  38.42857143,  38.42857143,  38.42857143])
        self.assertAlmostEqual(results.D2, .388656011675)
        self.assertAlmostEqual(results.adj_D2, 0.36207583826952761)

class TestBinomial(unittest.TestCase):

    def setUp(self):
        #London house price data
        #y: 'BATH2'
        y = np.array([0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 1, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0,
            0, 1, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0,
            0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0,
            0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0])
       	self.y = y.reshape((316,1))
        #X: 'FLOORSZ'
       	X = np.array([ 77,  75,  64,  95, 107, 100,  81, 151,  98, 260, 171, 161,  91,
            80,  50,  85,  52,  69,  60,  84, 155,  97,  69, 126,  90,  43,
            51,  41, 140,  80,  52,  86,  66,  60,  40, 155, 138,  97, 115,
            148, 206,  60,  53,  96,  88, 160,  31,  43, 154,  60, 131,  60,
            46,  61, 125, 150,  76,  92,  96, 100, 105,  72,  48,  41,  72,
            65,  60,  65,  98,  33, 144, 111,  91, 108,  38,  48,  95,  63,
            98, 129, 108,  51, 131,  66,  48, 127,  76,  68,  52,  64,  57,
            121,  67,  76, 112,  96,  90,  53,  93,  64,  97,  58,  44, 157,
            53,  70,  71, 167,  47,  70,  96,  77,  75,  71,  67,  47,  71,
            90,  69,  64,  65,  95,  60,  60,  65,  54, 121, 105,  50,  85,
            69,  69,  62,  65,  93,  93,  70,  62, 155,  68, 117,  80,  80,
            75,  98, 114,  86,  70,  50,  51, 163, 124,  59,  95,  51,  63,
            85,  53,  46, 102, 114,  83,  47,  40,  63, 123, 100,  63, 110,
            79,  98,  99, 120,  52,  48,  37,  81,  30,  88,  50,  35, 116,
            67,  45,  80,  86, 109,  59,  75,  60,  71, 141, 121,  50, 168,
            90,  51, 133,  75, 133, 127,  37,  68, 105,  61, 123, 151, 110,
            77, 220,  94,  77,  70, 100,  98, 126,  55, 105,  60, 176, 104,
            68,  62,  70,  48, 102,  80,  97,  66,  80, 102, 160,  55,  60,
            71, 125,  85,  85, 190, 137,  48,  41,  42,  51,  57,  60, 114,
            88,  84, 108,  66,  85,  42,  98,  90, 127, 100,  55,  76,  82,
            63,  80,  71,  76, 121, 109,  92, 160, 109, 185, 100,  90,  90,
            86,  88,  95, 116, 135,  61,  74,  60, 235,  76,  66, 100,  49,
            50,  37, 100,  88,  90,  52,  95,  81,  79,  96,  75,  91,  86,
            83, 180, 108,  80,  96,  49, 117, 117,  86,  46,  66,  95,  57,
            120, 137,  68, 240])
        self.X = X.reshape((316,1))

    def testIWLS(self):
        model = GLM(self.y, self.X, family=Binomial())
        results = model.fit()
        self.assertAlmostEqual(results.n, 316)
        self.assertAlmostEqual(results.df_model, 1)
        self.assertAlmostEqual(results.df_resid, 314)
        self.assertAlmostEqual(results.aic, 155.19347530342466)
        self.assertAlmostEqual(results.bic, -1656.1095797628657)
        self.assertAlmostEqual(results.deviance, 151.19347530342466)
        self.assertAlmostEqual(results.llf, -75.596737651712331)
        self.assertAlmostEqual(results.null_deviance, 189.16038985881212)
        self.assertAlmostEqual(results.scale, 1.0)
        np.testing.assert_allclose(results.params, [-5.33638276,  0.0287754 ])
        np.testing.assert_allclose(results.bse, [ 0.64499904,  0.00518312],
                atol=1.0e-8)
        np.testing.assert_allclose(results.cov_params(),
            [[  4.16023762e-01,  -3.14338457e-03],
            [ -3.14338457e-03,   2.68646833e-05]])
        np.testing.assert_allclose(results.tvalues, [-8.27347396,  5.55175826])
        np.testing.assert_allclose(results.pvalues, [  1.30111233e-16,
            2.82810512e-08])
        np.testing.assert_allclose(results.conf_int(),
            [[-6.60055765, -4.07220787],
            [ 0.01861668,  0.03893412]], atol=1.0e-8)
        np.testing.assert_allclose(results.normalized_cov_params,
            [[  4.16023762e-01,  -3.14338457e-03],
            [ -3.14338457e-03,   2.68646833e-05]])
        np.testing.assert_allclose(results.mu,
            [ 0.04226237,  0.03999333,  0.02946178,  0.0689636 ,  0.09471181,
            0.07879431,  0.04717464,  0.27065598,  0.07471691,  0.89522144,
            0.39752487,  0.33102718,  0.06192993,  0.04589793,  0.01988679,
            0.0526265 ,  0.02104007,  0.03386636,  0.02634295,  0.05121018,
            0.29396682,  0.07275173,  0.03386636,  0.15307528,  0.06027915,
            0.01631789,  0.02045547,  0.01541937,  0.2128508 ,  0.04589793,
            0.02104007,  0.05407977,  0.0311527 ,  0.02634295,  0.01498855,
            0.29396682,  0.20336776,  0.07275173,  0.11637537,  0.25395607,
            0.64367488,  0.02634295,  0.02164101,  0.07083428,  0.05710047,
            0.32468619,  0.01160845,  0.01631789,  0.28803008,  0.02634295,
            0.17267234,  0.02634295,  0.01776301,  0.02709115,  0.14938186,
            0.26501331,  0.04111287,  0.06362285,  0.07083428,  0.07879431,
            0.08989109,  0.03680743,  0.0187955 ,  0.01541937,  0.03680743,
            0.03029581,  0.02634295,  0.03029581,  0.07471691,  0.01228768,
            0.23277197,  0.10505173,  0.06192993,  0.09720799,  0.01416217,
            0.0187955 ,  0.0689636 ,  0.02865003,  0.07471691,  0.16460503,
            0.09720799,  0.02045547,  0.17267234,  0.0311527 ,  0.0187955 ,
            0.15684317,  0.04111287,  0.03293737,  0.02104007,  0.02946178,
            0.02421701,  0.1353385 ,  0.03203302,  0.04111287,  0.10778798,
            0.07083428,  0.06027915,  0.02164101,  0.06535882,  0.02946178,
            0.07275173,  0.02490638,  0.01678627,  0.30605146,  0.02164101,
            0.03482061,  0.03580075,  0.37030921,  0.0182721 ,  0.03482061,
            0.07083428,  0.04226237,  0.03999333,  0.03580075,  0.03203302,
            0.0182721 ,  0.03580075,  0.06027915,  0.03386636,  0.02946178,
            0.03029581,  0.0689636 ,  0.02634295,  0.02634295,  0.03029581,
            0.02225873,  0.1353385 ,  0.08989109,  0.01988679,  0.0526265 ,
            0.03386636,  0.03386636,  0.02786   ,  0.03029581,  0.06535882,
            0.06535882,  0.03482061,  0.02786   ,  0.29396682,  0.03293737,
            0.12242534,  0.04589793,  0.04589793,  0.03999333,  0.07471691,
            0.11344884,  0.05407977,  0.03482061,  0.01988679,  0.02045547,
            0.34389327,  0.14576223,  0.02561486,  0.0689636 ,  0.02045547,
            0.02865003,  0.0526265 ,  0.02164101,  0.01776301,  0.08307425,
            0.11344884,  0.04982997,  0.0182721 ,  0.01498855,  0.02865003,
            0.14221564,  0.07879431,  0.02865003,  0.10237696,  0.04465416,
            0.07471691,  0.07673078,  0.13200634,  0.02104007,  0.0187955 ,
            0.01376599,  0.04717464,  0.01128289,  0.05710047,  0.01988679,
            0.01300612,  0.11936722,  0.03203302,  0.01726786,  0.04589793,
            0.05407977,  0.09976271,  0.02561486,  0.03999333,  0.02634295,
            0.03580075,  0.21771181,  0.1353385 ,  0.01988679,  0.37704374,
            0.06027915,  0.02045547,  0.18104935,  0.03999333,  0.18104935,
            0.15684317,  0.01376599,  0.03293737,  0.08989109,  0.02709115,
            0.14221564,  0.27065598,  0.10237696,  0.04226237,  0.72991785,
            0.06713876,  0.04226237,  0.03482061,  0.07879431,  0.07471691,
            0.15307528,  0.02289366,  0.08989109,  0.02634295,  0.43243779,
            0.08756457,  0.03293737,  0.02786   ,  0.03482061,  0.0187955 ,
            0.08307425,  0.04589793,  0.07275173,  0.0311527 ,  0.04589793,
            0.08307425,  0.32468619,  0.02289366,  0.02634295,  0.03580075,
            0.14938186,  0.0526265 ,  0.0526265 ,  0.53268924,  0.19874565,
            0.0187955 ,  0.01541937,  0.01586237,  0.02045547,  0.02421701,
            0.02634295,  0.11344884,  0.05710047,  0.05121018,  0.09720799,
            0.0311527 ,  0.0526265 ,  0.01586237,  0.07471691,  0.06027915,
            0.15684317,  0.07879431,  0.02289366,  0.04111287,  0.04848506,
            0.02865003,  0.04589793,  0.03580075,  0.04111287,  0.1353385 ,
            0.09976271,  0.06362285,  0.32468619,  0.09976271,  0.49676673,
            0.07879431,  0.06027915,  0.06027915,  0.05407977,  0.05710047,
            0.0689636 ,  0.11936722,  0.18973955,  0.02709115,  0.03890304,
            0.02634295,  0.80625182,  0.04111287,  0.0311527 ,  0.07879431,
            0.0193336 ,  0.01988679,  0.01376599,  0.07879431,  0.05710047,
            0.06027915,  0.02104007,  0.0689636 ,  0.04717464,  0.04465416,
            0.07083428,  0.03999333,  0.06192993,  0.05407977,  0.04982997,
            0.46087756,  0.09720799,  0.04589793,  0.07083428,  0.0193336 ,
            0.12242534,  0.12242534,  0.05407977,  0.01776301,  0.0311527 ,
            0.0689636 ,  0.02421701,  0.13200634,  0.19874565,  0.03293737,
            0.82774282], atol=1.0e-8)
        self.assertAlmostEqual(results.pearson_chi2, 271.21110541713801)
        np.testing.assert_allclose(results.resid_response,
            [-0.04226237, -0.03999333, -0.02946178, -0.0689636 , -0.09471181,
            -0.07879431, -0.04717464, -0.27065598, -0.07471691,  0.10477856,
            -0.39752487,  0.66897282, -0.06192993, -0.04589793, -0.01988679,
            -0.0526265 , -0.02104007, -0.03386636, -0.02634295, -0.05121018,
            -0.29396682,  0.92724827, -0.03386636, -0.15307528, -0.06027915,
            -0.01631789, -0.02045547, -0.01541937, -0.2128508 , -0.04589793,
            -0.02104007, -0.05407977, -0.0311527 , -0.02634295, -0.01498855,
            -0.29396682,  0.79663224, -0.07275173, -0.11637537,  0.74604393,
            -0.64367488, -0.02634295, -0.02164101, -0.07083428, -0.05710047,
            -0.32468619, -0.01160845, -0.01631789, -0.28803008, -0.02634295,
            -0.17267234, -0.02634295, -0.01776301, -0.02709115,  0.85061814,
            0.73498669, -0.04111287, -0.06362285, -0.07083428, -0.07879431,
            0.91010891, -0.03680743, -0.0187955 , -0.01541937, -0.03680743,
            -0.03029581, -0.02634295, -0.03029581, -0.07471691, -0.01228768,
            0.76722803, -0.10505173, -0.06192993, -0.09720799, -0.01416217,
            -0.0187955 , -0.0689636 , -0.02865003, -0.07471691, -0.16460503,
            -0.09720799, -0.02045547,  0.82732766, -0.0311527 , -0.0187955 ,
            -0.15684317, -0.04111287, -0.03293737, -0.02104007, -0.02946178,
            -0.02421701, -0.1353385 , -0.03203302, -0.04111287, -0.10778798,
            -0.07083428, -0.06027915, -0.02164101, -0.06535882, -0.02946178,
            -0.07275173, -0.02490638, -0.01678627, -0.30605146, -0.02164101,
            -0.03482061, -0.03580075,  0.62969079, -0.0182721 , -0.03482061,
            -0.07083428, -0.04226237, -0.03999333, -0.03580075, -0.03203302,
            -0.0182721 , -0.03580075, -0.06027915, -0.03386636, -0.02946178,
            -0.03029581, -0.0689636 , -0.02634295, -0.02634295, -0.03029581,
            -0.02225873, -0.1353385 , -0.08989109, -0.01988679, -0.0526265 ,
            -0.03386636, -0.03386636, -0.02786   , -0.03029581, -0.06535882,
            -0.06535882, -0.03482061, -0.02786   , -0.29396682, -0.03293737,
            -0.12242534, -0.04589793, -0.04589793, -0.03999333, -0.07471691,
            -0.11344884, -0.05407977, -0.03482061, -0.01988679, -0.02045547,
            0.65610673,  0.85423777, -0.02561486, -0.0689636 , -0.02045547,
            -0.02865003, -0.0526265 , -0.02164101, -0.01776301, -0.08307425,
            -0.11344884, -0.04982997, -0.0182721 , -0.01498855, -0.02865003,
            -0.14221564, -0.07879431, -0.02865003, -0.10237696, -0.04465416,
            -0.07471691, -0.07673078, -0.13200634, -0.02104007, -0.0187955 ,
            -0.01376599, -0.04717464, -0.01128289,  0.94289953, -0.01988679,
            -0.01300612, -0.11936722, -0.03203302, -0.01726786, -0.04589793,
            -0.05407977, -0.09976271, -0.02561486, -0.03999333, -0.02634295,
            -0.03580075, -0.21771181,  0.8646615 , -0.01988679,  0.62295626,
            -0.06027915, -0.02045547, -0.18104935,  0.96000667, -0.18104935,
            -0.15684317, -0.01376599, -0.03293737, -0.08989109, -0.02709115,
            -0.14221564,  0.72934402, -0.10237696, -0.04226237, -0.72991785,
            -0.06713876, -0.04226237, -0.03482061, -0.07879431, -0.07471691,
            -0.15307528,  0.97710634,  0.91010891, -0.02634295, -0.43243779,
            -0.08756457, -0.03293737, -0.02786   , -0.03482061, -0.0187955 ,
             0.91692575, -0.04589793, -0.07275173, -0.0311527 , -0.04589793,
            -0.08307425,  0.67531381, -0.02289366, -0.02634295, -0.03580075,
            -0.14938186, -0.0526265 , -0.0526265 ,  0.46731076, -0.19874565,
            -0.0187955 , -0.01541937, -0.01586237, -0.02045547, -0.02421701,
            -0.02634295, -0.11344884, -0.05710047, -0.05121018, -0.09720799,
            0.9688473 , -0.0526265 , -0.01586237, -0.07471691, -0.06027915,
            -0.15684317, -0.07879431, -0.02289366, -0.04111287, -0.04848506,
            -0.02865003, -0.04589793, -0.03580075, -0.04111287, -0.1353385 ,
            -0.09976271, -0.06362285,  0.67531381, -0.09976271, -0.49676673,
            -0.07879431, -0.06027915, -0.06027915, -0.05407977, -0.05710047,
            -0.0689636 , -0.11936722, -0.18973955, -0.02709115, -0.03890304,
            -0.02634295,  0.19374818, -0.04111287, -0.0311527 , -0.07879431,
            -0.0193336 , -0.01988679, -0.01376599, -0.07879431,  0.94289953,
            -0.06027915, -0.02104007, -0.0689636 , -0.04717464, -0.04465416,
            0.92916572, -0.03999333, -0.06192993, -0.05407977, -0.04982997,
            -0.46087756, -0.09720799, -0.04589793, -0.07083428, -0.0193336 ,
            -0.12242534, -0.12242534, -0.05407977, -0.01776301, -0.0311527 ,
            -0.0689636 , -0.02421701, -0.13200634, -0.19874565, -0.03293737,
            -0.82774282], atol=1.0e-8)
        np.testing.assert_allclose(results.resid_working,
            [ -1.71062283e-03,  -1.53549840e-03,  -8.42423701e-04,
            -4.42798906e-03,  -8.12073047e-03,  -5.71934606e-03,
            -2.12046213e-03,  -5.34278480e-02,  -5.16550074e-03,
            9.82823035e-03,  -9.52067472e-02,   1.48142818e-01,
            -3.59779501e-03,  -2.00993083e-03,  -3.87619325e-04,
            -2.62379729e-03,  -4.33370579e-04,  -1.10808799e-03,
            -6.75670103e-04,  -2.48818484e-03,  -6.10129090e-02,
            6.25511612e-02,  -1.10808799e-03,  -1.98451739e-02,
            -3.41454749e-03,  -2.61928659e-04,  -4.09867263e-04,
            -2.34090923e-04,  -3.56621577e-02,  -2.00993083e-03,
            -4.33370579e-04,  -2.76645832e-03,  -9.40257152e-04,
            -6.75670103e-04,  -2.21289369e-04,  -6.10129090e-02,
            1.29061842e-01,  -4.90775251e-03,  -1.19671283e-02,
            1.41347263e-01,  -1.47631680e-01,  -6.75670103e-04,
            -4.58198217e-04,  -4.66208406e-03,  -3.07429001e-03,
            -7.11923401e-02,  -1.33191898e-04,  -2.61928659e-04,
            -5.90659690e-02,  -6.75670103e-04,  -2.46673839e-02,
            -6.75670103e-04,  -3.09919962e-04,  -7.14047519e-04,
            1.08085429e-01,   1.43161630e-01,  -1.62077632e-03,
            -3.79032977e-03,  -4.66208406e-03,  -5.71934606e-03,
            7.44566288e-02,  -1.30492035e-03,  -3.46630910e-04,
            -2.34090923e-04,  -1.30492035e-03,  -8.90029618e-04,
            -6.75670103e-04,  -8.90029618e-04,  -5.16550074e-03,
            -1.49131762e-04,   1.37018624e-01,  -9.87652847e-03,
            -3.59779501e-03,  -8.53083698e-03,  -1.97726627e-04,
            -3.46630910e-04,  -4.42798906e-03,  -7.97307494e-04,
            -5.16550074e-03,  -2.26348718e-02,  -8.53083698e-03,
            -4.09867263e-04,   1.18189219e-01,  -9.40257152e-04,
            -3.46630910e-04,  -2.07414715e-02,  -1.62077632e-03,
            -1.04913757e-03,  -4.33370579e-04,  -8.42423701e-04,
            -5.72261321e-04,  -1.58375811e-02,  -9.93244730e-04,
            -1.62077632e-03,  -1.03659408e-02,  -4.66208406e-03,
            -3.41454749e-03,  -4.58198217e-04,  -3.99257703e-03,
            -8.42423701e-04,  -4.90775251e-03,  -6.04877746e-04,
            -2.77048947e-04,  -6.50004229e-02,  -4.58198217e-04,
            -1.17025566e-03,  -1.23580799e-03,   1.46831486e-01,
            -3.27769165e-04,  -1.17025566e-03,  -4.66208406e-03,
            -1.71062283e-03,  -1.53549840e-03,  -1.23580799e-03,
            -9.93244730e-04,  -3.27769165e-04,  -1.23580799e-03,
            -3.41454749e-03,  -1.10808799e-03,  -8.42423701e-04,
            -8.90029618e-04,  -4.42798906e-03,  -6.75670103e-04,
            -6.75670103e-04,  -8.90029618e-04,  -4.84422741e-04,
            -1.58375811e-02,  -7.35405096e-03,  -3.87619325e-04,
            -2.62379729e-03,  -1.10808799e-03,  -1.10808799e-03,
            -7.54555329e-04,  -8.90029618e-04,  -3.99257703e-03,
            -3.99257703e-03,  -1.17025566e-03,  -7.54555329e-04,
            -6.10129090e-02,  -1.04913757e-03,  -1.31530576e-02,
            -2.00993083e-03,  -2.00993083e-03,  -1.53549840e-03,
            -5.16550074e-03,  -1.14104800e-02,  -2.76645832e-03,
            -1.17025566e-03,  -3.87619325e-04,  -4.09867263e-04,
            1.48037813e-01,   1.06365931e-01,  -6.39314594e-04,
            -4.42798906e-03,  -4.09867263e-04,  -7.97307494e-04,
            -2.62379729e-03,  -4.58198217e-04,  -3.09919962e-04,
            -6.32800839e-03,  -1.14104800e-02,  -2.35929680e-03,
            -3.27769165e-04,  -2.21289369e-04,  -7.97307494e-04,
            -1.73489362e-02,  -5.71934606e-03,  -7.97307494e-04,
            -9.40802551e-03,  -1.90495384e-03,  -5.16550074e-03,
            -5.43585191e-03,  -1.51253748e-02,  -4.33370579e-04,
            -3.46630910e-04,  -1.86893696e-04,  -2.12046213e-03,
            -1.25867293e-04,   5.07657192e-02,  -3.87619325e-04,
            -1.66959104e-04,  -1.25477263e-02,  -9.93244730e-04,
            -2.93030065e-04,  -2.00993083e-03,  -2.76645832e-03,
            -8.95970087e-03,  -6.39314594e-04,  -1.53549840e-03,
            -6.75670103e-04,  -1.23580799e-03,  -3.70792339e-02,
            1.01184411e-01,  -3.87619325e-04,   1.46321062e-01,
            -3.41454749e-03,  -4.09867263e-04,  -2.68442736e-02,
            3.68583645e-02,  -2.68442736e-02,  -2.07414715e-02,
            -1.86893696e-04,  -1.04913757e-03,  -7.35405096e-03,
            -7.14047519e-04,  -1.73489362e-02,   1.43973473e-01,
            -9.40802551e-03,  -1.71062283e-03,  -1.43894386e-01,
            -4.20497779e-03,  -1.71062283e-03,  -1.17025566e-03,
            -5.71934606e-03,  -5.16550074e-03,  -1.98451739e-02,
            2.18574168e-02,   7.44566288e-02,  -6.75670103e-04,
            -1.06135519e-01,  -6.99614755e-03,  -1.04913757e-03,
            -7.54555329e-04,  -1.17025566e-03,  -3.46630910e-04,
            6.98449121e-02,  -2.00993083e-03,  -4.90775251e-03,
            -9.40257152e-04,  -2.00993083e-03,  -6.32800839e-03,
            1.48072729e-01,  -5.12120512e-04,  -6.75670103e-04,
            -1.23580799e-03,  -1.89814939e-02,  -2.62379729e-03,
            -2.62379729e-03,   1.16328328e-01,  -3.16494123e-02,
            -3.46630910e-04,  -2.34090923e-04,  -2.47623705e-04,
            -4.09867263e-04,  -5.72261321e-04,  -6.75670103e-04,
            -1.14104800e-02,  -3.07429001e-03,  -2.48818484e-03,
            -8.53083698e-03,   2.92419496e-02,  -2.62379729e-03,
            -2.47623705e-04,  -5.16550074e-03,  -3.41454749e-03,
            -2.07414715e-02,  -5.71934606e-03,  -5.12120512e-04,
            -1.62077632e-03,  -2.23682205e-03,  -7.97307494e-04,
            -2.00993083e-03,  -1.23580799e-03,  -1.62077632e-03,
            -1.58375811e-02,  -8.95970087e-03,  -3.79032977e-03,
            1.48072729e-01,  -8.95970087e-03,  -1.24186489e-01,
            -5.71934606e-03,  -3.41454749e-03,  -3.41454749e-03,
            -2.76645832e-03,  -3.07429001e-03,  -4.42798906e-03,
            -1.25477263e-02,  -2.91702648e-02,  -7.14047519e-04,
            -1.45456868e-03,  -6.75670103e-04,   3.02653681e-02,
            -1.62077632e-03,  -9.40257152e-04,  -5.71934606e-03,
            -3.66561274e-04,  -3.87619325e-04,  -1.86893696e-04,
            -5.71934606e-03,   5.07657192e-02,  -3.41454749e-03,
            -4.33370579e-04,  -4.42798906e-03,  -2.12046213e-03,
            -1.90495384e-03,   6.11546973e-02,  -1.53549840e-03,
            -3.59779501e-03,  -2.76645832e-03,  -2.35929680e-03,
            -1.14513988e-01,  -8.53083698e-03,  -2.00993083e-03,
            -4.66208406e-03,  -3.66561274e-04,  -1.31530576e-02,
            -1.31530576e-02,  -2.76645832e-03,  -3.09919962e-04,
            -9.40257152e-04,  -4.42798906e-03,  -5.72261321e-04,
            -1.51253748e-02,  -3.16494123e-02,  -1.04913757e-03,
            -1.18023417e-01])
        np.testing.assert_allclose(results.resid_pearson,
            [-0.21006498, -0.20410641, -0.17423009, -0.27216147, -0.3234511 ,
            -0.29246179, -0.22250903, -0.60917574, -0.28416602,  0.3421141 ,
            -0.81229277,  1.42158361, -0.25694055, -0.21933056, -0.142444  ,
            -0.23569027, -0.14660243, -0.18722578, -0.16448609, -0.2323235 ,
            -0.64526275,  3.57006696, -0.18722578, -0.42513819, -0.25327023,
            -0.12879668, -0.14450826, -0.12514332, -0.5200069 , -0.21933056,
            -0.14660243, -0.23910582, -0.17931646, -0.16448609, -0.12335569,
            -0.64526275,  1.97919183, -0.28010679, -0.36290807,  1.71396874,
            -1.3440334 , -0.16448609, -0.14872695, -0.27610555, -0.24608613,
            -0.69339243, -0.1083734 , -0.12879668, -0.63604537, -0.16448609,
            -0.45684893, -0.16448609, -0.13447767, -0.16686977,  2.3862634 ,
            1.66535145, -0.20706426, -0.26066405, -0.27610555, -0.29246179,
            3.18191348, -0.19548397, -0.13840353, -0.12514332, -0.19548397,
            -0.17675498, -0.16448609, -0.17675498, -0.28416602, -0.11153719,
            1.81550268, -0.34261205, -0.25694055, -0.32813846, -0.11985666,
            -0.13840353, -0.27216147, -0.17174127, -0.28416602, -0.44389026,
            -0.32813846, -0.14450826,  2.18890738, -0.17931646, -0.13840353,
            -0.43129917, -0.20706426, -0.18455132, -0.14660243, -0.17423009,
            -0.1575374 , -0.39562855, -0.18191506, -0.20706426, -0.34757708,
            -0.27610555, -0.25327023, -0.14872695, -0.26444152, -0.17423009,
            -0.28010679, -0.15982038, -0.13066317, -0.66410018, -0.14872695,
            -0.189939  , -0.19269154,  1.30401147, -0.13642648, -0.189939  ,
            -0.27610555, -0.21006498, -0.20410641, -0.19269154, -0.18191506,
            -0.13642648, -0.19269154, -0.25327023, -0.18722578, -0.17423009,
            -0.17675498, -0.27216147, -0.16448609, -0.16448609, -0.17675498,
            -0.15088226, -0.39562855, -0.3142763 , -0.142444  , -0.23569027,
            -0.18722578, -0.18722578, -0.169288  , -0.17675498, -0.26444152,
            -0.26444152, -0.189939  , -0.169288  , -0.64526275, -0.18455132,
            -0.3735026 , -0.21933056, -0.21933056, -0.20410641, -0.28416602,
            -0.35772404, -0.23910582, -0.189939  , -0.142444  , -0.14450826,
            1.38125991,  2.42084442, -0.16213645, -0.27216147, -0.14450826,
            -0.17174127, -0.23569027, -0.14872695, -0.13447767, -0.30099975,
            -0.35772404, -0.22900483, -0.13642648, -0.12335569, -0.17174127,
            -0.4071783 , -0.29246179, -0.17174127, -0.33771794, -0.21619749,
            -0.28416602, -0.28828407, -0.38997712, -0.14660243, -0.13840353,
            -0.11814455, -0.22250903, -0.10682532,  4.06361781, -0.142444  ,
            -0.11479334, -0.36816723, -0.18191506, -0.1325567 , -0.21933056,
            -0.23910582, -0.33289374, -0.16213645, -0.20410641, -0.16448609,
            -0.19269154, -0.52754269,  2.52762346, -0.142444  ,  1.28538406,
            -0.25327023, -0.14450826, -0.47018591,  4.89940505, -0.47018591,
            -0.43129917, -0.11814455, -0.18455132, -0.3142763 , -0.16686977,
            -0.4071783 ,  1.64156241, -0.33771794, -0.21006498, -1.6439517 ,
            -0.26827373, -0.21006498, -0.189939  , -0.29246179, -0.28416602,
            -0.42513819,  6.53301013,  3.18191348, -0.16448609, -0.87288109,
            -0.30978696, -0.18455132, -0.169288  , -0.189939  , -0.13840353,
             3.32226189, -0.21933056, -0.28010679, -0.17931646, -0.21933056,
            -0.30099975,  1.44218477, -0.1530688 , -0.16448609, -0.19269154,
            -0.41906522, -0.23569027, -0.23569027,  0.93662539, -0.4980393 ,
            -0.13840353, -0.12514332, -0.12695686, -0.14450826, -0.1575374 ,
            -0.16448609, -0.35772404, -0.24608613, -0.2323235 , -0.32813846,
            5.57673284, -0.23569027, -0.12695686, -0.28416602, -0.25327023,
            -0.43129917, -0.29246179, -0.1530688 , -0.20706426, -0.22573357,
            -0.17174127, -0.21933056, -0.19269154, -0.20706426, -0.39562855,
            -0.33289374, -0.26066405,  1.44218477, -0.33289374, -0.99355423,
            -0.29246179, -0.25327023, -0.25327023, -0.23910582, -0.24608613,
            -0.27216147, -0.36816723, -0.48391225, -0.16686977, -0.20119082,
            -0.16448609,  0.49021146, -0.20706426, -0.17931646, -0.29246179,
            -0.14040923, -0.142444  , -0.11814455, -0.29246179,  4.06361781,
            -0.25327023, -0.14660243, -0.27216147, -0.22250903, -0.21619749,
            3.6218033 , -0.20410641, -0.25694055, -0.23910582, -0.22900483,
            -0.92458976, -0.32813846, -0.21933056, -0.27610555, -0.14040923,
            -0.3735026 , -0.3735026 , -0.23910582, -0.13447767, -0.17931646,
            -0.27216147, -0.1575374 , -0.38997712, -0.4980393 , -0.18455132,
            -2.19209332])
        np.testing.assert_allclose(results.resid_anscombe,
            [-0.31237627, -0.3036605 , -0.25978208, -0.40240831, -0.47552289,
            -0.43149255, -0.33053793, -0.85617194, -0.41962951,  0.50181328,
            -1.0954382 ,  1.66940149, -0.38048321, -0.3259044 , -0.21280762,
            -0.34971301, -0.21896842, -0.27890356, -0.2454118 , -0.34482158,
            -0.90063409,  2.80452413, -0.27890356, -0.61652596, -0.37518169,
            -0.19255932, -0.2158664 , -0.18713159, -0.74270558, -0.3259044 ,
            -0.21896842, -0.35467084, -0.2672722 , -0.2454118 , -0.18447466,
            -0.90063409,  2.05763941, -0.41381347, -0.53089521,  1.88552083,
            -1.60654218, -0.2454118 , -0.22211425, -0.40807333, -0.3647888 ,
            -0.95861559, -0.16218047, -0.19255932, -0.88935802, -0.2454118 ,
            -0.65930821, -0.2454118 , -0.20099345, -0.24892975,  2.28774016,
            1.85167195, -0.30798858, -0.38585584, -0.40807333, -0.43149255,
            2.65398426, -0.2910267 , -0.20681747, -0.18713159, -0.2910267 ,
            -0.26350118, -0.2454118 , -0.26350118, -0.41962951, -0.16689207,
             1.95381191, -0.50251231, -0.38048321, -0.48214234, -0.17927213,
            -0.20681747, -0.40240831, -0.25611424, -0.41962951, -0.64189694,
            -0.48214234, -0.2158664 ,  2.18071204, -0.2672722 , -0.20681747,
            -0.62488429, -0.30798858, -0.27497271, -0.21896842, -0.25978208,
            -0.23514749, -0.57618899, -0.27109582, -0.30798858, -0.50947546,
            -0.40807333, -0.37518169, -0.22211425, -0.39130036, -0.25978208,
            -0.41381347, -0.2385213 , -0.19533116, -0.92350689, -0.22211425,
            -0.28288904, -0.28692985,  1.5730846 , -0.20388497, -0.28288904,
            -0.40807333, -0.31237627, -0.3036605 , -0.28692985, -0.27109582,
            -0.20388497, -0.28692985, -0.37518169, -0.27890356, -0.25978208,
            -0.26350118, -0.40240831, -0.2454118 , -0.2454118 , -0.26350118,
            -0.22530448, -0.57618899, -0.46253505, -0.21280762, -0.34971301,
            -0.27890356, -0.27890356, -0.25249702, -0.26350118, -0.39130036,
            -0.39130036, -0.28288904, -0.25249702, -0.90063409, -0.27497271,
            -0.5456246 , -0.3259044 , -0.3259044 , -0.3036605 , -0.41962951,
            -0.52366614, -0.35467084, -0.28288904, -0.21280762, -0.2158664 ,
            1.63703418,  2.30570989, -0.24194253, -0.40240831, -0.2158664 ,
            -0.25611424, -0.34971301, -0.22211425, -0.20099345, -0.44366892,
            -0.52366614, -0.33999576, -0.20388497, -0.18447466, -0.25611424,
            -0.59203547, -0.43149255, -0.25611424, -0.49563627, -0.32133344,
            -0.41962951, -0.42552227, -0.56840788, -0.21896842, -0.20681747,
            -0.17672552, -0.33053793, -0.15987433,  2.9768074 , -0.21280762,
            -0.17173916, -0.53821445, -0.27109582, -0.19814236, -0.3259044 ,
            -0.35467084, -0.48884654, -0.24194253, -0.3036605 , -0.2454118 ,
            -0.28692985, -0.75249089,  2.35983933, -0.21280762,  1.55726719,
            -0.37518169, -0.2158664 , -0.67712261,  3.23165236, -0.67712261,
            -0.62488429, -0.17672552, -0.27497271, -0.46253505, -0.24892975,
            -0.59203547,  1.83482464, -0.49563627, -0.31237627, -1.83652534,
            -0.39681759, -0.31237627, -0.28288904, -0.43149255, -0.41962951,
            -0.61652596,  3.63983609,  2.65398426, -0.2454118 , -1.16171662,
            -0.45616505, -0.27497271, -0.25249702, -0.28288904, -0.20681747,
            2.71015945, -0.3259044 , -0.41381347, -0.2672722 , -0.3259044 ,
            -0.44366892,  1.68567947, -0.22853969, -0.2454118 , -0.28692985,
            -0.60826548, -0.34971301, -0.34971301,  1.2290223 , -0.71397735,
            -0.20681747, -0.18713159, -0.1898263 , -0.2158664 , -0.23514749,
            -0.2454118 , -0.52366614, -0.3647888 , -0.34482158, -0.48214234,
            3.41271513, -0.34971301, -0.1898263 , -0.41962951, -0.37518169,
            -0.62488429, -0.43149255, -0.22853969, -0.30798858, -0.3352348 ,
            -0.25611424, -0.3259044 , -0.28692985, -0.30798858, -0.57618899,
            -0.48884654, -0.38585584,  1.68567947, -0.48884654, -1.28709718,
            -0.43149255, -0.37518169, -0.37518169, -0.35467084, -0.3647888 ,
            -0.40240831, -0.53821445, -0.69534436, -0.24892975, -0.29939131,
            -0.2454118 ,  0.70366797, -0.30798858, -0.2672722 , -0.43149255,
            -0.2097915 , -0.21280762, -0.17672552, -0.43149255,  2.9768074 ,
            -0.37518169, -0.21896842, -0.40240831, -0.33053793, -0.32133344,
            2.82351017, -0.3036605 , -0.38048321, -0.35467084, -0.33999576,
            -1.21650102, -0.48214234, -0.3259044 , -0.40807333, -0.2097915 ,
            -0.5456246 , -0.5456246 , -0.35467084, -0.20099345, -0.2672722 ,
            -0.40240831, -0.23514749, -0.56840788, -0.71397735, -0.27497271,
            -2.18250381])
        np.testing.assert_allclose(results.resid_deviance,
            [-0.29387552, -0.2857098 , -0.24455876, -0.37803944, -0.44609851,
            -0.40514674, -0.31088148, -0.79449324, -0.39409528,  0.47049798,
            -1.00668653,  1.48698001, -0.35757692, -0.30654405, -0.20043547,
            -0.32882173, -0.20622595, -0.26249995, -0.23106769, -0.32424676,
            -0.83437766,  2.28941155, -0.26249995, -0.57644334, -0.35262564,
            -0.18139734, -0.20331052, -0.17629229, -0.69186337, -0.30654405,
            -0.20622595, -0.33345774, -0.251588  , -0.23106769, -0.17379306,
            -0.83437766,  1.78479093, -0.38867448, -0.4974393 ,  1.65565332,
            -1.43660134, -0.23106769, -0.20918228, -0.38332275, -0.34291558,
            -0.88609006, -0.15281596, -0.18139734, -0.82428104, -0.23106769,
            -0.61571821, -0.23106769, -0.18932865, -0.234371  ,  1.94999969,
            1.62970871, -0.2897651 , -0.36259328, -0.38332275, -0.40514674,
            2.19506559, -0.27386827, -0.19480442, -0.17629229, -0.27386827,
            -0.24804925, -0.23106769, -0.24804925, -0.39409528, -0.15725009,
            1.7074519 , -0.47114617, -0.35757692, -0.4522457 , -0.16889886,
            -0.19480442, -0.37803944, -0.24111595, -0.39409528, -0.59975102,
            -0.4522457 , -0.20331052,  1.87422489, -0.251588  , -0.19480442,
            -0.5841272 , -0.2897651 , -0.25881274, -0.20622595, -0.24455876,
            -0.22142749, -0.53929061, -0.25517563, -0.2897651 , -0.47760126,
            -0.38332275, -0.35262564, -0.20918228, -0.36767536, -0.24455876,
            -0.38867448, -0.2245965 , -0.18400413, -0.85481866, -0.20918228,
            -0.26623785, -0.27002708,  1.40955093, -0.19204738, -0.26623785,
            -0.38332275, -0.29387552, -0.2857098 , -0.27002708, -0.25517563,
            -0.19204738, -0.27002708, -0.35262564, -0.26249995, -0.24455876,
            -0.24804925, -0.37803944, -0.23106769, -0.23106769, -0.24804925,
            -0.21218006, -0.53929061, -0.43402996, -0.20043547, -0.32882173,
            -0.26249995, -0.26249995, -0.23772023, -0.24804925, -0.36767536,
            -0.36767536, -0.26623785, -0.23772023, -0.83437766, -0.25881274,
            -0.51106408, -0.30654405, -0.30654405, -0.2857098 , -0.39409528,
            -0.49074728, -0.33345774, -0.26623785, -0.20043547, -0.20331052,
            1.46111186,  1.96253843, -0.22780971, -0.37803944, -0.20331052,
            -0.24111595, -0.32882173, -0.20918228, -0.18932865, -0.41648237,
            -0.49074728, -0.31973217, -0.19204738, -0.17379306, -0.24111595,
            -0.55389988, -0.40514674, -0.24111595, -0.46476893, -0.30226435,
            -0.39409528, -0.39958581, -0.53211065, -0.20622595, -0.19480442,
            -0.16650295, -0.31088148, -0.15064545,  2.39288231, -0.20043547,
            -0.16181126, -0.5042114 , -0.25517563, -0.18664773, -0.30654405,
            -0.33345774, -0.45846897, -0.22780971, -0.2857098 , -0.23106769,
            -0.27002708, -0.7007597 ,  1.99998811, -0.20043547,  1.39670618,
            -0.35262564, -0.20331052, -0.63203077,  2.53733821, -0.63203077,
            -0.5841272 , -0.16650295, -0.25881274, -0.43402996, -0.234371  ,
            -0.55389988,  1.61672923, -0.46476893, -0.29387552, -1.61804148,
            -0.37282386, -0.29387552, -0.26623785, -0.40514674, -0.39409528,
            -0.57644334,  2.74841605,  2.19506559, -0.23106769, -1.06433539,
            -0.42810736, -0.25881274, -0.23772023, -0.26623785, -0.19480442,
            2.23070414, -0.30654405, -0.38867448, -0.251588  , -0.30654405,
            -0.41648237,  1.49993075, -0.21521982, -0.23106769, -0.27002708,
            -0.5688444 , -0.32882173, -0.32882173,  1.12233423, -0.66569789,
            -0.19480442, -0.17629229, -0.17882689, -0.20331052, -0.22142749,
            -0.23106769, -0.49074728, -0.34291558, -0.32424676, -0.4522457 ,
            2.63395309, -0.32882173, -0.17882689, -0.39409528, -0.35262564,
            -0.5841272 , -0.40514674, -0.21521982, -0.2897651 , -0.3152773 ,
            -0.24111595, -0.30654405, -0.27002708, -0.2897651 , -0.53929061,
            -0.45846897, -0.36259328,  1.49993075, -0.45846897, -1.17192274,
            -0.40514674, -0.35262564, -0.35262564, -0.33345774, -0.34291558,
            -0.37803944, -0.5042114 , -0.64869028, -0.234371  , -0.28170899,
            -0.23106769,  0.65629132, -0.2897651 , -0.251588  , -0.40514674,
            -0.19760028, -0.20043547, -0.16650295, -0.40514674,  2.39288231,
            -0.35262564, -0.20622595, -0.37803944, -0.31088148, -0.30226435,
            2.30104857, -0.2857098 , -0.35757692, -0.33345774, -0.31973217,
            -1.11158678, -0.4522457 , -0.30654405, -0.38332275, -0.19760028,
            -0.51106408, -0.51106408, -0.33345774, -0.18932865, -0.251588  ,
            -0.37803944, -0.22142749, -0.53211065, -0.66569789, -0.25881274,
            -1.87550882])
        np.testing.assert_allclose(results.null,
            [ 0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759,  0.08860759,  0.08860759,  0.08860759,  0.08860759,
            0.08860759])
        self.assertAlmostEqual(results.D2, .200712816165)
        self.assertAlmostEqual(results.adj_D2, 0.19816731557930456)



if __name__ == '__main__':
	unittest.main()
