"""Implementation of GLU Nurbs structure and callback methods

Same basic pattern as seen with the gluTess* functions, just need to
add some bookkeeping to the structure class so that we can keep the
Python function references alive during the calling process.
"""
from OpenGL.raw import GLU as simple
from OpenGL import platform, converters, wrapper
from OpenGL.GLU import glustruct
from OpenGL import arrays
import ctypes
import weakref

__all__ = (
	'GLUnurbs',
	'gluNewNurbsRenderer',
	'gluNurbsCallback',
	'gluNurbsCallbackData',
	'gluNurbsCallbackDataEXT',
	'gluNurbsCurve',
	'gluNurbsSurface',
	'gluPwlCurve',
)

# /usr/include/GL/glu.h 242
class GLUnurbs(simple.GLUnurbs,glustruct.GLUStruct):
	"""GLU Nurbs structure with oor and callback storage support
	
	IMPORTANT NOTE: the texture coordinate callback receives a raw ctypes 
	data-pointer, as without knowing what type of evaluation is being done 
	(1D or 2D) we cannot safely determine the size of the array to convert 
	it.  This is a limitation of the C implementation.  To convert to regular 
	data-pointer, just call yourNurb.ptrAsArray( ptr, size, arrays.GLfloatArray )
	with the size of data you expect.
	"""
	CALLBACK_TYPES = {
		# mapping from "which" GLU enumeration to a ctypes function type
		simple.GLU_NURBS_BEGIN: ctypes.CFUNCTYPE( 
			None, simple.GLenum 
		),
		simple.GLU_NURBS_BEGIN_DATA: ctypes.CFUNCTYPE( 
			None, simple.GLenum, ctypes.POINTER(simple.GLvoid) 
		),
		simple.GLU_NURBS_VERTEX: ctypes.CFUNCTYPE( 
			None, ctypes.POINTER(simple.GLfloat)
		),
		simple.GLU_NURBS_VERTEX_DATA: ctypes.CFUNCTYPE( 
			None, ctypes.POINTER(simple.GLfloat), ctypes.POINTER(simple.GLvoid) 
		),
		simple.GLU_NURBS_NORMAL: ctypes.CFUNCTYPE( 
			None, ctypes.POINTER(simple.GLfloat)
		),
		simple.GLU_NURBS_NORMAL_DATA: ctypes.CFUNCTYPE( 
			None, ctypes.POINTER(simple.GLfloat), ctypes.POINTER(simple.GLvoid) 
		),
		simple.GLU_NURBS_COLOR: ctypes.CFUNCTYPE( 
			None, ctypes.POINTER(simple.GLfloat)
		),
		simple.GLU_NURBS_COLOR_DATA: ctypes.CFUNCTYPE( 
			None, ctypes.POINTER(simple.GLfloat), ctypes.POINTER(simple.GLvoid) 
		),
		simple.GLU_NURBS_TEXTURE_COORD: ctypes.CFUNCTYPE( 
			None, ctypes.POINTER(simple.GLfloat)
		),
		simple.GLU_NURBS_TEXTURE_COORD_DATA: ctypes.CFUNCTYPE( 
			None, ctypes.POINTER(simple.GLfloat), ctypes.POINTER(simple.GLvoid) 
		),
		simple.GLU_NURBS_END:ctypes.CFUNCTYPE( 
			None
		),
		simple.GLU_NURBS_END_DATA: ctypes.CFUNCTYPE( 
			None, ctypes.POINTER(simple.GLvoid) 
		),
		simple.GLU_NURBS_ERROR:ctypes.CFUNCTYPE( 
			None, simple.GLenum, 
		),
	}
	WRAPPER_METHODS = {
		simple.GLU_NURBS_BEGIN: None,
		simple.GLU_NURBS_BEGIN_DATA: '_justOOR',
		simple.GLU_NURBS_VERTEX: '_vec3',
		simple.GLU_NURBS_VERTEX_DATA: '_vec3',
		simple.GLU_NURBS_NORMAL: '_vec3',
		simple.GLU_NURBS_NORMAL_DATA: '_vec3',
		simple.GLU_NURBS_COLOR: '_vec4',
		simple.GLU_NURBS_COLOR_DATA: '_vec4',
		simple.GLU_NURBS_TEXTURE_COORD: '_tex',
		simple.GLU_NURBS_TEXTURE_COORD_DATA: '_tex',
		simple.GLU_NURBS_END: None,
		simple.GLU_NURBS_END_DATA: '_justOOR',
		simple.GLU_NURBS_ERROR: None,
	}
	def _justOOR( self, function ):
		"""Just do OOR on the last argument..."""
		def getOOR( *args ):
			args = args[:-1] + (self.originalObject(args[-1]),)
			return function( *args )
		return getOOR
	def _vec3( self, function, size=3 ):
		"""Convert first arg to size-element array, do OOR on arg2 if present"""
		def vec( *args ):
			vec = self.ptrAsArray(args[0],size,arrays.GLfloatArray)
			if len(args) > 1:
				oor = self.originalObject(args[1])
				return function( vec, oor )
			else:
				return function( vec )
		return vec
	def _vec4( self, function ):
		"""Size-4 vector version..."""
		return self._vec3( function, 4 )
	def _tex( self, function ):
		"""Texture coordinate callback 
		
		NOTE: there is no way for *us* to tell what size the array is, you will 
		get back a raw data-point, not an array, as you do for all other callback 
		types!!!
		"""
		def oor( *args ):
			if len(args) > 1:
				oor = self.originalObject(args[1])
				return function( args[0], oor )
			else:
				return function( args[0] )
		return oor

def _callbackWithType( funcType ):
	"""Get gluNurbsCallback function with set last arg-type"""
	result =  platform.copyBaseFunction(
		simple.gluNurbsCallback
	)
	result.argTypes = [ctypes.POINTER(GLUnurbs), simple.GLenum, funcType]
	return result

GLUnurbs.CALLBACK_FUNCTION_REGISTRARS = dict([
	(c,_callbackWithType( funcType ))
	for (c,funcType) in GLUnurbs.CALLBACK_TYPES.items()
])
del c, funcType


def gluNurbsCallback( nurb, which, CallBackFunc ):
	"""Dispatch to the nurb's callback operation"""
	return nurb.addCallback( which, CallBackFunc )

def gluNewNurbsRenderer( ):
	"""Return a new nurbs renderer for the system (dereferences pointer)"""
	return simple.gluNewNurbsRenderer()[0]

def gluNurbsCallbackData( nurb, userData ):
	"""Note the Python object for use as userData by the nurb"""
	return simple.gluNurbsCallbackData( 
		nurb, nurb.noteObject( userData ) 
	)

def gluNurbsCallbackDataEXT( nurb, userData ):
	"""Note the Python object for use as userData by the nurb"""
	return simple.gluNurbsCallbackDataEXT( 
		nurb, nurb.noteObject( userData ) 
	)

def gluNurbsCurve( nurb, knots, control, type ):
	"""Pythonic version of gluNurbsCurve
	
	Calculates knotCount, stride, and order automatically
	"""
	knots = arrays.GLfloatArray.asArray( knots )
	knotCount = arrays.GLfloatArray.arraySize( knots )
	control = arrays.GLfloatArray.asArray( control )
	length,step = arrays.GLfloatArray.dimensions( control )
	order = knotCount - length
	return simple.gluNurbsCurve(
		nurb, knotCount, knots, step, control, order, type,
	)

def gluNurbsSurface( nurb, sKnots, tKnots, control, type ):
	"""Pythonic version of gluNurbsSurface
	
	Calculates knotCount, stride, and order automatically
	"""
	sKnots = arrays.GLfloatArray.asArray( sKnots )
	sKnotCount = arrays.GLfloatArray.arraySize( sKnots )
	tKnots = arrays.GLfloatArray.asArray( tKnots )
	tKnotCount = arrays.GLfloatArray.arraySize( tKnots )
	control = arrays.GLfloatArray.asArray( control )

	length,width,step = arrays.GLfloatArray.dimensions( control )
	sOrder = sKnotCount - length 
	tOrder = tKnotCount - width 
	sStride = width*step
	tStride = step
	
	assert (sKnotCount-sOrder)*(tKnotCount-tOrder) == length*width, (
		nurb, sKnotCount, sKnots, tKnotCount, tKnots,
		sStride, tStride, control,
		sOrder,tOrder,
		type
	)

	result = simple.gluNurbsSurface(
		nurb, sKnotCount, sKnots, tKnotCount, tKnots,
		sStride, tStride, control,
		sOrder,tOrder,
		type
	)
	return result

def gluPwlCurve( nurb, data, type ):
	"""gluPwlCurve -- piece-wise linear curve within GLU context
	
	data -- the data-array 
	type -- determines number of elements/data-point
	"""
	data = arrays.GLfloatArray.asArray( data )
	if type == simple.GLU_MAP1_TRIM_2:
		divisor = 2
	elif type == simple.GLU_MAP_TRIM_3:
		divisor = 3
	else:
		raise ValueError( """Unrecognised type constant: %s"""%(type))
	size = arrays.GLfloatArray.arraySize( data )
	size = int(size/divisor)
	return simple.gluPwlCurve( nurb, size, data, divisor, type )
