#-*- coding: utf-8 -*-

licence={}
licence['en']="""\
pyacidobasic version %s:

a program to simulate acido-basic equilibria

Copyright (C) 2010-2011 Georges Khaznadar <georgesk@ofset.org>

This program is free software: you can redistribute it and/or
modify it under the terms of the GNU General Public License as
published by the Free Software Foundation, either version 3 of the
License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see
<http://www.gnu.org/licenses/>.
"""

licence['fr']=u"""\
pyacidobasic version %s:

un programme pour simuler des équilibres acido-basiques

Copyright (C) 2010-2011 Georges Khaznadar <georgesk@ofset.org>

Ce projet est un logiciel libre : vous pouvez le redistribuer, le
modifier selon les terme de la GPL (GNU Public License) dans les
termes de la Free Software Foundation concernant la version 3 ou
plus de la dite licence.

Ce programme est fait avec l'espoir qu'il sera utile mais SANS
AUCUNE GARANTIE. Lisez la licence pour plus de détails.

<http://www.gnu.org/licenses/>.
"""

from PyQt4.QtCore import *
from PyQt4.QtGui import *
from PyQt4.Qwt5 import QwtPlot, QwtPlotCurve, QwtPlotGrid, QwtLegend
import PyQt4.Qwt5.anynumpy as np
import phplot
from Ui_main import Ui_MainWindow
from curveControl import CurveControl
import reactifs, acidebase, version
import sys, pickle, os.path, re



locale="" # cette variable est globale, elle est initialisée au démarrage

class acidoBasicMainWindow(QMainWindow):
    def __init__(self, parent, argv):
        """
        Le constructeur
        @param parent une fenêtre parente
        @argv une liste d'arguments
        """
         ######QT
        QMainWindow.__init__(self)
        self.windowTitle="pyAcidoBasic"
        QWidget.__init__(self, parent)
        self.ui = Ui_MainWindow()
        self.filename=""
        self.courbesActives=[] # la liste des courbes qu'on peut montrer/cacher
        self.ui.setupUi(self)
        # état initial des boutons radio concentrations/quantités
        self.showConcentrations=True
       # substitution des traceurs de courbes
        ####### traceur orienté pH
        self.ui.horizontalLayout_labCourbes.removeWidget(self.ui.qwtPlotpH)
        self.ui.qwtPlotpH.close()
        php=phplot.phPlot(self.ui.tabWidget)
        self.ui.qwtPlotpH=php
        self.ui.horizontalLayout_labCourbes.insertWidget (0,php)
        ####### traceur orienté concentrations
        self.ui.horizontalLayout_concentrationCourbes.removeWidget(self.ui.qwtConcentrationPlot)
        self.ui.qwtConcentrationPlot.close()
        php=phplot.phPlot(self.ui.tabWidget)
        self.ui.qwtConcentrationPlot=php
        self.ui.horizontalLayout_concentrationCourbes.insertWidget (1,php)
        ######
        self.plots=(self.ui.qwtPlotpH,self.ui.qwtConcentrationPlot)
        # volume max versé de la burette
        self.maxBurette=20
        self.configurePlot()
        self.abListe=acidebase.listFromFile("/usr/share/pyacidobasic/acides-bases.html")
        self.filterChanged("") # met en place la liste des acides et des bases
        self.ui_connections()
        if len(argv)>1:
            if os.path.exists(argv[1]):
                self.load(argv[1])

    def ui_connections(self):
        QObject.connect(self.ui.toolButtonBurette,SIGNAL("clicked()"), self.videBurette)
        QObject.connect(self.ui.toolButtonBecher,SIGNAL("clicked()"), self.videBecher)
        QObject.connect(self.ui.toolButtonPlusV,SIGNAL("clicked()"), self.plusV)
        QObject.connect(self.ui.toolButtonMoinsV,SIGNAL("clicked()"), self.moinsV)
        QObject.connect(self.ui.toolButtonPlusV_2,SIGNAL("clicked()"), self.plusV)
        QObject.connect(self.ui.toolButtonMoinsV_2,SIGNAL("clicked()"), self.moinsV)
        QObject.connect(self.ui.PDFbutton,SIGNAL('clicked()'),self.plots[0].exportPDF)
        QObject.connect(self.ui.SVGbutton,SIGNAL('clicked()'),self.plots[0].exportSVG)
        QObject.connect(self.ui.JPGbutton,SIGNAL('clicked()'),self.plots[0].exportJPG)
        QObject.connect(self.ui.TitleButton,SIGNAL('clicked()'),self.plots[0].newTitle)
        QObject.connect(self.ui.PDFbutton_2,SIGNAL('clicked()'),self.plots[1].exportPDF)
        QObject.connect(self.ui.SVGbutton_2,SIGNAL('clicked()'),self.plots[1].exportSVG)
        QObject.connect(self.ui.JPGbutton_2,SIGNAL('clicked()'),self.plots[1].exportJPG)
        QObject.connect(self.ui.TitleButton_2,SIGNAL('clicked()'),self.plots[1].newTitle)
        QObject.connect(self.ui.razFiltreButton,SIGNAL('clicked()'),self.razFiltre)
        QObject.connect(self.ui.actionQuitter,SIGNAL('triggered()'),self.close)
        QObject.connect(self.ui.actionEnregistrer_Sous,SIGNAL('triggered()'),self.saveAs)
        QObject.connect(self.ui.actionEnregistrer,SIGNAL('triggered()'),self.save)
        QObject.connect(self.ui.actionOuvrir,SIGNAL('triggered()'),self.load)
        QObject.connect(self.ui.actionExemples,SIGNAL('triggered()'),self.loadExample)
        QObject.connect(self.ui.action_propos,SIGNAL('triggered()'),self.apropos)
        QObject.connect(self.ui.filtreEdit,SIGNAL('textChanged (QString)'),self.filterChanged)
        QObject.connect(self.ui.concentrationsButton,SIGNAL('toggled (bool)'),self.cqChanged)


    def cqChanged(self, concentration):
        """
        Fonction de rappel pour le changement d'état du bouton radio de
        concentrations.
        @param concentration booléen vrai si on doit afficher des concentrations
        """
        self.showConcentrations=concentration
        self.configurePlot()
        self.courbesSysteme()
        self.ajouteSolutes()
        
    def razFiltre(self):
        """
        Remet à zéro le filtre pour choisir les réactifs
        """
        self.ui.filtreEdit.clear()

    def filterChanged(self, newtext):
        """
        Fonction de rappel quand le texte du filtre change
        @param newtext le texte actuel du filtre
        """
        reactifs.setupUi(self.ui,self.abListe, newtext)
       
    def colorChanged(self, controle):
        """
        réponse au clic et au double clic
        @param controle l'instance de CurveControl qu'on mettra à jour
        """
        qd=QColorDialog(controle.color)
        qd.exec_()
        color=qd.selectedColor()
        controle.setColor(color)
        for c in controle.courbes:
            p=QPen(c.pen())
            b=QBrush(p.brush())
            b.setColor(color)
            p.setBrush(b)
            c.setPen(p)
            self.replot()
                
        
    def nouvelleCourbe(self, titre="", yAxis=QwtPlot.yRight, color="black",
                       width=1, yData=[], cell=None,
                       qwtp=None, enabled=True, checked=False):
        """
        crée une nouvelle courbe active qu'on peut cacher/montrer
        @param titre label pour la courbe
        @param yaxis ordonnées pour la courbe, QwtPlot.yRight par défaut
        @param color la couleur, "black" par défaut
        @param width l'épaisseur, 1 par défaut
        @param yData la liste de données pour Y (c'est self.vData pour X)
        @param cell un curveControl (None par défaut, alors le widget est créée)
        @param qwtp un qwtPlotWidget, celui des Ĥ par défaut
        @param enabled si on doit mettre une case active dans le tableau (vrai par défaut)
        @param checked si la case est pré-cochée (faux par défaut)
        @return la cellule de tableau éventuellement créée
        """
        if self.row > 3 and color=="black":
            # couleurs variables
            color=self.varColor()
        c=QwtPlotCurve(titre)
        if cell==None:
            i=CurveControl(None,
                           mw=self,
                           html=titre,
                           color=color,
                           modifiable=enabled,
                           checked=checked,
                           courbes=[c]
                           )
            self.ui.curveControlLayout.addWidget(i)
            c.cell=i
            self.row+=1
        else:
            c.cell=cell
            c.cell.courbes.append(c)
        self.courbesActives.append(c)
        self.legend.insert(c, QLabel(titre))
        c.setYAxis(yAxis)
        p=QPen(QColor(color))
        p.setWidth(width)
        c.setPen(p)
        c.setData(self.vData, yData)
        if checked:
            c.setStyle(QwtPlotCurve.Lines)
        else:
            c.setStyle(QwtPlotCurve.NoCurve)
        if qwtp==None:
            qwtp=self.ui.qwtPlotpH
        c.attach(qwtp)
        return c.cell

    def varColor(self):
        """
        @return une nouvelle couleur pour chaque self.row
        """
        r=np.sin(self.row*np.pi/2)
        v=np.sin(self.row*np.pi/3)
        b=np.sin(self.row*np.pi/5)
        r=(1+r)/2*255
        v=(1+v)/2*255
        b=(1+b)/2*255
        return QColor(r,v,b)

    def setCurvesVisibility(self,courbes, state):
        """
        Affiche ou cache les courbes selon le contexte des cases cochées
        @param une liste de courbes contrôlées
        @param state l'état du contrôle
        """
        for c in courbes:
            if state:
                c.setStyle(QwtPlotCurve.Lines)
            else:
                c.setStyle(QwtPlotCurve.NoCurve)
        self.replot()
            
    def saveAs(self):
        """
        Enregistre les données du graphique sous un nouveau fichier
        """
        fname=QFileDialog.getSaveFileName (None, QApplication.translate("pyacidobasic","Fichier pour enregistrer", None, QApplication.UnicodeUTF8)
, filter=QApplication.translate("pyacidobasic","Fichiers Acidobasic [*.acb] (*.acb);; Tous types de fichiers (*.* *)", None, QApplication.UnicodeUTF8))
        if fname:
            if not re.match(".*\.\S*$",fname):
                fname+=".acb" #on ajoute un suffixe par défaut
            self.filename=fname
            self.updateWindowTitle()
            self.save()

    def updateWindowTitle(self):
        """
        Met à jour le titre de la fenêtre
        """
        self.setWindowTitle(QString("Pyacidobasic : %1").arg(os.path.basename(u"%s" %self.filename)))
        
    def save(self):
        """
        Enregistre les données du graphique
        """
        if self.filename:
            p=pickle.Pickler(open(self.filename,"w"))
            p.dump(version.versionString)
            burette=self.ui.listWidgetBurette.dump()
            p.dump(burette)
            p.dump(self.ui.listWidgetBecher.dump())
        else:
            self.saveAs()

    def loadExample(self):
        """
        Propose l'ouverture d'un exemple
        """
        self.load(dir="/usr/share/pyacidobasic")

    def load(self, fname="", dir=""):
        """
        Récupère les données depuis un fichier
        """
        if not fname:
            fname=QFileDialog.getOpenFileName (None, QApplication.translate("pyacidobasic", "Fichier pour enregistrer", None, QApplication.UnicodeUTF8), filter=QApplication.translate("pyacidobasic", "Fichiers Acidobasic [*.acb] (*.acb);; Tous types de fichiers (*.* *)", None, QApplication.UnicodeUTF8), directory=dir)
        if fname:
            self.filename=fname
            p=pickle.Unpickler(open(self.filename,"r"))
            try:
                versionString=p.load()
                if versionString!=version.versionString:
                    raise EnvironmentError
                self.ui.listWidgetBurette.load(p.load())
                self.ui.listWidgetBecher.load(p.load())
                self.updateWindowTitle()
            except:
                QMessageBox.warning (None, QApplication.translate("pyacidobasic", "Erreur de version", None, QApplication.UnicodeUTF8), QApplication.translate("pyacidobasic", "Ce fichier n'est pas un fichier Pyacidobasic valide, de version %1.", None, QApplication.UnicodeUTF8).arg(version.version))

    def apropos(self):
        """
        Donne un message d'information au sujet de la licence de pyacidobasic
        """
        global locale
        if locale[:2]=="fr":
            l="fr"
        else:
            l="en"
        msg=licence[l] %version.version
        QMessageBox.information(None, QApplication.translate("pyacidobasic", "À propos", None, QApplication.UnicodeUTF8), msg)
            
    def configurePlot(self):
        """
        Choisit les axes et les gradue
        @param burette une référence à la burette quand celle-ci n'est pas vide
        @param becher une référence au bécher quand celui-ci n'est pas vide
        """
        burette=self.ui.listWidgetBurette
        becher=self.ui.listWidgetBecher
        maxi=self.maxConcentration()
        if maxi:
            yRange=(0,1.25*maxi)
        else:
            yRange=(0,0.1)
        self.setGrid(self.ui.qwtPlotpH,
                     yLeftRange=(0,14),
                     yRightRange=yRange)
        if self.showConcentrations:
            maxi=self.maxConcentration()
            if maxi:
                yRange=(0,1.25*maxi)
            else:
                yRange=(0,0.1)
            self.setGrid(self.ui.qwtConcentrationPlot,
                         yLeftRange=yRange,
                         yLeftTitre="Concentration (mol/L)",
                         yRightRange=(0,14),
                         yRightTitre="pH")
        else:
            maxi=self.maxQuantite()
            if maxi:
                yRange=(0,1.25*maxi)
            else:
                yRange=(0,1e-3)
            self.setGrid(self.ui.qwtConcentrationPlot,
                         yLeftRange=yRange,
                         yLeftTitre=u"Quantité (mol)",
                         yRightRange=(0,14),
                         yRightTitre="pH")
        self.replot()

    def setGrid(self, qwtp, titre=QApplication.translate("pyacidobasic", "Courbe du dosage", None, QApplication.UnicodeUTF8),
                yLeftRange=(0,14),
                yRightRange=None,
                xTitre="V (mL)",
                yLeftTitre="pH",
                yRightTitre="Concentration (mol/L)",
                ):
        """
        définit la grille et les axes
        @param qwtp un qwtPlotWidget
        @param le titre initial
        @param yLeftRange l'intervalle des valeurs sur l'axe Y gauche, (0,14) par défaut
        @param yRightRange l'intervalle des valeurs sur l'axe Y droit, None par défaut
        @param xTitre le label de l'axe X, "V (mL)" par défaut
        @param yLeftTitre le label de l'axe Y de gauche, "pH" par défaut
        @param yRightTitre le label de l'axe Y de droite, "Concentration (mol/L)" par défaut
        """
        qwtp.setAxisScale(QwtPlot.yLeft,yLeftRange[0],yLeftRange[1])
        qwtp.setAxisTitle(QwtPlot.xBottom,xTitre)
        qwtp.setAxisTitle(QwtPlot.yLeft,yLeftTitre)
        if yRightRange!=None:
            qwtp.enableAxis(QwtPlot.yRight)
            qwtp.setAxisScale(QwtPlot.yRight,yRightRange[0],yRightRange[1])
            qwtp.setAxisTitle(QwtPlot.yRight,yRightTitre)
        grid=QwtPlotGrid()
        p1=QPen(QColor(100,100,100,100)) # noir transparent 
        p1.setWidthF(2.5)
        p2=QPen(QColor(200,200,200,100)) # noir transparent 
        p1.setWidthF(0)
        grid.setMajPen(p1)
        grid.setMinPen(p2)
        grid.enableX(True)
        grid.enableY(True)
        grid.enableXMin(True)
        grid.enableYMin(True)
        grid.attach(qwtp)
        qwtp.setTitle(titre)
        qwtp.setAxisScale(QwtPlot.xBottom,0,self.maxBurette)
        
    def replot(self):
        for qwtp in self.plots:
            qwtp.setAxisScale(QwtPlot.xBottom,0,self.maxBurette)
        self.legend=QwtLegend()
        self.ui.qwtPlotpH.insertLegend(self.legend)
        for qwtp in self.plots:
            qwtp.replot()

    def videBurette(self):
        self.ui.listWidgetBurette.vide()
        self.simule()

    def videBecher(self):
        self.ui.listWidgetBecher.vide()
        self.simule()
       
    def plusV(self):
        if self.maxBurette < 5:
            self.maxBurette+=1
        else:
            self.maxBurette+=5
        self.replot()

    def moinsV(self):
        if self.maxBurette >= 10:
            self.maxBurette-=5
        elif self.maxBurette >=2:
            self.maxBurette-=1
        self.replot()

    def simule(self):
        self.delCourbesEtControles()
        if not self.simulationPossible():
            self.configurePlot()
            self.replot()
            return
        
        self.vData=[]; self.pHData=[]; self.derivData=[]
        self.maxpH=0.0
        for i in range (1401):
            pH=0.01*i # pH varie de 0 à 14
            try:
                v=-self.ui.listWidgetBecher.charge(pH)/self.ui.listWidgetBurette.charge(pH,1)
                if v>0:
                    self.vData.append(v)
                    self.pHData.append(pH)
                    if len(self.pHData)>1:
                        self.derivData.append(0.01/(self.vData[-1]-self.vData[-2]))
                    else:
                        self.derivData.append(0.0)
                    if pH > self.maxpH: self.maxpH=pH
            except ZeroDivisionError:
                pass
        self.courbesSysteme()
        self.ajouteSolutes()

        # essaie d'ajuster le volume de burette pour voir la partie intéressante
        try:
            # trouve l'abscisse pour avoir ph < maxpH - 0.5
            i=0
            while self.pHData[i] < self.maxpH - 0.5:
                i+=1
                v= self.vData[i]
            # ajuste le volume versé depuis la burette à 5mL près
            self.maxBurette=5*(1+int(v)/5)
        except:
            pass
        self.configurePlot()
        self.replot()

    def maxConcentration(self):
        """
        @return la concentration du plus concentré des réactifs
        """
        burette=self.ui.listWidgetBurette
        becher=self.ui.listWidgetBecher

        maxi = 0
        if len (burette.contenu)>0:
            maxi=burette.contenu[0].c
        for b in becher.contenu:
            if b.c > maxi :
                maxi=b.c
        if maxi <= 0:
            return None
        else:
            return maxi
        
    def maxQuantite(self):
        """
        @return la quantité du plus concentré des réactifs à considérer
        """
        burette=self.ui.listWidgetBurette
        becher=self.ui.listWidgetBecher

        maxi = 0
        if len (burette.contenu)>0:
            maxi=burette.contenu[0].c*20e-3 # pour 20 mL max de burette !
        for b in becher.contenu:
            if b.c > maxi :
                maxi=b.c*1e-3*b.v
        if maxi <= 0:
            return None
        else:
            return maxi
        
    def vTotal(self):
        """
        @return un vecteur de valeurs de volume total de la solution
        """
        becher=self.ui.listWidgetBecher
        totalBecher=0
        for c in becher.contenu:
            totalBecher += c.v
        return totalBecher*np.ones(len(self.pHData))+self.vData

    def ajouteSolutes(self):
        """
        Ajoute toutes les espèces présentes, dans le menu du graphique
        """
        burette=self.ui.listWidgetBurette
        becher=self.ui.listWidgetBecher

        c=burette.contenu[0]
        for i in range(len(c.formes)):
            yData=c.concentrationParPH(i,self.pHData,self.vTotal(), vBurette=self.vData)
            if self.showConcentrations==False: # montre les quantités
                yData *= 1e-3*self.vTotal()
            self.nouvelleCourbe(titre="[%s]" %c.formes[i],
                                yAxis=QwtPlot.yLeft,
                                yData=yData,
                                qwtp=self.ui.qwtConcentrationPlot,
                                cell=None
                                )
        
        for c in becher.contenu:
            for i in range(len(c.formes)):
                yData=c.concentrationParPH(i,self.pHData,self.vTotal())
                if self.showConcentrations == False: # montre les quantités
                    yData *= 1e-3*self.vTotal()
                self.nouvelleCourbe(titre="[%s]" %c.formes[i],
                                    yAxis=QwtPlot.yLeft,
                                    yData=yData,
                                    qwtp=self.ui.qwtConcentrationPlot,
                                    cell=None
                                    )
    def delCourbesEtControles(self):
        """
        Supprime les courbes et leurs contôles
        """
        s=self.ui.curveControlLayout
        #nettoyage des anciens contrôles
        child=s.takeAt(0)
        while child:
            child.widget().close()
            del child
            child=s.takeAt(0)
        #nettoyage des anciennes courbes
        for c in self.courbesActives:
            c.detach()
            del c

    def courbesSysteme(self):
        """
        Crée les courbes "système" : pH, dpH/dV, [H<sub>3</sub>O<sup>+</sup>], [HO<sup>-</sup>]
        """
        self.delCourbesEtControles()
        self.row=0
        cell=self.nouvelleCourbe(titre="pH",
                                 color="red",
                                 yAxis=QwtPlot.yLeft,
                                 yData=self.pHData,
                                 width=2,
                                 qwtp=self.ui.qwtPlotpH,
                                 enabled=False,
                                 checked=True,
                                 cell=None
                                 )
        cell=self.nouvelleCourbe(titre="pH",
                                 color=QColor(255,0,0,50),
                                 yAxis=QwtPlot.yRight,
                                 yData=self.pHData,
                                 width=2,
                                 qwtp=self.ui.qwtConcentrationPlot,
                                 checked=True,
                                 cell=cell
                              )
        d=np.array(self.derivData)
        if d.min()<0:
            d= 14.0+d # décalage en cas de dosage par un acide !
            titre="14+dpH/dV"
        else:
            titre="dpH/dV"
        cell=self.nouvelleCourbe(titre=titre,
                                 color="blue",
                                 yAxis=QwtPlot.yLeft,
                                 yData=d,
                                 cell=None,
                                 qwtp=self.ui.qwtPlotpH,
                                 )
        cell=self.nouvelleCourbe(titre=titre,
                                 color=QColor(0,0,255,50),
                                 yAxis=QwtPlot.yRight,
                                 yData=d,
                                 cell=cell,
                                 qwtp=self.ui.qwtConcentrationPlot,
                                 )
        H3OData=np.exp(- 2.302585093*np.array(self.pHData))
        cell=self.nouvelleCourbe(titre="[H<sub>3</sub>O<sup>+</sup>]",
                            color="green",
                            yData=H3OData,
                            cell=None,
                            qwtp=self.ui.qwtPlotpH,
                            )
        if self.showConcentrations :
            yData = H3OData
        else:
            yData = 1e-3*H3OData*self.vTotal()
        cell=self.nouvelleCourbe(titre="[H<sub>3</sub>O<sup>+</sup>]",
                            color="green",
                            yData=yData,
                            yAxis=QwtPlot.yLeft,
                            cell=cell,
                            qwtp=self.ui.qwtConcentrationPlot,
                            )
        HOData=1e-14/H3OData
        cell=self.nouvelleCourbe(titre="[HO<sup>-</sup>]",
                            color="magenta",
                            yData=HOData,
                            cell=None,
                            qwtp=self.ui.qwtPlotpH,
                            )
        if self.showConcentrations :
            yData = HOData
        else:
            yData = 1e-3*HOData*self.vTotal()
        cell=self.nouvelleCourbe(titre="[HO<sup>-</sup>]",
                            color="magenta",
                            yData=yData,
                            yAxis=QwtPlot.yLeft,
                            cell=cell,
                            qwtp=self.ui.qwtConcentrationPlot,
                            )
        

    def simulationPossible(self):
        """
        @return un boolean vrai si une simulation est possible
        """
        return len(self.ui.listWidgetBecher.contenu)>0 and len(self.ui.listWidgetBurette.contenu)>0

    def event(self, ev):
        if ev.type()==QEvent.User:
            self.simule()
            return True
        else:
            return QMainWindow.event(self,ev)

def run(argv):
    global locale
    app = QApplication(sys.argv)

    ###translation##
    locale = QLocale.system().name()
    qtTranslator = QTranslator()

    if qtTranslator.load("qt_" + locale):
        app.installTranslator(qtTranslator)
        
    appTranslator = QTranslator()
    if appTranslator.load("/usr/share/pyacidobasic/lang/pyacidobasic_" + locale):
        app.installTranslator(appTranslator)
    
    w = acidoBasicMainWindow(None,argv)
    w.show()
    sys.exit(app.exec_())
    

if __name__=='__main__':
    run(sys.argv)
