// for finding memory leaks in debug mode with Visual Studio 
#if defined _DEBUG && defined _MSC_VER
#include <crtdbg.h>
#endif

#include <stdio.h>
#include <stdlib.h>
#include <stdint.h>
#include <sys/stat.h>
#ifdef _WIN32
#define WIN32_MEAN_AND_LEAN
#include <windows.h>
#else
#include <unistd.h>
#endif
#include <SDL2/SDL.h>
#include "config.h"
#include "palette.h"
#include "gui.h"

#define CLAMP(x, low, high) (((x) > (high)) ? (high) : (((x) < (low)) ? (low) : (x)))

#ifdef _MSC_VER
#pragma warning(disable: 4204)
#pragma warning(disable: 4221)
#endif

volatile uint8_t programRunning, bigEndian, redrawScreen, allowSigterm = 1;
uint32_t *frameBuffer;

static SDL_Window *window;
static SDL_Renderer *renderer;
static SDL_Texture *texture;

static int8_t setupVideo(void);
static void readMouseXY(void);
static void handleInput(void);

#ifdef __APPLE__
static void osxSetDirToProgramDirFromArgs(char **argv);
#endif

int main(int argc, char *argv[])
{
    // for finding memory leaks in debug mode with Visual Studio
#if defined _DEBUG && defined _MSC_VER
    _CrtSetDbgFlag(_CRTDBG_ALLOC_MEM_DF | _CRTDBG_LEAK_CHECK_DF);
#endif

    // very first thing to do is to set a big endian flag using a well-known hack
    // DO *NOT* run this test later in the code, as some macros depend on the flag!
    union
    {
        uint32_t a;
        uint8_t b[4];
    } endianTest;

    endianTest.a = 1;
    bigEndian = endianTest.b[3];
    // ----------------------------

    (void)(argc);
    (void)(argv);

#ifdef _WIN32
    if (!IsProcessorFeaturePresent(PF_XMMI64_INSTRUCTIONS_AVAILABLE))
    {
        showErrorMsgBox("Your computer's processor is too old and doesn't have the SSE2 instruction set\n" \
                        "which is needed for this program to run. Sorry!");
        return (0);
    }
#endif

#ifdef __APPLE__
    osxSetDirToProgramDirFromArgs(argv);
#endif

    if (!setupVideo())
    {
        SDL_Quit();
        return (0);
    }

    setupGUI();
    
#ifdef __APPLE__
    /* amazingly dumb kludge to prevent window from being non-painted (black)
    ** on macOS Mojave. It will be fixed in SDL2.0.9 */

    SDL_PumpEvents();
    SDL_SetWindowSize(window, SCREEN_W * 2, SCREEN_H * 2);
#endif

    programRunning = 1;
    while (programRunning)
    {
        readMouseXY();
        handleInput();

        if (redrawScreen)
        {
            redrawScreen = 0;

            // redraw screen
            SDL_UpdateTexture(texture, NULL, frameBuffer, SCREEN_W * sizeof (int32_t));
            SDL_RenderClear(renderer);
            SDL_RenderCopy(renderer, texture, NULL, NULL);
            SDL_RenderPresent(renderer);
        }

        SDL_Delay(1000 / 50); // ~50Hz
    }

    if (loadedFile != NULL) free(loadedFile);

    free(frameBuffer);
    SDL_Quit();

    return (0);
}

static void readMouseXY(void)
{
    int32_t mx, my;

    SDL_PumpEvents();
    SDL_GetMouseState(&mx, &my);

    mx = (int32_t)(mx * input.mouse.fXScaleMul);
    my = (int32_t)(my * input.mouse.fYScaleMul);

    mx = CLAMP(mx, 0, SCREEN_W - 1);
    my = CLAMP(my, 0, SCREEN_H - 1);

    input.mouse.x = (int16_t)(mx);
    input.mouse.y = (int16_t)(my);
}

static void showAskToSaveDialog(void)
{
    int32_t buttonPressed;

    const SDL_MessageBoxButtonData buttons[2] =
    {
        { SDL_MESSAGEBOX_BUTTON_ESCAPEKEY_DEFAULT, 0, "No"  },
        { SDL_MESSAGEBOX_BUTTON_RETURNKEY_DEFAULT, 1, "Yes" },
    };

    const SDL_MessageBoxData messageBoxData =
    {
        SDL_MESSAGEBOX_WARNING, /* .flags */
        window, /* .window */
        "Warning", /* .title */
        "Colors are unsaved. Save before quitting?", /* .message */
        SDL_arraysize(buttons), /* .numbuttons */
        buttons, /* .buttons */
        NULL /* .colorScheme */
    };

    if (SDL_ShowMessageBox(&messageBoxData, &buttonPressed) < 0)
    {
        allowSigterm   = 1;
        programRunning = 0;
        return;
    }

    if (buttonPressed == 1)
        savePalette(0);

    programRunning = 0;
}

void showErrorMsgBox(const char *fmt, ...)
{
    char strBuf[1024];
    va_list args;

    // format the text string
    va_start(args, fmt);
    vsnprintf(strBuf, sizeof (strBuf), fmt, args);
    va_end(args);

    // window can be NULL here, no problem...
    SDL_ShowSimpleMessageBox(SDL_MESSAGEBOX_ERROR, "Error", strBuf, NULL);
}

static void handleInput(void)
{
    SDL_Event inputEvent;

    while (SDL_PollEvent(&inputEvent))
    {
        if ((inputEvent.type == SDL_QUIT) && allowSigterm)
        {
            if (!configIsSaved)
            {
                allowSigterm = 0;
                showAskToSaveDialog();
            }
            else
            {
                programRunning = 0;
            }
        }
        else if (inputEvent.type == SDL_KEYDOWN)
        {
            keyDownHandler(inputEvent.key.keysym.sym);
        }
        else if (inputEvent.type == SDL_MOUSEBUTTONUP)
        {
            if (inputEvent.button.button == SDL_BUTTON_LEFT)
                mouseButtonUpHandler();
        }
        else if (inputEvent.type == SDL_MOUSEBUTTONDOWN)
        {
            if (inputEvent.button.button == SDL_BUTTON_LEFT)
                mouseButtonDownHandler();
        }
    }

    handleSlidersHeldDown();
    handleRainbowHeldDown();
    handleRainbowUpDownButtons();
}

// macOS/OS X specific routines
#ifdef __APPLE__
static void osxSetDirToProgramDirFromArgs(char **argv)
{
    char *tmpPath;
    int32_t i, tmpPathLen;

    /* OS X/macOS: hackish way of setting the current working directory to the place where we double clicked
    ** on the icon (for protracker.ini loading)
    */

    // if we launched from the terminal, argv[0][0] would be '.'
    if ((argv[0] != NULL) && (argv[0][0] == '/')) // don't do the hack if we launched from the terminal
    {
        tmpPath = strdup(argv[0]);
        if (tmpPath != NULL)
        {
            // cut off program filename
            tmpPathLen = strlen(tmpPath);
            for (i = tmpPathLen - 1; i >= 0; --i)
            {
                if (tmpPath[i] == '/')
                {
                    tmpPath[i] = '\0';
                    break;
                }
            }

            chdir(tmpPath);     // path to binary
            chdir("../../../"); // we should now be in the directory where the config can be.

            free(tmpPath);
        }
    }
}
#endif

static int8_t setupVideo(void)
{
    int32_t w, h;

    if (SDL_Init(SDL_INIT_VIDEO) < 0)
    {
        showErrorMsgBox("Couldn't initialize SDL: %s", SDL_GetError());
        return (0);
    }

    window = SDL_CreateWindow("Palette editor for ProTracker 2.3D clone",
        SDL_WINDOWPOS_CENTERED, SDL_WINDOWPOS_CENTERED, SCREEN_W * 2, SCREEN_H * 2, SDL_WINDOW_SHOWN | SDL_WINDOW_ALLOW_HIGHDPI);

    if (window == NULL)
    {
        showErrorMsgBox("Couldn't create an SDL2 window: %s", SDL_GetError());
        return (0);
    }

    renderer = SDL_CreateRenderer(window, -1, SDL_RENDERER_ACCELERATED);
    if (renderer == NULL)
    {
        showErrorMsgBox("Couldn't create an SDL2 renderer: %s", SDL_GetError());
        return (0);
    }

    SDL_RenderSetLogicalSize(renderer, SCREEN_W, SCREEN_H);

#if SDL_PATCHLEVEL >= 5
    SDL_RenderSetIntegerScale(renderer, SDL_TRUE);
#endif

    SDL_SetRenderDrawBlendMode(renderer, SDL_BLENDMODE_NONE);

    SDL_SetHint("SDL_RENDER_SCALE_QUALITY", "nearest");

    texture = SDL_CreateTexture(renderer, SDL_PIXELFORMAT_ARGB8888, SDL_TEXTUREACCESS_STREAMING, SCREEN_W, SCREEN_H);
    if (texture == NULL)
    {
        showErrorMsgBox("Couldn't create an SDL2 texture: %s", SDL_GetError());
        return (0);
    }

    SDL_SetTextureBlendMode(texture, SDL_BLENDMODE_NONE);

    // frame buffer used by SDL (for texture)
    frameBuffer = (uint32_t *)(malloc(SCREEN_W * SCREEN_H * sizeof (int32_t)));
    if (frameBuffer == NULL)
    {
        showErrorMsgBox("Out of memory!");
        return (0);
    }

    SDL_GetWindowSize(window, &w, &h);

    input.mouse.fXScaleMul = 1.0f / (w / (float)(SCREEN_W));
    input.mouse.fYScaleMul = 1.0f / (h / (float)(SCREEN_H));

    return (1);
}
