#include "policyd.h"


/*
 *
 *
 *                           Policy Daemon
 *
 *  policy daemon is used in conjuction with postfix to combat spam.
 *
 *  Copyright (C) 2004 Cami Sardinha (cami@mweb.co.za)
 *
 *
 *  This program is free software; you can redistribute it and/or modify it
 *  under the terms of the  GNU General  Public License as published by the
 *  Free Software Foundation;  either version 2 of the License, or (at your
 *  option) any later version.
 *
 *  This program  is  distributed  in the hope that  it will be useful, but
 *  WITHOUT  WARRANTY; without even the implied warranty of MERCHANTABILITY
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License
 *  for more details.
 *
 *  You should have received a copy of the GNU General Public License along
 *  with this program; if not, write to the Free  Software Foundation Inc.,
 *  59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 *
 *
 */

/*
 * function: w_mysql_query
 *  purpose: wrapper function for mysql_query()
 *   return: 0=success, -1=failure
 */
int
w_mysql_query(int volatile fd, const char *function)
{
  /* increase mysql query counter */
  mysql_count++;
  timenow=gettime();

  /* catch query timeouts */
  if (sigsetjmp (sjmp, 1))
  {
    if(DEBUG > 2)
      logmessage("%s()/mysql_query(): mysql_query timeout number: %d\n",
	  function, mysql_failure_count);
    
    mysql_failure_count++;
    last_mysql_failure=gettime();

    mysql_array[fd][0] = -1;
    return (-1); /* failure */
  }

  /* set the SIGALRM handler */
  signal (SIGALRM, (void *) sigalrm_handler);
  
  /* mysql timeout */
  alarm(5);
  
  /* cut down latency, do not do any queries for 5 minutes if 3 failures occur */
  if(mysql_failure_count >= 3)
  {
    /* has 300 seconds passed? */
    if(timenow >= (last_mysql_failure+300))
    {
                           /* timer has expired */
      mysql_failure_count=0;
    } else {               /* timer has not expired */

      /* reset the timer & unset the signal handler */
      alarm (0);     signal (SIGALRM, SIG_DFL);
      return (-1);
    }
  }
    
  /* fire off query */
  if (mysql_query(mysql, mysqlquery_array[fd]) != 0)
    return (-1);
  
  /* reset the timer */
  alarm (0);

  /* unset the signal handler */
  signal (SIGALRM, SIG_DFL);

  return (0); /* success */
}




/*
 * function: db_doquery
 *  purpose: do mysql queries
 *   return: 0=success, -1=failure
 */
int
db_doquery(int volatile fd)
{
  
  if(DEBUG > 1)
    logmessage("DEBUG: fd: %d query=%ld, db_doquery(): %s\n", fd, mysql_count, mysqlquery_array[fd]);

  /* fire off query */
  if (w_mysql_query(fd, "db_doquery") != 0)
    goto err;

  /* select() result */
  if (mysql_field_count(mysql) > 0)
  {
    int num_fields, i=0;
    MYSQL_RES   *res;
    MYSQL_ROW    row, end_row;

    if (!(res = mysql_store_result(mysql)))
      goto err;
    
    num_fields = mysql_num_fields(res);
    while ((row = mysql_fetch_row(res)))
    {
      for (end_row=row+num_fields;row<end_row;++row,i++)
      {
	if(DEBUG > 1)
	  logmessage("DEBUG: fd: %d row: %d data: %d (recieved)\n", fd, i, atol((char *)*row));
	
	mysql_array[fd][i]=atol(row ? (char*)*row : "0"); /* return seconds from epoch */

	if(DEBUG > 1)
	  logmessage("DEBUG: fd: %d row: %d data: %d (extracted)\n", fd, i, mysql_array[fd][i]);
      }
    }
    mysql_free_result(res);
  } else {
    
    /* update() result */
    mysql_array[fd][0]=mysql_affected_rows(mysql);
  }


  return (0); /* success */
  
err:
  
  if(DEBUG > 1)
    logmessage("db_doquery()/mysql_query(): %s -> %s\n", mysql_error(mysql), mysqlquery_array[fd]);

  mysql_array[fd][0] = -1;
  return (-1); /* failure */
}




/*
 * function: db_charquery
 *  purpose: do mysql query
 *   return: 0=success, -1=failure
 *   return: multi-dimensional char array
 */
int
db_charquery(int volatile fd)
{      

  if(DEBUG > 1)
    logmessage("DEBUG: fd: %d query=%ld, db_charquery(): %s\n", fd, mysql_count, mysqlquery_array[fd]);

  /* fire off query */
  if (w_mysql_query(fd, "db_charquery") != 0)
    goto err;
 
  /* select() result */
  if (mysql_field_count(mysql) > 0)  
  {
    int num_fields, i=0;
    MYSQL_RES   *res;
    MYSQL_ROW    row, end_row;
  
    if (!(res = mysql_store_result(mysql)))
      goto err;
    
    num_fields = mysql_num_fields(res);
    while ((row = mysql_fetch_row(res)))
    {
      for (end_row=row+num_fields;row<end_row;++row)
      {
	if(DEBUG > 1)
	  logmessage("DEBUG: fd: %d row: %d data: %d (recieved)\n", fd, i, (char *)*row);
	
        strncpy(mysqlchar_array[fd][i], (char*)*row, 256);

	if(DEBUG > 1)
          logmessage("DEBUG: fd: %d row: %d data: %s (extracted)\n", fd, i, mysqlchar_array[fd][i]);
	i++;
      }
    }
    mysql_free_result(res);
  }

  return (0); /* success */
  
err:

  if(DEBUG > 1)
    logmessage("db_charquery()/mysql_query(): %s -> %s\n", mysql_error(mysql), mysqlquery_array[fd]);
  
  mysql_array[fd][0] = -1;
  return (-1); /* failure */
}




/*
 * function: db_optquery
 *  purpose: do mysql opt in/out queries
 *   return: 0=success, -1=failure
 */
int
db_optquery(int volatile fd)
{
  
  if(DEBUG > 1)
    logmessage("DEBUG: fd: %d query=%ld, db_optquery(): %s\n", fd, mysql_count, mysqlquery_array[fd]);

  /* fire off query */
  if (w_mysql_query(fd, "db_optquery") != 0)
    goto err;


  /* select() result */
  if (mysql_field_count(mysql) > 0)
  {
    int num_fields, i=0;
    MYSQL_RES   *res;
    MYSQL_ROW    row, end_row;

    if (!(res = mysql_store_result(mysql)))
      goto err;

    num_fields = mysql_num_fields(res);
    while ((row = mysql_fetch_row(res)))
    {
      for (end_row=row+num_fields;row<end_row;++row,i++)
      {
	if(DEBUG > 1)
  	  logmessage("DEBUG: fd: %d row: %d data: %d (recieved)\n", fd, i, atoi((char *)*row));
	
	mysql_optarray[fd][0]=atol(row ? (char*)*row : "0");

	if(DEBUG > 1)
	  logmessage("DEBUG: fd: %d row: %d data: %d (extracted)\n", fd, i, mysql_optarray[fd][0]);
      }
    }
    mysql_free_result(res);
  }

  return (0); /* success */
  
err:
  
  if(DEBUG > 1)
    logmessage("db_optquery()/mysql_query(): %s -> %s\n", mysql_error(mysql), mysqlquery_array[fd]);
  
  mysql_array[fd][0] = -1;
  return (-1); /* failure */
}




/*
 * function: db_deletequery
 *  purpose: expire triplet in database
 *   return: 0=success, -1=failure
 */
int
db_deletequery(int volatile fd)
{
  count=0;

start:
  
  if(DEBUG > 1)
    logmessage("DEBUG: fd: %d query=%ld, db_deletequery(): %s\n", fd, mysql_count, mysqlquery_array[fd]);

  /* fire off query */
  if (w_mysql_query(fd, "db_deletequery") != 0)
    goto err;

  /* select() result */
  if (mysql_field_count(mysql) > 0)
  {
    int num_fields, i=0;
    MYSQL_RES   *res;
    MYSQL_ROW    row, end_row;

    if (!(res = mysql_store_result(mysql)))
      goto err;

    num_fields = mysql_num_fields(res);
    while ((row = mysql_fetch_row(res)))
    {
      for (end_row=row+num_fields;row<end_row;++row,i++)
      {
        logmessage("%ld\n", atol(row ? (char*)*row : "0"));
      }
    }
    mysql_free_result(res);
  } else {
    /* MySQL does not handle extremely large deletes very well */
    if((int)mysql_affected_rows(mysql) == 100000)
    {
      count=count+(int)mysql_affected_rows(mysql);
      goto start;
    }
    
    /* didnt reach 100000 deletes */
    count=count+(int)mysql_affected_rows(mysql);
    
  }
  
  logmessage("expired: %d records\n", count);

  return (0); /* success */
  
err:

  if(count != 0)
    logmessage("expired: %d records\n", count);

  if(DEBUG > 1)
    logmessage("db_deletequery()/mysql_query(): %s -> %s\n", mysql_error(mysql), mysqlquery_array[fd]);
  
  mysql_array[fd][0] = -1;
  return (-1); /* failure */
}




/*
 * function: db_connect
 *  purpose: open connection to mysql database
 *   return: MySQL structure
 */
MYSQL *db_connect(const char *dbname)
{
  /* initializes mysql structure */
  
  /*
   * volatile:
   *
   * It's a potential problem, because Sun doesn't guarrantee that automatic
   * variables are preserved when the saved function's context is restored.
   *
   *                                                  Leandro Santi
   */
  MYSQL * volatile db=mysql_init(NULL);
  if(!db)
  {
    logmessage("mysql_init(): no memory\n");
    exit(-1);
  }
  logmessage("connecting to mysql database: %s\n", MYSQLHOST);

  /* catch connection timeouts */
  if (sigsetjmp (sjmp, 1))
  {
    if(DEBUG > 1)
      logmessage("db_connect()/mysql_real_connect(): connection timeout\n");

    /* if the connection fails, that counts as a hard failure */
    mysql_failure_count++;
    last_mysql_failure=gettime();
    
    logmessage("NOT connected..\n");
    return (db); /* failure */
  }

  /* set the SIGALRM handler */
  signal (SIGALRM, (void *) sigalrm_handler);
  
  /* mysql timeout */
  alarm(8);
  
  /* fire off query */
  /* connect to mysql server */
  if(!mysql_real_connect(db, MYSQLHOST, MYSQLUSER, MYSQLPASS, dbname, MYSQLPORT, NULL, 0))
  {
    logmessage("mysql_real_connect(): %s\n", mysql_error(db));
    logmessage("NOT connected..\n");
  } else {
    logmessage("connected..\n");
  }

  /* reset the timer & unset the signal handler */
  alarm (0);           signal (SIGALRM, SIG_DFL);

  return (db);
}




/*
 * function: database_probe
 *  purpose: check to see if connection to the database is open
 *   return: 0=success, -20=failure
 */
int
database_probe(int fd)
{
  if(DEBUG > 1)
    logmessage("DEBUG: fd: %d database_probe(): mysql_ping\n", fd);

  /* catch query timeouts */
  if (sigsetjmp (sjmp, 1))
  {
    if(DEBUG > 2)
      logmessage("db_doquery()/mysql_query(): mysql_query timeout -> %s\n", mysqlquery_array[fd]);

    /* if the probe fails, that counts as a hard failure */
    mysql_failure_count++;
    last_mysql_failure=gettime();

    return (-20); /* failure */
  }
  
  /* set the SIGALRM handler */
  signal (SIGALRM, (void *) sigalrm_handler);

  /* mysql timeout */
  alarm (4);

  /* ping the mysql server, if connection is down, try and reconnect */
  if(mysql_ping(mysql) != 0)
    if(DEBUG > 1)
      logmessage("DEBUG: fd: %d database_probe(): reconnecting..\n", fd);
  
  /* reset the timer & unset the signal handler */
  alarm (0);
  signal (SIGALRM, SIG_DFL);

  return (0);
}

/* EOF */
