#include <unistd.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <errno.h>
#include <sys/mman.h>
#include <getopt.h>
#include <selinux/selinux.h>
#include <sepol/sepol.h>
#ifdef USE_NLS
#include <locale.h>			    /* for setlocale() */
#include <libintl.h>			    /* for gettext() */
#define _(msgid) gettext (msgid)
#else
#define _(msgid) (msgid)
#endif
#ifndef PACKAGE
#define PACKAGE "policycoreutils"   /* the name of this package lang translation */
#endif

void usage(char *progname) 
{
	fprintf(stderr, _("usage:  %s [-q] [-b] policyfile [booleanfile]\n"), progname);
	exit(1);
}

int main(int argc, char **argv) 
{
	int fd, ret, opt, quiet=0, setbools = 0, *values;
	int i, len;
	size_t data_size;
	struct stat sb;
	void *map, *data;
	char *polpath, *boolpath = NULL, **names;

#ifdef USE_NLS
	setlocale (LC_ALL, "");
	bindtextdomain (PACKAGE, LOCALEDIR);
	textdomain (PACKAGE);
#endif

	while ((opt = getopt(argc, argv, "qb")) > 0) {
		switch (opt) {
		case 'b':
			setbools = 1;
			break;
		case 'q':
			quiet = 1;
			sepol_debug(0);
			break;
		default:
			usage(argv[0]);
		}
	}

	if ((argc-optind) != 1 && (argc-optind) != 2) {
		usage(argv[0]);
	}

	if ((argc - optind) == 2) {
		/* Implicit enable when they specify a boolean file */
		setbools = 1;
	}

	polpath = argv[optind++];
	fd = open(polpath, O_RDONLY);
	if (fd < 0) {
		fprintf(stderr, _("Can't open '%s':  %s\n"),
			polpath, strerror(errno));
		exit(2);
	}

	if (fstat(fd, &sb) < 0) {
		fprintf(stderr, _("Can't stat '%s':  %s\n"),
			polpath, strerror(errno));
		exit(2);
	}

	map = mmap(NULL, sb.st_size, PROT_READ | PROT_WRITE, MAP_PRIVATE, fd, 0);
	if (map == MAP_FAILED) {
		fprintf(stderr, _("Can't map '%s':  %s\n"),
			polpath, strerror(errno));
		exit(2);
	}

	ret = sepol_genusers(map, sb.st_size, selinux_users_path(), &data, &data_size);
	if (ret < 0) {
		/* No users file; non-fatal. */
		if (! quiet) 
			fprintf(stderr, _("%s:  Error while setting user configuration from %s/{local.users,system.users}:  %s\n"), argv[0], selinux_users_path(), strerror(errno));
		data = map;
		data_size = sb.st_size;
	}

	if (setbools) {
		/* Set booleans based on a booleans configuration file. */
		boolpath = (optind < argc) ? argv[optind++] : (char*) selinux_booleans_path();
		ret = sepol_genbools(data, data_size, boolpath);
		if (ret < 0) {
			if (errno == ENOENT || errno == EINVAL) {
				/* No booleans file or stale booleans in the file; non-fatal. */
				if (! quiet) 
					fprintf(stderr, _("%s:  Warning!  Error while setting booleans from %s:  %s\n"), argv[0], boolpath, strerror(errno));
			} else {
				fprintf(stderr, _("%s:  Error while setting booleans from %s:  %s\n"), argv[0], boolpath, strerror(errno));
				exit(2);
			}
		}
	} else {
		/* Preserve current boolean values. */
		ret = security_get_boolean_names(&names, &len);
		if (ret) {
			/* Possibly ok, as there may be no booleans. */
			if (! quiet) 
				fprintf(stderr, _("%s:  Warning!  Error while getting boolean names:  %s\n"), argv[0], strerror(errno));
			goto load;
		}
		if (!len)
			goto load;
		values = malloc(sizeof(int)*len);
		if (!values) {
			fprintf(stderr, _("%s:  out of memory\n"), argv[0]);
			exit(2);
		}
		for (i = 0; i < len; i++) {
			values[i] = security_get_boolean_active(names[i]);
			if (values[i] < 0) {
				fprintf(stderr, _("%s:  Error while getting value for boolean %s:  %s\n"), argv[0], names[i], strerror(errno));
				exit(2);
			}
		}
		ret = sepol_genbools_array(data, data_size, names, values, len);
		if (ret < 0) {
			if (errno == EINVAL) {
				/* Stale booleans in the file; non-fatal. */
				if (! quiet) 
					fprintf(stderr, _("%s:  Warning!  Error while setting booleans:  %s\n"), argv[0], strerror(errno));
			} else {
				fprintf(stderr, _("%s:  Error while setting booleans:  %s\n"), argv[0], strerror(errno));
				exit(2);
			}
		}
	}
load:	
	ret = security_load_policy(data, data_size);
	if (ret < 0) {
		fprintf(stderr, _("%s:  security_load_policy failed\n"), argv[0]);
		exit(3);
	}
	
	exit(0);
}
