#include "stock_quote_engine.h"

#include <QDateTime>
#include <QHttp>

//------------------------------------------------------------------------------
// public:
//------------------------------------------------------------------------------
StockQuoteEngine::StockQuoteEngine(QObject* parent, const QVariantList& args)
    :
    Plasma::DataEngine(parent, args),
    m_http(0),
    m_timer(),
    m_add_timer()
{
    Q_UNUSED(args);

    m_http = new QHttp("download.finance.yahoo.com");

//    setMinimumPollingInterval(30 * 1000);
//    setPollingInterval(5 * 60 * 1000);

    connect(&m_timer,     SIGNAL(timeout()), this, SLOT(update()));

    connect(&m_add_timer, SIGNAL(timeout()), this, SLOT(update()));

    connect(m_http, SIGNAL(requestFinished(int, bool)),
            this,   SLOT  (requestFinished(int, bool)));

    // set a default update interval
    m_timer.setInterval(60 * 1000);

    // setup a single shot addition timer of 1/2 second.
    m_add_timer.setSingleShot(true);
    m_add_timer.setInterval(500);
    
    // create the status source
    setData("status", DataEngine::Data());
}
    
//------------------------------------------------------------------------------
StockQuoteEngine::~StockQuoteEngine()
{
    m_http->deleteLater();
}
    
//------------------------------------------------------------------------------
// protected:
//------------------------------------------------------------------------------
bool
StockQuoteEngine::sourceRequestEvent(const QString& name)
{
    if ("status" != name)
    {
        setData(name, DataEngine::Data());

        // restart the main timer, and trigger the add timer
        m_timer.start();
        m_add_timer.start();
    }

    return true;
}

//------------------------------------------------------------------------------
// private slots:
//------------------------------------------------------------------------------
void
StockQuoteEngine::update()
{
    const QString stocks = sources().join(",");

    if (!stocks.isEmpty())
    {
        m_http->get("/d/quotes.csv?f=snl1c1&e=.csv&s=" + stocks);
    }
}

//------------------------------------------------------------------------------
void
StockQuoteEngine::requestFinished(int id, bool error)
{
    Q_UNUSED(id);

    if (error)
    {
        setData("status", "valid", false);
        return;
    }

    setData("status", "time", QDateTime::currentDateTime());
    setData("status", "valid", true);

    QString     blob  = m_http->readAll();
    QStringList lines = blob.split("\r\n");

    for (QStringList::iterator iter = lines.begin(); iter != lines.end(); ++iter)
    {
        // if there is a commar in a quoted string, then we drop the commar as
        // it breaks the tokeniser
        *iter = fixQuotes(*iter);

        QStringList fields = iter->split(",");

        if (fields.size() != 4) continue;

        fields[0].remove(0, 1).chop(1);
        fields[1].remove(0, 1).chop(1);

        const QString symbol = fields[0];
        const QString name   = fields[1];
        const double  price  = fields[2].toDouble();
        const double  change = fields[3].toDouble();

        setData(symbol, "valid",          true);
        setData(symbol, "symbol",         symbol);
        setData(symbol, "name",           name);
        setData(symbol, "price",          price);
        setData(symbol, "change",         change);
        setData(symbol, "percent_change", 100 * change / (price - change)); 
    }
}
    
//------------------------------------------------------------------------------
// private:
//------------------------------------------------------------------------------
QString
StockQuoteEngine::fixQuotes(const QString& string)
{
    QString ret;
    bool quote = false;

    for (QString::const_iterator iter = string.begin(); iter != string.end(); ++iter)
    {
        if ('"' == *iter) quote = !quote;
        if (',' == *iter && true == quote) continue;

        ret += *iter;
    }

    return ret;
}

K_EXPORT_PLASMA_DATAENGINE(stockquote, StockQuoteEngine);

#include "stock_quote_engine.moc"

