/***********************************************************************************
* Fancy Tasks: Plasmoid providing a fancy representation of your tasks and launchers.
* Copyright (C) 2009 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "FancyTasksPreview.h"
#include "FancyTasksTask.h"
#include "FancyTasksLauncher.h"

#include <QPainter>

#include <Plasma/Theme>
#include <Plasma/Animator>
#include <Plasma/PaintUtils>

FancyTasksPreview::FancyTasksPreview(const QPixmap &icon, const QString &text, QWidget *parent) : QWidget(parent),
    m_task(NULL),
    m_launcher(NULL)
{
    init(QPixmap(), icon, text, 0);
}

FancyTasksPreview::FancyTasksPreview(const QPixmap &pixmap, const QPixmap &icon, const QString &text, FancyTasksTask *task, QWidget *parent) : QWidget(parent),
    m_task(task),
    m_launcher(NULL)
{
    init(pixmap, icon, text, 0);
}

FancyTasksPreview::FancyTasksPreview(const QPixmap &pixmap, const QPixmap &icon, const QString &text, int offset, FancyTasksTask *task, QWidget *parent) : QWidget(parent),
    m_task(task),
    m_launcher(NULL)
{
    init(pixmap, icon, text, offset);
}

FancyTasksPreview::FancyTasksPreview(const QPixmap &pixmap, const QPixmap &icon, const QString &text, int offset, FancyTasksLauncher *launcher, QWidget *parent) : QWidget(parent),
    m_task(NULL),
    m_launcher(launcher)
{
    init(pixmap, icon, text, offset);
}

void FancyTasksPreview::init(const QPixmap &pixmap, const QPixmap &icon, const QString &text, int offset)
{
    setAcceptDrops(true);

    m_pixmap = pixmap;

    m_icon = icon;

    m_offset = offset;

    m_animationId = -1;

    m_opacity = 0;

    m_highlightTimer = 0;

    if (!m_pixmap.isNull())
    {
        m_background = new Plasma::FrameSvg(this);
        m_background->setImagePath("widgets/button");
        m_background->setElementPrefix("normal");
        m_background->setCacheAllRenderedFrames(true);

        qreal left, top, right, bottom;

        m_background->getMargins(left, top, right, bottom);

        m_previewSize = QSize((m_pixmap.width() + left + right), (m_pixmap.height() + top + bottom));

        m_background->setElementPrefix("active");
        m_background->resizeFrame(m_previewSize);
        m_background->setElementPrefix("normal");
        m_background->resizeFrame(m_previewSize);

        connect(m_background, SIGNAL(repaintNeeded()), this, SLOT(updateTheme()));
    }
    else
    {
        m_previewSize = QSize(0, 0);
    }

    m_document = new QTextDocument(this);

    setText(text);

    updateTheme();

    connect(this, SIGNAL(startWindowsHighlight(QList<WId>)), parentWidget(), SIGNAL(startWindowsHighlight(QList<WId>)));
    connect(this, SIGNAL(stopWindowsHighlight(QList<WId>)), parentWidget(), SIGNAL(stopWindowsHighlight(QList<WId>)));
}

void FancyTasksPreview::paintEvent(QPaintEvent *event)
{
    Q_UNUSED(event)

    QPainter painter(this);
    painter.setRenderHints(QPainter::SmoothPixmapTransform | QPainter::Antialiasing | QPainter::TextAntialiasing);

    if (!m_pixmap.isNull())
    {
        qreal left, top, right, bottom;

        m_background->setElementPrefix("normal");
        m_background->getMargins(left, top, right, bottom);

        QPointF point = QPointF(((size().width() - m_previewSize.width()) / 2), m_offset);

        if (m_animationId != -1)
        {
            QPixmap normalPixmap = m_background->framePixmap();

            m_background->setElementPrefix("active");

            painter.drawPixmap(point, Plasma::PaintUtils::transition(m_background->framePixmap(), normalPixmap, (1 - m_opacity)));
        }
        else
        {
            if (underMouse())
            {
                m_background->setElementPrefix("active");
            }

            m_background->paintFrame(&painter, point);
        }

        painter.drawPixmap((((size().width() - m_previewSize.width()) / 2) + left), (top + m_offset), m_pixmap);
    }

    painter.drawPixmap(0, (m_previewSize.height() + m_offset + ((size().height() - m_offset - m_previewSize.height() - m_icon.height()) / 2)), m_icon);

    QPixmap description(m_document->size().toSize());
    description.fill(Qt::transparent);

    QPainter descriptionPainter(&description);
    descriptionPainter.setRenderHints(QPainter::SmoothPixmapTransform | QPainter::Antialiasing | QPainter::TextAntialiasing);

    m_document->drawContents(&descriptionPainter);

    descriptionPainter.end();

    painter.drawPixmap(m_icon.width(), (m_previewSize.height() + m_offset), description);
}

void FancyTasksPreview::enterEvent(QEvent *event)
{
    const int FadeInDuration = 75;

    if (m_animationId != -1)
    {
        Plasma::Animator::self()->stopCustomAnimation(m_animationId);
    }

    m_animationId = Plasma::Animator::self()->customAnimation(40 / (1000 / FadeInDuration), FadeInDuration, Plasma::Animator::LinearCurve, this, "updateAnimation");

    if (m_task)
    {
        m_highlightTimer = startTimer(250);
    }

    QWidget::enterEvent(event);
}

void FancyTasksPreview::leaveEvent(QEvent *event)
{
    const int FadeOutDuration = 150;

    killTimer(m_highlightTimer);

    if (m_animationId != -1)
    {
        Plasma::Animator::self()->stopCustomAnimation(m_animationId != -1);
    }

    m_fadeIn = false;

    m_animationId = Plasma::Animator::self()->customAnimation(40 / (1000 / FadeOutDuration), FadeOutDuration, Plasma::Animator::LinearCurve, this, "updateAnimation");

    if (m_task)
    {
        QList<WId> windows;
        windows.append(parentWidget()->winId());
        windows.append(m_task->windows());

        emit stopWindowsHighlight(windows);
    }

    QWidget::leaveEvent(event);
}

void FancyTasksPreview::dragEnterEvent(QDragEnterEvent *event)
{
    if (m_task)
    {
        m_task->activateWindow();
    }

    event->ignore();
}

void FancyTasksPreview::mousePressEvent(QMouseEvent *event)
{
    event->ignore();

    m_opacity = 0.5;

    if (event->button() == Qt::LeftButton)
    {
        if (event->modifiers() & Qt::ShiftModifier && m_task)
        {
            m_task->close();
        }
        else
        {
            if (m_task)
            {
                m_task->activate();
            }
            else if (m_launcher)
            {
                m_launcher->activate();
            }
        }
    }
    else if (event->button() == Qt::MidButton && m_launcher)
    {
        m_launcher->activate();
    }
}

void FancyTasksPreview::contextMenuEvent(QContextMenuEvent *event)
{
    if (m_task)
    {
        KMenu *taskMenu = m_task->contextMenu();
        taskMenu->addTitle(m_task->icon(), m_task->title().left(20), taskMenu->actions().at(0));
        taskMenu->addActions(taskMenu->actions());
        taskMenu->exec(mapToGlobal(event->pos()));

        delete taskMenu;
    }
    else if (m_launcher)
    {
        KMenu *launcherMenu = m_launcher->contextMenu();
        launcherMenu->addTitle(m_launcher->icon(), m_launcher->title().left(20), launcherMenu->actions().at(0));
        launcherMenu->exec(mapToGlobal(event->pos()));

        delete launcherMenu;
    }
}

void FancyTasksPreview::timerEvent(QTimerEvent *event)
{
    if (event->timerId() == m_highlightTimer && m_task)
    {
        QList<WId> windows;
        windows.append(parentWidget()->winId());
        windows.append(m_task->windows());

        emit startWindowsHighlight(windows);
    }

    killTimer(event->timerId());
}

void FancyTasksPreview::setText(const QString &text)
{
    m_document->setHtml(text);
    m_document->setTextWidth((m_pixmap.width() < 100)?100:m_pixmap.width());
    m_document->adjustSize();

    setFixedSize(QSize((((m_document->size().width() + m_icon.width()) > m_previewSize.width())?(m_document->size().width() + m_icon.width()):m_previewSize.width()), (m_offset + m_previewSize.height() + ((m_document->size().height() > m_icon.height())?m_document->size().height():m_icon.height()))));

    update();
}

void FancyTasksPreview::updateAnimation(qreal progress)
{
    if (progress == 1)
    {
        m_animationId = -1;

        m_fadeIn = true;
    }

    m_opacity = (m_fadeIn?progress:(1 - progress));

    update();
}

void FancyTasksPreview::updateTheme()
{
    if (!m_pixmap.isNull())
    {
        m_background->clearCache();
    }

    QPalette plasmaPalette = QPalette();
    plasmaPalette.setColor(QPalette::Window, Plasma::Theme::defaultTheme()->color(Plasma::Theme::BackgroundColor));
    plasmaPalette.setColor(QPalette::WindowText, Plasma::Theme::defaultTheme()->color(Plasma::Theme::TextColor));

    m_document->setDefaultStyleSheet(QString("p { color: %1; }").arg(Plasma::Theme::defaultTheme()->color(Plasma::Theme::TextColor).name()));
    m_document->setHtml(m_document->toHtml());

    update();
}

FancyTasksTask* FancyTasksPreview::task()
{
    return m_task;
}

Plasma::FrameSvg* FancyTasksPreview::background()
{
    return m_background;
}

QSize FancyTasksPreview::previewSize()
{
    return m_previewSize;
}

int FancyTasksPreview::offset()
{
    return m_offset;
}
