/***********************************************************************************
* Adjustable Clock: Plasmoid to show date and time in adjustable format.
* Copyright (C) 2008 - 2009 Michal Dutkiewicz aka Emdek <emdeck@gmail.com>
*
* This program is free software; you can redistribute it and/or
* modify it under the terms of the GNU General Public License
* as published by the Free Software Foundation; either version 2
* of the License, or (at your option) any later version.
*
* This program is distributed in the hope that it will be useful,
* but WITHOUT ANY WARRANTY; without even the implied warranty of
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
* GNU General Public License for more details.
*
* You should have received a copy of the GNU General Public License
* along with this program; if not, write to the Free Software
* Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
*
***********************************************************************************/

#include "plasma-adjustableclock.h"

#include <QPainter>
#include <QFontMetrics>
#include <QSizeF>
#include <QTimeLine>
#include <QClipboard>
#include <QHeaderView>

#include <Plasma/Theme>

#include <KGlobalSettings>
#include <KTimeZoneWidget>
#include <KDialog>
#include <KGlobalSettings>
#include <KConfigDialog>
#include <KDatePicker>
#include <KLocale>
#include <KMenu>
#include <KCalendarSystem>

AdjustableClock::AdjustableClock(QObject *parent, const QVariantList &args) : ClockApplet(parent, args)
{
    KGlobal::locale()->insertCatalog("plasmaclock");

    setHasConfigurationInterface(true);

    resize(150, 75);
}

void AdjustableClock::init()
{
    KConfigGroup configuration = config();

    ClockApplet::init();

    m_updateToolTip = false;

    m_clipboardFormats.append("%c");
    m_clipboardFormats.append("%F");
    m_clipboardFormats.append("%X");
    m_clipboardFormats.append("%f");
    m_clipboardFormats.append("%x");
    m_clipboardFormats.append("%a %b %e %Y");
    m_clipboardFormats.append("%H:%M:%S");
    m_clipboardFormats.append("%a %b %e %H:%M:%S %Y");
    m_clipboardFormats.append("%Y-%m-%d %H:%M:%S");
    m_clipboardFormats.append("%t");

    m_timeFormat = configuration.readEntry("timeFormat", "<center><big>%H:%M:%S</big><br />\n<small>%d.%m.%Y</small></center>");
    m_toolTipFormat = configuration.readEntry("toolTipFormat", "%Y-%m-%d %H:%M:%S");
    m_fastCopyFormat = configuration.readEntry("fastCopyFormat", "%Y-%m-%d %H:%M:%S");
    m_timeDifference = configuration.readEntry("timeDifference", 0);
    m_clipboardFormats = configuration.readEntry("clipboardFormats", m_clipboardFormats);

    m_clock = new Plasma::Label(this);
    m_clock->nativeWidget()->setTextFormat(Qt::RichText);
    m_clock->nativeWidget()->setTextInteractionFlags(Qt::NoTextInteraction);
    m_clock->nativeWidget()->setWordWrap(false);
    m_clock->setText(formatDateTime(QDateTime::currentDateTime(), m_timeFormat));

    dataEngine("time")->connectSource(currentTimezone(), this, 1000, Plasma::NoAlignment);

    m_clipboardMenu = new QAction(SmallIcon("edit-copy"), i18n("C&opy to Clipboard"), this);
    m_clipboardMenu->setMenu(new KMenu());
    m_clipboardMenu->setVisible(!m_clipboardFormats.isEmpty());

    m_actions.append(m_clipboardMenu);

    updateSize();

    connect(m_clipboardMenu, SIGNAL(hovered()), this, SLOT(updateClipboardMenu()));
}

void AdjustableClock::dataUpdated(const QString &source, const Plasma::DataEngine::Data &data)
{
    Q_UNUSED(source);

    m_dateTime = QDateTime(data["Date"].toDate(), data["Time"].toTime());
    m_dateTime = m_dateTime.addSecs(m_timeDifference);

    m_clock->setText(formatDateTime(m_dateTime, m_timeFormat));

    if (m_updateToolTip)
    {
        updateToolTipContent();
    }
}

void AdjustableClock::constraintsEvent(Plasma::Constraints constraints)
{
    if (constraints & Plasma::SizeConstraint)
    {
        updateSize();
    }
}

void AdjustableClock::createConfigurationInterface(KConfigDialog *parent)
{
    KConfigGroup configuration = config();
    QString preview;
    int row;

    QWidget *appearanceConfiguration = new QWidget();
    m_appearanceConfigurationUi.setupUi(appearanceConfiguration);

    QWidget *clipboardActions = new QWidget();
    m_clipboardActionsUi.setupUi(clipboardActions);

    QWidget *advancedConfiguration = new QWidget();
    m_advancedConfigurationUi.setupUi(advancedConfiguration);

    m_appearanceConfigurationUi.timeFormat->setPlainText(m_timeFormat);

    m_advancedConfigurationUi.toolTipFormat->setPlainText(m_toolTipFormat);
    m_advancedConfigurationUi.fastCopyFormat->setText(m_fastCopyFormat);
    m_advancedConfigurationUi.timeDifference->setValue(m_timeDifference);

    m_clipboardActionsUi.clipboardActionsTable->setAlternatingRowColors(true);
    m_clipboardActionsUi.clipboardActionsTable->setSelectionMode(QAbstractItemView::SingleSelection);
    m_clipboardActionsUi.clipboardActionsTable->setColumnCount(2);
    m_clipboardActionsUi.clipboardActionsTable->setHorizontalHeaderItem(0, new QTableWidgetItem(i18n("Format")));
    m_clipboardActionsUi.clipboardActionsTable->setHorizontalHeaderItem(1, new QTableWidgetItem(i18n("Preview")));
    m_clipboardActionsUi.clipboardActionsTable->setColumnWidth(0, 150);

    QHeaderView *header = m_clipboardActionsUi.clipboardActionsTable->horizontalHeader();
    header->setResizeMode(QHeaderView::Stretch);
    header->resizeSection(1, 150);

    m_clipboardActionsUi.moveUpButton->setIcon(KIcon("arrow-up"));

    m_clipboardActionsUi.moveDownButton->setIcon(KIcon("arrow-down"));

    foreach (const QString &string, m_clipboardFormats)
    {
        row = m_clipboardActionsUi.clipboardActionsTable->rowCount();
        m_clipboardActionsUi.clipboardActionsTable->insertRow(row);
        m_clipboardActionsUi.clipboardActionsTable->setItem(row, 0, new QTableWidgetItem(string));

        preview = formatDateTime(m_dateTime, string);

        QTableWidgetItem *item = new QTableWidgetItem(preview);
        item->setFlags(0);
        item->setToolTip(preview);

        m_clipboardActionsUi.clipboardActionsTable->setItem(row, 1, item);
    }

    itemSelectionChanged();

    parent->addPage(appearanceConfiguration, i18n("Appearance"), "preferences-desktop-theme");

    ClockApplet::createConfigurationInterface(parent);

    parent->addPage(clipboardActions, i18n("Clipboard actions"), "edit-copy");
    parent->addPage(advancedConfiguration, i18n("Advanced"), "configure");

    parent->setInitialSize(QSize(550, 450));

    connect(parent, SIGNAL(finished()), this, SLOT(updateClock()));
    connect(m_appearanceConfigurationUi.previewButton, SIGNAL(clicked()), this, SLOT(updateClock()));
    connect(m_clipboardActionsUi.addButton, SIGNAL(clicked()), this, SLOT(insertRow()));
    connect(m_clipboardActionsUi.deleteButton, SIGNAL(clicked()), this, SLOT(deleteRow()));
    connect(m_clipboardActionsUi.moveUpButton, SIGNAL(clicked()), this, SLOT(moveRowUp()));
    connect(m_clipboardActionsUi.moveDownButton, SIGNAL(clicked()), this, SLOT(moveRowDown()));
    connect(m_clipboardActionsUi.clipboardActionsTable, SIGNAL(itemSelectionChanged()), this, SLOT(itemSelectionChanged()));
    connect(m_clipboardActionsUi.clipboardActionsTable, SIGNAL(cellChanged(int, int)), this, SLOT(updateRow(int, int)));
}

void AdjustableClock::configAccepted()
{
    KConfigGroup configuration = config();

    dataEngine("time")->disconnectSource(currentTimezone(), this);

    m_timeFormat = m_appearanceConfigurationUi.timeFormat->toPlainText();
    m_toolTipFormat = m_advancedConfigurationUi.toolTipFormat->toPlainText();
    m_fastCopyFormat = m_advancedConfigurationUi.fastCopyFormat->text();
    m_timeDifference = m_advancedConfigurationUi.timeDifference->value();

    configuration.writeEntry("timeFormat", m_timeFormat);
    configuration.writeEntry("toolTipFormat", m_toolTipFormat);
    configuration.writeEntry("fastCopyFormat", m_fastCopyFormat);
    configuration.writeEntry("timeDifference", m_timeDifference);

    m_clipboardFormats.clear();

    for (int i = 0; i < m_clipboardActionsUi.clipboardActionsTable->rowCount(); ++i)
    {
        m_clipboardFormats.append(m_clipboardActionsUi.clipboardActionsTable->item(i, 0)->text());
    }

    configuration.writeEntry("clipboardFormats", m_clipboardFormats);

    ClockApplet::configAccepted();

    dataEngine("time")->connectSource(currentTimezone(), this, 1000, Plasma::NoAlignment);

    m_clipboardMenu->setVisible(!m_clipboardFormats.isEmpty());

    updateClock();

    emit configNeedsSaving();
}

void AdjustableClock::mousePressEvent(QGraphicsSceneMouseEvent *event)
{
    if (event->buttons() == Qt::MidButton)
    {
        QApplication::clipboard()->setText(formatDateTime(m_dateTime, m_fastCopyFormat));

        event->setAccepted(true);

        return;
    }

    ClockApplet::mousePressEvent(event);
}

void AdjustableClock::itemSelectionChanged()
{
    QList<QTableWidgetItem *> selectedItems = m_clipboardActionsUi.clipboardActionsTable->selectedItems();

    m_clipboardActionsUi.moveUpButton->setEnabled(!selectedItems.isEmpty() && m_clipboardActionsUi.clipboardActionsTable->row(selectedItems.first()) != 0);
    m_clipboardActionsUi.moveDownButton->setEnabled(!selectedItems.isEmpty() && m_clipboardActionsUi.clipboardActionsTable->row(selectedItems.last()) != (m_clipboardActionsUi.clipboardActionsTable->rowCount() - 1));
    m_clipboardActionsUi.deleteButton->setEnabled(!selectedItems.isEmpty());
}

void AdjustableClock::insertRow()
{
    int row = (m_clipboardActionsUi.clipboardActionsTable->rowCount()?m_clipboardActionsUi.clipboardActionsTable->currentRow():0);

    m_clipboardActionsUi.clipboardActionsTable->insertRow(row);

    m_clipboardActionsUi.clipboardActionsTable->setItem(row, 0, new QTableWidgetItem(QString("")));

    QTableWidgetItem *item = new QTableWidgetItem(QString(""));
    item->setFlags(0);

    m_clipboardActionsUi.clipboardActionsTable->setItem(row, 1, item);

    m_clipboardActionsUi.clipboardActionsTable->setCurrentCell(row, 0);
}

void AdjustableClock::deleteRow()
{
    m_clipboardActionsUi.clipboardActionsTable->removeRow(m_clipboardActionsUi.clipboardActionsTable->row(m_clipboardActionsUi.clipboardActionsTable->selectedItems().at(0)));
}

void AdjustableClock::moveRow(bool up)
{
    int sourceRow = m_clipboardActionsUi.clipboardActionsTable->row(m_clipboardActionsUi.clipboardActionsTable->selectedItems().at(0));
    int destinationRow = (up?(sourceRow - 1):(sourceRow + 1));

    QList<QTableWidgetItem*> sourceItems;
    QList<QTableWidgetItem*> destinationItems;

    for (int i = 0; i < 2; ++i)
    {
        sourceItems.append(m_clipboardActionsUi.clipboardActionsTable->takeItem(sourceRow, i));
        destinationItems.append(m_clipboardActionsUi.clipboardActionsTable->takeItem(destinationRow, i));
    }

    for (int i = 0; i < 2; ++i)
    {
        m_clipboardActionsUi.clipboardActionsTable->setItem(sourceRow, i, destinationItems.at(i));
        m_clipboardActionsUi.clipboardActionsTable->setItem(destinationRow, i, sourceItems.at(i));
    }

    m_clipboardActionsUi.clipboardActionsTable->setCurrentCell(destinationRow, 0);
}

void AdjustableClock::moveRowUp()
{
    moveRow(true);
}

void AdjustableClock::moveRowDown()
{
    moveRow(false);
}

void AdjustableClock::updateRow(int row, int column)
{
    Q_UNUSED(column);

    if (!m_clipboardActionsUi.clipboardActionsTable->item(row, 1))
    {
        return;
    }

    QString preview = formatDateTime(m_dateTime, m_clipboardActionsUi.clipboardActionsTable->item(row, 0)->text());

    m_clipboardActionsUi.clipboardActionsTable->item(row, 1)->setText(preview);
    m_clipboardActionsUi.clipboardActionsTable->item(row, 1)->setToolTip(preview);
}

void AdjustableClock::toolTipAboutToShow()
{
    updateToolTipContent();

    m_updateToolTip = true;
}

void AdjustableClock::toolTipHidden()
{
    Plasma::ToolTipManager::self()->clearContent(this);

    m_updateToolTip = false;
}

void AdjustableClock::copyToClipboard(QAction* action)
{
    QApplication::clipboard()->setText(action->text());
}

void AdjustableClock::updateClipboardMenu()
{
    KMenu *menu = new KMenu();
    Plasma::DataEngine::Data data = dataEngine("time")->query(currentTimezone());

    QDateTime dateTime = QDateTime(data["Date"].toDate(), data["Time"].toTime());
    dateTime = dateTime.addSecs(m_timeDifference);

    foreach (const QString &string, m_clipboardFormats)
    {
        menu->addAction(formatDateTime(dateTime, string));
    }

    connect(menu, SIGNAL(triggered(QAction*)), this, SLOT(copyToClipboard(QAction*)));

    m_clipboardMenu->setMenu(menu);
}

void AdjustableClock::updateClock()
{
    m_timeFormat = m_appearanceConfigurationUi.timeFormat->toPlainText();

    m_clock->setText(formatDateTime(m_dateTime, m_timeFormat));

    updateSize();
}

void AdjustableClock::updateToolTipContent()
{
    Plasma::ToolTipContent toolTipData;

    toolTipData.setImage(KIcon("chronometer").pixmap(IconSize(KIconLoader::Desktop)));
    toolTipData.setMainText(formatDateTime(m_dateTime, m_toolTipFormat));
    toolTipData.setAutohide(false);

    Plasma::ToolTipManager::self()->setContent(this, toolTipData);
}

void AdjustableClock::updateSize()
{
    m_clock->hide();

    m_clock->setText(formatDateTime(m_timeFormat));

    m_clock->nativeWidget()->adjustSize();

    setPreferredSize(m_clock->nativeWidget()->sizeHint());

    if (formFactor() == Plasma::Horizontal || formFactor() == Plasma::Vertical)
    {
        m_clock->nativeWidget()->setMargin(3);
    }
    else
    {
        m_clock->nativeWidget()->setMargin(12);

        resize(m_clock->nativeWidget()->sizeHint());
    }

    m_clock->setText(formatDateTime(m_dateTime, m_timeFormat));

    m_clock->show();
}

QString AdjustableClock::formatDateTime(const QDateTime dateTime, QString format)
{
    if (format.isEmpty())
    {
        return "";
    }

    QString string;
    int length = (format.length() - 1);

    for (int index = 0; index < length; ++index)
    {
        if (format.at(index) == '%' && format.at(index + 1) != '%')
        {
            index++;

            switch (format.at(index).unicode())
            {
                case 'a': // weekday, short form
                    string.append(KGlobal::locale()->calendar()->weekDayName(dateTime.date(), KCalendarSystem::ShortDayName));
                break;
                case 'A': // weekday, long form
                    string.append(KGlobal::locale()->calendar()->weekDayName(dateTime.date(), KCalendarSystem::LongDayName));
                break;
                case 'b': // month, short form
                    string.append(KGlobal::locale()->calendar()->monthName(KGlobal::locale()->calendar()->month(dateTime.date()), KGlobal::locale()->calendar()->year(dateTime.date()), ((KGlobal::locale()->nounDeclension() && KGlobal::locale()->dateMonthNamePossessive())?KCalendarSystem::ShortNamePossessive:KCalendarSystem::ShortName)));
                break;
                case 'B': // month, long form
                    string.append(KGlobal::locale()->calendar()->monthName(KGlobal::locale()->calendar()->month(dateTime.date()), KGlobal::locale()->calendar()->year(dateTime.date()), ((KGlobal::locale()->nounDeclension() && KGlobal::locale()->dateMonthNamePossessive())?KCalendarSystem::LongNamePossessive:KCalendarSystem::LongName)));
                break;
                case 'c': // date and time representation, short
                    string.append(KGlobal::locale()->formatDateTime(dateTime, KLocale::LongDate));
                break;
                case 'C': // date and time representation, long
                    string.append(KGlobal::locale()->formatDateTime(dateTime, KLocale::ShortDate));
                break;
                case 'd': // day of the month, two digits
                    string.append(KGlobal::locale()->calendar()->dayString(dateTime.date(), KCalendarSystem::LongFormat));
                break;
                case 'e': // day of the month, one digit
                    string.append(KGlobal::locale()->calendar()->dayString(dateTime.date(), KCalendarSystem::ShortFormat));
                break;
                case 'f': // date representation, short
                    string.append(KGlobal::locale()->formatDate(dateTime.date(), KLocale::ShortDate));
                break;
                case 'F': // time representation, short
                    string.append(KGlobal::locale()->formatTime(dateTime.time(), false));
                break;
                case 'H': // hour, 24h format
                    if (dateTime.time().hour() < 10)
                    {
                        string.append('0');
                    }

                    string.append(QString::number(dateTime.time().hour()));
                break;
                case 'I': // hour, 12h format
                    if ((((dateTime.time().hour() + 11) % 12) + 1) < 10)
                    {
                        string.append('0');
                    }

                    string.append(QString::number(((dateTime.time().hour() + 11) % 12) + 1));
                break;
                case 'j': // day of the year
                    string.append(QString::number(KGlobal::locale()->calendar()->dayOfYear(dateTime.date())));
                break;
                case 'k': // hour, 24h format, one digit
                    string.append(QString::number(dateTime.time().hour()));
                break;
                case 'l': // hour, 12h format, one digit
                    string.append(QString::number(((dateTime.time().hour() + 11) % 12) + 1));
                break;
                case 'm': // month, two digits
                    string.append(KGlobal::locale()->calendar()->monthString(dateTime.date(), KCalendarSystem::LongFormat));
                break;
                case 'M': // minute, two digits
                    if (dateTime.time().minute() < 10)
                    {
                        string.append('0');
                    }

                    string.append(QString::number(dateTime.time().minute()));
                break;
                case 'n': // month, one digit
                    string.append(KGlobal::locale()->calendar()->monthString(dateTime.date(), KCalendarSystem::ShortFormat));
                break;
                case 'p': // pm or am
                    string.append((dateTime.time().hour() >= 12)?i18n("pm"):i18n("am"));
                break;
                case 'S': // second, two digits
                    if (dateTime.time().second() < 10)
                    {
                        string.append('0');
                    }

                    string.append(QString::number(dateTime.time().second()));
                break;
                case 't': // UNIX timestamp
                    string.append(QString::number(dateTime.toTime_t()));
                break;
                case 'w': // day of week
                    string.append(KGlobal::locale()->calendar()->dayOfWeek(dateTime.date()));
                break;
                case 'W': // week number
                case 'U':
                    string.append(QString::number(KGlobal::locale()->calendar()->weekNumber(dateTime.date())));
                break;
                case 'x': // date representation, long
                    string.append(KGlobal::locale()->formatDate(dateTime.date(), KLocale::LongDate));
                break;
                case 'X': // time representation, long
                    string.append(KGlobal::locale()->formatTime(dateTime.time(), true));
                break;
                case 'Y': // year, four digits
                    string.append(KGlobal::locale()->calendar()->yearString(dateTime.date(), KCalendarSystem::LongFormat));
                break;
                case 'y': // year, two digits
                    string.append(KGlobal::locale()->calendar()->yearString(dateTime.date(), KCalendarSystem::ShortFormat));
                break;
                case 'Z': // timezone city
                    string.append(prettyTimezone());
                break;
                default:
                    string.append(format.at(index));
                break;
            }
        }
        else
        {
            string.append(format.at(index));
        }
    }

    if (format.at(length - 1) != '%')
    {
        string.append(format.at(length));
    }

    return string;
}

QString AdjustableClock::formatDateTime(QString format)
{
    if (format.isEmpty())
    {
        return "";
    }

    QString string;
    int length = (format.length() - 1);

    for (int index = 0; index < length; ++index)
    {
        if (format.at(index) == '%' && format.at(index + 1) != '%')
        {
            index++;

            switch (format.at(index).unicode())
            {
                case 'a':
                    string.append(KGlobal::locale()->calendar()->weekDayName(m_dateTime.date(), KCalendarSystem::ShortDayName));
                break;
                case 'A':
                    string.append(KGlobal::locale()->calendar()->weekDayName(m_dateTime.date(), KCalendarSystem::LongDayName));
                break;
                case 'b':
                    string.append(KGlobal::locale()->calendar()->monthName(KGlobal::locale()->calendar()->month(m_dateTime.date()), KGlobal::locale()->calendar()->year(m_dateTime.date()), ((KGlobal::locale()->nounDeclension() && KGlobal::locale()->dateMonthNamePossessive())?KCalendarSystem::ShortNamePossessive:KCalendarSystem::ShortName)));
                break;
                case 'B':
                    string.append(KGlobal::locale()->calendar()->monthName(KGlobal::locale()->calendar()->month(m_dateTime.date()), KGlobal::locale()->calendar()->year(m_dateTime.date()), ((KGlobal::locale()->nounDeclension() && KGlobal::locale()->dateMonthNamePossessive())?KCalendarSystem::LongNamePossessive:KCalendarSystem::LongName)));
                break;
                case 'c':
                    string.append(KGlobal::locale()->formatDateTime(m_dateTime, KLocale::LongDate));
                break;
                case 'C':
                    string.append(KGlobal::locale()->formatDateTime(m_dateTime, KLocale::ShortDate));
                break;
                case 'd':
                case 'e':
                case 'H':
                case 'I':
                case 'k':
                case 'l':
                case 'm':
                case 'M':
                case 'n':
                case 'S':
                case 'W':
                case 'U':
                case 'y':
                    string.append("00");
                break;
                case 'f':
                    string.append(KGlobal::locale()->formatDate(m_dateTime.date(), KLocale::ShortDate));
                break;
                case 'F':
                    string.append(KGlobal::locale()->formatTime(m_dateTime.time(), false));
                break;
                case 'j':
                    string.append("000");
                break;
                case 'p':
                    string.append((m_dateTime.time().hour() >= 12)?i18n("pm"):i18n("am"));
                break;
                case 't':
                    string.append(QString::number(m_dateTime.toTime_t()));
                break;
                case 'w':
                    string.append("0");
                break;
                case 'x':
                    string.append(KGlobal::locale()->formatDate(m_dateTime.date(), KLocale::LongDate));
                break;
                case 'X':
                    string.append(KGlobal::locale()->formatTime(m_dateTime.time(), true));
                break;
                case 'Y':
                    string.append("0000");
                break;
                case 'Z':
                    string.append(prettyTimezone());
                break;
                default:
                    string.append(format.at(index));
                break;
            }
        }
        else
        {
            string.append(format.at(index));
        }
    }

    if (format.at(length - 1) != '%')
    {
        string.append(format.at(length));
    }

    return string;
}

QList<QAction*> AdjustableClock::contextualActions()
{
    return m_actions;
}

#include "plasma-adjustableclock.moc"
