# ***** BEGIN LICENSE BLOCK *****
# Version: RCSL 1.0/RPSL 1.0/GPL 2.0
#
# Portions Copyright (c) 1995-2002 RealNetworks, Inc. All Rights Reserved.
# Portions Copyright (c) 2004 Robert Kaye. All Rights Reserved.
#
# The contents of this file, and the files included with this file, are
# subject to the current version of the RealNetworks Public Source License
# Version 1.0 (the "RPSL") available at
# http://www.helixcommunity.org/content/rpsl unless you have licensed
# the file under the RealNetworks Community Source License Version 1.0
# (the "RCSL") available at http://www.helixcommunity.org/content/rcsl,
# in which case the RCSL will apply. You may also obtain the license terms
# directly from RealNetworks.  You may not use this file except in
# compliance with the RPSL or, if you have a valid RCSL with RealNetworks
# applicable to this file, the RCSL.  Please see the applicable RPSL or
# RCSL for the rights, obligations and limitations governing use of the
# contents of the file.
#
# This file is part of the Helix DNA Technology. RealNetworks is the
# developer of the Original Code and owns the copyrights in the portions
# it created.
#
# This file, and the files included with this file, is distributed and made
# available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
# EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS ALL SUCH WARRANTIES,
# INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
#
# Technology Compatibility Kit Test Suite(s) Location:
#    http://www.helixcommunity.org/content/tck
#
# --------------------------------------------------------------------
#
# picard is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# picard is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with picard; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
# Contributor(s):
#   Lukas Lalinsky
#
#
# ***** END LICENSE BLOCK *****

import time
import Queue
import wx
from threading import Thread, Lock
from musicbrainz2.utils import extractUuid
from musicbrainz2.webservice import Query, WebServiceError,	TrackFilter
from picard.events import SetStatusTextEvent
from tunepimp import tunepimp, metadata

class SubmitDialog(wx.Dialog):
    
    def __init__(self, parent):
        wx.Dialog.__init__(self, parent, -1, _("Submit PUIDs"),
            style = wx.DEFAULT_DIALOG_STYLE)
        staticText = wx.StaticText(self, -1, _("Submitting PUID information to MusicBrainz server ..."), style=wx.ALIGN_CENTER_HORIZONTAL)
        cancelButton =  wx.Button(self, -1, _("Close"))
        self.Bind(wx.EVT_BUTTON, self.onCancel, cancelButton) 
        vbox = wx.BoxSizer(wx.VERTICAL)
        vbox.Add(staticText, 1, wx.EXPAND | wx.LEFT | wx.RIGHT | wx.TOP, 8)
        vbox.Add(cancelButton, 0, wx.ALIGN_CENTER | wx.ALL, 8)
        self.SetSizer(vbox)
        self.Fit()     
     
    def onCancel(self, event):
        self.EndModal(0)

class PUIDManager(Thread):

    LOOKUP = 0
    SUBMIT = 1
    
    def __init__(self, context):
        Thread.__init__(self)
        self.exitThread = 0
        self.queue = Queue.Queue()
        self.context = context
        self.lock = Lock()
        self.puids = {}
        self.tunePimp = context.config.getTunePimp()
        self.submitDialog = None

    def add(self, fileId, puid, trackId, check=True):
        self.lock.acquire()
        self.puids[fileId] = [puid, trackId, trackId]
        if check:
            self.checkUnsubmitted()
        self.lock.release()

    def update(self, fileId, trackId, check=True):
        self.lock.acquire()
        try:
            self.puids[fileId][2] = trackId
        except KeyError:
            pass
        if check:
            self.checkUnsubmitted()
        self.lock.release()

    def remove(self, fileId, check=True):
        self.lock.acquire()
        del self.puids[fileId]
        if check:
            self.checkUnsubmitted()
        self.lock.release()
    
    def checkUnsubmitted(self):
        """Check number of unsubmitted PUIDs"""
        n = self.getNumUnsubmitted()
        self.context.config.toolbar.enableSubmit(n > 0)
            
    def getNumUnsubmitted(self):
        """Get number of unsubmitted PUIDs"""
        n = 0
        tunePimp = self.context.config.getTunePimp()
        for (puid, origTrackId, trackId) in self.puids.values():
            if trackId and origTrackId != trackId:
                n += 1
        return n
            
    def submit(self):
        """Submit PUIDs to the server."""
        puids = {}
        self.lock.acquire()
        for (puid, origTrackId, trackId) in self.puids.values():
            if trackId and origTrackId != trackId:
                puids[trackId] = puid
        self.lock.release()
        self.queue.put((self.SUBMIT, puids))

    def lookup(self, fileId, puid):
        """Adds a PUID to the lookup queue."""
        self.queue.put((self.LOOKUP, puid, fileId))        
        
    def stop(self):
        if self.isAlive():
            self.exitThread = 1
            self.join()
         
    def _convertMetadata(self, mdata, track):
        mdata.trackId = extractUuid(track.getId(), 'track')
        mdata.track = track.getTitle()
        mdata.duration = track.getDuration() or 0
        mdata.artistId = extractUuid(track.getArtist().getId(), 'artist')
        mdata.artist = track.getArtist().getName()
        mdata.sortName = track.getArtist().getSortName()
        mdata.albumId = extractUuid(track.getReleases()[0].getId(), 'release')
        mdata.album = track.getReleases()[0].getTitle()
            
    def run(self):
        while not self.exitThread:
            
            try:
                action = self.queue.get(0)
            except Queue.Empty:
                time.sleep(.01)
                continue
            
            ws = self.context.config.getWebService()
            clientId = "%s-%s" % (self.context.config.getAppName(),
                                  self.context.config.getAppVersion().replace('-', ''))
            q = Query(ws, clientId=clientId)
                
            if action[0] == self.SUBMIT:
                wx.PostEvent(self.context.frame, SetStatusTextEvent(_("Submitting PUID information to MusicBrainz server ...")))
                wx.WakeUpIdle()
                puids = action[1]
                try:
                    q.submitPuids(puids)
                except:
                    wx.PostEvent(self.context.frame, SetStatusTextEvent(_("PUIDs submission failed.")))
                    wx.WakeUpIdle()
                else:
                    self.lock.acquire()
                    for fileId in self.puids.keys():
                        self.puids[fileId][1] = self.puids[fileId][2]
                    self.lock.release()
                    self.context.config.toolbar.enableSubmit(False)
                    wx.PostEvent(self.context.frame, SetStatusTextEvent(_("PUIDs successfully submitted.")))
                    wx.WakeUpIdle()
                
            elif action[0] == self.LOOKUP:
                puid, fileId = action[1], action[2]
                track = self.tunePimp.getTrack(fileId)
                
                # Set status bar text
                track.lock()
                fileName = track.getFileName() 
                track.unlock()
                wx.PostEvent(self.context.frame, SetStatusTextEvent(_("Performing a PUID lookup on file %s ...") % (fileName,)))
                wx.WakeUpIdle()

                # Lookup the file at the server
                recognized = False
                self.add(fileId, puid, None, check=False)
                try: 
                    flt = TrackFilter(puid=puid)
                    result = q.getTracks(flt)
                except WebServiceError, e:
                    result = []
                
                # Check to see if there is more than one matching track, and if only
                # one track then we're set                     
                if len(result) == 1:
                    mdata = metadata.metadata(self.tunePimp) 
                    track.lock() 
                    ldata = track.getLocalMetadata()
                    self._convertMetadata(mdata, result[0].track)
                    sim = ldata.compare(mdata)
                    if sim < self.context.config.settingPUIDMatchThreshold:
                        track.setStatus(tunepimp.eUnrecognized)
                        track.unlock()
                    else:
                        self.add(fileId, puid, mdata.trackId, check=False)
                        track.setServerMetadata(mdata)
                        track.setStatus(tunepimp.eRecognized)
                        track.unlock()
                        
                # Crap, more than one track matches. Loop through them and find the
                # best match and return that if it is higher then the collision threshold 
                elif len(result) > 1:
                    track.lock() 
                    ldata = track.getLocalMetadata()
                    track.unlock()
                    bestSim = -1
                    bestMdata = None
                    for res in result:
                        mdata = metadata.metadata(self.tunePimp) 
                        self._convertMetadata(mdata, res.track)
                        sim = ldata.compare(mdata)
                        if sim > bestSim:
                            bestSim = sim
                            bestMdata = mdata
                    track.lock()
                    if bestSim < self.context.config.settingPUIDCollisionThreshold:
                        track.setStatus(tunepimp.ePUIDCollision)
                        track.unlock()
                    else:
                        self.add(fileId, puid, bestMdata.trackId, check=False)
                        track.setServerMetadata(bestMdata)
                        track.setStatus(tunepimp.eRecognized)
                        track.unlock()
                       
                # No matching tracks?                        
                else:
                    track.lock() 
                    track.setStatus(tunepimp.eUnrecognized)
                    track.unlock()
                        
                self.tunePimp.wake(track)
                self.tunePimp.releaseTrack(track)


