# ***** BEGIN LICENSE BLOCK *****
# Version: RCSL 1.0/RPSL 1.0/GPL 2.0
#
# Portions Copyright (c) 1995-2002 RealNetworks, Inc. All Rights Reserved.
# Portions Copyright (c) 2004 Robert Kaye. All Rights Reserved.
#
# The contents of this file, and the files included with this file, are
# subject to the current version of the RealNetworks Public Source License
# Version 1.0 (the "RPSL") available at
# http://www.helixcommunity.org/content/rpsl unless you have licensed
# the file under the RealNetworks Community Source License Version 1.0
# (the "RCSL") available at http://www.helixcommunity.org/content/rcsl,
# in which case the RCSL will apply. You may also obtain the license terms
# directly from RealNetworks.  You may not use this file except in
# compliance with the RPSL or, if you have a valid RCSL with RealNetworks
# applicable to this file, the RCSL.  Please see the applicable RPSL or
# RCSL for the rights, obligations and limitations governing use of the
# contents of the file.
#
# This file is part of the Helix DNA Technology. RealNetworks is the
# developer of the Original Code and owns the copyrights in the portions
# it created.
#
# This file, and the files included with this file, is distributed and made
# available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
# EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS ALL SUCH WARRANTIES,
# INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
#
# Technology Compatibility Kit Test Suite(s) Location:
#    http://www.helixcommunity.org/content/tck
#
# --------------------------------------------------------------------
#
# picard is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# picard is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with picard; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
# Contributor(s):
#   Robert Kaye
#
#
# ***** END LICENSE BLOCK *****

import wx
import Queue
import time
import copy
from threading import Thread
import album, artist, events, cluster, debug
from tunepimp import metadata

pendingFiles = u"[pending]"
pendingFilesText = N_("New files (drag files to tag here)")
unmatchedFiles = u"[unmatched]"
unmatchedFilesText = N_("Unclustered files")
errorFiles = u"[error]"
errorFilesText = N_("Error files")
clusteredFiles = u"[clustered]"
clusteredFilesText = N_("Album clusters")

MAIN_SERVER_NAME = u"musicbrainz.org"
MAIN_SERVER_PORT = 80

# XXX not used
class DuplicateAlbumError(KeyError):
    '''This error is raised when a duplicate album is added to the AlbumManager.'''

# Does this class need to refcount the albums??
class AlbumManager(Thread):
    '''This class keeps track of all the album objects that are currently open in the tagger.'''

    # XXX Don't these belong inside the instance?
    # This dict contains mbid -> albums
    albumIndex = {}

    # This one has track mbid => album
    trackIndex = {}

    coverArtCache = None;
    exitThread = 0;
    loadQueue = Queue.Queue()
   
    def __init__(self, context, parent, config, tpmanager):
        Thread.__init__(self)
        self.context = context
        self.parent = parent
        self.tunePimp = config.getTunePimp()
        self.config = config
        self.tpmanager = tpmanager

        ar = artist.Artist(config, u"", u"", u"[internal_artist]");

        pending = album.Album(parent, config)
        pending.setArtist(ar)
        pending.setName(_(pendingFilesText))
        pending.setId(pendingFiles)
        pending.setDuration(-1)
        pending.setAmazonAsin(u"<mblogo>")
        self.addAlbum(pending)

        unmatched = album.Album(parent, config)
        unmatched.setArtist(ar)
        unmatched.setName(_(unmatchedFilesText))
        unmatched.setId(unmatchedFiles)
        unmatched.setDuration(-1)
        unmatched.setAmazonAsin(u"<mblogo>")
        self.addAlbum(unmatched)
        
        error = album.Album(parent, config)
        error.setArtist(ar)
        error.setName(_(errorFilesText))
        error.setId(errorFiles)
        error.setDuration(-1)
        error.setAmazonAsin(u"<mblogo>")
        self.addAlbum(error)

        clustered = album.Album(parent, config)
        clustered.setArtist(ar)
        clustered.setName(_(clusteredFilesText))
        clustered.setId(clusteredFiles)
        clustered.setDuration(-1)
        clustered.setAmazonAsin(u"<mblogo>")
        self.addAlbum(clustered)

    def stop(self):
        if self.isAlive():
            self.exitThread = 1;
            self.join()

    def getAlbumList(self):
        return self.albumIndex.keys()

    def dumpState(self, d):
        d(u"albummanager object: %s" % ( repr(self) ))

        d = d.nest()
        d(u"number of albums: %d" % len(self.albumIndex))

        keys = self.albumIndex.keys()
        keys.sort()

        for mbid in keys:
            al = self.albumIndex[mbid]
            al.dumpState(d)

    def addAlbum(self, al):
        '''Add the specified album and return the album added. If the album already exists,
        the existing album is returned.'''
        
        mbid = al.getId() 
        try:
            cur = self.albumIndex[mbid]
            return cur
        except KeyError:
            pass

        self.albumIndex[mbid] = al

        wx.PostEvent(self.parent, events.AlbumAddEvent(al.getId()))
        wx.WakeUpIdle()

    def add(self, mbid):
        '''Add the specified album and return the album added. If the album already exists,
        the existing album is returned.'''

        al = album.Album(self.parent, self.config)
        al.setId(mbid) 
        try:
            cur = self.albumIndex[mbid]
            return cur
        except KeyError:
            pass

        if not mbid.startswith("["):
            self.loadQueue.put(al)
            al.setName(u"[ " + _("loading album information") + " ]")

        self.albumIndex[mbid] = al

        wx.PostEvent(self.parent, events.AlbumAddEvent(al.getId()))
        wx.WakeUpIdle()

        return al

    def updateFromMainServer(self, mbid):
        '''Update the specified album from the main server since it may have changed'''

        al = None
        try:
            al = self.albumIndex[mbid]
        except KeyError:
            return 

        al.setServerAffinity(MAIN_SERVER_NAME, MAIN_SERVER_PORT)
        self.loadQueue.put(al)
        al.setName(u"[ " + _("updating album information") + " ]")
        wx.PostEvent(self.parent, events.AlbumUpdateEvent(mbid))
        wx.WakeUpIdle()

    def isLoaded(self, mbid):

        try:
            al = self.albumIndex[mbid]
        except KeyError:
            return False

        return al.isLoaded()

    def get(self, mbid):
        '''Get the album with the passed in mbid. If album is not in index, raises LookupError.'''

        return self.albumIndex[mbid]

    def getFromTrackId(self, mbid):
        '''Get the album with the track with the given mbid.  Raises LookupError if not found'''

        albumid = self.trackIndex[mbid]
        return self.albumIndex[albumid]

    def remove(self, mbid):
        '''Remove the specified album from the AlbumManager. Returns if mbid is not found.'''

        try:
            al = self.albumIndex[mbid]
        except LookupError:
            return

        debug.debug("Removing album %s from system" % mbid, "albummanager.remove");
        al.removeAllFilesFromSystem()
        for trackId in al.getTrackIds():
            del self.trackIndex[trackId]
        del self.albumIndex[mbid]

        wx.PostEvent(self.parent, events.AlbumRemovedEvent(al.getId()))
        wx.WakeUpIdle()

    def len(self):
        return len(self.albumIndex)

    def clusterUnmatchedFiles(self, threshold):
        try:
            pendingAlbum = self.get(pendingFiles)
        except KeyError:
            return

        # If there are pending files still, then don't cluster yet
        if pendingAlbum.getNumUnmatchedFiles():
            # XXX this behaviour is not yet obvious to the user
            # Show a message box?  Ask the user if they want to try clustering
            # anyway?  Disable the "cluster" button when it won't do anything?
            return

        # The pending album is empty, so let's cluster
        try:
            unmatchedAlbum = self.get(unmatchedFiles)
        except KeyError:
            return

        # If there are no unmatched files, there's nothing to do
        fileList = unmatchedAlbum.getUnmatchedFiles()
        if not fileList:
            return

        engine = cluster.FileClusterEngine(self.context)
        engine.cluster(fileList, threshold)
        for clusterAlbum in engine.getClusterAlbums():
            self.context.clustermanager.addCluster(clusterAlbum)
            clusterAlbum.moveFilesToMe(self.context)

    def checkUnmatchedAlbumFiles(self, al):

        # XXX surely locking is required here?
        unmatchedFiles = copy.copy(al.getUnmatchedFiles())
        for fileId in unmatchedFiles:
            tpfile = self.tpmanager.findFile(fileId)

            tr = tpfile.getTrack()
            ldata = tr.getLocalMetadata()
            sdata = tr.getServerMetadata()
            tpfile.releaseTrack(tr)

            trackId = ldata.trackId
            if not trackId:
                trackId = sdata.trackId
            if not trackId:
                continue

            for tr in al.tracks:
                if tr.getId() == trackId:
                    # XXX inefficient
                    seq = al.findSeqOfTrack(trackId)
                    tpfile.moveToAlbumAsLinked(al, seq)
                    break

    def matchFilesToAlbum(self, fileList, albumId):

        # First, get tunepimp to read the metadata for these files
        fileIdList = []
        for file in fileList:
            if not file:
                continue
            fileId = self.tunePimp.addFile(file, 1)
            if fileId >= 0:
                debug.debug("Added file #%d" % fileId, "tunepimp.add")
                fileIdList.append(fileId)

        return self.matchFileIdsToAlbum(fileIdList, albumId)

    def matchFileIdsToAlbum(self, fileIdList, albumId):

        # match items in fileIdList to albumId
        matchDict = {}
        matches = []
        try:
            al = self.get(albumId)
        except LookupError:
            return

        num = al.getNumTracks()

        # Now compare each file with each track in the target album and create a list of matches
        for fileId in fileIdList:

            tr = self.tunePimp.getTrack(fileId)
            tr.lock()
            ldata = tr.getLocalMetadata()
            tr.unlock()
            self.tunePimp.releaseTrack(tr)

            mdata = metadata.metadata(self.tunePimp)
            mdata.album = al.getName()

            for i in xrange(num):
                tr = al.getTrack(i)
                mdata.artist = tr.getArtist().getName()
                mdata.track = tr.getName()
                mdata.duration = tr.getDuration()
                mdata.trackNum = tr.getNum()
                matches.append([ldata.compare(mdata), fileId, tr.getId()])

        # Sort the matches and reverse the list so the best matches are at the top
        matches.sort() 
        matches.reverse()

        # Create the match dict fileId -> trackId
        for match in matches:
            if match[0] < self.config.settingMatchThreshold:
                break

            if not matchDict.has_key(match[1]):
                matchDict[match[1]] = match[2]
                fileIdList.remove(match[1])
                num = num - 1
                if not num: 
                    break

        for fileId in fileIdList:
            tpfile = self.context.tpmanager.findFile(fileId)
            tpfile.moveToAlbumAsUnlinked(al)

        return matchDict

    def isUUID(self, id):
        if len(id) != 36:
            return False

        if id[8] != u'-' or id[13] != u'-' or id[18] != u'-' or id[14] != u'4':
            return False

        return True

    def run(self):

        while not self.exitThread:

            al = None
            try:
                al = self.loadQueue.get(0)
            except Queue.Empty:
                time.sleep(.01)
                continue

            try:
                al.load()
                for trackId in al.getTrackIds():
                    self.trackIndex[trackId] = al.getId()
                debug.debug("Album %s loaded successfully" % al.getId(), "album.load")
            except album.AlbumLoadError:
                al.setName(_("Could not load album (%s)") % al.getId());
                debug.debug("Album %s failed to load" % al.getId(), "album.load")

            wx.PostEvent(self.parent, events.AlbumUpdateEvent(al.getId()))
            wx.WakeUpIdle()
            
