# ***** BEGIN LICENSE BLOCK *****
# Version: RCSL 1.0/RPSL 1.0/GPL 2.0
#
# Portions Copyright (c) 1995-2002 RealNetworks, Inc. All Rights Reserved.
# Portions Copyright (c) 2004 Robert Kaye. All Rights Reserved.
#
# The contents of this file, and the files included with this file, are
# subject to the current version of the RealNetworks Public Source License
# Version 1.0 (the "RPSL") available at
# http://www.helixcommunity.org/content/rpsl unless you have licensed
# the file under the RealNetworks Community Source License Version 1.0
# (the "RCSL") available at http://www.helixcommunity.org/content/rcsl,
# in which case the RCSL will apply. You may also obtain the license terms
# directly from RealNetworks.  You may not use this file except in
# compliance with the RPSL or, if you have a valid RCSL with RealNetworks
# applicable to this file, the RCSL.  Please see the applicable RPSL or
# RCSL for the rights, obligations and limitations governing use of the
# contents of the file.
#
# This file is part of the Helix DNA Technology. RealNetworks is the
# developer of the Original Code and owns the copyrights in the portions
# it created.
#
# This file, and the files included with this file, is distributed and made
# available on an 'AS IS' basis, WITHOUT WARRANTY OF ANY KIND, EITHER
# EXPRESS OR IMPLIED, AND REALNETWORKS HEREBY DISCLAIMS ALL SUCH WARRANTIES,
# INCLUDING WITHOUT LIMITATION, ANY WARRANTIES OF MERCHANTABILITY, FITNESS
# FOR A PARTICULAR PURPOSE, QUIET ENJOYMENT OR NON-INFRINGEMENT.
#
# Technology Compatibility Kit Test Suite(s) Location:
#    http://www.helixcommunity.org/content/tck
#
# --------------------------------------------------------------------
#
# picard is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# picard is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with picard; if not, write to the Free Software
# Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301  USA
#
# Contributor(s):
#   Robert Kaye
#
#
# ***** END LICENSE BLOCK *****

import wx
import events
import dataobjs, artist, track
from musicbrainz2.webservice import Query, WebServiceError, ReleaseIncludes
from musicbrainz2.model import VARIOUS_ARTISTS_ID, NS_MMD_1
from musicbrainz2.utils import extractUuid, extractFragment
import unicodedata

NON_ALBUM_TRACKS = "[non-album tracks]"

class AlbumLoadError(LookupError):
    "This is used to indicate that a given album was not found on the server"
   
class AlbumUnmatched(dataobjs.DataObject):
    '''The AlbumUnmatched class is a data object used in the album panel tree to identify
       the tree node'''
    def __init__(self, config, id):
        dataobjs.DataObject.__init__(self, config, "", id)

class Album(dataobjs.DataObject):
    '''The Album class is a representation of an album that contains metadata for tracks and
       the list of tunepimp file ids that have been associated with a track in this album.'''
   
    def __init__(self, parent, config):
        dataobjs.DataObject.__init__(self, config, "", "")
        self.parent = parent
        self.tracks = []			# track objects
        self.trackIds = {}		# mbid => index into self.tracks
        self.unmatchedFiles = []		# TP file IDs
        self.unsavedFiles = []		# TP file IDs
        self.TPFileIdToTrackSeq = {}	# TP file ID => index into self.tracks, or None
        self.duration = 0
        self.asin = u''
        self.artist = artist.Artist(config, u'', u'', u'')
        self.releaseYear = 0
        self.releaseDate = u''
        self.releaseType = u''
        self.releaseStatus = u''
        self.releaseCountry = u''
        self.numLinked = 0
        self.loaded = False
        self.hasLoadError = False

        # For server 'affinity' -- see setServerAffinity
        self.affServer = u''
        self.affPort = u''

    def dumpState(self, writer):
        writer(u"album object: %s" % ( repr(self) ))
        writer = writer.nest()
        writer(u"id: %s" % ( repr(self.id) ))
        writer(u"name: %s" % ( repr(self.name) ))
        writer(u"trackIds: %s" % ( repr(self.trackIds) ))
        writer(u"unmatchedFiles: %s" % ( repr(self.unmatchedFiles) ))
        writer(u"duration: %s" % ( repr(self.duration) ))
        writer(u"asin: %s" % ( repr(self.asin) ))
        writer(u"artist: %s" % ( repr(self.artist) ))
        self.artist.dumpState(writer)
        writer(u"releaseYear: %s" % ( repr(self.releaseYear) ))
        writer(u"releaseDate: %s" % ( repr(self.releaseDate) ))
        writer(u"releaseType: %s" % ( repr(self.releaseType) ))
        writer(u"releaseStatus: %s" % ( repr(self.releaseStatus) ))
        writer(u"releaseCountry: %s" % ( repr(self.releaseCountry) ))
        writer(u"numLinked: %s" % ( repr(self.numLinked) ))
        writer(u"loaded: %s" % ( repr(self.loaded) ))
        if len(self.tracks):
            writer(u"track count: %d" % ( len(self.tracks) ))
            for i in xrange(0, len(self.tracks)):
                t = self.tracks[i]
                t.dumpState(writer)
        else:
            writer(u"this album has no tracks")

    def setArtist(self, ar):
        self.artist = ar
       
    def getArtist(self):
        return self.artist

    def getNumTracks(self):
        '''Return the number of tracks in this album.'''
        return len(self.tracks)

    def getTrackIds(self):
        return self.trackIds.keys()

    def getTrack(self, arg):
        '''Get a specific track from the album. index is the 0 based track number of the track to get.'''
        if arg.__class__.__name__ == "int":
            try:
                return self.tracks[arg]
            except LookupError:
                return None
        else:
            try:
                return self.tracks[self.trackIds[arg]]
            except LookupError:
                return None

    def getTrackFromFileId(self, fileId):
        seq = self.TPFileIdToTrackSeq[fileId]
        if seq is None: return None
        return self.tracks[seq]

    def findSeqOfTrack(self, trackid):
        for i in xrange(0, len(self.tracks)):
            if self.tracks[i].getId() == trackid: return i
        return None

    def isSpecial(self):
        return self.id == u"" or self.id.startswith(u"[")

    def isLoaded(self):
        return self.loaded

    def hasError(self):
        return self.hasLoadError

    def setVariousArtists(self, isVA):
        self.isVA = isVA

    def isVariousArtists(self):
        return self.isVA

    def getArtist(self):
        return self.artist

    def getAmazonAsin(self):
        return self.asin

    def setAmazonAsin(self, asin):
        self.asin = asin

    def getDuration(self):
        return self.duration

    def getDurationString(self):
        if self.duration >= 0:
            ret = u"%d:%02d" % ((self.duration / 60000), ((self.duration % 60000) / 1000))
        else:
            ret = u""

        return ret

    def setDuration(self, duration):
        self.duration = duration

    def getReleaseYear(self):
        return self.releaseYear

    def getReleaseDate(self):
        return self.releaseDate

    def getReleaseMonth(self):
        date = self.releaseDate.split('-')
        return len(date) >= 2 and int(date[1]) or 0

    def getReleaseDay(self):
        date = self.releaseDate.split('-')
        return len(date) >= 3 and int(date[2]) or 0

    def getReleaseCountry(self):
        return self.releaseCountry

    def getReleaseStatus(self):
        return self.releaseStatus

    def getReleaseType(self):
        return self.releaseType

    def getNumUnsavedTracks(self):
        return len(self.unsavedFiles)

    def getNumLinkedTracks(self):
        return self.numLinked

    def getUnmatchedFiles(self):
        return list(self.unmatchedFiles)

    def getNumUnmatchedFiles(self):
        return len(self.unmatchedFiles)

    def addLinkedFile(self, tpfile, seq):
        fileId = tpfile.getFileId()
        assert(fileId >= 0)
        assert(fileId not in self.TPFileIdToTrackSeq)

        track = self.tracks[seq]
        if track.isLinked():
            othertpfile = self.parent.tpmanager.findFile(track.getFileId())
            self.removeFile(othertpfile)
            self.addUnlinkedFile(othertpfile)

	    assert(not track.isLinked())

        self.TPFileIdToTrackSeq[fileId] = seq
        self.numLinked = self.numLinked + 1
        track.link(fileId)
        self.checkTrackUnsaved(tpfile)

        wx.PostEvent(self.parent, events.AlbumTrackCountsChangedEvent(self.getId()))
        wx.PostEvent(self.parent, events.AlbumFilesAddedEvent(self.getId(), [fileId]))
        wx.WakeUpIdle()

    def addUnlinkedFile(self, tpfile):
        fileId = tpfile.getFileId()
        assert(fileId >= 0)

        assert(fileId not in self.TPFileIdToTrackSeq)
        self.TPFileIdToTrackSeq[fileId] = None
        self.unmatchedFiles.append(fileId)

        wx.PostEvent(self.parent, events.AlbumTrackCountsChangedEvent(self.getId()))
        wx.PostEvent(self.parent, events.AlbumFilesAddedEvent(self.getId(), [fileId]))
        wx.WakeUpIdle()

    def removeFile(self, tpfile):
        fileId = tpfile.getFileId()
        assert(fileId >= 0)
        assert(fileId in self.TPFileIdToTrackSeq)

        seq = self.TPFileIdToTrackSeq[fileId]
        del self.TPFileIdToTrackSeq[fileId]

        if seq is None:
            self.unmatchedFiles.remove(fileId)
            wx.PostEvent(self.parent, events.AlbumTrackCountsChangedEvent(self.getId()))
            assert(fileId not in self.unsavedFiles)
        else:
            track = self.tracks[seq]
            track.unlink()
            self.numLinked = self.numLinked - 1
            if fileId in self.unsavedFiles:
                self.unsavedFiles.remove(fileId)

        wx.PostEvent(self.parent, events.AlbumTrackCountsChangedEvent(self.getId()))
        wx.PostEvent(self.parent, events.AlbumFilesRemovedEvent(self.getId(), [fileId]))
        wx.WakeUpIdle()

    def removeAllFilesFromSystem(self):
        fileIds = self.TPFileIdToTrackSeq.keys()
        for fileId in fileIds:
            tpfile = self.parent.tpmanager.findFile(fileId)
            tpfile.removeFromSystem()

    def checkTrackUnsaved(self, tpfile):
        fileId = tpfile.getFileId()
        assert(fileId >= 0)
        assert(self.TPFileIdToTrackSeq[fileId] is not None)

        tr = tpfile.getTrack()
        hasChanged = tr.hasChanged()
        tpfile.releaseTrack(tr)

        if hasChanged:
            if fileId not in self.unsavedFiles:
                self.unsavedFiles.append(fileId)
                wx.PostEvent(self.parent, events.AlbumTrackCountsChangedEvent(self.getId()))
                wx.WakeUpIdle()
        else:
            if fileId in self.unsavedFiles:
                self.unsavedFiles.remove(fileId)
                wx.PostEvent(self.parent, events.AlbumTrackCountsChangedEvent(self.getId()))
                wx.WakeUpIdle()

    def _reverseSortName(self, sortName):
        chunks = sortName.split(',')
        if len(chunks) == 2:
            return "%s %s" % (chunks[1].strip(), chunks[0].strip())
        if len(chunks) == 3:
            return "%s %s %s" % (chunks[2].strip(), chunks[1].strip(), \
                                 chunks[0].strip())
        if len(chunks) == 4:
            return "%s %s, %s %s" % (chunks[1].strip(), chunks[0].strip(), \
                                     chunks[3].strip(), chunks[2].strip())
        else:
            return sortName
                
    def _translateArtist(self, name, sortName):
        isLatin = True
        for c in name:
            ctg = unicodedata.category(c)
            if ctg[0] in ['P', 'Z'] or ctg in ['Nd'] or \
                unicodedata.name(c).find('LATIN') != -1:
                continue
            isLatin = False
            break
        if not isLatin:
            result = []
            chunks = sortName.split('&')
            for chunk in chunks:
                result.append(self._reverseSortName(chunk.strip()))
            return ' & '.join(result)
        return name
                
    def load(self):
        if not self.getId():
            raise ValueError, "No album id given."

        self.hasLoadError = False
        
        ws = self.config.getWebService()
        query = Query(ws)
        try:
            inc = ReleaseIncludes(artist=True, releaseEvents=True, discs=True, tracks=True)
            mbAlbum = query.getReleaseById(self.getId(), inc)
        except WebServiceError, e:
            self.hasLoadError = True
            raise AlbumLoadError, u'Error: ' + str(e)

        self.name = mbAlbum.getTitle()
        if self.name == NON_ALBUM_TRACKS:
            self.name = self.config.settingNonAlbumTracks
        numTracks = len(mbAlbum.getTracks())

        # VA check
        self.isVA = not mbAlbum.isSingleArtistRelease()

        # Artist
        mbAlbumArtist = mbAlbum.getArtist()
        if mbAlbumArtist.getId() == VARIOUS_ARTISTS_ID:
            artistName = self.config.settingVA
            sortName = self.config.settingVA
        else:
            artistName = mbAlbumArtist.getName()
            sortName = mbAlbumArtist.getSortName()
            if self.config.settingArtistTranslation:
                artistName = self._translateArtist(artistName, sortName)
        self.artist = artist.Artist(self.config, artistName, sortName, extractUuid(mbAlbumArtist.getId(), 'artist'))

        # ASIN
        self.asin = mbAlbum.getAsin()

        # Release type and status        
        self.releaseType = u''
        self.releaseStatus = u''
        for type_ in mbAlbum.getTypes():
            if type_ in [mbAlbum.TYPE_OFFICIAL, mbAlbum.TYPE_PROMOTION, mbAlbum.TYPE_BOOTLEG]:
                self.releaseStatus = extractFragment(type_, NS_MMD_1)
            else:
                self.releaseType = extractFragment(type_, NS_MMD_1)
        
        # Release date and country
        self.releaseCountry = u''
        self.releaseYear = 0
        self.releaseDate = u''
        for event in mbAlbum.getReleaseEvents():
            releaseDate = event.getDate()
            releaseCountry = event.getCountry()
            releaseYear = int(releaseDate[0:4])
            if releaseYear < self.releaseYear or self.releaseYear == 0:
                self.releaseYear = releaseYear
                self.releaseCountry = releaseCountry
                self.releaseDate = releaseDate            
        
        # Tracks
        self.trackIds = {}
        self.duration = 0
        i = 1
        for mbTrack in mbAlbum.getTracks():
            
            name = mbTrack.getTitle()
            duration = mbTrack.getDuration() or 0
            trackId = extractUuid(mbTrack.getId(), 'track')

            self.duration += int(duration)

            try:
                tr = self.tracks[i-1]
                tr.setName(name)
                tr.setId(trackId)
            except IndexError:
                tr = track.Track(self.config, name, trackId)
                self.tracks.append(tr)
               
            tr.setNum(i)
            tr.setDuration(duration)
            tr.setAlbum(self)
            if self.isVA:
                mbArtist = mbTrack.getArtist()
                if mbArtist == None:
                    mbArtist = mbAlbumArtist
                artistName = mbArtist.getName()
                artistId = extractUuid(mbArtist.getId(), 'artist')
                sortName = mbArtist.getSortName()
                if self.config.settingArtistTranslation:
                    artistName = self._translateArtist(artistName, sortName)
                tr.setArtist(artist.Artist(self.config, artistName, sortName, artistId))
            else:
                tr.setArtist(self.artist)
               
            tr.update()
            if tr.hasChanged() and (tr.getFileId() not in self.unsavedFiles):
                self.unsavedFiles.append(tr.getFileId())
               
            self.trackIds[tr.getId()] = i-1
            i += 1

        for i in xrange(numTracks + 1, len(self.tracks)):
            tr = self.tracks[i - 1]
            if tr.isLinked():
                fileId = tpfile.getFileId()
                del self.TPFileIdToTrackSeq[fileId]
                track.unlink()
                self.numLinked = self.numLinked - 1
            else:
                self.unmatchedFiles.remove(fileId)
            if fileId in self.unsavedFiles:
                self.unsavedFiles.remove(fileId)
               
        self.loaded = True

    def setServerAffinity(self, server, port):
        '''
        This sets the server that this album was loaded from. Future updates about
        this album should be pulled from the same server. If no affinity servers are
        set, the server in the config settings should be used.
        '''
        self.affServer = server
        self.affPort = port
  
    def getServerAffinity(self):
        '''
        Returns a tuple of the server and port that this album was loaded from. 
        See setServerAffinity()
        '''
        return (self.affServer, self.affPort)

