<?php
/*  
 *  COPYRIGHT
 *  ---------
 *
 *  See ../AUTHORS file
 *
 *
 *  LICENSE
 *  -------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Revision: 1.13 $
 *
 *  ABOUT
 *  -----
 *
 *  Provides functions for language detection.
 *
 */

/* We dont have any better place to put this right now... */
function str_rand($length = 8, $seeds = 'abcdefghijklmnopqrstuvwxyz0123456789') {
    $str = '';
    $seeds_count = strlen($seeds);
  
    // Seed
    //list($usec, $sec) = explode(' ', microtime());
    //$seed = (float) $sec + ((float) $usec * 100000);
    //mt_srand($seed);
  
    // Generate
    for ($i = 0; $length > $i; $i++) {
        $str .= $seeds{mt_rand(0, $seeds_count - 1)};
    }
  
    return $str;
   }

/**
 * This class provides the basic connectivity to the Kolab LDAP
 * server. It also offers some common utilities.
 *
 * $Header: /home/kroupware/jail/kolabrepository/server/php-kolab/Kolab_Webadmin/Webadmin/ldap.class.php,v 1.13 2007/09/18 06:08:48 gunnar Exp $
 *
 * @author  Steffen Hansen <steffen@klaralvdalens-datakonsult.se>
 * @author  Gunnar Wrobel  <wrobel@pardus.de>
 * @package Kolab_Webadmin
 */
class KolabLDAP {

    /**
     * The LDAP connection id.
     *
     * @var int
     */
    var $_ldap;

    /**
     * The base dn of the server.
     *
     * @var string
     */
    var $base_dn;

    /**
     * Default DN for binding.
     *
     * @var string
     */
    var $_default_dn;

    /**
     * Default password for binding.
     *
     * @var string
     */
    var $_default_pw;

    /**
     * Are we bound to the server?
     *
     * @var boolean
     */
    var $_bound = false;

    /**
     * Current DN bound to the server.
     *
     * @var mixed
     */
    var $_bind_dn = false;

    /**
     * A cache for search results.
     *
     * @var mixed
     */
    var $_search_result = false;

    /**
     * A cache for domains.
     *
     * @var mixed
     */
    var $_domains = false;

    /**
     * Initialize the LDAP connection
     *
     * @param array $params The configuration parameters for the kolab
     * web admin
     */
    function KolabLDAP($uri, $base_dn, $default_dn = '', $default_pw = '') 
    {
        $this->base_dn     = $base_dn;
        $this->_default_dn = $default_dn;
        $this->_default_pw = $default_pw;

        // Always connect to master server
        $this->_ldap = ldap_connect($uri);

        if (!ldap_set_option($this->_ldap, LDAP_OPT_PROTOCOL_VERSION, 3)) {
            return PEAR::raiseError(_("Error setting LDAP protocol to v3. Please contact your system administrator"));
        }
    }

    /**
     * Close the LDAP connection
     */
    function close() {
        if ($this->_search_result) {
            @ldap_free_result($this->_search_result);
        }
        $rc = @ldap_close($this->_ldap);
        $this->_ldap = $this->_bound = false;
    }

    /**
     * Get the last LDAP error
     *
     * @return string The last LDAP error
     */
    function error() {
        return ldap_error($this->_ldap);
    }

    /**
     * Get the last LDAP error number
     *
     * @return int The last LDAP error number
     */
    function errno() {
        return ldap_errno($this->_ldap);
    }

    /**
     * Escape LDAP values
     *
     * @param string $str The string to escape
     *
     * @return string The safe string
     */
    function escape($str) {
        /*
         From RFC-2254:

         If a value should contain any of the following characters

         Character       ASCII value
         ---------------------------
         *               0x2a
         (              0x28
         )               0x29
         \               0x5c
         NUL             0x00

         the character must be encoded as the backslash '\' character (ASCII
         0x5c) followed by the two hexadecimal digits representing the ASCII
         value of the encoded character. The case of the two hexadecimal
         digits is not significant.
        */
        $str = str_replace('\\', '\\5c', $str);
        $str = str_replace('*',  '\\2a', $str);
        $str = str_replace('(',  '\\28', $str);
        $str = str_replace(')',  '\\29', $str);
        $str = str_replace('\0', '\\00', $str);
        return $str;
    }

    /**
     * Escape LDAP DN values
     *
     * @param string $str The string to escape
     *
     * @return string The safe string
     */
    function dn_escape($str) {
        /*
         DN component escaping as described in RFC-2253
        */
        $str = str_replace('\\', '\\\\', $str);
        $str = str_replace(',', '\\,', $str);
        $str = str_replace('+', '\\,', $str);
        $str = str_replace('<', '\\<', $str);
        $str = str_replace('>', '\\>', $str);
        $str = str_replace(';', '\\;', $str);
        if ($str[0] == '#') $str = '\\'.$str;
        // PENDING(steffen): Escape leading/trailing spaces
        return $str;
    }
  
    /**
     * Bind to the LDAP server
     *
     * @param string $dn The DN used for binding
     * @param string $pw The password used for binding
     *
     * @return mixed True if binding succeeds, a PEAR error otherwise
     */
    function bind($dn = false, $pw = '') {
        if (!$dn) {
            // Default ldap auth
            $dn = $this->_default_dn;
            $pw = $this->_default_pw;
        }
        $this->_bound = @ldap_bind($this->_ldap, $dn, $pw);
        if ($this->_bound) {
            $this->_bind_dn = $dn;
        } else {
            $this->_bind_dn = false;
            return PEAR::raiseError(sprintf(_("LDAP Error: Binding failed: %s"),
                                            $this->error()));
        }
        return true;
    }

    /**
     * Search for an object.
     *
     * @param string  $base_dn    The base for the search
     * @param string  $filter     Filter criteria
     * @param array   $attributes Restrict the search result to 
     *                            these attributes
     * @param boolean $attrsonly  Don't return the values
     * @param int     $sizelimit  Limit to this number of entries
     * @param int     $timelimit  Limit to this number of seconds
     *
     * @return mixed A LDAP search result or a PEAR error
     */
    function search($base_dn, 
                    $filter,
                    $attributes = null) {
        if (isset($attributes)) {
            $result = ldap_search($this->_ldap, $base_dn, $filter, $attributes);
        } else {
            $result = ldap_search($this->_ldap, $base_dn, $filter);
        }
        if (!$result && $this->errno()) {
            return PEAR::raiseError(sprintf(_("LDAP Error: Failed to search using filter %s. Error was: %s"),
                                            $filter, $this->error()));
        }
        return $result;
    }

    /**
     * Sort a result
     *
     * @param int    $result The result ID
     * @param string $attr   The attribute used as sort criterium
     */
    function sort($result, $attr) 
    {
        ldap_sort($this->_ldap, $result, $attr);
    }

    /**
     * Retrieve results.
     *
     * @param int  $result The result ID
     *
     * @return array The result entries
     */
    function getEntries($result) 
    {
        return ldap_get_entries($this->_ldap, $result);
    }

    /**
     * Retrieve a certain section of the results.
     *
     * @param int  $result  The result ID
     * @param int  $from    Get the results starting here, 
     *                      including this entry
     * @param int  $to      Get the results ending here,
     *                      excluding this entry
     *
     * @return array The result entries
     */
    function getEntrySection($result, $from, $to)
    {
        $entries = array();
        if ($to <= $from) {
            return $entries;
        }
        
        $entry = ldap_first_entry($this->_ldap, $result);
        $current = 0;
        
        while( $entry ) {
            if ($current >= $to) {
                break;
            }
            if ($current >= $from) {
                $entries[] = ldap_get_attributes($this->_ldap, $entry);
            }
            $entry = ldap_next_entry($this->_ldap, $entry);
            $current++;
        }
        return $entries;
    }

    /**
     * Retrieve a certain section of the results.
     *
     * @param int    $result     The result ID
     * @param string $base_dn    Base DN for the search
     * @param array  $attributes Attributes to return with the search
     * @param string $sort       Sort criterium
     * @param int    $perpage    Maximal entries per page
     * @param int    $page       Start with this page
     *
     * @return array The result entries
     */
    function getPagedResult($base_dn, $filter, $attributes, 
                            $sort, $perpage, $page)
    {
        $result = $this->search($base_dn, $filter, $attributes);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        if( $result ) {

            $this->sort($result, $sort);

            $from = $page * $perpage;
            $to   = ($page + 1) * $perpage;

            return $this->getEntrySection($result, $from, $to);
        }
        return array();
    }

    /**
     * Read object data.
     *
     * @param string $dn The object to retrieve.
     *
     * @return mixed An array of attributes if reading succeeds, a
     * PEAR error otherwise
     */
    function read($dn, $attributes = null) {
        if (isset($attributes)) {
            $result = @ldap_read($this->_ldap, $dn, '(objectclass=*)', $attributes);
        } else {
            $result = @ldap_read($this->_ldap, $dn, '(objectclass=*)');
        }
        if (!$result) {
            return PEAR::raiseError(sprintf(_("LDAP Error: No such object: %s."),
                                            $dn));
        }
        $entry = @ldap_first_entry($this->_ldap, $result);
        if (!$entry) {
            ldap_free_result($result);
            return PEAR::raiseError(sprintf(_("LDAP Error: Empty result for: %s."),
                                            $dn));
        }
        $ldap_object = ldap_get_attributes($this->_ldap, $entry);
        ldap_free_result($result);
        return $ldap_object;
    }

    /**
     * Count the number of entries in a LDAP search result.
     *
     * @param array $result   The search result
     *
     * @return int The number of entries
     */
    function count($result) {
        return ldap_count_entries($this->_ldap, $result);
    }

    /**
     * Get the LDAP object classes for the given DN
     *
     * @param string $dn DN of the object
     *
     * @return array An array of object classes
     */
    function getObjectClasses($dn) {
        $ldap_object = $this->read($dn, array('objectClass'));
        if (is_a($ldap_object, 'PEAR_Error')) {
            return $ldap_object;
        }
        if( !$ldap_object && $this->errno()) {
            return PEAR::raiseError(sprintf(_("LDAP Error: No such dn: %s: %s"), 
                                            $dn, $this->error()));
        }
        unset($ldap_object['count']);
        unset($ldap_object['objectClass']['count']);
        return $ldap_object['objectClass'];
    }

    /**
     * Retrieve a specified attribute from a LDAP object
     *
     * @param string $dn   DN of the object
     * @param string $attr The attribute to be read
     *
     * @return mixed The attribute value or a PEAR error
     */
    function attrForDn($dn, $attr) {
        $result = ldap_read($this->_ldap, $dn,
                            '(objectclass=*)',
                            array($attr));
        if (!$result) {
            return PEAR::raiseError(sprintf(_("LDAP Error reading %s for DN %s: %s"), 
                                            $attr, $dn, ldap_error($this->_ldap)));
        }
        $entries = ldap_get_entries($this->_ldap, $result);
        ldap_free_result($result);
        if ($entries['count'] != 1) {
            return PEAR::raiseError(sprintf(_("No or several object(s) with DN %s."), 
                                            $dn));
        }
        return $entries[0][$attr][0];
    }
    
    /**
     * Retrieve the uid from a LDAP object
     *
     * @param string $dn   DN of the object
     *
     * @return mixed The uid or a PEAR error
     */
    function uidForDn($dn) {
        return $this->attrForDn($dn, 'uid');
    }

    /**
     * Retrieve the mail address from a LDAP object
     *
     * @param string $dn   DN of the object
     *
     * @return mixed The mail address or a PEAR error
     */
    function mailForDn($dn) {
        return $this->attrForDn($dn, 'mail');
    }

    /**
     * Retrieve the mail alias from a LDAP object
     *
     * @param string $dn   DN of the object
     *
     * @return mixed The mail alias or a PEAR error
     */
    function aliasForDn($dn) {
        return $this->attrForDn($dn, 'alias');
    }

    /**
     * Identify the DN of the first result entry
     *
     * @param array $result The LDAP search result
     *
     * @return mixed The DN, false or a PEAR error
     */
    function dnFromResult($result) {
        $entry = @ldap_first_entry($this->_ldap, $result);
        if (!$entry && $this->errno()) {
            return PEAR::raiseError(sprintf(_("Search result contained no entries. Error was: %s"),
                                            $this->error()));
        }
        if (!$entry) {
            return false;
        }
        return ldap_get_dn($this->_ldap, $entry);
    }
    
    /**
     * Identify the DN for the first object found using a filter
     *
     * @param string $filter The LDAP filter to use
     *
     * @return mixed The DN or a PEAR error
     */
    function dnForFilter($filter) {
        $result = $this->search($this->base_dn, $filter, array());
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        return $this->dnFromResult($result);
    }


    /**
     * Count the number of occurences of an email address
     * in users' mail and alias attributes and in dist. lists.
     * This can be used to check for uniqueness etc.
     *
     * @param string $mail      Search for objects with this mail address
     *                          alias, or uid.
     * @param string $base      Search base.
     * @param string $excludedn Exclude this DN when counting
     *
     * @return int The number of occurences
     */
    function countMail($mail, $base = null, $excludedn = false) {
        if (empty($base)) {
            $base = $this->base_dn;
        }

        // First count users
        $filter = '(|(|(mail=' . $this->escape($mail) . 
            ')(alias=' . $this->escape($mail) . 
            '))(uid=' . $this->escape($mail) . '))';

        $result = $this->search($base, $filter, array());
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        $entries = $this->getEntries($result);
        $count = $entries['count'];

        $dns = array();
        for ($i = 0; $i < count($entries); $i++) {
            if (isset($entries[$i]['dn'])) {
                $dns[] = $entries[$i]['dn'];
                if ($excludedn && $entries[$i]['dn'] == $excludedn) {
                    $count--;
                }
            }
        } 

        // Now count dist. lists (this might still be relevant for old
        // servers where dist lists don't have a mail attribute.)
        $cn = substr($mail, 0, strpos($mail, '@'));
        $filter = '(&(objectClass=kolabGroupOfNames)(cn=' . 
            $this->escape($cn) . '))';
        $result = $this->search($base, $filter, array());
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        $entries = $this->getEntries($result);

        for ($i = 0; $i < count($entries); $i++) {
            if (isset($entries[$i]['dn'])) {
                if (in_arry($entries[$i]['dn'], $dns) ||
                    ($excludedn && $entries[$i]['dn'] == $excludedn)) {
                    continue;
                } else {
                    $count++;
                }
            }
        }
        return $count;
    }

    // Set deleflag on object, or if $delete_now is
    // true, just delete it
    function deleteObject($dn, $delete_now = false) {
        return $this->_doDeleteObject($dn, $delete_now, true);
    }

    // Private
    function _doDeleteObject($dn, $delete_now = false, $nuke_password = false) {
        if ($delete_now) {
            if (!ldap_delete($this->_ldap, $dn)) {
                return false;
            }
        } else {
            // Look up hostnames in this setup
            $kolab_obj = $this->read('k=kolab,'.$this->base_dn);
            if (!$kolab_obj) return false;
            $delete_template = array();
            $delete_template['kolabDeleteflag'] = $kolab_obj['kolabHost'];      
            unset($delete_template['kolabDeleteflag']['count']);
            if ($nuke_password) {
                // Write random garbage into passwd field to lock the user out
                $delete_template['userPassword'] = '{sha}'.base64_encode(pack('H*', 
                                                                              sha1(str_rand(32))));
            }
            if (!ldap_modify($this->_ldap,$dn,$delete_template)) {
                return false;
            }
        }
        return true;
    }

    /**
     * Get first name
     *
     * @param string  $sn The last name
     * @param string  $cn The common name
     *
     * @return string The extracted first name
     */
    function getFirstName($sn, $cn) 
    {
        return substr($cn, 0, strlen($cn) - strlen($sn));
    }
};

/*
 Local variables:
 mode: php
 indent-tabs-mode: f
 tab-width: 4
 buffer-file-coding-system: utf-8
 End:
 vim:encoding=utf-8:
*/
?>
