<?php
/*  
 *  COPYRIGHT
 *  ---------
 *
 *  See ../AUTHORS file
 *
 *
 *  LICENSE
 *  -------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Revision: 1.4 $
 *
 *  ABOUT
 *  -----
 *
 *  A class for handling Kolab objects in the LDAP db.
 *
 */

/** Provide access to the Kolab specific objects. */
require_once 'Kolab/Webadmin/address.class.php';
require_once 'Kolab/Webadmin/administrator.class.php';
require_once 'Kolab/Webadmin/domainmaintainer.class.php';
require_once 'Kolab/Webadmin/group.class.php';
require_once 'Kolab/Webadmin/maintainer.class.php';
require_once 'Kolab/Webadmin/sharedfolder.class.php';
require_once 'Kolab/Webadmin/user.class.php';

/**
 * This class provides methods to deal with Kolab objects stored in
 * the LDAP db.
 *
 * $Header: /home/kroupware/jail/kolabrepository/server/php-kolab/Kolab_Webadmin/Webadmin/db.php,v 1.4 2007/09/18 06:08:48 gunnar Exp $
 *
 * @author  Gunnar Wrobel  <wrobel@pardus.de>
 * @package Kolab_Webadmin
 */
class KolabObjectDb {

    /**
     * Link into the Kolab LDAP db
     *
     * @var KolabLDAP
     */
    var $_ldap;

    /**
     * Valid Kolab object types
     *
     * @var array
     */
    var $valid_types = array(
        KOLAB_OBJECT_ADDRESS,
        KOLAB_OBJECT_ADMINISTRATOR,
        KOLAB_OBJECT_DOMAINMAINTAINER,
        KOLAB_OBJECT_GROUP,
        KOLAB_OBJECT_MAINTAINER,
        KOLAB_OBJECT_SHAREDFOLDER,
        KOLAB_OBJECT_USER,
    );

    /**
     * Initialize the Kolab Object DB
     *
     * @param KolabLDAP $ldap The link into the Kolab LDAP db
     */
    function KolabObjectDb($ldap)
    {
        $this->_ldap = $ldap;
    }

    /**
     * Fetch a Kolab object
     *
     * @param string $dn   The DN of the object to fetch
     * @param string $type The type of the object to fetch
     *
     * @return KolabObject The corresponding Kolab object
     */
    function fetch($dn, $type = null)
    {
        if (empty($type)) {
            $type = $this->determineType($dn);
            if (is_a($type, 'PEAR_Error')) {
                return $type;
            }
        } else {
            if (!in_array($type, $this->valid_types)) {
                return PEAR::raiseError(sprintf(_("Invalid Kolab object type \"%s\"."), 
                                                $type));
            }
        }

        $class = 'Kolab' . $type;
        if (class_exists($class)) {
            $object = &new $class($this->_ldap, $dn);
            return $object;
        } else {
            return PEAR::raiseError(sprintf(_("Failed to load Kolab object type %s"),
                                            $type));
        }
    }

    /**
     * List all objects of a specific type
     *
     * @param string $type The type of the objects to be listed
     *
     * @return mixed An array of Kolab objects or a PEAR error
     */
    function listObjects($type, $base_dn = null, $addfilter = '', $perpage = 50, $page = 1)
    {
        if (!in_array($type, $this->valid_types)) {
            return PEAR::raiseError(sprintf(_("Invalid Kolab object type \"%s\"."), 
                                            $type));
        }

        if (empty($base_dn)) {
            $base_dn = $this->_ldap->base_dn;
        }

        $class = 'Kolab' . $type;
        $const = get_class_vars($class);
        if (empty($addfilter)) {
            $filter = $const['filter'];
        } else {
            $filter = '(&(' . $addfilter . ')(' . $const['filter'] . ')';
        }
        $attributes = $const['list_attributes'];
        $sort = $const['sort'];

        $result = $this->_ldap->getPagedResult($base_dn, $filter, $attributes,
                                               $sort, $perpage, $page);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        $entries = array();
        foreach($result as $attrs) {
            $entries[] =&new $class($this->_ldap, null, $attrs);
        }
        return $entries;
    }

    /**
     * Determine the type of a Kolab object
     *
     * @param string $dn The DN of the object to fetch
     *
     * @return KolabObject The corresponding Kolab object
     */
    function determineType($dn) 
    {
        $oc = $this->_ldap->getObjectClasses($dn);
        if (is_a($oc, 'PEAR_Error')) {
            return $oc;
        }
        // Not a user type?
        if (!in_array('kolabInetOrgPerson', $oc)) {
            // Is it a group?
            if (in_array('kolabGroupOfNames')) {
                return KOLAB_OBJECT_GROUP;
            }
            // Is it a shared Folder?
            if (in_array('kolabSharedFolder')) {
                return KOLAB_OBJECT_SHAREDFOLDER;
            }
            return PEAR::raiseError(sprintf(_("Unkown Kolab object type for DN %s."), 
                                            $dn));
        }

       
        $filter = '(member=' . $this->_ldap->escape($dn) . ')';
        $result = $this->_ldap->search('cn=admin,cn=internal,' . 
                                       $this->_ldap->base_dn, $filter);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if ($this->_ldap->count($result) > 0) {
            return KOLAB_OBJECT_ADMINISTRATOR;
        }
        $result = $this->_ldap->search('cn=maintainer,cn=internal,' . 
                                       $this->_ldap->base_dn, $filter);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if ($this->_ldap->count($result) > 0) {
            return KOLAB_OBJECT_MAINTAINER;
        }
        $result = $this->_ldap->search('cn=domain-maintainer,cn=internal,' . 
                                       $this->_ldap->base_dn, $filter);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        if ($this->_ldap->count($result) > 0) {
            return KOLAB_OBJECT_DOMAINMAINTAINER;
        }

        if (strpos($dn, 'cn=internal') !== false) {
            return KOLAB_OBJECT_ADDRESS;
        }
        
        return KOLAB_OBJECT_USER;
    }

    /**
     * Identify the DN for the first object found using a specified
     * attribute value.
     *
     * @param string $attr  The name of the attribute used for searching
     * @param string $value The desired value of the attribute
     *
     * @return mixed The DN or a PEAR error
     */
    function dnForAttr($attr, $value) {
        $filter = '(&(objectclass=kolabInetOrgPerson)(' . $attr . 
            '=' . $this->_ldap->escape($value) . '))';
        $result = $this->_ldap->search($this->_ldap->base_dn, $filter, array());
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
        return $this->_ldap->dnFromResult($result);
    }
    
    /**
     * Identify the DN for the first object found with the given uid.
     *
     * @param string $uid  Search for objects with this uid
     *
     * @return mixed The DN or a PEAR error
     */
    function dnForUid($uid) {
        return $this->dnForAttr('uid', $uid);
    }

    /**
     * Identify the DN for the first object found with the given mail.
     *
     * @param string $mail  Search for objects with this mail address.
     *
     * @return mixed The DN or a PEAR error
     */
    function dnForMail($mail) {
        return $this->dnForAttr('mail', $mail);
    }

    /**
     * Identify the DN for the first object found with the given uid or mail
     *
     * @param string $id  Search for objects with this uid/mail
     *
     * @return mixed The DN or a PEAR error
     */
    function dnForUidOrMail($id) {
        $dn = $this->dnForAttr('uid', $id);
        if (!$dn) {
            $dn = $this->dnForAttr('mail', $id);
        }
        return $dn;
    }

    /**
     * Identify the DN for the first object found with the given alias.
     *
     * @param string $mail  Search for objects with this mail alias.
     *
     * @return mixed The DN or a PEAR error
     */
    function dnForAlias($mail) {
        return $this->dnForAttr('alias', $mail);
    }

    /**
     * Identify the DN for the first object found with the given mail
     * address or alias.
     *
     * @param string $mail Search for objects with this mail address
     * or alias.
     *
     * @return mixed The DN or a PEAR error
     */
    function dnForMailOrAlias($mail) {
        return $this->_ldap->dnForFilter('(&(objectclass=kolabInetOrgPerson)(|(mail=' . 
                                         $this->_ldap->escape($mail) . ')(alias=' . 
                                         $this->_ldap->escape($mail) . ')))');
    }
};

?>
