<?php
/*  
 *  COPYRIGHT
 *  ---------
 *
 *  See ../AUTHORS file
 *
 *
 *  LICENSE
 *  -------
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 *  $Revision: 1.10 $
 *
 *  ABOUT
 *  -----
 *
 *  This provides the necessary utilities to handle the file based
 *  free/busy cache.
 *
 */

/* We require the iCalendar library to build the free/busy list */
require_once 'Horde/iCalendar.php';
require_once 'Horde/iCalendar/vfreebusy.php';

class FreeBusyCache {

    var $_cache_dir;

    function FreeBusyCache($cache_dir)
    {
        $this->_cache_dir = $cache_dir;
    }

    function store($access) 
    {
        /* Now we really need the free/busy library */
        require_once('Horde/Kolab/Freebusy.php');

        $fb = &new Horde_Kolab_Freebusy();
        $result = $fb->connect($access->imap_folder);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        $vCal = $fb->generate(null, null, $access->fbpast, $access->fbfuture,
                              $access->owner, $access->cn);
        if (is_a($vCal, 'PEAR_Error')) {
            $vCal;
        }

        $fbfilename = $this->_getFilename($access->folder, $access->owner);
        
        $c_pvcal = &new FreeBusyCacheFile_pvcal($this->_cache_dir, $fbfilename);
        $c_acl   = &new FreeBusyCacheFile_acl($this->_cache_dir, $fbfilename);
        $c_xacl  = &new FreeBusyCacheFile_xacl($this->_cache_dir, $fbfilename);
        
        /* missing data means delete the cache files */
        if (empty($vCal)) {
            Horde::logMessage(sprintf(_("No events. Purging cache %s."), 
                                      $fbfilename),
                              __FILE__, __LINE__, PEAR_LOG_DEBUG);

            $result = $c_pvcal->purge();
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
            $result = $c_acl->purge();
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
            $result = $c_xacl->purge();
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }

        } else {

            $result = $c_pvcal->storePVcal($vCal);
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }

            $relevance = $fb->getRelevance();
            if (is_a($relevance, 'PEAR_Error')) {
                return $relevance;
            }
            $acl = $fb->getACL();
            if (is_a($acl, 'PEAR_Error')) {
                return $acl;
            }

            $result = $c_acl->storeACL($acl, $relevance);
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
            
            $xacl = $fb->getExtendedACL();
            if (is_a($xacl, 'PEAR_Error')) {
                return $xacl;
            }

            $result = $c_xacl->storeXACL($xacl, $acl);
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
    
            Horde::logMessage(sprintf(_("FreeBusyCache::store(file=%s, relevance=%s, acl=%s, xacl=%s)"), 
                                      $fbfilename, $relevance, $acl, $xacl), 
                              __FILE__, __LINE__, PEAR_LOG_DEBUG);
        }
    }

    function &loadPartial(&$access, $extended) 
    {
        $file = $this->_getFilename($access->folder, $access->owner);

        $aclcache = &FreeBusyCacheDB_acl::singleton('acl', $this->_cache_dir);

        $relevant = false;
        foreach ($access->groups as $id) {
            if ($aclcache->has($file, $id)) {
                $relevant = true;
                break;
                }
        }

        if (!$relevant) {
            return PEAR::raiseError(sprintf(_("Folder %s is irrelevant for user %s."),
                                            $access->imap_folder, $access->user));
        }

        if ($extended) {
            $extended = $this->_allowExtended($file, $access);
        }

        $c_pvcal = &new FreeBusyCacheFile_pvcal($this->_cache_dir, $file);
        $pvCal = $c_pvcal->loadPVcal($extended);
        if (is_a($pvCal, 'PEAR_Error')) {
            return $pvCal;
        }
        return $pvCal;
    }

    function &load(&$access, $extended) 
    {
        global $conf;

        /* Which files will we access? */
        $aclcache = &FreeBusyCacheDB_acl::singleton('acl', $this->_cache_dir);
        $files = $aclcache->get($access->owner);
        if (is_a($files, 'PEAR_Error')) {
            return $files;
        }
        $filesnames = array();

        $owner = $access->owner;
        if (ereg('(.*)@(.*)', $owner, $regs)) {
            $owner = $regs[2] . '/' . $regs[1];
        }
        $user = $access->user;
        if (ereg('(.*)@(.*)', $user, $regs)) {
            $user = $regs[2] . '/' . $regs[1];
        }
        $c_file = str_replace("\0", '', str_replace('.', '^', $user . '/' . $owner));

        $c_vcal = &new FreeBusyCacheFile_vcal($this->_cache_dir, $c_file);

        /* If the current vCal cache did not expire, we can deliver it */
        if (!$c_vcal->expired($files)) {
            return $c_vcal->loadVcal();
        }
        
        // Create the new iCalendar.
        $vCal = &new Horde_iCalendar();
        $vCal->setAttribute('PRODID', '-//proko2//freebusy 1.0//EN');
        $vCal->setAttribute('METHOD', 'PUBLISH');
        
        // Create new vFreebusy.
        $vFb = &Horde_iCalendar::newComponent('vfreebusy', $vCal);
        $params = array();
        if ($access->cn) {
            $params['cn'] = $access->cn;
        }
        $vFb->setAttribute('ORGANIZER', 'MAILTO:' . $access->owner, $params);
        
        $vFb->setAttribute('DTSTAMP', time());
        $vFb->setAttribute('URL', 'http://' . $_SERVER['SERVER_NAME'] . $_SERVER['REQUEST_URI']);

        $mtimes = array();
        foreach ($files as $file) {
            if ($extended) {
                $extended = $this->_allowExtended($file, $access);
            }
            $c_pvcal = &new FreeBusyCacheFile_pvcal($this->_cache_dir, $file);
            $pvCal = $c_pvcal->loadPVcal($extended);
            if (is_a($pvCal, 'PEAR_Error')) {
                    Horde::logMessage(sprintf(_("Ignoring partial free/busy file %s: %s)"), 
                                              $file, $pvCal->getMessage()), 
                                      __FILE__, __LINE__, PEAR_LOG_INFO);
                    continue;
            }
            $pvFb = &$pvCal->findComponent('vfreebusy');
            if( !$pvFb ) {
                    Horde::logMessage(sprintf(_("Could not find free/busy info in file %s.)"), 
                                              $file), __FILE__, __LINE__, PEAR_LOG_INFO);
                    continue;
            }
            if ($ets = $pvFb->getAttributeDefault('DTEND', false) !== false) {
                // PENDING(steffen): Make value configurable
                if ($ets < time()) {
                    Horde::logMessage(sprintf(_("Free/busy info in file %s is too old.)"), 
                                              $file), __FILE__, __LINE__, PEAR_LOG_INFO);
                    $c_pvcal->purge();
                    continue;
                }
            }
            $vFb->merge($pvFb);

            /* Store last modification time */
            $mtimes[$file] = array($c_pvcal->getFile(), $c_pvcal->getMtime());
        }

        if (!empty($conf['fb']['remote_servers'])) {
            $remote_vfb = $this->_fetchRemote($conf['fb']['remote_servers'],
                                              $access);
            if (is_a($remote_vfb, 'PEAR_Error')) {
                Horde::logMessage(sprintf(_("Ignoring remote free/busy files: %s)"), 
                                          $remote_vfb->getMessage()), 
                                  __FILE__, __LINE__, PEAR_LOG_INFO);
            } else {
                $vFb->merge($remote_vfb);
            }
        }

        if (!(boolean)$vFb->getBusyPeriods()) {
            /* No busy periods in fb list. We have to add a
             * dummy one to be standards compliant
             */
            $vFb->setAttribute('COMMENT', 'This is a dummy vfreebusy that indicates an empty calendar');
            $vFb->addBusyPeriod('BUSY', 0,0, null);
        }

        $vCal->addComponent($vFb);

        $c_vcal->storeVcal($vCal, $mtimes);

        return $vCal;
    }

    function _allowExtended($file, &$access) 
    {
        $xaclcache = &FreeBusyCacheDB_xacl::singleton('xacl', $this->_cache_dir);
            
            Horde::logMessage(sprintf(_("GROUPS: %s."), 
                                      serialize($access->user_groups)),
                              __FILE__, __LINE__, PEAR_LOG_DEBUG);
        foreach ($access->user_groups as $id) {
            if ($xaclcache->has($file, $id)) {
                return true;
            }
        }
        Horde::logMessage(sprintf(_("Extended attributes on folder %s disallowed for user %s."),
                                  $access->folder, $access->user), __FILE__, __LINE__, PEAR_LOG_DEBUG);
        return false;
    }

    /*************** Private API below this line *************/
    function _getFilename($folder, $owner)
    {
        if (ereg('(.*)@(.*)', $owner, $regs)) {
            $owner = $regs[2] . '/' . $regs[1];
        }
        
        return str_replace("\0", '', str_replace('.', '^', $owner . '/' . $folder));
    }

    function &_fetchRemote($servers, $access) 
    {
        $vFb = null;
        
        foreach ($servers as $server) {
            
            $url = 'https://' . urlencode($access->user) . ':' . urlencode($access->pass)
            . '@' . $server . $_SERVER['REQUEST_URI'];
            $remote = @file_get_contents($url);
            if (!$remote) {
                $message = sprintf(_("Unable to read free/busy information from %s"), 
                                   'https://' . urlencode($access->user) . ':XXX'
                                   . '@' . $server . $_SERVER['REQUEST_URI']);
                Horde::logMessage($message, __FILE__, __LINE__, PEAR_LOG_INFO);
            }

            $rvCal = &new Horde_iCalendar();
            $result = $rvCal->parsevCalendar($remote);
            
            if (is_a($result, 'PEAR_Error')) {
                $message = sprintf(_("Unable to parse free/busy information from %s: %s"), 
                                   'https://' . urlencode($access->user) . ':XXX'
                                   . '@' . $server . $_SERVER['REQUEST_URI'], 
                                   $result->getMessage());
                Horde::logMessage($message, __FILE__, __LINE__, PEAR_LOG_INFO);
            }
            
            $rvFb = &$rvCal->findComponent('vfreebusy');
            if (!$pvFb) {
                $message = sprintf(_("Unable to find free/busy information in data from %s."), 
                                   'https://' . urlencode($access->user) . ':XXX'
                                   . '@' . $server . $_SERVER['REQUEST_URI']);
                Horde::logMessage($message, __FILE__, __LINE__, PEAR_LOG_INFO);
            }
            if ($ets = $rvFb->getAttributeDefault('DTEND', false) !== false) {
                // PENDING(steffen): Make value configurable
                if ($ets < time()) {
                    $message = sprintf(_("free/busy information from %s is too old."), 
                                       'https://' . urlencode($access->user) . ':XXX'
                                       . '@' . $server . $_SERVER['REQUEST_URI']);
                    Horde::logMessage($message, __FILE__, __LINE__, PEAR_LOG_INFO);
                }
            }
            if (!empty($vFb)) {
                $vFb->merge($rvFb);
            } else {
                $vFb = $rvFb;
            }
        }
        return $vFb;
    }
};

class FreeBusyCacheDB {

    var $_cache_dir;
    var $_db;
    var $_dbformat;
    var $_type = '';

    function FreeBusyCacheDB($cache_dir) {
        global $conf;

        $this->_cache_dir = $cache_dir;
    
        if (!empty($conf['fb']['dbformat'])) {
            $this->_dbformat = $conf['fb']['dbformat'];
        } else {
            $this->_dbformat = 'db4';
        }

        /* make sure that a database really exists before accessing it */
        if (!file_exists($this->_cache_dir . '/' . $this->_type . 'cache.db')) {
            $result = $this->_open();
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
            $this->_close();
        }

    }

    function _open() 
    {
        if ($this->_db) {
            return;
        }

        $dbfile = $this->_cache_dir . '/' . $this->_type . 'cache.db';
        $this->_db = @dba_open($dbfile, 'cd', $this->_dbformat);
        if ($this->_db === false) {
            return PEAR::raiseError(sprintf(_("Unable to open freebusy cache db %s"), $dbfile));
        }
    }
    
    function _close() 
    {
        @dba_close($this->_db);
        $this->_db = null;
    }

    function _remove($filename, $uid) 
    {
        $result = $this->_open();
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        if (dba_exists($uid, $this->_db)) {
            $lst = dba_fetch($uid, $this->_db);
            $lst = split(',', $lst);
            $lst = array_diff($lst, array($filename));
            $result = dba_replace($uid, join(',', $lst), $this->_db);
            if ($result === false) {
                return PEAR::raiseError(sprintf(_("Unable to set db value for uid %s"), $uid));
            }
        }
    }

    function _add($filename, $uid) 
    {
        $result = $this->_open();
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        if (dba_exists($uid, $this->_db)) {
            $lst = dba_fetch($uid, $this->_db);
            $lst = split(',', $lst);
            $lst[] = $filename;
            $result = dba_replace($uid, join(',', array_unique($lst)), $this->_db);
            if ($result === false) {
                return PEAR::raiseError(sprintf(_("Unable to set db value for uid %s"), $uid));
            }
        } else {
            $result = dba_insert($uid, $filename, $this->_db);
            if ($result === false) {
                return PEAR::raiseError(sprintf(_("Unable to set db value for uid %s"), $uid));
            }
        }
    }

    function has($filename, $uid) 
    {
        $result = $this->_open();

        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        if (dba_exists($uid, $this->_db)) {
            $lst = dba_fetch($uid, $this->_db);
            $lst = split(',', $lst);
            return in_array($filename, $lst);
        }
        return false;
    }
    
    function get($uid) 
    {
        $result = $this->_open();
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        if (dba_exists($uid, $this->_db)) {
            $lst = dba_fetch($uid, $this->_db);
            return split(',', $lst);
        }
        return array();
    }
    
    /**
     * Attempts to return a reference to a concrete FreeBusyACLCache
     * instance. It will only create a new instance if no
     * FreeBusyACLCache instance currently exists.
     *
     * This method must be invoked as:
     *   <code>$var = &FreeBusyACLCache::singleton($cache_dir);</code>
     *
     * @static
     *
     * @param string $type       The type of the cache.
     * @param string $cache_dir  The directory for storing the cache.
     *
     * @return FreeBusyACLCache The concrete FreeBusyACLCache
     *                          reference, or false on an error.
     */
    function &singleton($type, $cache_dir)
    {
        static $cachedb = array();

        $signature = $type . $cache_dir;

        if (empty($cachedb[$signature])) {
            $class = 'FreeBusyCacheDB_' . $type;
            $cachedb[$signature] = &new $class($cache_dir);
        }

        return $cachedb[$signature];
    }
}

class FreeBusyCacheDB_acl extends FreeBusyCacheDB {

    var $_type = 'acl';

    function store($filename, $acl, $oldacl, $perm)
    {
        /* We remove the filename from all users listed in the old ACL first */
        foreach ($oldacl as $ac) {
            $result = $this->_remove($filename, $ac[0]);
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
        }

        /* Now add the filename for all users with the correct permissions */
        if ($perm !== false ) {
            foreach ($acl as $user => $ac) {
                if (strpos($ac, $perm) !== false) {
                    if (!empty($user)) {
                        $result = $this->_add($filename, $user);
                        if (is_a($result, 'PEAR_Error')) {
                            return $result;
                        }
                    }
                }
            }
        }

        $this->_close();
    }
}

class FreeBusyCacheDB_xacl extends FreeBusyCacheDB {

    var $_type = 'xacl';

    function store($filename, $xacl, $oldxacl)
    {
        $xacl = split(' ', $xacl);
        $oldxacl = split(' ', $oldxacl);
        $both = array_intersect($xacl, $oldxacl);

        /* Removed access rights */
        foreach (array_diff($oldxacl, $both) as $uid) {
            if (!empty($uid)) {
                $result = $this->_remove($filename, $uid);
                if (is_a($result, 'PEAR_Error')) {
                    return $result;
                }
            }
        }

        /* Added access rights */
        foreach (array_diff($xacl, $both) as $uid) {
            if (!empty($uid)) {
                $result = $this->_add($filename, $uid);
                if (is_a($result, 'PEAR_Error')) {
                    return $result;
                }
            }
        }

        $this->_close();
    }
}

class FreeBusyCacheFile {

    var $_suffix = '';
    var $_filename;
    var $_file;
    var $_version = 1;

    function FreeBusyCacheFile($cache_dir, $filename, $suffix = null)
    {
        if (!empty($suffix)) {
            $this->_suffix = $suffix;
        }

        $this->_cache_dir = $cache_dir;
        $this->_filename  = $filename;
        $this->_file = $this->_cache_dir . '/' . $this->_filename . '.' . $this->_suffix;
    }

    function getFile()
    {
        return $this->_file;
    }

    function purge()
    {
        if (file_exists($this->_file)) {
            $result = @unlink($this->_file);
            if (!$result) {
                return PEAR::raiseError(sprintf(_("Failed removing file %s"),
                                                $this->_file));
            }
        }
    }

    function store(&$data) 
    {
        /* Create directories if missing */
        $fbdirname = dirname($this->_file);
        if (!is_dir($fbdirname)) {
            $result = $this->_makeTree($fbdirname);
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
        }

        /* Store the cache data */
        $fh = fopen($this->_file, 'w');
        if (!$fh) {
            return PEAR::raiseError(sprintf(_("Failed creating cache file %s!"), 
                                            $this->_file));
        }
        fwrite($fh, serialize(array('version' => $this->_version,
                                    'data' => $data)));
        fclose($fh);
    }

    function &load()
    {
        $file = @file_get_contents($this->_file);
        if ($file === false) {
            return PEAR::raiseError(sprintf(_("%s failed reading cache file %s!"), 
                                            get_class($this), $this->_file));
        }
        $cache = @unserialize($file);
        if ($cache === false) {
            return PEAR::raiseError(sprintf(_("%s failed to unserialize cache data from file %s!"), 
                                            get_class($this), $this->_file));
        }
        if (!isset($cache['version'])) {
            return PEAR::raiseError(sprintf(_("Cache file %s lacks version data!"), 
                                            $this->_file));
        }
        $this->_version = $cache['version'];
        if (!isset($cache['data'])) {
            return PEAR::raiseError(sprintf(_("Cache file %s lacks data!"), 
                                            $this->_file));
        }
        if ($cache['version'] != $this->_version) {
            return PEAR::raiseError(sprintf(_("Cache file %s has version %s while %s is required!"), 
                                            $this->_file, $cache['version'], $this->_version));
        }
        return $cache['data'];
    }
    
    function _maketree($dirname)
    {
        $base = substr($dirname, 0, strrpos($dirname, '/'));
        $base = str_replace(".", "^", $base);
        if (!empty($base) && !is_dir($base)) {
            $result = $this->_maketree($base);
            if (is_a($result, 'PEAR_Error')) {
                return $result;
            }
        }
        if (!file_exists($dirname)) {
            $result = @mkdir($dirname, 0755);
            if (!$result) {
                return PEAR::raiseError(sprintf(_("Error creating directory %s"), $dirname));
            }
        }
    }
}

class FreeBusyCacheFile_pvcal extends FreeBusyCacheFile {

    var $_suffix = 'pvc';

    function storePVcal(&$pvcal)
    {
        return $this->store($pvcal);
    }

    function &loadPVcal($extended)
    {
        $pvcal = $this->load();
        if (is_a($pvcal, 'PEAR_Error')) {
            return $pvcal;
        }
        if (!$extended) {
            $components = &$pvcal->getComponents();
            foreach ($components as $component) {
                if ($component->getType() == 'vFreebusy') {
                    $component->_extraParams = array();
                }
            }
        }
        return $pvcal;
    }

    function getMtime() 
    {
        return filemtime($this->_file);
    }
}

class FreeBusyCacheFile_vcal extends FreeBusyCacheFile {

    var $_suffix = 'vc';

    var $_version = 2;

    var $_data;

    function storeVcal(&$vcal, &$mtimes) 
    {
        $data = array('vcal' => $vcal,
                      'mtimes' => $mtimes);
        return $this->store($data);
    }
    
    function &loadVcal()
    {
        if ($this->_data) {
            return $this->_data;
        }
        
        $result = $this->load();
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        $this->_data = $result['vcal'];

        return $this->_data;
    }

    function expired($files)
    {
        $result = $this->load();
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        /* Check the cache version */
        if ($this->_version < 2) {
            return true;
        }

        $this->_data = $result['vcal'];

        /* Files changed? */
        $keys = array_keys($result['mtimes']);
        $changes = array_diff($keys, $files);        
        if (count($keys) != count($files) || !empty($changes)) {
            return true;
        }

        /* Check the file ctimes */
        foreach ($files as $file) {
            if (filemtime($result['mtimes'][$file][0]) != $result['mtimes'][$file][1]) {
                return true;
            }
        }

        /* Older than three days? */
        $components = $this->_data->getComponents();
        foreach ($components as $component) {
            if ($component->getType() == 'vFreebusy') {
                $attr = $component->getAttribute('DTSTAMP');
                if (!empty($attr) && !is_a($attr, 'PEAR_Error')) {
                    //Should be configurable
                    if (time() - (int)$attr > 259200) {
                        return true;
                    }
                }
            }
        }


        return false;
    }
}

class FreeBusyCacheFile_acl extends FreeBusyCacheFile {

    var $_suffix = 'acl';

    var $_acls;
    
    function FreeBusyCacheFile_acl($cache_dir, $filename) 
    {
        $this->_acls = &FreeBusyCacheDB::singleton('acl', $cache_dir);
        parent::FreeBusyCacheFile($cache_dir, $filename, 'acl');
    }

    function purge()
    {
        $oldacl = $this->load();
        if (is_a($oldacl, 'PEAR_Error')) {
            $oldacl = array();
        }
        
        $result = $this->_acls->store($this->_filename, array(), $oldacl, false);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        return parent::purge();
    }
    
    function storeACL(&$acl, $relevance) 
    {
        $oldacl = $this->load();
        if (is_a($oldacl, 'PEAR_Error')) {
            $oldacl = array();
        }

        /* Handle relevance */
        switch ($relevance) {
        case 'readers':
            $perm = 'r';
            break;
        case 'nobody':
            $perm = false;
            break;
        case 'admins':  
        default: 
            $perm = 'a';
        }

        $result = $this->_acls->store($this->_filename, $acl, $oldacl, $perm);
        if (is_a($oldacl, 'PEAR_Error')) {
            return $oldacl;
        }

        $result = $this->store($acl);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
    }
}

class FreeBusyCacheFile_xacl extends FreeBusyCacheFile {

    var $_suffix = 'xacl';

    var $_xacls;
    
    function FreeBusyCacheFile_xacl($cache_dir, $filename) 
    {
        $this->_xacls = &FreeBusyCacheDB::singleton('xacl', $cache_dir);
        parent::FreeBusyCacheFile($cache_dir, $filename, 'xacl');
    }

    function purge()
    {
        $oldxacl = $this->load();
        if (is_a($oldxacl, 'PEAR_Error')) {
            $oldxacl = '';
        }
        
        $result = $this->_xacls->store($this->_filename, '', $oldxacl);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        return parent::purge();
    }
    
    function storeXACL(&$xacl, &$acl)
    {
        $oldxacl = $this->load();
        if (is_a($oldxacl, 'PEAR_Error')) {
            $oldxacl = '';
        }

        /* Users with read access to the folder may also access the extended information */
        foreach ($acl as $user => $ac) {
            if (strpos($ac, 'r') !== false) {
                if (!empty($user)) {
                    $xacl .= ' ' . $user;
                }
            }
        }

        $result = $this->_xacls->store($this->_filename, $xacl, $oldxacl);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }

        $result = $this->store($xacl);
        if (is_a($result, 'PEAR_Error')) {
            return $result;
        }
    }
}

?>
