
/* ---------------------------------------------------------------- */

%header %{
EXTERN_C_BEGIN

SWIGINTERNINLINE int
_obj_valid(PetscObject obj)
{
  return ((obj != PETSC_NULL) &&
	  (obj->cookie != PETSCFREEDHEADER) &&
	  (obj->cookie >= PETSC_SMALLEST_COOKIE) &&
	  (obj->cookie <= PETSC_LARGEST_COOKIE));
}

SWIGINTERNINLINE void
_obj_sync(PyPetscObjectObject *self)
{
  if (!_obj_valid(self->obj)) {
    self->obj = PETSC_NULL;
    self->own = Py_True;
  }
}

EXTERN_C_END
%}

%header %{
EXTERN_C_BEGIN

static PyTypeObject* Py_PetscObject_Type;

SWIGINTERNINLINE PetscErrorCode
_obj_destroy(PetscObject obj)
{
  if (_obj_valid(obj) && obj->bops->destroy)
    if (!PetscFinalizeCalled)
      return PetscObjectDestroy(obj);
  return 0;
}

static void
obj_dealloc(PyPetscObjectObject *self)
{
  if (self->own == Py_True && _obj_destroy(self->obj) != 0)
    PyErr_Format(PyExc_RuntimeError,"destroying a %s object",
		 self->ob_type->tp_name);
  Py_XDECREF(self->swig);
  self->obj  = PETSC_NULL;
  self->own  = NULL;
  self->swig = NULL;
  self->ob_type->tp_free(self);
}

static PyObject *
obj_new(PyTypeObject *type, PyObject *args, PyObject *kwds)
{
  PyPetscObjectObject *self;
  self = (PyPetscObjectObject*) type->tp_alloc(type, 0);
  if (self != NULL)  {
    self->obj  = PETSC_NULL;
    self->own  = Py_True;
    self->swig = NULL;
  }
  return (PyObject*)self;
}

#define obj_alloc PyType_GenericAlloc
#define obj_init  (initproc)0
#define obj_free  PyObject_Del

static PyObject*
obj_repr(PyPetscObjectObject *self) 
{
  _obj_sync(self); /*@*/
  return PyString_FromFormat("<%s object at %p (%p)>",
			     self->ob_type->tp_name,
			     (void*)self, (void*)self->obj);
}

static int
obj_nonzero(PyPetscObjectObject *ob)
{
  _obj_sync(ob); /*@*/
  return ob->obj != PETSC_NULL;
}

static PyNumberMethods obj_number_methods = {
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
	(inquiry)obj_nonzero,	/*nb_nonzero*/
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 
	0, 0, 0, 0, 0, 0, 0,
};

#define obj_as_number   (&obj_number_methods)
#define obj_as_sequence (PySequenceMethods*)0
#define obj_as_mapping  (PyMappingMethods*)0

static PyObject *
obj_richcompare(PyPetscObjectObject *o1, 
		PyPetscObjectObject *o2, int op)
{
  PyObject *res;

  /* Make sure both arguments are the same type */
  if (!(PyObject_TypeCheck((PyObject*)o1,
			   Py_PetscObject_Type) &&
	PyObject_TypeCheck((PyObject*)o2,
			   Py_PetscObject_Type))) {
    Py_INCREF(Py_NotImplemented);
    return Py_NotImplemented;
  }

  _obj_sync(o1); /*@*/
  _obj_sync(o2); /*@*/

  if (op != Py_EQ && op != Py_NE) {
    PyErr_SetString(PyExc_TypeError,
		    "cannot compare PETSc objects "
		    "using <, <=, >, >=");
    return NULL;
  }
  
  if ((op == Py_EQ) == 
      (PyPetscObject_OBJ(o1) == PyPetscObject_OBJ(o2)))
    res = Py_True;
  else
    res = Py_False;
  
  Py_INCREF(res);
  return res;
}

static PyObject * 
obj_get_this(PyPetscObjectObject *self, void *closure) 
{ 
  if (self->swig != NULL) { /* sanity check */
    PySwigObject *sobj = (PySwigObject*)self->swig;
    if (((PetscObject*)sobj->ptr) != (&self->obj)) {
      printf("cleanig 'this'... Why this happens???\n");
      printf("&self->obj: %p\n", (void*)(&self->obj));
      printf(" swig->ptr: %p\n", (void*)sobj->ptr);
      Py_DECREF(self->swig); self->swig = NULL;
    }
  }
  if (self->swig == NULL) { /* create SWIG object */
    PyObject *_getter;
    _getter = PyObject_GetAttrString((PyObject*)self->ob_type,"__swig_this__");
    if (_getter == NULL) { PyErr_Clear(); Py_RETURN_NONE; }
    self->swig = PyObject_CallFunction(_getter, "O", (PyObject*)self);
    Py_DECREF(_getter);
    if (self->swig == NULL) return NULL;
  }
  Py_INCREF(self->swig);
  return self->swig;
}

static int 
obj_set_this(PyPetscObjectObject *self, PyObject *value,
	     void *closure)
{ 
  if (value == NULL) { 
    PyErr_SetString(PyExc_TypeError, "cannot delete attribute");
    return -1;
  } 
  _obj_sync(self); /*@*/
  /* ignore, do nothing */
  if (PyErr_Occurred()) return -1;
  return 0; 
}

static PyObject *
obj_get_own(PyPetscObjectObject *self, void *closure)
{ 
  _obj_sync(self); /*@*/
  Py_XINCREF(self->own);
  return self->own;
} 

static int 
obj_set_own(PyPetscObjectObject *self, PyObject *value, void *closure) {
  _obj_sync(self); /*@*/
  if (value == NULL) {
    PyErr_SetString(PyExc_TypeError, "cannot delete attribute");
    return -1;
  } 
  if (self->obj != PETSC_NULL) {
    int truth;
    if ((truth = PyObject_IsTrue(value)) == -1) return -1;
    self->own = truth ? Py_True : Py_False;
  }
  return 0;
}

static PyGetSetDef obj_getset[] = {
  {"this",
   (getter)obj_get_this, (setter)obj_set_this,
   "SWIG pointer object", NULL},
  {"thisown",
   (getter)obj_get_own,  (setter)obj_set_own,
   "SWIG pointer object ownership", NULL},

  {NULL}  /* Sentinel */
};

#define obj_methods 0
#define obj_members 0

PyDoc_STRVAR(obj_doc, "Base type for PETSc objects");

static PyTypeObject _Py_PetscObject_Type = {
  PyObject_HEAD_INIT(&PyType_Type)
  0,					/*ob_size*/
  SWIG_name".Object",			/*tp_name*/
  sizeof(PyPetscObjectObject),		/*tp_basicsize*/
  0,					/*tp_itemsize*/

  (destructor)obj_dealloc,		/*tp_dealloc*/
  (printfunc)0,				/*tp_print*/
  (getattrfunc)0,			/*tp_getattr*/
  (setattrfunc)0,			/*tp_setattr*/
  (cmpfunc)0,				/*tp_compare*/
  (reprfunc)obj_repr,			/*tp_repr*/

  obj_as_number,			/*tp_as_number*/
  obj_as_sequence,			/*tp_as_sequence*/
  obj_as_mapping,			/*tp_as_mapping*/

  (hashfunc)0,				/*tp_hash*/
  (ternaryfunc)0,			/*tp_call*/
  (reprfunc)0,				/*tp_str*/
  (getattrofunc)0,			/*tp_getattro*/
  (setattrofunc)0,			/*tp_setattro*/

  (PyBufferProcs*)0,			/*tp_as_buffer*/

  (Py_TPFLAGS_DEFAULT  |
   Py_TPFLAGS_BASETYPE |
   Py_TPFLAGS_CHECKTYPES),		/*tp_flags*/

  obj_doc, 				/*tp_doc*/

  (traverseproc)0,			/*tp_traverse */

  (inquiry)0,				/*tp_clear */

  (richcmpfunc)obj_richcompare, 	/*tp_richcompare */

  (long)0,				/*tp_weaklistoffset */

  (getiterfunc)0,			/*tp_iter */
  (iternextfunc)0,			/*tp_iternext */

  obj_methods,				/*tp_methods */
  obj_members,				/*tp_members */
  obj_getset,				/*tp_getset */
  0,					/*tp_base */
  0,					/*tp_dict */
  (descrgetfunc)0,			/*tp_descr_get */
  (descrsetfunc)0,			/*tp_descr_set */
  (long)0,				/*tp_dictoffset */

  obj_init,				/*tp_init */
  obj_alloc,				/*tp_alloc */
  obj_new,				/*tp_new */
  obj_free,           			/*tp_free */
};

#undef obj_alloc
#undef obj_init
#undef obj_free

#undef obj_as_number
#undef obj_as_sequence
#undef obj_as_mapping

#undef obj_methods
#undef obj_members

EXTERN_C_END
%}

/* ---------------------------------------------------------------- */

%init %{
Py_PetscObject_Type = &_Py_PetscObject_Type;
if (PyType_Ready(Py_PetscObject_Type) < 0)  return;
PyModule_AddObject(m, "Object", (PyObject*)Py_PetscObject_Type);
if(PyErr_Occurred()) return;
%}

/* ---------------------------------------------------------------- */

/*
 * Local Variables:
 * mode: C
 * End:
 */
