/*
**  POP3CacheManager.m
**
**  Copyright (c) 2001, 2002
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This library is free software; you can redistribute it and/or
**  modify it under the terms of the GNU Lesser General Public
**  License as published by the Free Software Foundation; either
**  version 2.1 of the License, or (at your option) any later version.
**  
**  This library is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
**  Lesser General Public License for more details.
**  
**  You should have received a copy of the GNU Lesser General Public
**  License along with this library; if not, write to the Free Software
**  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
*/

#include <Pantomime/POP3CacheManager.h>

#include <Pantomime/Constants.h>
#include <Pantomime/POP3CacheObject.h>

#include <Foundation/NSArchiver.h>
#include <Foundation/NSException.h>

@implementation POP3CacheManager

- (id) init
{
  [super init];

  [self setPOP3CacheObjects: [NSArray array]];

  return self;
}

- (id) initWithPath: (NSString *) thePath
{
  [self init];
  
  [self setPathToCache: thePath];

  return self;
}

- (void) dealloc
{
  RELEASE(pop3CacheObjects);
  RELEASE(pathToCache);

  [super dealloc];
}


- (BOOL) synchronize
{
  return [NSArchiver archiveRootObject: self toFile: [self pathToCache]];
}

//
// NSCoding protocol
//

- (void) encodeWithCoder: (NSCoder *) theCoder
{
  [theCoder encodeObject: [self pop3CacheObjects]];
}

- (id) initWithCoder: (NSCoder *) theCoder
{
  self = [super init];

  [self setPOP3CacheObjects: [theCoder decodeObject]];

  return self;
}


//
// access/mutation methods
//
- (void) addObject: (POP3CacheObject *) thePOP3CacheObject
{
  [pop3CacheObjects addObject: thePOP3CacheObject];
}


//
//
//
- (void) removeObject: (POP3CacheObject *) thePOP3CacheObject
{
  [pop3CacheObjects removeObject: thePOP3CacheObject];
}


//
//
//
- (POP3CacheObject *) findPOP3CacheObject: (NSString *) theUID
{
  POP3CacheObject *anObject;
  int i;

  for (i = 0; i < [[self pop3CacheObjects] count]; i++)
    {
      anObject = [[self pop3CacheObjects] objectAtIndex: i];
      
      if ( [[anObject uid] isEqualToString: theUID] )
	{
	  return anObject;
	}
    }
  
  return nil;
}


//
//
//
- (NSArray *) pop3CacheObjects
{
  return pop3CacheObjects;
}


//
//
//
- (void) setPOP3CacheObjects: (NSArray *) thePOP3CacheObjects
{
  if ( thePOP3CacheObjects )
    {
      NSMutableArray *newPOP3CacheObjects;

      newPOP3CacheObjects = [[NSMutableArray alloc] initWithArray: thePOP3CacheObjects];
      RELEASE(pop3CacheObjects);
      pop3CacheObjects = newPOP3CacheObjects;
    }
  else
    {
      RELEASE(pop3CacheObjects);
      pop3CacheObjects = nil;
    }
}


//
//
//
- (NSString *) pathToCache
{
  return pathToCache;
}


//
//
//
- (void) setPathToCache: (NSString *) thePathToCache
{
  RETAIN(thePathToCache);
  RELEASE(pathToCache);
  pathToCache = thePathToCache;
}


//
// class methods
//
+ (id) pop3CacheObjectsFromDiskWithPath: (NSString *) thePath;
{
  POP3CacheManager *aPOP3CacheManager;

  NS_DURING
    aPOP3CacheManager = [NSUnarchiver unarchiveObjectWithFile: thePath ];
  NS_HANDLER
    NSLog(@"Caught exception while unarchiving the POP3 cache. Ignoring.");
    aPOP3CacheManager = nil;
  NS_ENDHANDLER

  if ( aPOP3CacheManager )
    {
      [aPOP3CacheManager setPathToCache: thePath];
    }
  else
    {
      aPOP3CacheManager = [[POP3CacheManager alloc] initWithPath: thePath];
      AUTORELEASE(aPOP3CacheManager);
    }

  return aPOP3CacheManager;
}


@end
