
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: main_menu.c 1571 2006-11-19 17:54:55Z mschwerin $
 *
 */
#include "config.h"

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

#include "disc.h"
#include "environment.h"
#include "gui_utils.h"
#include "heap.h"
#include "i18n.h"
#include "logger.h"
#include "oxine.h"

#include "help_menu.h"
#include "main_menu.h"
#include "filelist_menu.h"
#include "playback_menu.h"
#include "playlist_menu.h"

extern oxine_t *oxine;

static otk_widget_t *eject_list;
static otk_widget_t *shutdown_list;
static char *shutdown_commands[3] = { NULL, NULL, NULL };

void
execute_cb (void *cmd_p)
{
    char *cmd = (char *) cmd_p;

    if (cmd && strlen (cmd) > 0) {
        debug ("executing shell command %s", cmd);
        system (cmd);
    }
}


void
play_dvb_cb (void *cb_data)
{
    const char *channels_conf = get_file_dvb_channels ();

    if (access (channels_conf, R_OK) != 0) {
        show_message_dialog (show_main_menu_cb, oxine, NULL, NULL, DIALOG_OK,
                             NULL, _("Could not find DVB channels file!"));
        return;
    }

    oxine->backto_menu = show_main_menu_cb;
    oxine->playback_ended_menu = show_main_menu_cb;

    please_wait ();

    int default_channel = 0;
    playitem_t *default_item = NULL;

    /* Look up the last channel played. */
    xine_cfg_entry_t lastchannel;
    xine_config_lookup_entry (oxine->xine, "media.dvb.last_channel",
                              &lastchannel);
    default_channel = lastchannel.num_value;

    /* We add all channels from .xine/channels.conf to the playlist. */
    FILE *f = fopen (channels_conf, "r");
    if (f) {
        playlist_clear (oxine->ro_playlist);

        int i;
        for (i = 1; !feof (f); i++) {
            char line[256];
            char title[64];
            char mrl[16];
            if (!fgets (line, 256, f))
                break;
            sscanf (line, "%[^:]", title);
            snprintf (mrl, 16, "dvb://%d", (i - 1));
            playitem_t *item = playlist_add (oxine->ro_playlist, title, mrl);

            if (default_channel == i)
                default_item = item;
        }
        fclose (f);

        /* If we found the last channel we play that. */
        if (default_item)
            playlist_play_item (oxine, show_playback_menu_cb,
                                oxine->ro_playlist, default_item);
        else
            playlist_play_first (oxine, show_playback_menu_cb,
                                 oxine->ro_playlist);
    }
}


void
play_v4l_cb (void *cb_data)
{
    oxine->backto_menu = show_main_menu_cb;
    oxine->playback_ended_menu = show_main_menu_cb;

    please_wait ();

    playlist_clear (oxine->ro_playlist);
    playlist_add (oxine->ro_playlist, "Analog Television", "v4l://");
    playlist_play_first (oxine, show_playback_menu_cb, oxine->ro_playlist);
}


#ifdef HAVE_VDR
void
play_vdr_cb (void *cb_data)
{
    oxine->backto_menu = show_main_menu_cb;
    oxine->playback_ended_menu = show_main_menu_cb;

    please_wait ();

    playlist_clear (oxine->ro_playlist);

    int num_mrls;
    char **mrls = xine_get_autoplay_mrls (oxine->xine, "VDR", &num_mrls);
    if (mrls) {
        int i;
        for (i = 0; i < num_mrls; i++) {
            playlist_add (oxine->ro_playlist, "VDR Video Disc Recorder",
                          mrls[i]);
        }
    }

    playlist_play_first (oxine, show_playback_menu_cb, oxine->ro_playlist);
}
#endif


void
play_mrl_cb (void *mrl_p)
{
    char *mrl = (char *) mrl_p;

    oxine->backto_menu = show_main_menu_cb;
    oxine->playback_ended_menu = show_main_menu_cb;

    please_wait ();

    playlist_clear (oxine->ro_playlist);
    playlist_add (oxine->ro_playlist, NULL, mrl);
    playlist_play_first (oxine, show_playback_menu_cb, oxine->ro_playlist);
}


void
play_audiocd_cb (void *device_p)
{
    oxine->backto_menu = show_main_menu_cb;
    oxine->playback_ended_menu = show_main_menu_cb;

    please_wait ();

    char *device = NULL;
    if (!device) {
        xine_cfg_entry_t centry;
        xine_config_lookup_entry (oxine->xine, "media.audio_cd.device",
                                  &centry);
        device = ho_strdup (centry.str_value);
    } else {
        device = ho_strdup ((char *) device_p);
    }

    device_type_t type = disc_get_type (device);
    /* If we are certain that the disc in the specified audio CD drive is
     * indeed an audio CD we play this disc. */
    if (type == DEVICE_TYPE_AUDIO_CD) {
        char mrl[1024];
        snprintf (mrl, 1024, "cdda:/%s", device);

        playlist_clear (oxine->ro_playlist);
        disc_autoscan_load (oxine->ro_playlist, mrl);
        playlist_play_first (oxine, show_playback_menu_cb,
                             oxine->ro_playlist);
    }
    /* If there is no disc in the drive or if there is a disc in the drive but
     * we know what kind of disc this is we tell the user to insert a audio CD. */
    else if (type != DEVICE_TYPE_UNKNOWN) {
        show_message_dialog (play_audiocd_cb, device,
                             backto_menu_cb, oxine,
                             DIALOG_OK_CANCEL, NULL,
                             _("Please insert an Audio CD!"));
    }
    /* If none of the above apply we just try our luck and hope xine-lib can
     * solve this problem for us. */
    else {
        playlist_clear (oxine->ro_playlist);
        disc_autoscan_load (oxine->ro_playlist, "cdda:/");
        playlist_play_first (oxine, show_playback_menu_cb,
                             oxine->ro_playlist);
    }

    ho_free (device);
}


void
play_vcd_cb (void *device_p)
{
    oxine->backto_menu = show_main_menu_cb;
    oxine->playback_ended_menu = show_main_menu_cb;

    please_wait ();

    char *device = NULL;
    if (!device) {
        xine_cfg_entry_t centry;
        xine_config_lookup_entry (oxine->xine, "media.vcd.device", &centry);
        device = ho_strdup (centry.str_value);
    } else {
        device = ho_strdup ((char *) device_p);
    }

    device_type_t type = disc_get_type (device);
    /* If we are certain that the disc in the specified VCD drive is indeed a
     * VCD we play this disc. */
    if (type == DEVICE_TYPE_VIDEO_CD) {
        char mrl[1024];
        snprintf (mrl, 1024, "vcd://%s", device);

        playlist_clear (oxine->ro_playlist);
        playlist_add (oxine->ro_playlist, _("Video CD"), mrl);
        playlist_play_first (oxine, show_playback_menu_cb,
                             oxine->ro_playlist);
    }
    /* If there is no disc in the drive or if there is a disc in the drive but
     * we know what kind of disc this is we tell the user to insert a VCD. */
    else if (type != DEVICE_TYPE_UNKNOWN) {
        show_message_dialog (play_vcd_cb, device,
                             backto_menu_cb, oxine,
                             DIALOG_OK_CANCEL, NULL,
                             _("Please insert a Video CD!"));
    }
    /* If none of the above apply we just try our luck and hope xine-lib can
     * solve this problem for us. */
    else {
        playlist_clear (oxine->ro_playlist);
        playlist_add (oxine->ro_playlist, _("Video CD"), "vcd://");
        playlist_play_first (oxine, show_playback_menu_cb,
                             oxine->ro_playlist);
    }

    ho_free (device);
}


void
play_dvd_cb (void *device_p)
{
    oxine->backto_menu = show_main_menu_cb;
    oxine->playback_ended_menu = show_main_menu_cb;

    please_wait ();

    char *device = NULL;
    if (!device) {
        xine_cfg_entry_t centry;
        xine_config_lookup_entry (oxine->xine, "media.dvd.device", &centry);
        device = ho_strdup (centry.str_value);
    } else {
        device = ho_strdup ((char *) device_p);
    }

    device_type_t type = disc_get_type (device);
    /* If we are certain that the disc in the specified DVD drive is indeed a
     * DVD we play this disc. */
    if (type == DEVICE_TYPE_VIDEO_DVD) {
        char mrl[1024];
        snprintf (mrl, 1024, "dvd://%s", device);

        playlist_clear (oxine->ro_playlist);
        playlist_add (oxine->ro_playlist, _("DVD"), mrl);
        playlist_play_first (oxine, show_playback_menu_cb,
                             oxine->ro_playlist);
    }
    /* If there is no disc in the drive or if there is a disc in the drive but
     * we know what kind of disc this is we tell the user to insert a DVD. */
    else if (type != DEVICE_TYPE_UNKNOWN) {
        show_message_dialog (play_dvd_cb, device,
                             backto_menu_cb, oxine,
                             DIALOG_OK_CANCEL, NULL,
                             _("Please insert a DVD!"));
    }
    /* If none of the above apply we just try our luck and hope xine-lib can
     * solve this problem for us. */
    else {
        playlist_clear (oxine->ro_playlist);
        playlist_add (oxine->ro_playlist, _("DVD"), "dvd://");
        playlist_play_first (oxine, show_playback_menu_cb,
                             oxine->ro_playlist);
    }

    ho_free (device);
}


static void
shutdown_execute_cb (void *cb_data)
{
    int p = otk_list_get_pos (shutdown_list);

    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine, "shutdown.last", &centry);
    centry.num_value = p;
    xine_config_update_entry (oxine->xine, &centry);

    if (p > 0) {
        const char *cmd = shutdown_commands[p - 1];

        if (cmd && strlen (cmd) > 0) {
            oxine->shutdown_command = ho_strdup (cmd);
        }
    }

    odk_exit (oxine->odk);
}


void
shutdown_cb (void *cb_data)
{
    xine_cfg_entry_t centry;
    xine_config_lookup_entry (oxine->xine, "shutdown.ask", &centry);
    if (!centry.num_value) {
        odk_exit (oxine->odk);
        return;
    }

    hide_user_interface (oxine);

    create_new_window (false, true);
    otk_border_new (oxine->otk, 100, 200, 600, 200);
    otk_label_new (oxine->otk, 120, 250, 560,
                   OTK_ALIGN_LEFT | OTK_ALIGN_BOTTOM,
                   _("How do you want to proceed?"));

    shutdown_list = otk_selector_new (oxine->otk, 120, 270, 560, NULL, NULL);

    ho_free (shutdown_commands[0]);
    shutdown_commands[0] = NULL;
    ho_free (shutdown_commands[1]);
    shutdown_commands[1] = NULL;
    ho_free (shutdown_commands[2]);
    shutdown_commands[2] = NULL;

    int i = 0;

    {
        otk_listentry_new (shutdown_list, _("Quit oxine"),
                           shutdown_execute_cb, oxine,
                           NULL, NULL, NULL, NULL);
    }

    xine_config_lookup_entry (oxine->xine, "shutdown.shutdown_command",
                              &centry);
    if (centry.str_value && (strlen (centry.str_value) > 0)) {
        shutdown_commands[i++] = ho_strdup (centry.str_value);
        otk_listentry_new (shutdown_list, _("Shutdown the computer"),
                           shutdown_execute_cb, oxine,
                           NULL, NULL, NULL, NULL);
    }

    xine_config_lookup_entry (oxine->xine, "shutdown.standby_command",
                              &centry);
    if (centry.str_value && (strlen (centry.str_value) > 0)) {
        shutdown_commands[i++] = ho_strdup (centry.str_value);
        otk_listentry_new (shutdown_list, _("Switch computer to standby"),
                           shutdown_execute_cb, oxine,
                           NULL, NULL, NULL, NULL);
    }

    xine_config_lookup_entry (oxine->xine, "shutdown.reboot_command",
                              &centry);
    if (centry.str_value && (strlen (centry.str_value) > 0)) {
        shutdown_commands[i++] = ho_strdup (centry.str_value);
        otk_listentry_new (shutdown_list, _("Reboot the computer"),
                           shutdown_execute_cb, oxine,
                           NULL, NULL, NULL, NULL);
    }

    xine_config_lookup_entry (oxine->xine, "shutdown.last", &centry);

    otk_list_set_pos (shutdown_list, centry.num_value);
    otk_list_set_focus (shutdown_list, centry.num_value);

    otk_widget_t *button;
    button = otk_text_button_new (oxine->otk, 230, 340, 160, 40,
                                  _("OK"), shutdown_execute_cb, oxine);
    otk_widget_set_alignment (button, OTK_ALIGN_CENTER);

    button = otk_text_button_new (oxine->otk, 410, 340, 160, 40,
                                  _("Cancel"), backto_menu_cb, oxine);
    otk_widget_set_alignment (button, OTK_ALIGN_CENTER);

    oxine->current_menu = shutdown_cb;
    show_user_interface (oxine);
}


static void
eject_execute_cb (void *cb_data)
{
    int p = otk_list_get_pos (eject_list);
    int i = 0;

#ifdef HAVE_DISC_POLLING
    mutex_lock (&oxine->removable_discs->mutex);
#endif

    device_entry_t *entry = devicelist_first ();
    while (entry) {
        if (i == p) {
            otk_cb_t cb = backto_menu_cb;
            please_wait ();

            if (odk_current_is_playback_mode (oxine->odk)) {
                cb = oxine->playback_ended_menu;
                char *current_mrl = odk_current_get_mrl (oxine->odk);
                // are we playing from the disc we're about to eject?
                if ((entry->mountpoint
                     && strncmp (current_mrl, entry->mountpoint,
                                 strlen (entry->mountpoint)) == 0)
                    || (odk_current_is_dvd (oxine->odk)
                        && (strstr (current_mrl, entry->device) != NULL))
                    || (odk_current_is_vcd (oxine->odk)
                        && (strstr (current_mrl, entry->device) != NULL))
                    || (odk_current_is_cdda (oxine->odk)
                        && (strstr (current_mrl, entry->device) != NULL))) {

                    // we stop the stream
                    odk_stop (oxine->odk);
                }
            }

            /* Now we first try to unmount and then try to eject the
             * device. */
            if (disc_is_mounted (entry->device)
                && !disc_umount (entry->device)) {
                show_message_dialog (cb, oxine, NULL, NULL,
                                     DIALOG_OK, NULL,
                                     _("Could not unmount device!"));
            } else if (!disc_eject (entry->device)) {
                show_message_dialog (cb, oxine, NULL, NULL,
                                     DIALOG_OK, NULL,
                                     _("Could not eject device!"));
#ifdef HAVE_DISC_POLLING
            } else if (!oxine->use_polling) {
                cb (oxine);
            }
#else
            } else {
                cb (oxine);
            }
#endif

            break;
        }
        i += 1;
        entry = devicelist_next (entry);
    }

#ifdef HAVE_DISC_POLLING
    mutex_unlock (&oxine->removable_discs->mutex);
#endif
}


void
eject_cb (void *cb_data)
{
    hide_user_interface (oxine);
    create_new_window (false, true);

    otk_border_new (oxine->otk, 100, 200, 600, 200);
    otk_label_new (oxine->otk, 120, 250, 560,
                   OTK_ALIGN_LEFT | OTK_ALIGN_BOTTOM,
                   _("Select device to eject:"));

    eject_list = otk_selector_new (oxine->otk, 120, 270, 560, NULL, NULL);
    device_entry_t *entry = devicelist_first ();
    while (entry) {
        char *title = disc_get_title (entry->device);
        otk_listentry_new (eject_list, title, eject_execute_cb, oxine,
                           NULL, NULL, NULL, NULL);
        ho_free (title);
        entry = devicelist_next (entry);
    }
    otk_list_set_pos (eject_list, 0);
    otk_list_set_focus (eject_list, 0);

    otk_widget_t *button;
    button = otk_text_button_new (oxine->otk, 230, 340, 160, 40,
                                  _("Eject"), eject_execute_cb, oxine);
    otk_widget_set_alignment (button, OTK_ALIGN_CENTER);

    button = otk_text_button_new (oxine->otk, 410, 340, 160, 40,
                                  _("Cancel"), backto_menu_cb, oxine);
    otk_widget_set_alignment (button, OTK_ALIGN_CENTER);

    oxine->current_menu = eject_cb;
    show_user_interface (oxine);
}


static void
menulist_show (otk_widget_t * l, menulist_t * menulist)
{
    if (odk_current_is_playback_mode (oxine->odk)) {
        otk_widget_t *e = otk_listentry_new (l, _("Current title"),
                                             show_playback_menu_cb, oxine,
                                             show_playback_menu_cb, oxine,
                                             NULL, NULL);
        otk_widget_set_font (e, "sans", 30);
        otk_widget_set_alignment (e, OTK_ALIGN_CENTER | OTK_ALIGN_VCENTER);
    }

    menuitem_t *item = l_list_first (menulist->list);
    while (item) {
        otk_widget_t *e = otk_listentry_new (l, _(item->title),
                                             item->cb, item->cb_data,
                                             item->cb, item->cb_data,
                                             NULL, NULL);
        otk_widget_set_font (e, "sans", 30);
        otk_widget_set_alignment (e, OTK_ALIGN_CENTER | OTK_ALIGN_VCENTER);

        item = l_list_next (menulist->list, item);
    }

    otk_list_select_all (l);
}


static void
main_menu_event_handler (void *cb_data, oxine_event_t * event)
{
    if ((oxine->current_menu != show_main_menu_cb)
        && (oxine->current_menu != show_sub_menu_cb))
        return;

    if (!(event->type == OXINE_EVENT_KEY))
        return;

    switch (event->source.key) {
    case OXINE_KEY_BACK:
        if (oxine->current_menu == show_sub_menu_cb) {
            show_main_menu_cb (oxine);
        } else {
            if (!odk_current_is_logo_mode (oxine->odk)) {
                show_playback_menu_cb (oxine);
            }
        }
        event->source.key = OXINE_KEY_NULL;
        break;
    default:
        break;
    }
}


void
show_sub_menu_cb (void *menulist_p)
{
    menulist_t *menulist = (menulist_t *) menulist_p;

    hide_user_interface (oxine);

    oxine->backto_menu = show_main_menu_cb;
    if (odk_current_is_logo_mode (oxine->odk))
        oxine->playback_ended_menu = show_main_menu_cb;

    show_menu_background (OXINE_BACKGROUNDS "/mainmenu.png");
    create_new_window (false, true);

    otk_widget_t *l = otk_list_new (oxine->otk, 215, 150, 370, 400,
                                    45, 55, false, false,
                                    OTK_LIST_MULTIPLE_SELECTION,
                                    oxine);

    menulist_show (l, menulist);

    oxine->current_menu = show_main_menu_cb;
    show_user_interface (oxine);
}


void
show_main_menu_cb (void *cb_data)
{
    hide_user_interface (oxine);

    oxine->backto_menu = show_main_menu_cb;
    oxine->current_menu = show_main_menu_cb;
    if (!odk_current_is_playback_mode (oxine->odk))
        oxine->playback_ended_menu = show_main_menu_cb;

    show_menu_background (OXINE_BACKGROUNDS "/mainmenu.png");

    static otk_widget_t *main_menu_list = NULL;
    static otk_widget_t *main_menu_window = NULL;
    if (main_menu_window) {
        otk_window_set_current (oxine->otk, main_menu_window);
        int top_pos = otk_list_get_pos (main_menu_list);
        int cur_pos = otk_list_get_focus (main_menu_list);
        int length = otk_list_get_length (main_menu_list);
        otk_clear_list (main_menu_list);
        menulist_show (main_menu_list, oxine->main_menu_items);
        top_pos += otk_list_get_length (main_menu_list) - length;
        cur_pos += otk_list_get_length (main_menu_list) - length;
        otk_list_set_pos (main_menu_list, top_pos);
        otk_list_set_focus (main_menu_list, cur_pos);
        show_user_interface (oxine);
        return;
    }

    odk_add_event_handler (oxine->odk, main_menu_event_handler, oxine,
                           EVENT_HANDLER_PRIORITY_NORMAL);
    main_menu_window = create_new_window (true, true);

    /* These entries are needed so the titles used in the standard
     * mainmenu.xml file appear in the translation files. */
    _("Listen to Music");
    _("Watch Films");
    _("Watch Television");
    _("Play a DVD");
    _("Play a Video CD");
    _("Play an Audio CD");
    _("Show Help");
    _("Show Weather Report");
    _("Edit Settings");
    _("Shutdown");

    main_menu_list = otk_list_new (oxine->otk, 215, 150, 370, 400,
                                   45, 55, false, false,
                                   OTK_LIST_MULTIPLE_SELECTION, oxine);

    menulist_show (main_menu_list, oxine->main_menu_items);

    show_user_interface (oxine);
}


void
main_menu_init ()
{
    char mmpath[1024];
    snprintf (mmpath, 1024, "%s/mainmenu.xml", get_dir_oxine ());
    oxine->main_menu_items = menulist_new (mmpath, NULL);
    if (!oxine->main_menu_items) {
        snprintf (mmpath, 1024, "%s/mainmenu.xml", OXINE_DATADIR);
        oxine->main_menu_items = menulist_new (mmpath, NULL);
        if (!oxine->main_menu_items) {
            fatal (_("Could not load main menu!"));
            abort ();
        }
    }
}


void
main_menu_free ()
{
    if (oxine->main_menu_items)
        menulist_free (oxine->main_menu_items);
    oxine->main_menu_items = NULL;

    ho_free (shutdown_commands[0]);
    shutdown_commands[0] = NULL;
    ho_free (shutdown_commands[1]);
    shutdown_commands[1] = NULL;
    ho_free (shutdown_commands[2]);
    shutdown_commands[2] = NULL;
}
