///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////

#ifndef OPENVDB_TREE_LEAF_ARRAY_HAS_BEEN_INCLUDED
#define OPENVDB_TREE_LEAF_ARRAY_HAS_BEEN_INCLUDED

#warning This file is deprecated and should be replaced by tree/LeafManager.h

#include <iostream>
#include <algorithm> // for std::swap
#include <cstring> //  for std::memcpy()
#include <boost/shared_ptr.hpp>
#include <boost/static_assert.hpp>
#include <boost/bind.hpp>
#include <tbb/blocked_range.h>
#include <tbb/parallel_for.h>
#include <openvdb/tools/Morphology.h>
#include <openvdb/tree/LeafManager.h>

class TestLeaf;
template<typename> class TestLeafIO;

namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tree {

/// @brief This is a range and iterator class useful for multithreading computations
/// on leaf values that require the use of multiple temporal buffers but static tree
/// topology (i.e. only voxel values are dynamic). The internal buffer in LeafNodes
/// is referred to as the read buffer, and the external buffers allocated by the
/// LeafArray class are called write buffers. LeafArray::swapBuffers
/// implements a multi-threaded method to efficiently swap the read buffer with
/// any of the write buffers. See tools/Filter.h for an application.
template<typename TreeT, int BufferCount>
class LeafArray
{
public:
    typedef boost::shared_ptr<LeafArray>    Ptr;
    typedef TreeT                           TreeType;
    typedef typename TreeType::ValueType    ValueType;
    typedef typename TreeType::LeafNodeType LeafType;
    typedef typename LeafType::Buffer       BufferType;

    /// Simple structure of LeafNode pointers and corresponding write buffers
    struct BufferStruct {
        /// Constructor initializing write buffers to the leaf's read buffer!
        BufferStruct(LeafType& _leaf): leaf(&_leaf) {}
        /// Swap the leaf's read-buffer with the n'th write-buffer
        void swap(Index n) { assert(n < BufferCount); leaf->swap(buffer[n]); }
        void copy(Index n) { assert(n < BufferCount); buffer[n]=leaf->buffer(); }
        void sync() { for (Index n = 0; n < BufferCount; ++n) this->copy(n); }
        LeafType*  leaf;// pointer to leaf containing the internal read buffer
        BufferType buffer[BufferCount];//static array of external write buffers
    };

    typedef typename std::vector<BufferStruct>    ArrayType;
    typedef typename ArrayType::iterator          IterType;
    typedef typename tbb::blocked_range<IterType> IterRangeType;

    /// Constructor from a Tree reference
    OPENVDB_DEPRECATED LeafArray(TreeType &tree, bool sync=true) : mTree(NULL)
    {
        this->init(tree, sync);
    }
    void init(TreeType &tree, bool sync)
    {
        mTree = &tree;
        mArray.clear();
        mArray.reserve(tree.leafCount());
        for (typename TreeType::LeafIter i = tree.beginLeaf(); i; ++i) {
            mArray.push_back( BufferStruct(*i) );
        }
        if (sync) this->syncBuffers();
    }
    void clear() { mArray.clear(); }
    Index size() const { return mArray.size(); }
    TreeType& tree() { return *mTree; }
    LeafType* operator[](int i) { return mArray[i].leaf; }

    /// Return range of the buffer-array. Needed for multithreading by tbb
    IterRangeType getRange(size_t grainsize=1)
    {
        return IterRangeType(mArray.begin(), mArray.end(), grainsize);
    }

    /// For each leaf node in the given range, swap the leaf node's internal buffer
    /// with its nth external write buffer.
    /// @param r  the range of leaf nodes to process
    /// @param n  the index of the external write buffer
    /// @note Consider using the non-static method swapBuffers() instead, since
    /// it supports multithreading and has a simpler API.
    static void swapBufferRange(const IterRangeType& r, Index n)
    {
        if (n < BufferCount) {
            for (IterType i = r.begin(), e = r.end(); i != e; ++i) i->swap(n);
        }
    }
    /// For each leaf node in the given range, copy the leaf node's internal buffer
    /// into its nth external write buffer.
    /// @param r  the range of leaf nodes to process
    /// @param n  the index of the external write buffer
    /// @note Consider using the non-static method copyBuffers() instead, since
    /// it supports multithreading and has a simpler API.
    static void copyBufferRange(const IterRangeType& r, Index n)
    {
        if (n < BufferCount) {
            for (IterType i = r.begin(), e = r.end(); i != e; ++i) i->copy(n);
        }
    }
    /// For each leaf node in the given range, sync the leaf node's internal buffer
    /// with all the external write buffers.
    /// @param r  the range of leaf nodes to process
    /// @note Consider using the non-static method syncBuffers() instead, since
    /// it supports multithreading and has a simpler API.
    static void syncBufferRange(const IterRangeType& r)
    {
        for (IterType i = r.begin(), e = r.end(); i != e; ++i) i->sync();
    }
    /// Swap each leaf node's internal buffer with its nth external write buffer.
    /// @param n       the index of the external write buffer
    /// @param serial  if false, swap buffers in parallel using multiple threads
    void swapBuffers(Index n = 0, bool serial = false)
    {
        if (n < BufferCount) {
            if (serial) {
                this->swapBufferRange(this->getRange(), n);
            } else {
                tbb::parallel_for(this->getRange(512), boost::bind(&swapBufferRange, _1, n));
            }
        }
    }
    /// Copy each leaf node's internal buffer into its nth external write buffer.
    /// @param n       the index of the external write buffer
    /// @param serial  if false, copy buffers in parallel using multiple threads
    void copyBuffers(Index n = 0, bool serial = false)
    {
        if (n < BufferCount) {
            if (serial) {
                this->copyBufferRange(this->getRange(), n);
            } else {
                tbb::parallel_for(this->getRange(8), boost::bind(&copyBufferRange, _1, n));
            }
        }
    }
    /// Sync each leaf node's internal buffer with all its external write buffer.
    /// @param serial  if false, sync buffers in parallel using multiple threads
    void syncBuffers(bool serial = false)
    {
        if (serial) {
            this->syncBufferRange(this->getRange());
        } else {
            tbb::parallel_for(this->getRange(8), boost::bind(&syncBufferRange, _1));
        }
    }

private:
    ArrayType mArray;
    TreeType* mTree;
};//end of LeafArray class

/// @brief This template specialization of LeafArray contains no write buffers!
/// Hence this is effectively a thin wrapper for std::vector<LeafNode*>
template<typename TreeT>
class LeafArray<TreeT,0>
{
public:
    typedef boost::shared_ptr<LeafArray>    Ptr;
    typedef TreeT                           TreeType;
    typedef typename TreeType::ValueType    ValueType;
    typedef typename TreeType::LeafNodeType LeafType;
    typedef typename LeafType::Buffer       BufferType;

    /// Simple structure of LeafNode pointers and corresponding write buffers
    struct BufferStruct {
        /// Constructor initializing write buffers to the leaf's read buffer!
        BufferStruct(LeafType& _leaf): leaf(&_leaf) {}
        LeafType*  leaf;// pointer to leaf containing the internal read buffer
    };

    typedef typename std::vector<BufferStruct>    ArrayType;
    typedef typename ArrayType::iterator          IterType;
    typedef typename tbb::blocked_range<IterType> IterRangeType;

    /// Constructor from a Tree reference
    OPENVDB_DEPRECATED LeafArray(TreeType &tree) : mTree(NULL) { this->init(tree); }
    void init(TreeType &tree)
    {
        mTree = &tree;
        mArray.clear();
        mArray.reserve(tree.leafCount());
        for (typename TreeType::LeafIter i = tree.beginLeaf(); i; ++i) {
            mArray.push_back( BufferStruct(*i) );
        }
    }
    void clear() { mArray.clear(); }
    Index size() const { return mArray.size(); }
    TreeType& tree() { return *mTree; }
    LeafType* operator[](int i) { return mArray[i].leaf; }
    
    /// Return range of the leaf array. Needed for multithreading by tbb
    IterRangeType getRange(size_t grainsize=1)
    {
        return IterRangeType(mArray.begin(), mArray.end(), grainsize);
    }

private:
    ArrayType mArray;
    TreeType* mTree;
};//end of specialized LeafArray class

template<typename TreeType, int BufferCount>
inline void
OPENVDB_DEPRECATED dilateVoxels(tree::LeafArray<TreeType, BufferCount>& leafArray)
{
    tree::LeafManager<TreeType> manager(leafArray.tree());
    dilateVoxels<TreeType>(manager);
}
    
} // namespace tree
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb


#endif // OPENVDB_TREE_LEAF_ARRAY_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
