///////////////////////////////////////////////////////////////////////////
//
// Copyright (c) 2012 DreamWorks Animation LLC
//
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
//
// Redistributions of source code must retain the above copyright
// and license notice and the following restrictions and disclaimer.
//
// *     Neither the name of DreamWorks Animation nor the names of
// its contributors may be used to endorse or promote products derived
// from this software without specific prior written permission.
//
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
// "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
// LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
// A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
// OWNER OR CONTRIBUTORS BE LIABLE FOR ANY INDIRECT, INCIDENTAL,
// SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
// LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
// DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
// THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
// OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// IN NO EVENT SHALL THE COPYRIGHT HOLDERS' AND CONTRIBUTORS' AGGREGATE
// LIABILITY FOR ALL CLAIMS REGARDLESS OF THEIR BASIS EXCEED US$250.00.
//
///////////////////////////////////////////////////////////////////////////
//
/// @file GridOperators.h

#ifndef OPENVDB_TOOLS_GRID_OPERATORS_HAS_BEEN_INCLUDED
#define OPENVDB_TOOLS_GRID_OPERATORS_HAS_BEEN_INCLUDED

#include <openvdb/Grid.h>
#include <openvdb/math/Operators.h>
#include <openvdb/math/Stencils.h>
#include <openvdb/tree/LeafManager.h>
#include <openvdb/tree/ValueAccessor.h>
#include <tbb/parallel_for.h>
#include <tbb/blocked_range.h>


namespace openvdb {
OPENVDB_USE_VERSION_NAMESPACE
namespace OPENVDB_VERSION_NAME {
namespace tools {

/// @brief VectorToScalarConverter<VectorGridType>::Type is the type of a grid
/// having the same tree configuration as VectorGridType but a scalar value type, T,
/// where T is the type of the original vector components.
/// @details For example, VectorToScalarConverter<Vec3DGrid>::Type is equivalent to DoubleGrid.
template<typename VectorGridType> struct VectorToScalarConverter {
    typedef typename VectorGridType::ValueType::value_type VecComponentValueT;
    typedef typename VectorGridType::template ValueConverter<VecComponentValueT>::Type Type;
};

/// @brief ScalarToVectorConverter<ScalarGridType>::Type is the type of a grid
/// having the same tree configuration as ScalarGridType but value type Vec3<T>
/// where T is ScalarGridType::ValueType.
/// @details For example, ScalarToVectorConverter<DoubleGrid>::Type is equivalent to Vec3DGrid.
template<typename ScalarGridType> struct ScalarToVectorConverter {
    typedef math::Vec3<typename ScalarGridType::ValueType> VectorValueT;
    typedef typename ScalarGridType::template ValueConverter<VectorValueT>::Type Type;
};


/// @brief Compute the curl of the given vector-valued grid.
/// @return a new vector-valued grid
template<typename GridType> inline
typename GridType::Ptr
curl(const GridType& grid, bool threaded = true);


/// @brief Compute the divergence of the given vector-valued grid.
/// @return a new scalar-valued grid with the same numerical precision as the input grid
///     (for example, if the input grid is a Vec3DGrid, the output grid will be a DoubleGrid)
template<typename GridType> inline
typename VectorToScalarConverter<GridType>::Type::Ptr
divergence(const GridType& grid, bool threaded = true);


/// @brief Compute the gradient of the given scalar grid.
/// @return a new vector-valued grid with the same numerical precision as the input grid
///     (for example, if the input grid is a DoubleGrid, the output grid will be a Vec3DGrid)
template<typename GridType> inline
typename ScalarToVectorConverter<GridType>::Type::Ptr
gradient(const GridType& grid, bool threaded = true);


/// @brief Compute the Laplacian of the given scalar grid.
/// @return a new scalar grid
template<typename GridType> inline
typename GridType::Ptr
laplacian(const GridType& grid, bool threaded = true);


/// @brief Compute the mean curvature of the given grid.
/// @return a new grid
template<typename GridType> inline
typename GridType::Ptr
meanCurvature(const GridType& grid, bool threaded = true);
    

/// @brief Compute the magnitude of the given grid.
/// @return a new grid
template<typename GridType> inline
typename GridType::Ptr
magnitude(const GridType& grid, bool threaded = true);


/// @brief Normalizes the given vector grid.
/// @return a new grid
template<typename GridType> inline
typename GridType::Ptr
normalize(const GridType& grid, bool threaded = true);      

////////////////////////////////////////


namespace {

/// @brief Applies an operator on an input grid to produce an output
/// grid with the same topology but potentially different value type.
///
/// @note This is a helper class and should never be used directly
///
/// @note The current implementation assumes all the input
/// values are represented by leaf voxels and not tiles. In the
/// future we will expand this class to also handle tile values.
template<typename InGridT, typename OutGridT, typename MapT, typename OperatorT>
class GridOperator
{
public:
    typedef typename OutGridT::TreeType           OutTreeT;
    typedef typename OutTreeT::LeafNodeType       OutLeafT;
    typedef typename tree::LeafManager<OutTreeT>  LeafManagerT;
    typedef typename LeafManagerT::RangeType      RangeT;

    GridOperator(const InGridT& grid, const MapT& map):
        mAcc(grid.getConstAccessor()), mMap(map), mLeafs(NULL)
    {
    }
    virtual ~GridOperator() {}
    typename OutGridT::Ptr process(bool threaded = true)
    {
        // Derive background value of the output grid
        typename InGridT::TreeType tmp(mAcc.getTree()->getBackground());
        typename OutGridT::ValueType backg = OperatorT::result(mMap, tmp, math::Coord(0));
        // output tree = topology copy of input tree!
        typename OutTreeT::Ptr tree(new OutTreeT(*mAcc.getTree(), backg, TopologyCopy()));
        // create grid with output tree and unit transform
        typename OutGridT::Ptr result(new OutGridT(tree));
        // transform of output grid = transform of input grid
        result->setTransform(math::Transform::Ptr(new math::Transform( mMap.copy() )));

        mLeafs = new LeafManagerT(*tree);
        if (threaded) {
            tbb::parallel_for(mLeafs->getRange(), *this);
        } else {
            (*this)(mLeafs->getRange());
        }
        delete mLeafs; mLeafs=NULL;

        return result;
    }

    /// @brief Iterate sequentially over LeafNodes and voxels in the output
    /// grid and compute the laplacian using a valueAccessor for the
    /// input grid.
    ///
    /// @note Never call this public method directly - it is called by
    /// TBB threads only!
    void operator()(const RangeT& range) const
    {
        assert(mLeafs);
        for (size_t n=range.begin(), e=range.end(); n != e; ++n) {
            OutLeafT& leaf = mLeafs->leaf(n);
            for (typename OutLeafT::ValueOnIter iter = leaf.beginValueOn(); iter; ++iter) {
                iter.setValue(OperatorT::result(mMap, mAcc, iter.getCoord()));
            }
        }
    }

protected:
    typedef typename InGridT::ConstAccessor  AccessorT;

    mutable AccessorT  mAcc;
    const MapT&        mMap;
    LeafManagerT*      mLeafs;
}; // end of GridOperator class

} //end of anonymous namespace


////////////////////////////////////////


/// @brief Computes the Curl of a scalar grid
template<typename GridT>
class Curl
{
public:
    typedef GridT  InGridType;
    typedef GridT  OutGridType;
    Curl(const GridT& grid): mInputGrid(grid) {}
    virtual ~Curl() {}
    typename GridT::Ptr process(bool threaded = true)
    {
        Functor functor(mInputGrid, threaded);
        processTypedMap(mInputGrid.transform(), functor);
        return functor.mOutputGrid;
    }

protected:
    struct Functor
    {
        Functor(const GridT& grid, bool threaded): mThreaded(threaded), mInputGrid(grid) {}
        template<typename MapT>
        void operator()(const MapT& map)
        {
            typedef math::Curl<MapT, math::CD_2ND> OpT;
            GridOperator<GridT, GridT, MapT, OpT> op(mInputGrid, map);
            mOutputGrid = op.process(mThreaded); // cache the result
        }
        const bool           mThreaded;
        const GridT&         mInputGrid;
        typename GridT::Ptr  mOutputGrid;
    }; // Private Functor

    const GridT& mInputGrid;
}; // end of Curl class


////////////////////////////////////////


/// @brief Computes the Divergence of a scalar grid
template<typename InGridT>
class Divergence
{
public:
    typedef InGridT                                         InGridType;
    typedef typename VectorToScalarConverter<InGridT>::Type OutGridType;

    Divergence(const InGridT& grid): mInputGrid(grid) {}
    virtual ~Divergence() {}
    typename OutGridType::Ptr process(bool threaded = true)
    {
        if( mInputGrid.getGridClass() == GRID_STAGGERED ) {
            Functor<math::FD_1ST> functor(mInputGrid, threaded);
            processTypedMap(mInputGrid.transform(), functor);
            return functor.mOutputGrid;
        }
        else {
            Functor<math::CD_2ND> functor(mInputGrid, threaded);
            processTypedMap(mInputGrid.transform(), functor);
            return functor.mOutputGrid;
        }
    }

protected:
    template<math::DScheme DiffScheme>
    struct Functor
    {
        Functor(const InGridT& grid, bool threaded): mThreaded(threaded), mInputGrid(grid) {}
        template<typename MapT>
        void operator()(const MapT& map)
        {
            typedef math::Divergence<MapT, DiffScheme> OpT;
            GridOperator<InGridType, OutGridType, MapT, OpT> op(mInputGrid, map);
            mOutputGrid = op.process(mThreaded); // cache the result
        }
        const bool                 mThreaded;
        const InGridType&          mInputGrid;
        typename OutGridType::Ptr  mOutputGrid;
    }; // Private Functor

    const InGridType& mInputGrid;
}; // end of Divergence class


////////////////////////////////////////


/// @brief Computes the Gradient of a scalar grid
template<typename InGridT>
class Gradient
{
public:
    typedef InGridT                                         InGridType;
    typedef typename ScalarToVectorConverter<InGridT>::Type OutGridType;

    Gradient(const InGridT& grid): mInputGrid(grid) {}
    virtual ~Gradient() {}
    typename OutGridType::Ptr process(bool threaded = true)
    {
        Functor functor(mInputGrid, threaded);
        processTypedMap(mInputGrid.transform(), functor);
        return functor.mOutputGrid;
    }

protected:
    struct Functor
    {
        Functor(const InGridT& grid, bool threaded): mThreaded(threaded), mInputGrid(grid) {}
        template<typename MapT>
        void operator()(const MapT& map)
        {
            typedef math::Gradient<MapT, math::CD_2ND> OpT;
            GridOperator<InGridType, OutGridType, MapT, OpT> op(mInputGrid, map);
            mOutputGrid = op.process(mThreaded); // cache the result
        }
        const bool                 mThreaded;
        const InGridT&             mInputGrid;
        typename OutGridType::Ptr  mOutputGrid;
    }; // Private Functor

    const InGridT& mInputGrid;
}; // end of Gradient class


////////////////////////////////////////


/// @brief Computes the Laplacian of a scalar grid
template<typename GridT>
class Laplacian
{
public:
    typedef GridT  InGridType;
    typedef GridT  OutGridType;
    Laplacian(const GridT& grid): mInputGrid(grid) {}
    virtual ~Laplacian() {}
    typename GridT::Ptr process(bool threaded = true)
    {
        Functor functor(mInputGrid, threaded);
        processTypedMap(mInputGrid.transform(), functor);
        return functor.mOutputGrid;
    }

protected:
    struct Functor
    {
        Functor(const GridT& grid, bool threaded): mThreaded(threaded), mInputGrid(grid) {}
        template<typename MapT>
        void operator()(const MapT& map)
        {
            typedef math::Laplacian<MapT, math::CD_SECOND> OpT;
            GridOperator<GridT, GridT, MapT, OpT> op(mInputGrid, map);
            mOutputGrid = op.process(mThreaded); // cache the result
        }
        const bool           mThreaded;
        const GridT&         mInputGrid;
        typename GridT::Ptr  mOutputGrid;
    }; // Private Functor

    const GridT& mInputGrid;
}; // end of Laplacian class


////////////////////////////////////////


template<typename GridT>
class MeanCurvature
{
public:
    typedef GridT  InGridType;
    typedef GridT  OutGridType;
    MeanCurvature(const GridT& grid): mInputGrid(grid) {}
    virtual ~MeanCurvature() {}
    typename GridT::Ptr process(bool threaded = true)
    {
        Functor functor(mInputGrid, threaded);
        processTypedMap(mInputGrid.transform(), functor);
        return functor.mOutputGrid;
    }

protected:
    struct Functor
    {
        Functor(const GridT& grid, bool threaded): mThreaded(threaded), mInputGrid(grid) {}
        template<typename MapT>
        void operator()(const MapT& map)
        {
            typedef math::MeanCurvature<MapT, math::CD_SECOND, math::CD_2ND> OpT;
            GridOperator<GridT, GridT, MapT, OpT> op(mInputGrid, map);
            mOutputGrid = op.process(mThreaded); // cache the result
        }
        const bool           mThreaded;
        const GridT&         mInputGrid;
        typename GridT::Ptr  mOutputGrid;
    }; // Private Functor

    const GridT& mInputGrid;
}; // end of MeanCurvature class


////////////////////////////////////////
    
template<typename InGridT>
class Magnitude
{
public:
    typedef InGridT                                         InGridType;
    typedef typename VectorToScalarConverter<InGridT>::Type OutGridType;
    Magnitude(const InGridType& grid): mInputGrid(grid) {}
    virtual ~Magnitude() {}
    typename OutGridType::Ptr process(bool threaded = true)
    {
        Functor functor(mInputGrid, threaded);
        processTypedMap(mInputGrid.transform(), functor);
        return functor.mOutputGrid;
    }

protected:
    struct OpT
    {
        template<typename MapT, typename AccT>
        static typename OutGridType::ValueType
        result(const MapT&, const AccT& acc, const Coord& xyz) { return acc.getValue(xyz).length();}
    };
    struct Functor
    {
        Functor(const InGridT& grid, bool threaded): mThreaded(threaded), mInputGrid(grid) {}
        template<typename MapT>
        void operator()(const MapT& map)
        {
            GridOperator<InGridType, OutGridType, MapT, OpT> op(mInputGrid, map);
            mOutputGrid = op.process(mThreaded); // cache the result
        }
        const bool                 mThreaded;
        const InGridType&          mInputGrid;
        typename OutGridType::Ptr  mOutputGrid;
    }; // Private Functor

    const InGridType& mInputGrid;
}; // end of Magnitude class


////////////////////////////////////////
    
    
template<typename GridT>
class Normalize
{
public:
    typedef GridT  InGridType;
    typedef GridT  OutGridType;
    Normalize(const GridT& grid): mInputGrid(grid) {}
    virtual ~Normalize() {}
    typename GridT::Ptr process(bool threaded = true)
    {
        Functor functor(mInputGrid, threaded);
        processTypedMap(mInputGrid.transform(), functor);
        return functor.mOutputGrid;
    }

protected:
    struct OpT
    {
        template<typename MapT, typename AccT>
        static typename OutGridType::ValueType
        result(const MapT&, const AccT& acc, const Coord& xyz)
        {
            typename OutGridType::ValueType vec = acc.getValue(xyz);
            if ( !vec.normalize() ) vec.setZero();
            return vec;
        }
    };
    struct Functor
    {
        Functor(const GridT& grid, bool threaded): mThreaded(threaded), mInputGrid(grid) {}
        template<typename MapT>
        void operator()(const MapT& map)
        {
            GridOperator<GridT, GridT, MapT, OpT> op(mInputGrid, map);
            mOutputGrid = op.process(mThreaded); // cache the result
        }
        const bool           mThreaded;
        const GridT&         mInputGrid;
        typename GridT::Ptr  mOutputGrid;
    }; // Private Functor

    const GridT& mInputGrid;
}; // end of Normalize class

////////////////////////////////////////

    
template<typename GridType> inline
typename GridType::Ptr
curl(const GridType& grid, bool threaded)
{
    Curl<GridType> op(grid);
    return op.process(threaded);
}

template<typename GridType> inline
typename VectorToScalarConverter<GridType>::Type::Ptr
divergence(const GridType& grid, bool threaded)
{
    Divergence<GridType> op(grid);
    return op.process(threaded);
}

template<typename GridType> inline
typename ScalarToVectorConverter<GridType>::Type::Ptr
gradient(const GridType& grid, bool threaded)
{
    Gradient<GridType> op(grid);
    return op.process(threaded);
}

template<typename GridType> inline
typename GridType::Ptr
laplacian(const GridType& grid, bool threaded)
{
    Laplacian<GridType> op(grid);
    return op.process(threaded);
}

template<typename GridType> inline
typename GridType::Ptr
meanCurvature(const GridType& grid, bool threaded)
{
    MeanCurvature<GridType> op(grid);
    return op.process(threaded);
}

template<typename GridType> inline
typename GridType::Ptr
magnitude(const GridType& grid, bool threaded)
{
    Magnitude<GridType> op(grid);
    return op.process(threaded);
}

template<typename GridType> inline
typename GridType::Ptr
normalize(const GridType& grid, bool threaded)
{
    Normalize<GridType> op(grid);
    return op.process(threaded);
}  

} // namespace tools
} // namespace OPENVDB_VERSION_NAME
} // namespace openvdb

#endif // OPENVDB_TOOLS_GRID_OPERATORS_HAS_BEEN_INCLUDED

// Copyright (c) 2012 DreamWorks Animation LLC
// All rights reserved. This software is distributed under the
// Mozilla Public License 2.0 ( http://www.mozilla.org/MPL/2.0/ )
