/*
 *  Copyright 2001-2005 Internet2
 * 
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

/* SAMLException.cpp - SAML exception class

   Scott Cantor
   2/28/02

   $History:$
*/

#include "internal.h"

using namespace saml;
using namespace std;

params::params(int count,...)
{
    va_list args;
    va_start(args,count);
    while (count--)
        v.push_back(va_arg(args,char*));
    va_end(args);
}

namedparams::namedparams(int count,...)
{
    count*=2;
    va_list args;
    va_start(args,count);
    while (count--)
        v.push_back(va_arg(args,char*));
    va_end(args);
}

saml::QName SAMLException::SUCCESS(XML::SAMLP_NS,L(Success));
saml::QName SAMLException::REQUESTER(XML::SAMLP_NS,L(Requester));
saml::QName SAMLException::RESPONDER(XML::SAMLP_NS,L(Responder));
saml::QName SAMLException::VERSIONMISMATCH(XML::SAMLP_NS,L(VersionMismatch));

SAMLException::SAMLExceptionFactoryMap SAMLException::m_map;

void SAMLException::regFactory(const char* exceptionClass, SAMLExceptionFactory* factory)
{
    if (exceptionClass && factory)
        m_map.insert(SAMLExceptionFactoryMap::value_type(exceptionClass,factory));
}

void SAMLException::unregFactory(const char* exceptionClass)
{
    if (exceptionClass)
        m_map.erase(exceptionClass);
}

SAMLException* SAMLException::getInstance(DOMElement* e)
{
    // Find the StatusDetail element.
    DOMElement* detail=XML::getLastChildElement(e,XML::SAMLP_NS,L(StatusDetail));
    if (detail) {
        // Look for the special OpenSAML ExceptionClass element.
        DOMElement* eclass=XML::getFirstChildElement(detail,XML::OPENSAML_NS,L(ExceptionClass));
        if (eclass && eclass->hasChildNodes() && eclass->getFirstChild()->getNodeType()==DOMNode::TEXT_NODE) {
            auto_ptr_char cname(eclass->getFirstChild()->getNodeValue());
            if (cname.get()) {
                SAMLExceptionFactoryMap::const_iterator i=m_map.find(cname.get());
                if (i!=m_map.end())
                    return (i->second)(e);
            }
        }
    }
    
    // Default to base class.
    return new SAMLException(e);
}

SAMLException* SAMLException::getInstance(istream& in)
{
    XML::Parser p;
    XML::StreamInputSource src(in);
    Wrapper4InputSource dsrc(&src,false);
    DOMDocument* doc=p.parse(dsrc);
    try
    {
        SAMLException* e=getInstance(doc->getDocumentElement());
        if (e)
            e->setDocument(doc);
        return e;
    }
    catch(...)
    {
        doc->release();
        throw;
    }
}

SAMLException::SAMLException(const char* msg, const params& p, const Iterator<saml::QName>& codes, DOMElement* detail)
    : m_hr(E_FAIL), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    if (msg)
        m_msg=msg;
    addProperties(p);
    while (codes.hasNext())
        m_codes.push_back(codes.next());
    setDetail(detail);
}

SAMLException::SAMLException(const char* msg, const namedparams& p, const Iterator<saml::QName>& codes, DOMElement* detail)
    : m_hr(E_FAIL), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    if (msg)
        m_msg=msg;
    addProperties(p);
    while (codes.hasNext())
        m_codes.push_back(codes.next());
    setDetail(detail);
}

SAMLException::SAMLException(const std::string& msg, const params& p, const Iterator<saml::QName>& codes, DOMElement* detail)
    : m_hr(E_FAIL), m_msg(msg), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    addProperties(p);
    while (codes.hasNext())
        m_codes.push_back(codes.next());
    setDetail(detail);
}

SAMLException::SAMLException(const string& msg, const namedparams& p, const Iterator<saml::QName>& codes, DOMElement* detail)
    : m_hr(E_FAIL), m_msg(msg), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    addProperties(p);
    while (codes.hasNext())
        m_codes.push_back(codes.next());
    setDetail(detail);
}

SAMLException::SAMLException(const saml::QName& code, const char* msg, const params& p, DOMElement* detail)
    : m_hr(E_FAIL), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    m_codes.push_back(code);
    if (msg)
        m_msg=msg;
    addProperties(p);
    setDetail(detail);
}

SAMLException::SAMLException(const saml::QName& code, const char* msg, const namedparams& p, DOMElement* detail)
    : m_hr(E_FAIL), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    m_codes.push_back(code);
    if (msg)
        m_msg=msg;
    addProperties(p);
    setDetail(detail);
}

SAMLException::SAMLException(const saml::QName& code, const string& msg, const params& p, DOMElement* detail)
    : m_hr(E_FAIL), m_msg(msg), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    m_codes.push_back(code);
    addProperties(p);
    setDetail(detail);
}

SAMLException::SAMLException(const saml::QName& code, const string& msg, const namedparams& p, DOMElement* detail)
    : m_hr(E_FAIL), m_msg(msg), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    m_codes.push_back(code);
    addProperties(p);
    setDetail(detail);
}

SAMLException::SAMLException(HRESULT code, const char* msg, const params& p, DOMElement* detail)
    : m_hr(code), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    if (msg)
        m_msg=msg;
    addProperties(p);
    setDetail(detail);
}

SAMLException::SAMLException(HRESULT code, const char* msg, const namedparams& p, DOMElement* detail)
    : m_hr(code), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    if (msg)
        m_msg=msg;
    addProperties(p);
    setDetail(detail);
}

SAMLException::SAMLException(HRESULT code, const string& msg, const params& p, DOMElement* detail)
    : m_hr(code), m_msg(msg), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    addProperties(p);
    setDetail(detail);
}

SAMLException::SAMLException(HRESULT code, const string& msg, const namedparams& p, DOMElement* detail)
    : m_hr(code), m_msg(msg), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    addProperties(p);
    setDetail(detail);
}

SAMLException::SAMLException(const SAMLException& src)
    : m_hr(src.m_hr), m_msg(src.m_msg), m_processedmsg(src.m_processedmsg), m_codes(src.m_codes), m_params(src.m_params),
        m_detail(NULL), m_scratch(NULL)
{
    m_classname=src.m_classname;
    setDetail(src.m_detail);
}

SAMLException& SAMLException::operator=(const SAMLException& src)
{
    m_classname=src.m_classname;
    m_hr=src.m_hr;
    m_msg=src.m_msg;
    m_processedmsg=src.m_processedmsg;
    m_codes=src.m_codes;
    m_params=src.m_params;
    setDetail(src.m_detail);
    return *this;
}

SAMLException::SAMLException(DOMElement* e) : m_hr(E_FAIL), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    fromDOM(e);
}

SAMLException::SAMLException(istream& in) : SAMLObject(in), m_hr(E_FAIL), m_detail(NULL), m_scratch(NULL)
{
    RTTI(SAMLException);
    fromDOM(m_document->getDocumentElement());
}

SAMLException::~SAMLException() throw()
{
    if (m_scratch)
        m_scratch->release();
}

void SAMLException::fromDOM(DOMElement* e)
{
    SAMLObject::fromDOM(e);

    if (SAMLConfig::getConfig().strict_dom_checking &&
        (XMLString::compareString(XML::SAMLP_NS,e->getNamespaceURI()) || XMLString::compareString(L(Status),e->getLocalName())))
        throw MalformedException(SAMLException::RESPONDER,"SAMLException::fromDOM() requires samlp:Status at root");

    // Extract the status message.
    const XMLCh* msg=NULL;
    DOMElement* m=XML::getFirstChildElement(e,XML::SAMLP_NS,L(StatusMessage));
    if (m && m->hasChildNodes())
        msg=m->getFirstChild()->getNodeValue();
    auto_ptr_char trans(msg);
    if (trans.get())
        m_msg=trans.get();

    DOMNodeList* nlist=e->getElementsByTagNameNS(XML::SAMLP_NS,L(StatusCode));
    for (unsigned int i=0; nlist && i<nlist->getLength(); i++) {
        saml::QName* qptr=saml::QName::getQNameAttribute(static_cast<DOMElement*>(nlist->item(i)),NULL,L(Value));
        if (qptr)
            m_codes.push_back(*qptr);
        else
            throw MalformedException(SAMLException::RESPONDER,"SAMLException::fromDOM() unable to evaluate QName Value");
        delete qptr;
    }

    m_detail=XML::getLastChildElement(e,XML::SAMLP_NS,L(StatusDetail));
    if (m_detail) {
        // Look for message parameters.
        nlist=m_detail->getElementsByTagNameNS(XML::OPENSAML_NS,L(Param));
        for (unsigned int j=0; nlist && j<nlist->getLength(); j++) {
            if (nlist->item(j)->hasChildNodes()) {
                auto_ptr_char pname(static_cast<DOMElement*>(nlist->item(j))->getAttributeNS(NULL,L(Name)));
                auto_ptr_char pval(nlist->item(j)->getFirstChild()->getNodeValue());
                if (pname.get() && pval.get())
                    m_params[pname.get()]=pval.get();
            }
        }
        
        // Look for HRESULT
        const DOMElement* hr=XML::getFirstChildElement(m_detail,XML::OPENSAML_NS,L(Code));
        if (hr && hr->hasChildNodes()) {
            auto_ptr_char pval(hr->getFirstChild()->getNodeValue());
            if (pval.get())
                m_hr=strtoul(pval.get(),NULL,0); // 0 base means we can handle decimal or hex syntax
        }
    }
}

inline const char* get_digit_character()
{
    static const char  s_characters[19] = 
    {
            '9'
        ,   '8'
        ,   '7'
        ,   '6'
        ,   '5'
        ,   '4'
        ,   '3'
        ,   '2'
        ,   '1'
        ,   '0'
        ,   '1'
        ,   '2'
        ,   '3'
        ,   '4'
        ,   '5'
        ,   '6'
        ,   '7'
        ,   '8'
        ,   '9'
    };
    static const char  *s_mid  =   s_characters + 9;

    return s_mid;
}

inline const char* unsigned_integer_to_string(char* buf, size_t cchBuf, int i)
{
    char* psz=buf + cchBuf - 1;     // Set psz to last char
    *psz = 0;                       // Set terminating null

    do {
        unsigned int lsd = i % 10;  // Get least significant
                                    // digit

        i /= 10;                    // Prepare for next most
                                    // significant digit

        --psz;                      // Move back

        *psz = get_digit_character()[lsd]; // Place the digit

    } while(i!=0 && psz>buf);

    return psz;
}

void SAMLException::addProperties(const params& p)
{
    int i=m_params.size()+1;
    char buf[11];
    Iterator<const char*> pit=p.get();
    while (pit.hasNext()) {
        const char* value=pit.next();
        m_params[unsigned_integer_to_string(buf,sizeof(buf),i)]=value;
        i++;
    }
    ownStrings();
    setDirty();
}

void SAMLException::addProperties(const namedparams& p)
{
    Iterator<const char*> pit=p.get();
    while (pit.hasNext()) {
        const char* name=pit.next();
        m_params.erase(name);
        m_params[name]=pit.next();
    }
    ownStrings();
    setDirty();
}

const char* SAMLException::getProperty(unsigned int index) const
{
    char buf[11];
    map<string,string>::const_iterator i=m_params.find(unsigned_integer_to_string(buf,sizeof(buf),index));
    return (i==m_params.end()) ? NULL : i->second.c_str();
}

const char* SAMLException::getProperty(const char* name) const
{
    map<string,string>::const_iterator i=m_params.find(name);
    return (i==m_params.end()) ? NULL : i->second.c_str();
}

void SAMLException::setMessage(const char* msg)
{
    if (msg)
        m_msg=msg;
    else
        m_msg.erase();
    m_processedmsg.erase();
    ownStrings();
    setDirty();
}

void SAMLException::setStatus(HRESULT hr)
{
    m_hr=hr;
    ownStrings();
    setDirty();
}

void SAMLException::setCodes(const Iterator<saml::QName>& codes)
{
    while (m_codes.size())
        removeCode(0);
    while (codes.hasNext())
        addCode(codes.next());
}

void SAMLException::addCode(const saml::QName& code)
{
    m_codes.push_back(code);
    ownStrings();
    setDirty();
}

void SAMLException::removeCode(unsigned long index)
{
    m_codes.erase(m_codes.begin()+index);
    ownStrings();
    setDirty();
}

void SAMLException::setDetail(DOMElement* detail)
{
    if (detail) {
        if (!m_scratch)
            m_scratch=DOMImplementationRegistry::getDOMImplementation(NULL)->createDocument();
        m_detail=static_cast<DOMElement*>(m_scratch->importNode(detail,true));
    }
    ownStrings();
    setDirty();
}

DOMElement* SAMLException::buildRoot(DOMDocument* doc, bool xmlns) const
{
    DOMElement* e = doc->createElementNS(XML::SAMLP_NS, L(Status));
    if (xmlns)
        e->setAttributeNS(XML::XMLNS_NS, L(xmlns), XML::SAMLP_NS);
    e->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,samlp),XML::SAMLP_NS);
    return e;
}

DOMNode* SAMLException::toDOM(DOMDocument* doc, bool xmlns) const
{
    SAMLObject::toDOM(doc,xmlns);
    DOMElement* s=static_cast<DOMElement*>(m_root);
    doc=s->getOwnerDocument();

    if (m_bDirty) {
        if (m_codes.empty()) {
            DOMElement* sc=doc->createElementNS(XML::SAMLP_NS,L(StatusCode));
            sc->setAttributeNS(NULL,L(Value),L_QNAME(samlp,Responder));
            s->appendChild(sc);
        }
        else {
            DOMNode* base=s;
            for (vector<saml::QName>::const_iterator qcode=m_codes.begin(); qcode!=m_codes.end(); qcode++) {
                DOMElement* sc=doc->createElementNS(XML::SAMLP_NS,L(StatusCode));
                const XMLCh* codens=qcode->getNamespaceURI();
                if (XMLString::compareString(codens,XML::SAMLP_NS)) {
                    sc->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,code),codens);
                    static const XMLCh nspre[]={chLatin_c, chLatin_o, chLatin_d, chLatin_e, chColon, chNull};
                    codens=nspre;
                }
                else {
                    static const XMLCh samlpre[]={chLatin_s, chLatin_a, chLatin_m, chLatin_l, chLatin_p, chColon, chNull};
                    codens=samlpre;
                }
    
                XMLCh* qval=new XMLCh[XMLString::stringLen(codens) + XMLString::stringLen(qcode->getLocalName()) + 1];
                qval[0]=chNull;
                XMLString::catString(qval,codens);
                XMLString::catString(qval,qcode->getLocalName());
                sc->setAttributeNS(NULL,L(Value),qval);
                delete[] qval;
                base=base->appendChild(sc);
            }
        }
    
        if (!m_msg.empty()) {
            DOMElement* msg=doc->createElementNS(XML::SAMLP_NS,L(StatusMessage));
            auto_ptr_XMLCh xmsg(m_msg.c_str());
            msg->appendChild(doc->createTextNode(xmsg.get()));
            s->appendChild(msg);
        }
        
        if (m_detail && m_detail->getOwnerDocument()!=doc) {
            DOMElement* copy=static_cast<DOMElement*>(doc->importNode(m_detail,true));
            if (m_detail->getParentNode())
                m_detail->getParentNode()->removeChild(m_detail);
            m_detail->release();
            m_detail=copy;
        }
    
        if (typeid(this)!=typeid(SAMLException) || !m_params.empty() || m_hr!=E_FAIL) {
            // We use the StatusDetail element for custom object content.
            if (!m_detail)
                m_detail=doc->createElementNS(XML::SAMLP_NS,L(StatusDetail));
            m_detail->setAttributeNS(XML::XMLNS_NS,L_QNAME(xmlns,opensaml),XML::OPENSAML_NS);
    
            // First we create the exception class.
            if (typeid(this)!=typeid(SAMLException)) {
                DOMElement* eclass=XML::getFirstChildElement(m_detail,XML::OPENSAML_NS,L_QNAME(opensaml,ExceptionClass));
                if (!eclass) {
                    string temp=string("org.opensaml.") + m_classname;
                    auto_ptr_XMLCh type(temp.c_str());
                    eclass=doc->createElementNS(XML::OPENSAML_NS,L_QNAME(opensaml,ExceptionClass));
                    eclass->appendChild(doc->createTextNode(type.get()));
                    m_detail->appendChild(eclass);
                }
            }
            
            // Now serialize any parameters after removing what's there.
            DOMNodeList* plist=m_detail->getElementsByTagNameNS(XML::OPENSAML_NS,L_QNAME(opensaml,Param));
            for (unsigned int pi=0; plist && pi<plist->getLength(); pi++)
                m_detail->removeChild(plist->item(pi));
            for (map<string,string>::const_iterator i=m_params.begin(); i!=m_params.end(); i++) {
                auto_ptr_XMLCh pname(i->first.c_str());
                auto_ptr_XMLCh pval(i->second.c_str());
                DOMElement* p=doc->createElementNS(XML::OPENSAML_NS,L_QNAME(opensaml,Param));
                p->setAttributeNS(NULL,L(Name),pname.get());
                p->appendChild(doc->createTextNode(pval.get()));
                m_detail->appendChild(p);
            }
            
            // Finally, handle code.
            DOMElement* hr=XML::getFirstChildElement(m_detail,XML::OPENSAML_NS,L_QNAME(opensaml,Code));
            if (hr)
                m_detail->removeChild(hr);
            if (m_hr!=E_FAIL) {
                char hexbuf[16];
                sprintf(hexbuf,"0x%.8X",m_hr);
                hexbuf[10]=0;
                auto_ptr_XMLCh hexstr(hexbuf);
                hr=doc->createElementNS(XML::OPENSAML_NS,L_QNAME(opensaml,Code));
                hr->appendChild(doc->createTextNode(hexstr.get()));
                m_detail->appendChild(hr);
            }
        }
        else
            SAML_log.debug("skipping type name generation for generic exception with no code or parameters");

        if (m_detail)
            s->appendChild(m_detail);
        setClean();
    }
    else if (xmlns) {
        DECLARE_DEF_NAMESPACE(s,XML::SAMLP_NS);
        DECLARE_NAMESPACE(s,samlp,XML::SAMLP_NS);
    }
    return m_root;
}

const char* SAMLException::getMessage() const
{
    if (!m_processedmsg.empty())
        return m_processedmsg.c_str();
    else if (m_params.empty())
        return m_msg.c_str();

    static const char* legal="ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz1234567890_";

    // Replace any parameters in the message.
    string::size_type i=0,start=0;
    while (start!=string::npos && start<m_msg.length() && (i=m_msg.find("$",start))!=string::npos) {
        if (i>start)
            m_processedmsg += m_msg.substr(start,i-start);  // append everything in between
        start=i+1;                                  // move start to the beginning of the token name
        i=m_msg.find_first_not_of(legal,start);     // find token delimiter
        if (i==start) {                             // append a non legal character
           m_processedmsg+=m_msg[start++];
           continue;
        }
        
        // search for token in map
        map<string,string>::const_iterator param=m_params.find(m_msg.substr(start,(i==string::npos) ? i : i-start));
        if (param!=m_params.end()) {
            m_processedmsg+=param->second;
            start=i;
        }
    }
    if (start!=string::npos && start<m_msg.length())
        m_processedmsg += m_msg.substr(start,i);    // append rest of string
    return m_processedmsg.c_str();
}

SAMLObject* SAMLException::clone() const
{
    return new SAMLException(*this);
}
