import os

from nipype.testing import *
import nipype.interfaces.fsl as fsl


def test_fslversion():
    ver = fsl.fsl_info.version
    if ver:
        # If ver is None, fsl is not installed
        ver = ver.split('.')
        yield assert_equal, ver[0], '4'

    
def test_fsloutputtype():
    types = ['ANALYZE_GZ', 'NIFTI_PAIR_GZ', 'NIFTI', 'NIFTI_PAIR',
             'NIFTI_GZ', 'ANALYZE']
    out_type, ext = fsl.fsl_info.outputtype()
    if out_type is None:
        # Environment variable is not set.  FSL may not be installed.
        return
    yield assert_true, out_type in types
    env_type = os.environ.get('FSLOUTPUTTYPE')
    if env_type:
        # Set to same value for test.
        out_type, ext = fsl.fsl_info.outputtype(env_type)
        yield assert_equal, out_type, env_type


def test_FSLCommand():
    # Most methods in FSLCommand are tested in the subclasses.  Only
    # testing the one item that is not.
    cmd = fsl.FSLCommand()
    cmd.cmd = 'bet' # Set the cmd to something
    res = cmd.run()
    yield assert_equal, type(res), fsl.InterfaceResult
    

# test Bet
def test_bet():
    better = fsl.Bet()
    yield assert_equal, better.cmd, 'bet'

    # Test raising error with mandatory args absent
    yield assert_raises, ValueError, better.run

    # .inputs based parameter setting
    better.inputs.frac = 0.5
    better.inputs.infile = 'infile'
    better.inputs.outfile = 'outfile'
    yield assert_equal, better.cmdline, 'bet infile outfile -f 0.50'

    # .run() based parameter setting
    betted = better.run(infile='infile2', outfile='outfile')
    # Non-existant files, shouldn't finish cleanly
    yield assert_not_equal, betted.runtime.returncode, 0
    yield assert_equal, betted.interface.inputs.infile, 'infile2'
    yield assert_equal, betted.interface.inputs.outfile, 'outfile'
    yield assert_equal, betted.runtime.cmdline, 'bet infile2 outfile -f 0.50'
    
    # test that an outfile is autogenerated when inputs.outfile is None
    better = fsl.Bet()
    better.inputs.infile = 'infile'
    res = better.run()
    outfile = os.path.join(os.path.abspath('.'), 'infile_bet')
    # This behavior is bad, see Trac ticket #42
    # yield assert_equal, better.inputs.outfile, outfile

    # Our options and some test values for them
    # Should parallel the opt_map structure in the class for clarity
    opt_map = {
        'outline':            ('-o', True),
        'mask':               ('-m', True),
        'skull':              ('-s', True),
        'nooutput':           ('-n', True),
        'frac':               ('-f 0.40', 0.4),
        'vertical_gradient':  ('-g 0.75', 0.75),
        'radius':             ('-r 20', 20),
        'center':             ('-c 54 75 80', (54, 75, 80)),
        'threshold':          ('-t', True),
        'mesh':               ('-e', True),
        'verbose':            ('-v', True),
        'flags':              ('--i-made-this-up', '--i-made-this-up'),
            }
    # Currently we don't test -R, -S, -B, -Z, -F, -A or -A2
    

    # test each of our arguments
    for name, settings in opt_map.items():
        better = fsl.Bet(**{name: settings[1]})
        yield assert_equal, better.cmdline, ' '.join([better.cmd, settings[0]])
    
        
# test fast
def test_fast():
    faster = fsl.Fast()
    faster.inputs.verbose = True
    fasted = faster.run(infiles='infile')
    fasted2 = faster.run(infiles=['infile', 'otherfile'])
    
    yield assert_equal, faster.cmd, 'fast'
    yield assert_equal, faster.inputs.verbose, True
    yield assert_equal, faster.inputs.manualseg , None
    yield assert_not_equal, faster, fasted
    yield assert_equal, fasted.runtime.cmdline, 'fast -v infile'
    yield assert_equal, fasted2.runtime.cmdline, 'fast -v infile otherfile'

    faster = fsl.Fast()
    faster.inputs.infiles = 'foo.nii'
    yield assert_equal, faster.cmdline, 'fast foo.nii'
    faster.inputs.infiles = ['foo.nii', 'bar.nii']
    yield assert_equal, faster.cmdline, 'fast foo.nii bar.nii'
    
    # Our options and some test values for them
    # Should parallel the opt_map structure in the class for clarity
    opt_map = {'number_classes':       ('-n 4', 4),
               'bias_iters':           ('-I 5', 5),
               'bias_lowpass':         ('-l 15', 15),
               'img_type':             ('-t 2', 2),
               'init_seg_smooth':      ('-f 0.035', 0.035),
               'segments':             ('-g', True),
               'init_transform':       ('-a xform.mat', 'xform.mat'),
               'other_priors':         ('-A prior1.nii prior2.nii prior3.nii', 
                       ('prior1.nii', 'prior2.nii', 'prior3.nii')),
               'nopve':                ('--nopve', True),
               'output_biasfield':     ('-b', True),
               'output_biascorrected': ('-B', True),
               'nobias':               ('-N', True),
               'n_inputimages':        ('-S 2', 2),
               'out_basename':         ('-o fasted', 'fasted'),
               'use_priors':           ('-P', True),
               'segment_iters':        ('-W 14', 14),
               'mixel_smooth':         ('-R 0.25', 0.25),
               'iters_afterbias':      ('-O 3', 3),
               'hyper':                ('-H 0.15', 0.15),
               'verbose':              ('-v', True), 
               'manualseg':            ('-s intensities.nii',
                       'intensities.nii'),
               'probability_maps':     ('-p', True),
              }
   
    # test each of our arguments
    for name, settings in opt_map.items():
        faster = fsl.Fast(**{name: settings[1]})
        yield assert_equal, faster.cmdline, ' '.join([faster.cmd, settings[0]])


#test flirt
def test_flirt():
    flirter = fsl.Flirt()
    flirter.inputs.bins = 256
    flirter.inputs.cost = 'mutualinfo'
    flirted = flirter.run(infile='infile',reference='reffile',
                          outfile='outfile',outmatrix='outmat.mat')
    flirt_est = flirter.run(infile='infile',reference='reffile',
                            outfile=None,outmatrix='outmat.mat')

    yield assert_not_equal, flirter, flirted
    yield assert_not_equal, flirted, flirt_est

    yield assert_equal, flirter.cmd, 'flirt'
    yield assert_equal, flirter.inputs.bins, flirted.interface.inputs.bins
    yield assert_equal, flirter.inputs.cost, flirt_est.interface.inputs.cost
    yield assert_equal, flirted.runtime.cmdline, \
        'flirt -in infile -ref reffile -omat outmat.mat -out outfile ' \
        '-bins 256 -cost mutualinfo'

    flirter = fsl.Flirt()
    # infile not specified
    yield assert_raises, AttributeError, flirter.run
    flirter.inputs.infile = 'foo.nii'
    # reference not specified
    yield assert_raises, AttributeError, flirter.run
    flirter.inputs.reference = 'mni152.nii'
    res = flirter.run()
    realcmd = 'flirt -in foo.nii -ref mni152.nii'
    yield assert_equal, res.interface.cmdline, realcmd
    inputs = dict(flags='-v')
    res = flirter.run(**inputs)
    realcmd ='flirt -in foo.nii -ref mni152.nii -v'
    yield assert_equal, res.interface.cmdline, realcmd


def test_applyxfm():
    # ApplyXFM subclasses Flirt.
    flt = fsl.ApplyXFM(infile='subj.nii', inmatrix='xfm.mat', 
                       outfile='xfm_subj.nii', reference='mni152.nii')
    flted = flt.run()
    yield assert_equal, flt.cmdline, \
        'flirt -in subj.nii -ref mni152.nii -init xfm.mat ' \
        '-out xfm_subj.nii -applyxfm'
    flt = fsl.ApplyXFM()
    yield assert_raises, AttributeError, flt.run
    flt.inputs.infile = 'subj.nii'
    flt.inputs.outfile = 'xfm_subj.nii'
    # reference not specified
    yield assert_raises, AttributeError, flt.run
    flt.inputs.reference = 'mni152.nii'
    # inmatrix not specified
    yield assert_raises, AttributeError, flt.run
    flt.inputs.inmatrix = 'xfm.mat'
    res = flt.run()
    realcmd = 'flirt -in subj.nii -ref mni152.nii -init xfm.mat '\
        '-out xfm_subj.nii -applyxfm'
    yield assert_equal, res.interface.cmdline, realcmd


# Mcflirt
def test_mcflirt():
    frt = fsl.McFlirt()
    yield assert_equal, frt.cmd, 'mcflirt'
    
    opt_map = {
        'outfile':     ('-out bar.nii', 'bar.nii'),
        'cost':        ('-cost mutualinfo', 'mutualinfo'),
        'bins':        ('-bins 256', 256),
        'dof':         ('-dof 6', 6),
        'refvol':      ('-refvol 2', 2),
        'scaling':     ('-scaling 6.00', 6.00),
        'smooth':      ('-smooth 1.00', 1.00),
        'rotation':    ('-rotation 2', 2),
        'verbose':     ('-verbose', True),
        'stages':      ('-stages 3', 3),
        'init':        ('-init matrix.mat', 'matrix.mat'),
        'usegradient': ('-gdt', True),
        'usecontour':  ('-edge', True),
        'meanvol':     ('-meanvol', True),
        'statsimgs':   ('-stats', True),
        'savemats':    ('-mats', True),
        'saveplots':   ('-plots', True),
        'report':      ('-report', True),
        }

    for name, settings in opt_map.items():
        fnt = fsl.McFlirt(**{name : settings[1]})
        yield assert_equal, fnt.cmdline, ' '.join([fnt.cmd, settings[0]])

    # Test error is raised when missing required args
    fnt = fsl.McFlirt()
    yield assert_raises, AttributeError, fnt.run
    # Test run result
    fnt = fsl.McFlirt()
    fnt.inputs.infile = 'foo.nii'
    res = fnt.run()
    yield assert_equal, type(res), fsl.InterfaceResult
    res = fnt.run(infile='bar.nii')
    yield assert_equal, type(res), fsl.InterfaceResult


#test fnirt
def test_fnirt():
    fnirt = fsl.Fnirt()
    yield assert_equal, fnirt.cmd, 'fnirt'

    # Test inputs with variable number of values
    fnirt.inputs.sub_sampling = [8,6,4]
    yield assert_equal, fnirt.inputs.sub_sampling, [8,6,4]
    fnirtd = fnirt.run(infile='infile', reference='reference')
    realcmd = 'fnirt --in=infile --ref=reference --subsamp=8,6,4'
    yield assert_equal, fnirtd.runtime.cmdline, realcmd

    fnirt2 = fsl.Fnirt(sub_sampling=[8,2])
    fnirtd2 = fnirt2.run(infile='infile', reference='reference')
    realcmd = 'fnirt --in=infile --ref=reference --subsamp=8,2'
    yield assert_equal, fnirtd2.runtime.cmdline, realcmd

    # Test case where input that can be a list is just a single value
    params = [('sub_sampling', '--subsamp'),
              ('max_iter', '--miter'),
              ('referencefwhm', '--reffwhm'),
              ('imgfwhm', '--infwhm'),
              ('lambdas', '--lambda'),
              ('estintensity', '--estint'),
              ('applyrefmask', '--applyrefmask'),
              ('applyimgmask', '--applyinmask')]
    for item, flag in params:
        
        
        if item in ('sub_sampling', 'max_iter',
                    'referencefwhm', 'imgfwhm',
                    'lambdas', 'estintensity'):
            fnirt = fsl.Fnirt(**{item : 5})
            cmd = 'fnirt %s=%d' % (flag, 5)
        else:
            fnirt = fsl.Fnirt(**{item : 5})
            cmd = 'fnirt %s=%f' % (flag, 5)
        yield assert_equal, fnirt.cmdline, cmd

    # Test error is raised when missing required args
    fnirt = fsl.Fnirt()
    yield assert_raises, AttributeError, fnirt.run
    fnirt.inputs.infile = 'foo.nii'
    # I don't think this is correct. See Fnirt documentation -DJC
    # yield assert_raises, AttributeError, fnirt.run
    fnirt.inputs.reference = 'mni152.nii'
    res = fnirt.run()
    yield assert_equal, type(res), fsl.InterfaceResult

    opt_map = {
        'affine':           ('--aff=affine.mat', 'affine.mat'),
        'initwarp':         ('--inwarp=warp.mat', 'warp.mat'),
        'initintensity':    ('--intin=inten.mat', 'inten.mat'),
        'configfile':       ('--config=conf.txt', 'conf.txt'),
        'referencemask':    ('--refmask=ref.mat', 'ref.mat'),
        'imagemask':        ('--inmask=mask.nii', 'mask.nii'),
        'fieldcoeff_file':  ('--cout=coef.txt', 'coef.txt'),
        'outimage':         ('--iout=out.nii', 'out.nii'),
        'fieldfile':        ('--fout=fld.txt', 'fld.txt'),
        'jacobianfile':     ('--jout=jaco.txt', 'jaco.txt'),
        'reffile':          ('--refout=refout.nii', 'refout.nii'),
        'intensityfile':    ('--intout=intout.txt', 'intout.txt'),
        'logfile':          ('--logout=log.txt', 'log.txt'),
        'verbose':          ('--verbose', True),
        'flags':            ('--fake-flag', '--fake-flag')}

    for name, settings in opt_map.items():
        fnirt = fsl.Fnirt(**{name : settings[1]})
        yield assert_equal, fnirt.cmdline, ' '.join([fnirt.cmd, settings[0]])

