/* display an image in a window ... watching an iImage model.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

/* Define to trace button press events.
#define EVENT
 */

#include "ip.h"

static iWindowClass *parent_class = NULL;

/* All the magnification menus we have.
 */
typedef struct _ImageviewMagmenu {
	const char *path;
	int mag;
} ImageviewMagmenu;

static const ImageviewMagmenu imageview_mags[] = {
	{ "/View/Zoom/6%", -16 },
	{ "/View/Zoom/12%", -8 },
	{ "/View/Zoom/25%", -4 },
	{ "/View/Zoom/50%", -2 },
	{ "/View/Zoom/100%", 1 },
	{ "/View/Zoom/200%", 2 },
	{ "/View/Zoom/400%", 4 },
	{ "/View/Zoom/800%", 8 },
	{ "/View/Zoom/1600%", 16 }
};

static void
imageview_destroy( GtkObject *object )
{
	Imageview *iv;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_IMAGEVIEW( object ) );

	iv = IMAGEVIEW( object );

#ifdef DEBUG
	printf( "imageview_destroy\n" );
#endif /*DEBUG*/

	/* My instance destroy stuff.
	 */
	UNREF( iv->imagemodel );
	UNREF( iv->ifac );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

static void
imageview_class_init( ImageviewClass *class )
{
	GtkObjectClass *object_class = (GtkObjectClass *) class;

	parent_class = g_type_class_peek_parent( class );

	object_class->destroy = imageview_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
}

static void
imageview_init( Imageview *iv )
{
	iv->imagemodel = NULL;
	iv->ifac = NULL;
}

GtkType
imageview_get_type( void )
{
	static GtkType imageview_type = 0;

	if( !imageview_type ) {
		static const GtkTypeInfo sinfo = {
			"Imageview",
			sizeof( Imageview ),
			sizeof( ImageviewClass ),
			(GtkClassInitFunc) imageview_class_init,
			(GtkObjectInitFunc) imageview_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		imageview_type = gtk_type_unique( TYPE_IWINDOW, &sinfo );
	}

	return( imageview_type );
}

static void
imageview_refresh_title( Imageview *iv )
{
	Imagemodel *imagemodel = iv->imagemodel;
	Conversion *conv = imagemodel->conv;
	iImage *iimage = imagemodel->iimage;
	Imageinfo *ii = iimage->instance.ii;
	Row *row = HEAPMODEL( iimage )->row;
	Workspace *ws = row_get_workspace( row );

	BufInfo buf;
	char txt[512];

	buf_init_static( &buf, txt, 512 );
	row_qualified_name_relative( ws->sym, row, &buf );

	if( ii && IOBJECT( ii )->name && ii->from_file )
		buf_appendf( &buf, " - %s", IOBJECT( ii )->name );

	buf_appendf( &buf, " - %.0f%%", 100.0 * conversion_dmag( conv->mag ) );

	iwindow_set_title( IWINDOW( iv ), "%s", buf_all( &buf ) );
}

/* The model has changed ... update our menus and titlebar.
 */
static void
imageview_imagemodel_changed_cb( Imagemodel *imagemodel, Imageview *iv )
{
	Conversion *conv = imagemodel->conv;
	GtkWidget *item;
	int i;

	for( i = 0; i < IM_NUMBER( imageview_mags ); i++ )
		if( conv->mag == imageview_mags[i].mag ) {
			item = gtk_item_factory_get_widget( iv->ifac,
				imageview_mags[i].path );
			gtk_check_menu_item_set_active( 
				GTK_CHECK_MENU_ITEM( item ), TRUE );
		}

	item = gtk_item_factory_get_widget( iv->ifac, 
		"/View/Toolbar/Display control" );
	gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( item ), 
		imagemodel->show_convert );

	item = gtk_item_factory_get_widget( iv->ifac, "/View/Toolbar/Status" );
	gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( item ), 
		imagemodel->show_status );

	item = gtk_item_factory_get_widget( iv->ifac, "/View/Toolbar/Paint" );
	gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( item ), 
		imagemodel->show_paintbox );

	item = gtk_item_factory_get_widget( iv->ifac, "/View/Toolbar/Rulers" );
	gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( item ),
		imagemodel->show_rulers );

	if( imagemodel->state == IMAGEMODEL_SELECT )
		item = gtk_item_factory_get_widget( iv->ifac, 
			"/View/Mode/Select" );
	else if( imagemodel->state == IMAGEMODEL_PAN )
		item = gtk_item_factory_get_widget( iv->ifac, 
			"/View/Mode/Pan" );
	else if( imagemodel->state == IMAGEMODEL_MAGIN )
		item = gtk_item_factory_get_widget( iv->ifac, 
			"/View/Mode/Zoom In" );
	else if( imagemodel->state == IMAGEMODEL_MAGOUT )
		item = gtk_item_factory_get_widget( iv->ifac, 
			"/View/Mode/Zoom Out" );
	else if( imagemodel_state_paint( imagemodel->state ) )
		item = gtk_item_factory_get_widget( iv->ifac, 
			"/View/Mode/Paint" );
	else
		item = NULL;

	if( item )
		gtk_check_menu_item_set_active( GTK_CHECK_MENU_ITEM( item ),
			TRUE );

	imageview_refresh_title( iv );
}

/* The model has been destroyed ... kill us too.
 */
static void
imageview_imagemodel_destroy_cb( Imagemodel *imagemodel, Imageview *iv )
{
	gtk_widget_destroy( GTK_WIDGET( iv ) );
}

static void
imageview_replace_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	iImage *iimage = imagemodel->iimage;

	classmodel_graphic_replace( CLASSMODEL( iimage ), GTK_WIDGET( iv ) );
}

static const char *imageview_region_name[] = {
	CLASS_MARK,
	CLASS_HGUIDE,
	CLASS_VGUIDE,
	CLASS_ARROW,
	CLASS_REGION,
	CLASS_AREA
};

static void
imageview_new_arrow_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	iImage *iimage = imagemodel->iimage;
	Row *row = HEAPMODEL( iimage )->row;
	Workspace *ws = row_get_workspace( row );
	Conversion *conv = imagemodel->conv;
	iRegionType rt = (iRegionType) callback_action;
	int dx = imagemodel->visible.left + imagemodel->visible.width / 2;
	int dy = imagemodel->visible.top + imagemodel->visible.height / 2;

	char txt[MAX_STRSIZE];
	BufInfo buf;
	Symbol *sym;
	int ix, iy;

	conversion_disp_to_im( conv, dx, dy, &ix, &iy );

	buf_init_static( &buf, txt, MAX_STRSIZE );
	buf_appendf( &buf, "%s ", imageview_region_name[rt] );
	row_qualified_name_relative( ws->sym, row, &buf );
	switch( rt ) {
	case IREGION_MARK:
		buf_appendf( &buf, " (%d) (%d)", ix, iy );
		break;

	case IREGION_HGUIDE:
		buf_appendf( &buf, " (%d)", iy );
		break;

	case IREGION_VGUIDE:
		buf_appendf( &buf, " (%d)", ix );
		break;

	default:
		assert( FALSE );
	}

	if( !(sym = workspace_add_def( ws, buf_all( &buf ) )) ) {
		box_alert( GTK_WIDGET( iv ) );
		return;
	}

	workspace_deselect_all( ws );
}

static void
imageview_new_region_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	iImage *iimage = imagemodel->iimage;
	Row *row = HEAPMODEL( iimage )->row;
	Workspace *ws = row_get_workspace( row );
	Conversion *conv = imagemodel->conv;
	iRegionType rt = (iRegionType) callback_action;

	Rect dr, ir;
	char txt[MAX_STRSIZE];
	BufInfo buf;
	Symbol *sym;
	Column *col;

	dr.left = imagemodel->visible.left + imagemodel->visible.width / 4;
	dr.top = imagemodel->visible.top + imagemodel->visible.height / 4;
	dr.width = imagemodel->visible.width / 2;
	dr.height = imagemodel->visible.height / 2;
	conversion_disp_to_im_rect( conv, &dr, &ir );

	buf_init_static( &buf, txt, MAX_STRSIZE );
	buf_appendf( &buf, "%s ", imageview_region_name[rt] );
	row_qualified_name_relative( ws->sym, row, &buf );
	buf_appendf( &buf, " (%d) (%d) %d %d", 
		ir.left, ir.top, ir.width, ir.height );

	if( !(sym = workspace_add_def( ws, buf_all( &buf ) )) ) {
		box_alert( GTK_WIDGET( iv ) );
		return;
	}

	col = sym->expr->row->top_col;
	model_scrollto( MODEL( col ) );
}

static void
imageview_save_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	iImage *iimage = imagemodel->iimage;

	classmodel_graphic_save( CLASSMODEL( iimage ), GTK_WIDGET( iv ) );
}

static void
imageview_header_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	iImage *iimage = imagemodel->iimage;
	Row *row = HEAPMODEL( iimage )->row;
	Workspace *ws = row_get_workspace( row );
	GtkWidget *imageheader = imageheader_new( imagemodel->conv );
	BufInfo buf;
	char txt[512];

	buf_init_static( &buf, txt, 512 );
	row_qualified_name_relative( ws->sym, row, &buf );

	iwindow_set_title( IWINDOW( imageheader ), _( "Header for \"%s\"" ), 
		buf_all( &buf ) );
	idialog_set_callbacks( IDIALOG( imageheader ), 
		NULL, NULL, NULL, iv );
	idialog_add_ok( IDIALOG( imageheader ), iwindow_true_cb, _( "OK" ) );
	idialog_set_parent( IDIALOG( imageheader ), GTK_WIDGET( iv ) );
	iwindow_build( IWINDOW( imageheader ) );

	gtk_widget_show( imageheader );
}

static void
imageview_quit_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );

	iwindow_kill( IWINDOW( iv ) );
}

static void
imageview_recalc_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	iImage *iimage = imagemodel->iimage;
	Row *row = HEAPMODEL( iimage )->row;

        workspace_deselect_all( row->ws );
        row_select( row );
        if( !workspace_selected_recalc( row->ws ) )
                box_alert( GTK_WIDGET( iv ) );
        workspace_deselect_all( row->ws );
}

static void
imageview_mag_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	int mag = (int) callback_action;
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	if( item->active && iv->ip ) 
		imagepresent_zoom_to( iv->ip, mag );
}

/* ... make sure the specified mode is set.
 */
static void
imageview_mode_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	ImagemodelState state = (ImagemodelState) callback_action;
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	if( item->active ) 
		imagemodel_set_state( imagemodel, state, widget );
}

/* Make sure we're in one of the paint modes.
 */
static void
imageview_mode_paint_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	if( item->active && 
		!imagemodel_state_paint( imagemodel->state ) )
		imagemodel_set_state( imagemodel, IMAGEMODEL_PEN, widget );
}

static void
imageview_zoom_in_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	Conversion *conv = imagemodel->conv;

	conversion_set_mag( conv, conversion_double( conv->mag ) ); 
}

static void
imageview_zoom_out_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	Conversion *conv = imagemodel->conv;

	conversion_set_mag( conv, conversion_halve( conv->mag ) ); 
}

static void
imageview_zoom_100_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );

	if( iv->ip )
		imagepresent_zoom_to( iv->ip, 1 );
}

static void
imageview_zoom_fit_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );

	imagepresent_zoom_to( iv->ip, 0 );
}

static void
imageview_show_rulers_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	imagemodel_set_rulers( imagemodel, item->active );
}

static void
imageview_show_status_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	imagemodel_set_status( imagemodel, item->active );
}

static void
imageview_show_paintbox_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	imagemodel_set_paintbox( imagemodel, item->active );
}

static void
imageview_show_convert_ifcb( gpointer callback_data, guint callback_action,
        GtkWidget *widget )
{
	Imageview *iv = IMAGEVIEW( callback_data );
	Imagemodel *imagemodel = iv->imagemodel;
	GtkCheckMenuItem *item = GTK_CHECK_MENU_ITEM( widget );

	imagemodel_set_convert( imagemodel, item->active );
}

/* Menu bar.
 */
static GtkItemFactoryEntry imageview_menu_items[] = {
	{ N_( "/_File" ),            			NULL,         
		NULL,			0, "<Branch>" },
	{ N_( "/File/_New" ), 			NULL, 
		NULL, 			0, "<Branch>" },
	{ N_( "/File/New/_Mark" ), 			NULL,
		imageview_new_arrow_ifcb,	IREGION_MARK, 
			"<StockItem>", GTK_STOCK_NEW },
	{ N_( "/File/New/_HGuide" ), 			NULL,
		imageview_new_arrow_ifcb,	IREGION_HGUIDE, 
			"<StockItem>", GTK_STOCK_NEW },
	{ N_( "/File/New/_VGuide" ), 			NULL,
		imageview_new_arrow_ifcb,	IREGION_VGUIDE, 
			"<StockItem>", GTK_STOCK_NEW },
	{ N_( "/File/New/_Arrow" ), 			NULL,
		imageview_new_region_ifcb,	IREGION_ARROW, 
			"<StockItem>", GTK_STOCK_NEW },
	{ N_( "/File/New/_Region" ), 			NULL,
		imageview_new_region_ifcb,	IREGION_REGION, 
			"<StockItem>", GTK_STOCK_NEW },
	{ N_( "/File/sep3" ),        			NULL,         
		NULL,       		0, "<Separator>" },
	{ N_( "/File/_Replace Image ..." ),		"<control>R", 
		imageview_replace_ifcb, 0 },
	{ N_( "/File/_Save Image As ..." ),     	"<control>S", 
		imageview_save_ifcb,    0, 
			"<StockItem>", GTK_STOCK_SAVE_AS },
	{ N_( "/File/sep1" ),        			NULL,         
		NULL,       		0, "<Separator>" },
	{ N_( "/File/Re_calculate Image" ),		"<control>C", 
		imageview_recalc_ifcb,  0 },
	{ N_( "/File/sep2" ),        			NULL,         
		NULL,       		0, "<Separator>" },
	{ N_( "/File/_Close" ),   			"<control>W", 
		imageview_quit_ifcb,    0, 
			"<StockItem>", GTK_STOCK_CLOSE },

	{ N_( "/_View" ),       			NULL,		
		NULL,			0, "<Branch>" },

	{ N_( "/View/_Toolbar" ), 			NULL,
		NULL,			0, "<Branch>" },
	{ N_( "/View/Toolbar/_Status" ), 		NULL, 
		imageview_show_status_ifcb,    	0, "<CheckItem>" },
	{ N_( "/View/Toolbar/_Display control" ), 	NULL, 
		imageview_show_convert_ifcb,   	0, "<CheckItem>" },
	{ N_( "/View/Toolbar/_Paint" ), 		NULL, 
		imageview_show_paintbox_ifcb,  	0, "<CheckItem>" },
	{ N_( "/View/Toolbar/_Rulers" ), 		NULL, 
		imageview_show_rulers_ifcb,    	0, "<CheckItem>" },

	{ N_( "/View/M_ode" ),				NULL, 
		NULL,       		0, "<Branch>" },
	{ N_( "/View/Mode/_Select" ),			NULL, 
		imageview_mode_ifcb, 
			IMAGEMODEL_SELECT, "<RadioItem>" },
	{ N_( "/View/Mode/_Pan" ),			NULL, 
		imageview_mode_ifcb, 
			IMAGEMODEL_PAN, "/View/Mode/Select" },
	{ N_( "/View/Mode/Zoom _In" ),			NULL, 
		imageview_mode_ifcb, 
			IMAGEMODEL_MAGIN, "/View/Mode/Select" },
	{ N_( "/View/Mode/Zoom _Out" ),			NULL, 
		imageview_mode_ifcb, 
			IMAGEMODEL_MAGOUT, "/View/Mode/Select" },
	{ N_( "/View/Mode/P_aint" ),			NULL, 
		imageview_mode_paint_ifcb, 
			0, "/View/Mode/Select" },

	{ N_( "/View/Image _Header" ), 	    		NULL, 
		imageview_header_ifcb,  0 },

	{ N_( "/View/sep1" ),        			NULL,         
		NULL,       		0, "<Separator>" },

	{ N_( "/View/Zoom _In" ),			"<control>plus", 
		imageview_zoom_in_ifcb,	0, 
			"<StockItem>", GTK_STOCK_ZOOM_IN },
	{ N_( "/View/Zoom _Out" ),			"<control>minus", 
		imageview_zoom_out_ifcb,0, 
			"<StockItem>", GTK_STOCK_ZOOM_OUT },
	{ N_( "/View/Zoom _100%" ),			"<control>equal", 
		imageview_zoom_100_ifcb,0, 
			"<StockItem>", GTK_STOCK_ZOOM_100 },
	{ N_( "/View/Zoom to _Fit" ),			NULL, 
		imageview_zoom_fit_ifcb,0,
			"<StockItem>", GTK_STOCK_ZOOM_FIT },
	{ N_( "/View/_Zoom" ),				NULL, 
		NULL,       		0, "<Branch>" },
	{ N_( "/View/Zoom/6%" ),			NULL, 
		imageview_mag_ifcb,	-16, "<RadioItem>" },
	{ N_( "/View/Zoom/12%" ),			NULL, 
		imageview_mag_ifcb, 	-8, "/View/Zoom/6%" },
	{ N_( "/View/Zoom/25%" ),			NULL, 
		imageview_mag_ifcb,  	-4, "/View/Zoom/6%" },
	{ N_( "/View/Zoom/50%" ),			NULL, 
		imageview_mag_ifcb,  	-2, "/View/Zoom/6%" },
	{ N_( "/View/Zoom/100%" ),			NULL, 
		imageview_mag_ifcb,  	1, "/View/Zoom/6%" },
	{ N_( "/View/Zoom/200%" ),			NULL, 
		imageview_mag_ifcb,  	2, "/View/Zoom/6%" },
	{ N_( "/View/Zoom/400%" ),			NULL, 
		imageview_mag_ifcb,  	4, "/View/Zoom/6%" },
	{ N_( "/View/Zoom/800%" ),			NULL, 
		imageview_mag_ifcb,  	8, "/View/Zoom/6%" },
	{ N_( "/View/Zoom/1600%" ),			NULL, 
		imageview_mag_ifcb,  	16, "/View/Zoom/6%" },

	{ N_( "/_Help" ),            			NULL,         
		NULL,			0, "<LastBranch>" },
	{ N_( "/Help/_Image View" ),  			NULL,         
		box_help_ifcb,		GPOINTER_TO_UINT( "sec:view" ) },
	{ N_( "/Help/_Paint Bar" ),  			NULL,         
		box_help_ifcb,		GPOINTER_TO_UINT( "sec:paintbox" ) }
};

/* Spot mouse motion events, to update status bar.
 */
static gint
imageview_event( GtkWidget *widget, GdkEvent *event, Imageview *iv )
{
#ifdef EVENT
	if( event->type == GDK_BUTTON_PRESS )
		printf( "imageview_event: GDK_BUTTON_PRESS\n" );
#endif /*EVENT*/

	if( event->type == GDK_MOTION_NOTIFY ) { 
		Imagemodel *imagemodel = iv->imagemodel;
		Conversion *conv = imagemodel->conv;
		int ix, iy;

		conversion_disp_to_im( conv, 
			event->button.x, event->button.y, &ix, &iy );

		statusview_mouse( iv->sv, ix, iy );
	}

	return( FALSE );
}

static gboolean
imageview_filedrop( Imageview *iv, const char *file )
{
	gboolean result;

	if( (result = iimage_replace( iv->imagemodel->iimage, file )) )
		symbol_recalculate_all();

	return( result );
}

static void
imageview_build( Imageview *iv, GtkWidget *vbox, iImage *iimage )
{
	iWindow *iwnd = IWINDOW( iv );
	GtkWidget *mbar;
	GtkWidget *frame;
	GList *focus_chain;

	int w, h; 

	/* All the model parts for our set of views.
	 */
	iv->imagemodel = imagemodel_new( iimage );
	g_object_ref( G_OBJECT( iv->imagemodel ) );
	iobject_sink( IOBJECT( iv->imagemodel ) );
	iv->imagemodel_changed_sid = g_signal_connect( 
		G_OBJECT( iv->imagemodel ), "changed", 
		G_CALLBACK( imageview_imagemodel_changed_cb ), iv );
	iv->imagemodel_destroy_sid = g_signal_connect( 
		G_OBJECT( iv->imagemodel ), "destroy", 
		G_CALLBACK( imageview_imagemodel_destroy_cb ), iv );

	/* Make menu bar
	 */
	iv->ifac = gtk_item_factory_new( GTK_TYPE_MENU_BAR, 
		"<imageview>", iwnd->accel_group );
	g_object_ref( iv->ifac );
	gtk_object_sink( GTK_OBJECT( iv->ifac ) );
#ifdef ENABLE_NLS
	gtk_item_factory_set_translate_func( iv->ifac,
		(GtkTranslateFunc) gettext, NULL, NULL );
#endif /* ENABLE_NLS */
	gtk_item_factory_create_items( iv->ifac, 
		IM_NUMBER( imageview_menu_items ), imageview_menu_items, iv );
	mbar = gtk_item_factory_get_widget( iv->ifac, "<imageview>" );
	gtk_box_pack_start( GTK_BOX( vbox ), mbar, FALSE, FALSE, 0 );
	gtk_widget_show( mbar );

	/* Status bar.
	 */
	iv->sv = statusview_new( iv->imagemodel );
	gtk_box_pack_start( GTK_BOX( vbox ), 
		GTK_WIDGET( iv->sv ), FALSE, FALSE, 0 );

	/* Conversion bar.
	 */
	iv->cv = conversionview_new( iv->imagemodel );
	gtk_box_pack_start( GTK_BOX( vbox ), 
		GTK_WIDGET( iv->cv ), FALSE, FALSE, 0 );

	/* Paintbox bar.
	 */
	iv->pbv = paintboxview_new( iv->imagemodel );
	gtk_box_pack_start( GTK_BOX( vbox ), 
		GTK_WIDGET( iv->pbv ), FALSE, FALSE, 0 );

	/* Image area. 
	 */
	frame = gtk_frame_new( NULL );
	gtk_frame_set_shadow_type( GTK_FRAME( frame ), GTK_SHADOW_OUT );
	gtk_widget_show( frame );
	gtk_box_pack_start( GTK_BOX( vbox ), 
		GTK_WIDGET( frame ), TRUE, TRUE, 0 );
	iv->ip = imagepresent_new( iv->imagemodel );
	gtk_container_add( GTK_CONTAINER( frame ), GTK_WIDGET( iv->ip ) );
	gtk_widget_show( GTK_WIDGET( iv->ip ) );
	gtk_signal_connect( GTK_OBJECT( iv->ip->id ), "event",
		GTK_SIGNAL_FUNC( imageview_event ), iv );

	/* Position and size to restore?
	 */
	if( iimage->window_width != -1 ) {
		GdkScreen *screen = gtk_widget_get_screen( GTK_WIDGET( iv ) );

		/* We need to clip x/y against the desktop size ... we may be
		 * loading a workspace made on a machine with a big screen on
		 * a machine with a small screen.

		 	FIXME ... we could only clip if the window will be
			completely off the screen? ie. ignore
			iimage->window_width etc.

		 */
		int window_x = IM_CLIP( 0, iimage->window_x, 
			gdk_screen_get_width( screen ) - iimage->window_width );
		int window_y = IM_CLIP( 0, iimage->window_y, 
			gdk_screen_get_height( screen ) -
				iimage->window_height );

		gtk_widget_set_uposition( GTK_WIDGET( iv ), 
			window_x, window_y );

		gtk_window_set_default_size( GTK_WINDOW( iv ),
			iimage->window_width, iimage->window_height );

		iv->imagemodel->show_status = iimage->show_status;
		iv->imagemodel->show_paintbox = iimage->show_paintbox;
		iv->imagemodel->show_convert = iimage->show_convert;
		iv->imagemodel->show_rulers = iimage->show_rulers;
		iv->imagemodel->scale = iimage->scale;
		iv->imagemodel->offset = iimage->offset;

		imagepresent_set_mag_pos( iv->ip, 
			iimage->image_mag,
			iimage->image_left, iimage->image_top );
	}
	else {
		/* Set initial size. This is really hard to do right :-( These
		 * magic numbers will break with different themes.

		 	FIXME ... maybe realize the window but don't map it,
			calculate border size, then set default size and map?
			yuk!

		 */
		w = IM_MIN( IMAGE_WINDOW_WIDTH, 
			iv->imagemodel->conv->image.width + 16 );
		h = IM_MIN( IMAGE_WINDOW_HEIGHT, 
			iv->imagemodel->conv->image.height + 43 );
		gtk_window_set_default_size( GTK_WINDOW( iv ), w, h );
		conversion_set_mag( iv->imagemodel->conv, 1 );
	}

	/* Set as file drop destination 
	 */
	filedrop_register( GTK_WIDGET( iv ), 
		(FiledropFunc) imageview_filedrop, iv );

	/* Override the focus_chain ... we want the imagedisplay first.
	 */
	focus_chain = NULL;
	focus_chain = g_list_append( focus_chain, iv->ip );
	focus_chain = g_list_append( focus_chain, iv->cv );
	focus_chain = g_list_append( focus_chain, iv->pbv );
	gtk_container_set_focus_chain( GTK_CONTAINER( vbox ), focus_chain );
	gtk_widget_grab_focus( GTK_WIDGET( iv->ip->id ) );
}

static void *
imageview_add_region( Classmodel *classmodel, Imageview *iv )
{
	iRegionInstance *instance;

	if( MODEL( classmodel )->display &&
		(instance = classmodel_get_instance( classmodel )) ) {
		Regionview *regionview = regionview_new( classmodel, 
			&instance->area, iv->ip );
		PElement *root = &HEAPMODEL( classmodel )->row->expr->root;

		/* Look at the class we are drawing, set the display type.
		 */
		regionview_set_type( regionview, root );
	}

	return( NULL );
}

static void
imageview_popdown( iWindow *iwnd, void *client,
	iWindowNotifyFn nfn, void *sys )
{
	Imageview *iv = IMAGEVIEW( iwnd );
	Imagemodel *imagemodel = iv->imagemodel;
	iImage *iimage = imagemodel->iimage;
	Conversion *conv = imagemodel->conv;

	/* We have to note position/size in popdown rather than destroy, since
	 * the widgets have to all still be extant.
	 */

	/* Note position/size for later reuse.
	 */
	iimage->window_width = GTK_WIDGET( iv )->allocation.width;
	iimage->window_height = GTK_WIDGET( iv )->allocation.height;
	iimage->image_mag = conv->mag;
	gdk_window_get_root_origin( 
		gtk_widget_get_toplevel( GTK_WIDGET( iv ) )->window, 
		&iimage->window_x, &iimage->window_y );

	/* Save the centre of the window in image cods.
	 */
	conversion_disp_to_im( conv, 
		imagemodel->visible.left + imagemodel->visible.width / 2, 
		imagemodel->visible.top + imagemodel->visible.height / 2,
		&iimage->image_left, &iimage->image_top );

	iimage->show_status = imagemodel->show_status;
	iimage->show_paintbox = imagemodel->show_paintbox;
	iimage->show_rulers = imagemodel->show_rulers;

	/* Signal changed on iimage if we save the convert settings. This will
	 * make the thumbnail update.
	 */
	if( imagemodel->show_convert ) {
		if( !iimage->show_convert ||
			iimage->scale != imagemodel->scale ||
			iimage->offset != imagemodel->offset ) {
			iimage->show_convert = imagemodel->show_convert;
			iimage->scale = imagemodel->scale;
			iimage->offset = imagemodel->offset;
			iobject_changed( IOBJECT( iimage ) );
		}
	}
	else if( iimage->show_convert ) {
		iimage->show_convert = FALSE;
		iimage->scale = 1.0;
		iimage->offset = 0.0;
		iobject_changed( IOBJECT( iimage ) );
	}

	nfn( sys, IWINDOW_TRUE );
}

static void
imageview_link( Imageview *iv, iImage *iimage )
{
	iwindow_set_build( IWINDOW( iv ), 
		(iWindowBuildFn) imageview_build, iimage, NULL, NULL );
	iwindow_set_popdown( IWINDOW( iv ), imageview_popdown, NULL );
	iwindow_build( IWINDOW( iv ) );
	slist_map( iimage->classmodels,
		(SListMapFn) imageview_add_region, iv );

	/* Initial "changed" on the model to get all views to init.
	 */
	iobject_changed( IOBJECT( iv->imagemodel ) );
}

Imageview *
imageview_new( iImage *iimage )
{
	Imageview *iv = gtk_type_new( TYPE_IMAGEVIEW );

	imageview_link( iv, iimage );

	/* This is odd ... we wouldn't normally _show() the widget in _new(),
	 * but restoring the scroll position doesn't work unless the window is
	 * visible. We have to show here.
	 */
	gtk_widget_show( GTK_WIDGET( iv ) );

	if( iimage->window_width != -1 ) 
		imagepresent_set_mag_pos( iv->ip, 
			iimage->image_mag,
			iimage->image_left, iimage->image_top );

	return( iv );
}

/* Make an imageview, and try to make area (image cods) visible. width/height
 * can be -ve
 */
Imageview *
imageview_new_area( iImage *iimage, Rect *area )
{
	Imageview *iv = imageview_new( iimage );
	Imagemodel *imagemodel = iv->imagemodel;
	Conversion *conv = imagemodel->conv;
	int shrink_x, shrink_y, shrink;

	/* Calculate a shrink factor which should make all the region 
	 * visible ... don't zoom.
	 */
	shrink_x = (abs( area->width ) + conv->canvas.width) / 
		conv->canvas.width;
	shrink_y = (abs( area->height ) + conv->canvas.height) / 
		conv->canvas.height;
	shrink = -IM_MAX( 1, IM_MAX( shrink_x, shrink_y ) );
	if( shrink == -1 )
		shrink = 1;

	imagepresent_set_mag_pos( iv->ip, shrink, 
		area->left + area->width / 2, 
		area->top + area->height / 2 );

	return( iv );
}
