/* Call vips functions from the graph reducer.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

#include "ip.h"

/*
#define DEBUG
 */

/* Maxiumum number of args to a VIPS function.
 */
#define MAX_VIPS_ARGS (100)

/* Stuff we hold during a call to a VIPS function.
 */
typedef struct {
	/* Environment.
	 */
	const char *name;
	im_function *fn;		/* Function we call */
	Reduce *rc;			/* RC we run inside */
	PElement *out;			/* Write result here */

	/* Args we build.
	 */
	im_object *vargv;		/* vargv we build for VIPS */
	int nargs;			/* Number of args needed from ip */
	int nres;			/* Number of objects we write back */
	int inpos[MAX_VIPS_ARGS];	/* Positions of inputs */
	int outpos[MAX_VIPS_ARGS];	/* Positions of outputs */

	/* Input images.
	 */
	int ninii;			/* Number of input images */
	Imageinfo *inii[MAX_VIPS_ARGS];	/* Set of input images */
	int inposii[MAX_VIPS_ARGS];	/* Where each one comes in vargv */

	gboolean use_lut;		/* TRUE for using a lut */
} VipsInfo;

/* VIPS argument types we support.
 */
static im_arg_type vips_supported[] = {
	IM_TYPE_DOUBLE,		
	IM_TYPE_INT,	
	IM_TYPE_COMPLEX,
	IM_TYPE_STRING,
	IM_TYPE_IMAGE,
	IM_TYPE_DOUBLEVEC,
	IM_TYPE_DMASK,
	IM_TYPE_IMASK
};

/* Look up a VIPS type. Return the index in the above, or -1.
 */
static int
vips_type_find( im_arg_type type )
{
	int i;

	for( i = 0; i < IM_NUMBER( vips_supported ); i++ )
		if( strcmp( type, vips_supported[i] ) == 0 )
			return( i );

	ierrors( "VIPS type \"%s\" not supported", type );
	return( -1 );
}

static void
vips_destroy( VipsInfo *vi )
{
	int i;

	/* Free any VIPS args we built and haven't used.
	 */
	for( i = 0; i < vi->fn->argc; i++ ) {
		im_type_desc *ty = vi->fn->argv[i].desc;
		im_object *obj = vi->vargv[i];
		int vt = vips_type_find( ty->type );

		switch( vt ) {
		case -1:	/* IM_TYPE_DISPLAY */
		case 0: 	/* IM_TYPE_DOUBLE */
		case 1: 	/* IM_TYPE_INT */
		case 2: 	/* IM_TYPE_COMPLEX */
			/* Do nothing.
			 */
			break;

		case 3: 	/* IM_TYPE_STRING */
			FREE( obj );
			break;

		case 4:		/* IM_TYPE_IMAGE */
			/* Input images are from the heap; do nothing.
			 * Output image we made ... close if there.
			 */
			if( ty->flags & IM_TYPE_OUTPUT ) {
				IMAGE **im = (IMAGE **) &obj;

				FREEF( im_close, *im );
			}
			break;

		case 5: 	/* IM_TYPE_DOUBLEVEC */
			FREE( ((im_doublevec_object *) obj)->vec );
			break;

		case 6: 	/* IM_TYPE_DMASK */
			FREE( ((im_mask_object *) obj)->name );
			FREEF( im_free_dmask, ((im_mask_object *) obj)->mask );
			break;

		case 7: 	/* IM_TYPE_IMASK */
			FREE( ((im_mask_object *) obj)->name );
			FREEF( im_free_imask, ((im_mask_object *) obj)->mask );
			break;

		default:
			assert( FALSE );
		}
	}

	if( vi->vargv ) {
		im_free_vargv( vi->fn, vi->vargv );
		FREE( vi->vargv );
	}
	FREE( vi );
}

static VipsInfo *
vips_new( Reduce *rc, im_function *fn, PElement *out )
{
	VipsInfo *vi;
	int i;

	if( !fn || !(vi = IM_NEW( NULL, VipsInfo )) )
		return( NULL );
	vi->name = fn->name;
	vi->out = out;
	vi->fn = fn;
	vi->rc = rc;
	vi->vargv = NULL;
	vi->nargs = 0;
	vi->nres = 0;
	vi->ninii = 0;
	vi->use_lut = FALSE;		/* Set this properly later */

	/* Look over the args ... count the number of inputs we need, and 
	 * the number of outputs we generate. Note the position of each.
	 */
	for( i = 0; i < vi->fn->argc; i++ ) {
		im_type_desc *ty = vi->fn->argv[i].desc;

		if( ty->flags & IM_TYPE_OUTPUT ) {
			/* Found an output object.
			 */
			vi->outpos[vi->nres] = i;
			vi->nres++; 
		}
		else if( strcmp( ty->type, IM_TYPE_DISPLAY ) != 0 ) {
			/* Found an input object.
			 */
			vi->inpos[vi->nargs] = i;
			vi->nargs++; 
		}
	}

	/* Make the call spine, alloc memory. 
	 */
	if( !(vi->vargv = IM_ARRAY( NULL, MAX_VIPS_ARGS, im_object )) ||
		im_allocate_vargv( vi->fn, vi->vargv ) ) {
		verrors( "out of memory" );
		vips_destroy( vi );
		return( NULL );
	}

	return( vi );
}

/* Is this the sort of VIPS funtion we can call?
 */
gboolean
vips_is_callable( im_function *fn )
{
	int i;
	int nout;

        /* Check all argument types are supported. As well as the arg types
         * spotted by vips_type_find, we also allow IM_TYPE_DISPLAY.
         */
        for( i = 0; i < fn->argc; i++ ) {
                im_arg_desc *arg = &fn->argv[i];
                im_arg_type vt = arg->desc->type;
                int t = vips_type_find( vt );

                if( t < 0 ) {
                        /* Unknown type .. if DISPLAY it's OK.
                         */
                        if( strcmp( vt, IM_TYPE_DISPLAY ) != 0 )
                                return( FALSE );
                }
        }

        /* Must be at least one output argument.
         */
        for( nout = 0, i = 0; i < fn->argc; i++ )
                if( fn->argv[i].desc->flags & IM_TYPE_OUTPUT )
                        nout++;
        if( nout == 0 ) 
                return( FALSE );

	return( TRUE );
}

/* Count the number of args a VIPS function needs.
 */
int
vips_n_args( im_function *fn )
{
	int i;
	int nin;

        for( nin = 0, i = 0; i < fn->argc; i++ ) {
                im_arg_desc *arg = &fn->argv[i];
                im_arg_type vt = arg->desc->type;
                int t = vips_type_find( vt );

                if( t < 0 ) {
                        /* Unknown type .. if DISPLAY it's OK.
                         */
                        if( strcmp( vt, IM_TYPE_DISPLAY ) != 0 )
                                return( -1 );
                }
                else if( !(arg->desc->flags & IM_TYPE_OUTPUT) )
                        /* Found an input arg.
                         */
                        nin += 1;
        }

	return( nin );
}

/* Make a usage error for a VIPS function.
 */
void
vips_usage( BufInfo *buf, im_function *fn )
{
	char args[MAX_STRSIZE];
	char output[MAX_STRSIZE];
	int nout;
	int i;

	strcpy( output, "" );
	strcpy( args, "" );
	nout = 0;
	for( i = 0; i < fn->argc; i++ ) {
		im_arg_desc *arg = &fn->argv[i];
		char buf[MAX_STRSIZE];

		/* Format name, type message.
		 */
		im_snprintf( buf, MAX_STRSIZE, 
			"%s (%s)", arg->name, arg->desc->type );
		if( strcmp( arg->desc->type, IM_TYPE_DISPLAY ) == 0 ) {
			/* Special display argument. No action (we fill this
			 * in later, in secret).
			 */
		}
		else if( arg->desc->flags & IM_TYPE_OUTPUT ) {
			if( nout == 0 ) 
				/* Note on output string.
				 */
				strcpy( output, buf );
			else {
				strcat( output, ", " );
				strcat( output, buf );
			}
			nout++;
		}
		else {
			/* Note on args list.
			 */
			strcat( args, buf );
			strcat( args, " " );
		}
	}

	if( nout == 1 )
		buf_appendf( buf, 
			"%s =\n     %s %s\n", output, fn->name, args );
	else
		buf_appendf( buf, 
			"[%s] =\n     %s %s\n", output, fn->name, args );

        buf_appendf( buf, "%s\n", fn->desc );

        /* Print any flags this function has.
         */
        buf_appendf( buf, "flags: " );
        if( fn->flags & IM_FN_PIO )
                buf_appendf( buf, "(PIO function) " );
        else
                buf_appendf( buf, "(WIO function) " );
        if( fn->flags & IM_FN_TRANSFORM ) 
                buf_appendf( buf, "(coordinate transformer) " );
        else
                buf_appendf( buf, "(no coordinate transformation) " );
        if( fn->flags & IM_FN_PTOP )
                buf_appendf( buf, "(point-to-point operation) " );
        else
                buf_appendf( buf, "(area operation) " );
        buf_appendf( buf, "\n" );
}

/* Make an im_doublevec_object.
 */
static int
vips_make_doublevec( im_doublevec_object *dv, int n, double *vec )
{
	int i;

	dv->n = n;
	if( !(dv->vec = IM_ARRAY( NULL, n, double )) )
		return( -1 );
	for( i = 0; i < n; i++ )
		dv->vec[i] = vec[i];

	return( 0 );
}

/* ip types -> VIPS types. Write to obj. FALSE for no conversion possible.
 */
static gboolean
vips_fromip( Reduce *rc, PElement *arg, 
	im_type_desc *vips, im_object *obj )
{
	int vt = vips_type_find( vips->type );

	/* If vips_type_find failed, is it the special DISPLAY type?
	 */
	if( vt < 0 && strcmp( vips->type, IM_TYPE_DISPLAY ) != 0 ) 
		/* Unknown type, and it's not DISPLAY. Flag an error.
		 */
		return( FALSE );

	switch( vt ) {
	case -1:	/* IM_TYPE_DISPLAY */
		/* Just use IM_TYPE_sRGB.
		 */
		*obj = im_col_displays( 7 );
		break;

	case 0: 	/* IM_TYPE_DOUBLE */
	{
		double *a = *obj;

		if( !PEISREAL( arg ) )
			return( FALSE );
		*a = PEGETREAL( arg );
		break;
	}

	case 1: 	/* IM_TYPE_INT */
	{
		int *i = *obj;

		if( PEISREAL( arg ) ) {
			double t = PEGETREAL( arg );

			*i = (int) t;
		}
		else if( PEISBOOL( arg ) )
			*i = PEGETBOOL( arg );
		else
			return( FALSE );
		break;
	}

	case 2: 	/* IM_TYPE_COMPLEX */
	{
		double *c = *obj;

		if( !PEISCOMPLEX( arg ) )
			return( FALSE );
		c[0] = PEGETREALPART( arg );
		c[1] = PEGETIMAGPART( arg );
		break;
	}

	case 3: 	/* IM_TYPE_STRING */
	{
		char **c = (char **) obj;
		char buf[ MAX_STRSIZE ];

		(void) reduce_get_string( rc, arg, buf, MAX_STRSIZE );
		*c = im_strdup( NULL, buf );
		break;
	}

	case 4:		/* IM_TYPE_IMAGE */
	{
		/* Put Imageinfo in for now ... a later pass changes this to
		 * IMAGE* once we've checked all the LUTs.
		 */
		Imageinfo **im = (Imageinfo **) obj;

		if( !PEISIMAGE( arg ) )
			return( FALSE );

		*im = PEGETII( arg );
		break;
	}

	case 5: 	/* IM_TYPE_DOUBLEVEC */
	{
		double buf[ MAX_STRSIZE ];
		int n;

		n = reduce_get_realvec( rc, arg, buf, MAX_STRSIZE );
		if( vips_make_doublevec( *obj, n, buf ) )
			return( FALSE );

		break;
	}

	case 6: 	/* IM_TYPE_DMASK */
	case 7: 	/* IM_TYPE_IMASK */
	{
		im_mask_object **mo = (im_mask_object **) obj;

		if( vt == 6 ) {
			DOUBLEMASK *mask;

			if( !(mask = matrix_ip_to_dmask( arg )) )
				return( FALSE );
			(*mo)->mask = mask;
			(*mo)->name = im_strdupn( mask->filename );
		}
		else {
			INTMASK *mask;

			if( !(mask = matrix_ip_to_imask( arg )) )
				return( FALSE );
			(*mo)->mask = mask;
			(*mo)->name = im_strdupn( mask->filename );
		}

		break;
	}

	default:
		assert( FALSE );
	}

#ifdef DEBUG
	pgraph( arg );
#endif /*DEBUG*/

	return( TRUE );
}

/* VIPS types -> ip types. Write to arg.
 */
static void
vips_toip( VipsInfo *vi, int i, PElement *arg )
{
	im_object obj = vi->vargv[i];
	im_type_desc *vips = vi->fn->argv[i].desc;
	int vt = vips_type_find( vips->type );

#ifdef DEBUG
	printf( "vips_toip: arg[%d] (%s) = ", i, vips->type );
#endif /*DEBUG*/

	switch( vt ) {
	case 0: 	/* IM_TYPE_DOUBLE */
		if( !heap_real_new( vi->rc->hi, *((double*)obj), arg ) )
			everror( vi->rc, "%s", error_string );
		break;

	case 1: 	/* IM_TYPE_INT */
		if( !heap_real_new( vi->rc->hi, *((int*)obj), arg ) )
			everror( vi->rc, "%s", error_string );
		break;

	case 2: 	/* IM_TYPE_COMPLEX */
		if( !heap_complex_new( vi->rc->hi, 
			((double*)obj)[0], ((double*)obj)[1], arg ) )
			everror( vi->rc, "%s", error_string );
		break;

	case 3: 	/* IM_TYPE_STRING */
		if( !heap_string_new( vi->rc->hi, (char*)obj, arg ) )
			everror( vi->rc, "%s", error_string );
		break;

	case 4:		/* IM_TYPE_IMAGE */
	{
		IMAGE *im = obj;
		Imageinfo *outii;

		if( vi->use_lut ) {
			/* We are using a LUT. 
			 */
			if( !(outii = imageinfo_new_modlut( vi->rc->hi, 
				vi->inii[0], im )) )
				everror( vi->rc, "%s", error_string );
		}
		else {
			if( !(outii = imageinfo_new( vi->rc->hi, im )) )
				everror( vi->rc, "%s", error_string );
		}
		PEPUTP( arg, ELEMENT_IMAGE, outii );

		/* This output ii depends upon all of the input images.
		 */
		imageinfo_sub_add_all( outii, vi->ninii, vi->inii );

		/* We've got a pointer to the image from the heap ... junk
		 * the pointer in vargv to stop im_close() on vips end.
		 */
		vi->vargv[i] = NULL;

		/* Rewind the image.
		 */
		if( im_pincheck( im ) ) 
			veverror( vi->rc, "vips_toip: unable to rewind image" );

		break;
	}

	case 6:		/* IM_TYPE_DMASK */
	{
		im_mask_object *mo = obj;
		DOUBLEMASK *mask = mo->mask;

		if( !matrix_dmask_to_heap( vi->rc->hi, mask, arg ) )
			everror( vi->rc, "%s", error_string );

		break;
	}

	case 7:		/* IM_TYPE_IMASK */
	{
		im_mask_object *mo = obj;
		INTMASK *mask = mo->mask;

		if( !matrix_imask_to_heap( vi->rc->hi, mask, arg ) )
			everror( vi->rc, "%s", error_string );

		break;
	}

	case 5:		/* IM_TYPE_DOUBLEVEC */
	default:
		assert( FALSE );
	}

#ifdef DEBUG
	pgraph( arg );
#endif /*DEBUG*/
}

/* Look over the args ... make a note of all input Imageinfo. 
 */
static void
vips_gather( VipsInfo *vi ) 
{
	int i, j;

	for( j = 0, i = 0; i < vi->fn->argc; i++ ) {
		im_type_desc *ty = vi->fn->argv[i].desc;

		if( strcmp( ty->type, IM_TYPE_IMAGE ) == 0 && 
			!(ty->flags & IM_TYPE_OUTPUT) ) {
			/* Found an input image. Note which vargv 
			 * it is.
			 */
			vi->inii[vi->ninii] = vi->vargv[i];
			vi->inposii[vi->ninii] = i;
			vi->ninii++;
		}
	}
}

/* Look along a vargv, and get ready for any lutting. We check input images,
 * and transform them all from Imageinfo* to IMAGE* ready for the call to
 * VIPS. 
 */
static void
vips_build_luts( VipsInfo *vi )
{
	int i;

	/* No input images.
	 */
	if( vi->ninii == 0 )
		return;

	/* Can we LUT? Function needs to be LUTable, all input images
	 * have to be the same underlying image, and image must be uncoded
	 * IM_BANDFMT_UCHAR.
	 */
	vi->use_lut = (vi->fn->flags & IM_FN_PTOP) && 
		imageinfo_same_underlying( vi->inii, vi->ninii ) &&
		imageinfo_get_underlying( vi->inii[0] )->Coding == 
			IM_CODING_NONE &&
		imageinfo_get_underlying( vi->inii[0] )->BandFmt == 
			IM_BANDFMT_UCHAR;

	/* Now zap the vargv vector with the correct IMAGE pointers.
	 */
	for( i = 0; i < vi->ninii; i++ ) 
		if( !(vi->vargv[vi->inposii[i]] = 
			imageinfo_get( vi->use_lut, vi->inii[i] )) )
			everror( vi->rc, "%s", error_string );
}

/* Init an output slot in vargv.
 */
static void
vips_build_output( VipsInfo *vi, int i )
{
	im_type_desc *ty = vi->fn->argv[i].desc;
	int vt = vips_type_find( ty->type );
	char tname[FILENAME_MAX];

	/* Provide output objects for the function to write
	 * to.
	 */
	switch( vt ) {
	case 0:	/* IM_TYPE_DOUBLE */
	case 1:	/* IM_TYPE_INT */	
	case 2:	/* IM_TYPE_COMPLEX */
	case 3:	/* IM_TYPE_STRING */
		break;

	case 4:	/* IM_TYPE_IMAGE */
		if( !temp_name( tname, "v" ) || 
			!(vi->vargv[i] = im_open( tname, "p" )) )
			veverror( vi->rc, "unable to make temp" );
		break;

	case 6:	/* IM_TYPE_DMASK */
	case 7:	/* IM_TYPE_IMASk */
	{
		im_mask_object *mo = vi->vargv[i];

		mo->mask = NULL;
		mo->name = im_strdup( NULL, "tmp" );

		break;
	}

	case 5:	/* IM_TYPE_DOUBLEVEC */
		/* 

			FIXME ... add output doublevec

		 */
	default:
		assert( FALSE );
	}
}

/* Fill an argument vector from our stack frame. Number of args already
 * checked. 
 */
static void
vips_fill_spine( VipsInfo *vi, HeapNode **arg )
{
	int i, j;

	for( j = 0, i = 0; i < vi->fn->argc; i++ ) {
		im_type_desc *ty = vi->fn->argv[i].desc;

		if( ty->flags & IM_TYPE_OUTPUT ) 
			vips_build_output( vi, i ); 
		else if( strcmp( ty->type, IM_TYPE_DISPLAY ) == 0 ) {
			/* Special DISPLAY argument - don't fetch another ip
			 * argument for it.
			 */
			(void) vips_fromip( vi->rc, NULL, ty, &vi->vargv[i] );
		}
		else {
			PElement rhs;

#ifdef DEBUG
			printf( "vips_fill_spine: arg[%d] (%s) = ", 
				i, ty->type );
#endif /*DEBUG*/

			/* Convert ip type to VIPS type.
			 */
			PEPOINTRIGHT( arg[vi->nargs - j - 1], &rhs );
			if( !vips_fromip( vi->rc, &rhs, ty, &vi->vargv[i] ) ) {
				BufInfo buf1, buf2;
				char txt1[100], txt2[200];

				buf_init_static( &buf1, txt1, 100 );
				itext_decompile_ev( vi->rc, &buf1, &rhs );
				buf_init_static( &buf2, txt2, 200 );
				vips_usage( &buf2, vi->fn );

				everror( vi->rc, "argument %d to \"%s\" is "
					"the wrong type\n"
					"you passed:\n  %s\n"
					"usage:\n  %s",
					j, vi->name,
					buf_all( &buf1 ), buf_all( &buf2 ) );
			}

			j++;
		}
	}
}

/* Update the history of all output images.
 */
static void
vips_update_history( VipsInfo *vi )
{
	int i;

#ifdef DEBUG
	printf( "vips_update_history: %s\n", vi->name );
#endif /*DEBUG*/

	for( i = 0; i < vi->nres; i++ ) {
		int j = vi->outpos[i];
		im_type_desc *ty = vi->fn->argv[j].desc;
		int vt = vips_type_find( ty->type );

		/* Image output.
		 */
		if( vt == 4 ) {
#ifdef DEBUG
			printf( "vips_update_history: adding to arg %d\n", j );
#endif /*DEBUG*/

			/* Can't do args! Groan.
			 */
			im_updatehist( vi->vargv[j], 1, &vi->fn->name );
		}
	}
}

/* Write the results back to the heap.
 */
static void
vips_write_result( VipsInfo *vi )
{
	/* Write result.
	 */
	if( vi->nres == 1 )
		/* Single result.
		 */
		vips_toip( vi, vi->outpos[0], vi->out );
	else {
		/* Have to build a list of results.
		 */
		PElement list = *vi->out;
		PElement t;
		int i;

		heap_list_init( &list );
		for( i = 0; i < vi->nres; i++ ) {
			if( !heap_list_add( vi->rc->hi, &list, &t ) )
				everror( vi->rc, "%s", error_string );
			vips_toip( vi, vi->outpos[i], &t );

			(void) heap_list_next( &list );
		}
	}
}

/* Fill an argument vector from the C stack.
 */
static void
vips_fillva( VipsInfo *vi, va_list ap )
{
	int i, j;

	for( j = 0, i = 0; i < vi->fn->argc; i++ ) {
		im_type_desc *ty = vi->fn->argv[i].desc;
		int vt = vips_type_find( ty->type );

#ifdef DEBUG
		printf( "vips_fillva: arg[%d] (%s) = ", i, ty->type );
#endif /*DEBUG*/

		if( ty->flags & IM_TYPE_OUTPUT ) 
			vips_build_output( vi, i );
		else if( strcmp( ty->type, IM_TYPE_DISPLAY ) == 0 ) {
			/* DISPLAY argument ... just IM_TYPE_sRGB.
			 */
			vi->vargv[i] = im_col_displays( 7 );
#ifdef DEBUG
			printf( "ip-display-calib\n" );
#endif /*DEBUG*/
		}
		else if( vt == 0 ) {
			/* DOUBLE.
			 */
			double v = va_arg( ap, double );

			*((double*)vi->vargv[i]) = v;

			if( trace_flags & TRACE_BUILTIN ) 
				buf_appendf( trace_current(), "%g ", v );
		}
		else if( vt == 1 ) {
			/* INT
			 */
			int v = va_arg( ap, int );

			*((int*)vi->vargv[i]) = v;

			if( trace_flags & TRACE_BUILTIN ) 
				buf_appendf( trace_current(), "%d ", v );
		}
		else if( vt == 4 ) {
			/* Imageinfo
			 */
			Imageinfo *ii = va_arg( ap, Imageinfo * );

			vi->vargv[i] = ii;

			if( trace_flags & TRACE_BUILTIN ) {
				BufInfo *buf = trace_current();

				if( ii && ii->im )
					buf_appendf( buf,
						"<image \"%s\"> ",
						ii->im->filename ); 
				else
					buf_appends( buf, "<no image> " );
			}
	    	}
		else if( vt == 5 ) {
			/* doublevec ... get a number, and a double*
			 */
			int n = va_arg( ap, int );
			double *buf = va_arg( ap, double * );

			if( vips_make_doublevec( vi->vargv[i], n, buf ) )
				veverror( vi->rc, "unable to build args" );
			if( trace_flags & TRACE_BUILTIN ) {
				BufInfo *buf = trace_current();
				int i;

				buf_appendf( buf, "<doublevec" );
				for( i = 0; i < n; i++ )
					buf_appendf( buf, " %g", buf[i] );
				buf_appends( buf, ">" );
			}
		}
		else
			assert( FALSE );
	}
}

static gboolean
vips_spine_sub( VipsInfo *vi, HeapNode **arg )
{
	Reduce *rc = vi->rc;

	REDUCE_CATCH_START( FALSE );

	vips_fill_spine( vi, arg );

	/* Get set of input images ready.
	 */
	vips_gather( vi );
	vips_build_luts( vi ); 

	/* Call function!
	 */
#ifdef DEBUG
	printf( "vips_spine_sub: calling %s\n", vi->fn->name );
#endif /*DEBUG*/
	if( vi->fn->disp( vi->vargv ) )
		veverror( rc, "VIPS function \"%s\" failed", vi->fn->name );

	vips_update_history( vi );

	/* Write result back to heap. 
	 */
	vips_write_result( vi );

	REDUCE_CATCH_STOP;

	return( TRUE );
}

/* Call a VIPS function, pick up args from the graph. 
 */
void
vips_spine( Reduce *rc, const char *name, HeapNode **arg, PElement *out )
{
	VipsInfo *vi;
	gboolean res;

#ifdef DEBUG
	printf( "vips_spine: starting for %s\n", name );
#endif /*DEBUG*/

	if( !(vi = vips_new( rc, im_find_function( name ), out )) )
		everror( rc, "%s", error_string );

	if( trace_flags & TRACE_BUILTIN ) {
		BufInfo *buf = trace_push();

		buf_appendf( buf, "\"%s\" ", name );
		trace_args( arg, vi->nargs );
	}

	res = vips_spine_sub( vi, arg );
	vips_destroy( vi );

	if( !res )
		everror( rc, "%s", error_string );

	if( trace_flags & TRACE_BUILTIN ) {
		trace_result( TRACE_BUILTIN, out );
		trace_pop();
	}

#ifdef DEBUG
	printf( "vips_spine: success for %s\n", name );
#endif /*DEBUG*/
}

/* As an action function.
 */
void
vips_run( Reduce *rc, 
	int op, const char *name, HeapNode **arg, PElement *out,
	im_function *function )
{
	VipsInfo *vi;
	gboolean res;

#ifdef DEBUG
	printf( "vips_run: starting for %s\n", name );
#endif /*DEBUG*/

	if( !(vi = vips_new( rc, function, out )) )
		everror( rc, "%s", error_string );

	res = vips_spine_sub( vi, arg );
	vips_destroy( vi );

	if( !res )
		everror( rc, "%s", error_string );

#ifdef DEBUG
	printf( "vips_run: success for %s\n", name );
#endif /*DEBUG*/
}

static gboolean
vipsva_sub( VipsInfo *vi, va_list ap )
{
	Reduce *rc = vi->rc;

	REDUCE_CATCH_START( FALSE );

	if( trace_flags & TRACE_BUILTIN ) 
		buf_appendf( trace_current(), "\"%s\" ", vi->name );

	/* Fill argv ... input images go in as Imageinfo here, output as
	 * IMAGE.
	 */
	vips_fillva( vi, ap );

	if( trace_flags & TRACE_BUILTIN ) 
		buf_appends( trace_current(), " ->\n" ); 

	/* Look over the images we have ... turn input Imageinfo to IMAGE.
	 * If we can do this with a lut, set all that up.
	 */
	vips_gather( vi );
	vips_build_luts( vi ); 

	/* Call function!
	 */
#ifdef DEBUG
	printf( "vipsva_sub: calling %s\n", vi->fn->name );
#endif /*DEBUG*/
	if( vi->fn->disp( vi->vargv ) ) 
		veverror( rc, "VIPS function \"%s\" failed", vi->fn->name );

	vips_update_history( vi );

	/* Write result back to heap. Transform IMAGE outputs to Imageinfo.
	 */
	vips_write_result( vi );

	REDUCE_CATCH_STOP;

	return( TRUE );
}

/* Call a VIPS function, but pick up args from the function call.
 */
void
vipsva( Reduce *rc, PElement *out, const char *name, ... )
{
	va_list ap;
	VipsInfo *vi;
	gboolean res;

	if( trace_flags & TRACE_BUILTIN ) 
		trace_push();

#ifdef DEBUG
	printf( "vipsva: starting for %s\n", name );
#endif /*DEBUG*/

	if( !(vi = vips_new( rc, im_find_function( name ), out )) )
		everror( rc, "%s", error_string );
        va_start( ap, name );
	res = vipsva_sub( vi, ap );
        va_end( ap );
	vips_destroy( vi );

	if( !res )
		everror( rc, "%s", error_string );

	if( trace_flags & TRACE_BUILTIN ) {
		trace_result( TRACE_BUILTIN, out );
		trace_pop();
	}

#ifdef DEBUG
	printf( "vipsva: success for %s\n", name );
#endif /*DEBUG*/
}
