/* Manage toolkits and their display.
 */

/*

    Copyright (C) 1991-2003 The National Gallery

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

 */

/*

    These files are distributed with VIPS - http://www.vips.ecs.soton.ac.uk

 */

/*
#define DEBUG
 */

#include "ip.h"

static FilemodelClass *parent_class = NULL;

/* Remove a tool. Also strip the sym, if any.
 */
static void 
tool_destroy( GtkObject *object )
{	
	Tool *tool;

	g_return_if_fail( object != NULL );
	g_return_if_fail( IS_TOOL( object ) );

	tool = TOOL( object );

#ifdef DEBUG
	printf( "tool_destroy: destroying tool for " );
	if( tool->sym )
		symbol_name_print( tool->sym );
	else
		printf( "anonymous-tool" );
	printf( " at IM_REGION_ADDR 0x%x\n", (unsigned int) tool );
#endif /*DEBUG*/

	/* Unlink from symbol and toolkit. This changes the kit - mark it as
	 * dirty.
	 */
	if( tool->sym ) {
		Symbol *sym = tool->sym;

		sym->tool = NULL;
		tool->sym = NULL;
		symbol_strip( sym );
	}

	filemodel_set_modified( FILEMODEL( tool->kit ), TRUE );

	GTK_OBJECT_CLASS( parent_class )->destroy( object );
}

/* Save a tool's definition to a file. 
 */
static gboolean
tool_save_text( Model *model, iOpenFile *of )
{
	Tool *tool = TOOL( model );
	Symbol *sym = tool->sym;

	switch( tool->type ) {
	case TOOL_SYM:
		if( sym->expr )
			if( !file_write( of, 
				"%s;\n\n", sym->expr->compile->text ) )
				return( FALSE );
		break;

	case TOOL_SEP:
		if( !file_write( of, "#separator\n\n" ) )
			return( FALSE );
		break;

	case TOOL_DIA:
		if( !file_write( of, "#dialog \"%s\" \"%s\"\n\n",
			MODEL( tool )->name, FILEMODEL( tool )->filename ) )
			return( FALSE );
		break;

	default:
		assert( FALSE );
	}

	return( TRUE );
}

static char *
tool_type_to_char( ToolType type )
{
	switch( type ) {
	case TOOL_SYM:	return( "symbol" );
	case TOOL_DIA:	return( "dialog" );
	case TOOL_SEP:	return( "separator" );

	default:
		assert( FALSE );
	}
}

static void
tool_info( Model *model, BufInfo *buf )
{
	Tool *tool = TOOL( model );

	MODEL_CLASS( parent_class )->info( model, buf );

	buf_appendf( buf, "type = \"%s\"\n", tool_type_to_char( tool->type ) );
	if( tool->type == TOOL_SYM )
		buf_appendf( buf, "symbol = \"%s\"\n", 
			MODEL( tool->sym )->name );
}

static void
tool_parent_add( Model *child, Model *parent )
{
        Tool *tool = TOOL( child );
        Toolkit *kit = TOOLKIT( parent );

        tool->kit = kit;

        MODEL_CLASS( parent_class )->parent_add( child, parent );
}

static void
tool_class_init( ToolClass *klass )
{
	GtkObjectClass *object_class = (GtkObjectClass *) klass;
	ModelClass *model_class = (ModelClass *) klass;

	parent_class = gtk_type_class( TYPE_FILEMODEL );

	object_class->destroy = tool_destroy;

	/* Create signals.
	 */

	/* Init methods.
	 */
	model_class->view_new = toolview_new;
	model_class->info = tool_info;
	model_class->parent_add = tool_parent_add;
	model_class->save_text = tool_save_text;
}

static void
tool_init( Tool *tool )
{
        tool->type = TOOL_SEP;
        tool->sym = NULL;
        tool->kit = NULL;
}

GtkType
tool_get_type( void )
{
	static GtkType tool_type = 0;

	if( !tool_type ) {
		static const GtkTypeInfo tool_info = {
			"Tool",
			sizeof( Tool ),
			sizeof( ToolClass ),
			(GtkClassInitFunc) tool_class_init,
			(GtkObjectInitFunc) tool_init,
			/* reserved_1 */ NULL,
			/* reserved_2 */ NULL,
			(GtkClassInitFunc) NULL,
		};

		tool_type = gtk_type_unique( TYPE_FILEMODEL, &tool_info );
	}

	return( tool_type );
}

/* Add a tool to a toolkit.
 */
static void
tool_link( Tool *tool, Toolkit *kit, int pos, const char *name )
{
	filemodel_set_modified( FILEMODEL( kit ), TRUE );
	model_set( MODEL( tool ), name, NULL );
	model_child_add( MODEL( kit ), MODEL( tool ), pos );
}

/* Add a symbol to a toolkit. 
 */
Tool *
tool_new_sym( Toolkit *kit, int pos, Symbol *sym )
{
	Tool *tool;

	assert( kit && sym );

	/* Is there a tool we can reuse? Don't update pos .. assume we want to
	 * keep the old one.
	 */
	if( sym->tool && sym->tool->kit == kit ) 
		return( sym->tool );

	/* Junk any existing tool for this sym.
	 */
	if( (tool = sym->tool) ) {
		sym->tool = NULL;
		tool->sym = NULL;

		FREEO( tool );
	}

	tool = gtk_type_new( TYPE_TOOL );
	tool->type = TOOL_SYM;
	tool->sym = sym;
	sym->tool = tool;

	tool_link( tool, kit, pos, MODEL( sym )->name );

#ifdef DEBUG
	printf( "tool_new_sym: new tool for " );
	symbol_name_print( sym );
	printf( " at 0x%x\n", (unsigned int) tool );
#endif /*DEBUG*/

	return( tool );
}

/* Add a separator to a toolkit. 
 */
Tool *
tool_new_sep( Toolkit *kit, int pos )
{
	Tool *tool;

	assert( kit );

	tool = gtk_type_new( TYPE_TOOL );

	tool->type = TOOL_SEP;
	model_set( MODEL( tool ), "separator", NULL );
	tool_link( tool, kit, pos, NULL );

	return( tool );
}

/* Search a kit for a tool by tool name. Used for searching for dialogs ... we
 * can't use the symtable stuff, as they're not syms.
 */
static Tool *
tool_find( Toolkit *kit, const char *name )
{
	return( (Tool *) model_map( MODEL( kit ), 
		(model_map_fn) model_test_name, (char *) name, NULL ) );
}

/* Add a dialog entry to a toolkit. 
 */
Tool *
tool_new_dia( Toolkit *kit, int pos, 
	const char *name, const char *filename )
{
	Tool *tool;

	assert( kit && name && filename );

	if( (tool = tool_find( kit, name )) ) {
		if( tool->type != TOOL_DIA ) {
			ierrors( "can't create dialog with name \"%s\"\n"
				"an object with that name already exists in "
				"kit \"%s\"", 
				name, MODEL( kit )->name );
			return( NULL );
		}

		/* Just update the filename.
		 */
		filemodel_set_filename( FILEMODEL( tool ), filename );
		model_changed( MODEL( tool ) );
	}
	else {
		tool = gtk_type_new( TYPE_TOOL );

		tool->type = TOOL_DIA;
		filemodel_set_filename( FILEMODEL( tool ), filename );
		model_set( MODEL( tool ), name, NULL );
		tool_link( tool, kit, pos, NULL );
	}

	return( tool );
}

static void *
tool_get_menu_pos_sub( Tool *test, Tool *tool, int *nless )
{
	assert( IS_TOOL( test ) );
	assert( IS_TOOL( tool ) );

	if( test->sym &&
		MODEL( test )->pos < MODEL( tool )->pos &&
		!is_menuable( test->sym ) )
		*nless += 1;

	return( NULL );
}

/* Find the position of a tool in a menu ... MODEL( tool )->pos, minus all the
 * hidden items with pos less than us.

 	FIXME ... slow! yuk! ... but probably doesn't matter

 */
int
tool_get_menu_pos( Tool *tool )
{
	Toolkit *kit = tool->kit;
	int nless;

	nless = 0;
	(void) model_map( MODEL( kit ),
		(model_map_fn) tool_get_menu_pos_sub, tool, &nless );

	assert( MODEL( tool )->pos >= nless );

	return( MODEL( tool )->pos - nless );
}
